/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  determinate.pl --- predicates for determinate literals, which is          */
/*                     literals that have no gain but product information.    */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  call          : add_determinate_literals(+Clause, -Clause_ad)             */
/*                                                                            */
/*  arguments     : Clause    = a partial clause given with a sample,         */
/*                              which has the partial clause format.          */
/*                  Clause_ad = a partial clause made of the given clause and */
/*                              determinate literals,                         */
/*                              which has the partial clause format.          */
/*                                                                            */
/*  subpredicates : add_determinate_literals1                                 */
/*                                                                            */
/******************************************************************************/
/* It is a main predicate to generate and add determinate predicates to       */
/* a partial clause.                                                          */
/******************************************************************************/
add_determinate_literals(Clause, Clause_ad):-
    Clause=(_,Arity,_,_,_,_),
    add_determinate_literals1(Clause, Clause_ad, Arity).
    
add_determinate_literals1(Clause, Clause_ad, Arity):-
    add_a_determinate_literal(Clause, Clause_ad1, Arity),!,
    Clause_ad1=(_,_,_,[Determinate|_],_,_),
    format("A determinite literal ~p is added.~n", [Determinate]),
    add_determinate_literals1(Clause_ad1, Clause_ad, Arity).
add_determinate_literals1(Clause, Clause, _):-!.

/******************************************************************************/
/*                                                                            */
/*  call       : add_a_determinate_literal(+Clause, -Clause_ad, +Arity)       */
/*                                                                            */
/*  arguments  : Clause    = a partial clause given with a sample,            */
/*                           which has the partial clause format.             */
/*               Clause_ad = a partial clause made of the given clause and    */
/*                           a determinate literal generated here,            */
/*                           which has the partial clause format.             */
/*               Arity     = the number of variables occured in an original   */
/*                           clause before added any determinate literals.    */
/*                                                                            */
/******************************************************************************/
/* It generates a determinate literal and add it to a given clause.           */
/******************************************************************************/
add_a_determinate_literal(
    (Sample,     N_Vars,  Old_type, Literals,                   C, S),
    (New_Sample, NN_Vars, New_type, [(d, Pred, Args)|Literals], C, S),
    Arity):-              % The sign "d" means determinate.
    choose_a_determinate_predicate(Pred, Mode, Type, Condition),
    make_determinate_arguments(N_Vars, Mode, Args, Arity),
    check_determinacy(
	Sample,     N_Vars,  Old_type, Literals,
	New_Sample, NN_Vars, New_type, (d, Pred, Args),
	Mode, Type, Condition),
    (clause(determinate_predicate(Pred),_) ->
        true ; assert(determinate_predicate(Pred))).

/******************************************************************************/
/*                                                                            */
/*  call       : choose_a_determinate_predicate(-Pred, -Mode,                 */
/*                                              -Type, -Condition)            */
/*  arguments  : Pred      = a predicate name.                                */
/*               Mode      = a mode list of the predicate.                    */
/*               Type      = a type list of the predicate.                    */
/*               Condition = a condition of the predicate.                    */
/*                                                                            */
/******************************************************************************/
/* It chooses a predicate from a given database.                              */
/******************************************************************************/
choose_a_determinate_predicate(Pred, Mode, Type, Condition):-
    hojo_tuples(Pred, _Arity, Mode, Type, _, Condition),
    member('-', Mode).

/******************************************************************************/
/*                                                                            */
/*  call       : make_determinate_arguments(+N_Vars, +Mode, -Args, +Arity)    */
/*                                                                            */
/*  arguments  : N_Vars = the number of variables before adding this          */
/*                        determinate literal.                                */
/*               Mode   = a mode list of a determinate literal to be added.   */
/*               Args   = a argument list of a determinate literal.           */
/*               Arity  = the number of variables before adding any           */
/*                        determinate literals.                               */
/*                                                                            */
/******************************************************************************/
/* It makes a acandidate of argument list referring the mode.                 */
/******************************************************************************/
make_determinate_arguments(N_Vars, Mode, Args, Arity):-!,
    split_arguments(Mode, Old, New),
    make_a_n_tuple_of_integer_upto_m(Old, Arity, Old_Args),
    make_new_arguments(New, N_Vars, New_Args),
    merge_arguments(Mode, Old_Args, New_Args, Args).

/******************************************************************************/
/*                                                                            */
/*  call       : split_arguments(+Mode, -Old, -New)                           */
/*                                                                            */
/*  arguments  : Mode = a mode                                                */
/*               Old  = the number of + in the mode                           */
/*               New  = the number of - in the mode                           */
/*                                                                            */
/******************************************************************************/
/* It counts the number of + and - sign in the mode.                          */
/******************************************************************************/
split_arguments([],    0,   0  ):-!.
split_arguments([+|M], Old, New):-!,
    split_arguments(M, Old1, New), Old is Old1+1.
split_arguments([-|M], Old, New):-!,
    split_arguments(M, Old, New1), New is New1+1.

/******************************************************************************/
/*                                                                            */
/*  call       : merge_arguments(+Mode, +Old, +New, -Args)                    */
/*                                                                            */
/*  arguments  : Mode = a mode.                                               */
/*               Old  = a list of old arguments.                              */
/*               New  = a list of new arguments.                              */
/*               Args = a list of arguments merged.                           */
/*                                                                            */
/******************************************************************************/
/* It merges two argument lists one of which is for + mode and another one    */
/* is for - mode.                                                             */
/******************************************************************************/
merge_arguments(   [],      [],  [],       []):-!.
merge_arguments([+|M], [A|Old], New, [A|Args]):-
    !,merge_arguments(M, Old, New, Args).
merge_arguments([-|M], Old, [A|New], [A|Args]):-
    !,merge_arguments(M, Old, New, Args).

/******************************************************************************/
/*                                                                            */
/*  call       : make_new_arguments(+Num, +From, -Args)                       */
/*                                                                            */
/*  arguments  : Num  = the number of arguments required.                     */
/*               From = the id number of arguments beginning.                 */
/*               Args = a list of arguments merged.                           */
/*                                                                            */
/******************************************************************************/
/* It makes new arguments to use for a determinate literal.                   */
/******************************************************************************/
make_new_arguments(  0, _,        []):-!. 
make_new_arguments(New, N, [N1|Args]):-!,
    N1 is N + 1,
    New1 is New - 1,
    make_new_arguments(New1, N1, Args).

/******************************************************************************/
/*                                                                            */
/*  call       : check_determinacy(+Sample, +N_Vars,  +Old_type, +Literals,   */
/*                             -New_Sample, -NN_Vars, -New_type, +Literal,    */
/*                                 +Mode, +Type, +Condition)                  */
/*                                                                            */
/*  arguments  : Sample     = a sample before adding this determinate literal.*/
/*               N_Vars     = the number of vrbls before adding this dtrm lit.*/
/*               Old_type   = a type list before adding this dtrm lit.        */
/*               Literals   = literals constituting a clause before adding    */
/*                            this literal.                                   */
/*               New_Sample = a sample after adding this determinate literal. */
/*               NN_Vars    = the number of vrbls after adding this dtrm lit. */
/*               New_type   = a type list after before adding this dtrm lit.  */
/*               Literal    = a new determinate literal to be added.          */
/*               Mode       = a mode list of the predicate of the literal.    */
/*               Type       = a type list of the predicate of the literal.    */
/*               Condition  = a condition of the predicate of the literal.    */
/*                                                                            */
/******************************************************************************/
/* It checks determinacy of a literal and generate information of the literal.*/
/******************************************************************************/
check_determinacy(
	Sample,     N_Vars,  Old_type, Literals,
	New_Sample, NN_Vars, New_type, Literal,  Mode, Type, Condition):-
    Literal=(d, _Pred, Args),
    check_duplicates(Literal, Literals, N_Vars),!,
    check_arguments(Args, N_Vars, NN_Vars),!,
    check_mode(Mode, N_Vars, Args),!,
    check_type(Type, N_Vars, Args, Old_type, New_type),!,
    check_condition(Args, Condition),!,
    expand_sample(Literals, Literal, Sample, New_Sample,_,N_Vars,_,_),!,
    Sample=(P,_),New_Sample=(NP,_),!,
    length(P,N),length(NP,N).

/******************************************************************************/
/*                                                                            */
/*  call         : remove_nonuse_determinate(+Clauses, -No_non_uses)          */
/*                                                                            */
/*  arguments    : Clauses     = a set of clauses.                            */
/*                 No_non_uses = a set of clauses which does not include      */
/*                               redundant clauses.                           */
/*                                                                            */
/*  subpredicate : remove_nonuse_determinate1, remove_nonuse_determinate2     */
/*                                                                            */
/******************************************************************************/
/* It removes determinate literals that are not refered any other literals.   */
/* (this predicate can be used only for this purpose. If necessary Type       */
/*  and Sf should be reconsidered)                                            */
/******************************************************************************/
remove_nonuse_determinate([], []):-!.
remove_nonuse_determinate([Clause|Clauses], [No_non_use|No_non_uses]):-
    remove_nonuse_determinate1(Clause, No_non_use),
    remove_nonuse_determinate(Clauses, No_non_uses).

remove_nonuse_determinate1((S, Ay, _, Literals, Cv, _),
			   (S, Ay, _, No_nonuse_literals, Cv, _)):-!,
    remove_nonuse_determinate2(1, Literals, Literals, No_nonuse_literals).

remove_nonuse_determinate2(_, _Original, [], []):-!.
remove_nonuse_determinate2(N, Original, [L|Literals], No_nonuse_literals):-
    N1 is N+1,
    remove_nonuse_determinate2(N1, Original, Literals, No_nonuse_literals1),
    (L=(d,_,_) -> (
	 q_used_literal(L, Original, N) ->
	   No_nonuse_literals = [L|No_nonuse_literals1]
	 ; No_nonuse_literals = No_nonuse_literals1
     ) ; No_nonuse_literals = [L|No_nonuse_literals1]
    ).

/******************************************************************************/
/*                                                                            */
/*  call         : q_used_literal(+Literal, +Original, +N)                    */
/*                                                                            */
/*  arguments    : Literal  = a literal.                                      */
/*                 Original = a clause which includes the literal.            */
/*                 N        = number which indexes the position of the        */
/*                            literal in the clause.                          */
/*  subpredicate : only_output_ars                                            */
/*                                                                            */
/******************************************************************************/
/* It succeses if the literal is referred by some other literals in the       */
/* clause.                                                                    */
/******************************************************************************/
q_used_literal((_, Pred, Args), Original, N):-
    (target(Pred,_,Mode,_)
      ; hojo(Pred,_,Mode,_,_,_)),
    only_output_args(Args, Mode, Out),
    member(A, Out),
    nth(M, Original, (_,_,OArgs)),
    N =\= M,
    member(A, OArgs).

only_output_args([], [], []):-!.
only_output_args([A|Args], [-|M], [A|Out]):-!,
    only_output_args(Args, M, Out).
only_output_args([_|Args], [+|M], Out):-!,
    only_output_args(Args, M, Out).
