/* Copyright (C) 1997 Itoh Hidenori */
/******************************************************************************/
/*                                                                            */
/*  Predicates to transform clauses from internal syntax to the prolog syntax */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*  call          : transform_to_clauses(+Clauses, -Transformed)              */
/*                                                                            */
/*  arguments     : Clauses     = a clause in internal syntax.                */
/*                  Transformed = a clause in prolog syntax.                  */
/*                                                                            */
/*  subpredicates : construct_a_clause = to transform clause from internal    */
/*                                       syntax to the prolog syntax in a     */
/*                                       string                               */
/*                                                                            */
/******************************************************************************/
/* It transforms clauses from internal syntax to the prolog syntax.           */
/******************************************************************************/
transform_to_clauses(Clauses, Transformed):-
    transform_to_term(Clauses, Transformed).
    
transform_to_term([], []):-!.
transform_to_term([(_,_,_,Body,_,_)|Clauses], [Transformed|Transformeds]):-!,
    reverse(Body, Bodyr),
    construct_a_clause(Bodyr, Result_Clause_string),
    read_from_chars(Result_Clause_string, Transformed),
    transform_to_term(Clauses, Transformeds).

/******************************************************************************/
/*                                                                            */
/*  call          : construct_a_clause(Literals, Clause)                      */
/*                                                                            */
/*  arguments     : Literals = a list of body literals.                       */
/*                  Clause   = a string describing the clause in prolog       */
/*                  syntax.                                                   */
/*                                                                            */
/*  subpredicates : construct_a_body = to transform body literals to the      */
/*                                     prolog syntax in a string              */
/*                  construct_a_head = to make a head literal.                */
/*               construct_a_literal = to make a literal.                     */
/*                     make_a_format = to make a format to use 'format' to    */
/*                                     write in a string.                     */
/*                       vars_format = to make a format  for variables.       */
/*                                                                            */
/******************************************************************************/
/* It  transforms  a clause  from internal syntax  to the prolog syntax       */
/* in a string.                                                               */
/******************************************************************************/
construct_a_clause([], Clause):-
    !,(clause(target_head(Clause),_) ; assert_target_head(Clause)).
construct_a_clause(Literals, Clause):-
    !,(clause(target_head_string(Head),_) ; assert_target_head_string(Head)),
    construct_a_body(Literals, Body),
    format_to_chars("~s:-~s~s", [Head, Body, [46, 10]], Clause).

construct_a_body([], ""):-!.
construct_a_body([Literal], Body):-
    !,construct_a_literal(Literal, ConstructedLiteral),
    Body = ConstructedLiteral.
construct_a_body([Literal|Literals], Body):-
    !,construct_a_literal(Literal, ConstructedLiteral),
    construct_a_body(Literals, ConstructedLiterals),
    format_to_chars("~s, ~s",
		    [ConstructedLiteral, ConstructedLiterals],
		    Body).

construct_a_head((Target_pred, N_args), Head):-
    !,make_list_to_n(N_args, Args),
    construct_a_literal((p, Target_pred, Args), Head).

construct_a_literal((Sign, Pred, Args), Literal):-
    !,make_a_format(Sign, Args, Format),
    format_to_chars(Format, [Pred|Args], Literal).

make_a_format(p, Args, Format):-
    !,vars_format(Args, ArgsFormat),
    append("~p(", ArgsFormat, Temp),
    append(Temp, ")", Format).
make_a_format(d, Args, Format):-
    !,vars_format(Args, ArgsFormat),
    append("~p(", ArgsFormat, Temp),
    append(Temp, ")", Format).
make_a_format(n, Args, Format):-
    !,vars_format(Args, ArgsFormat),
    append("(\+~p(", ArgsFormat, Temp),
    append(Temp, "))", Format).

vars_format([_], "X~p"):-!.
vars_format([_|Args], ArgsFormat):-
    !,vars_format(Args, AF1),
    append("X~p,", AF1, ArgsFormat).
