/* ----------------------------------------------------------
%        Copyright (C) 1996-1997 Kazuhiko Ohno, Masahiko Ikawa,
%        and Computer Architecture Laboratory, Depertment of
%        Information Science, Kyoto University.
%----------------------------------------------------------- */  

/* ---------------------------------------------------------- 
%   (C)1993,1994,1995 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */

/*
  struct.h

  Definition of data structures for KL1/C system

  fix for DEC alpha
*/

#include <klic/param.h>

/****************************************
  UNIVERSAL ONE-WORD DATA

  All the one-word data is represented as type "q".
  This pointer type will always be used after type conversion.
  Using "struct gazonk *" type is for reducing confusion.
****************************************/

typedef struct gazonk *q;

/****************************************
  POINTER TAGS

  [Pointer Tags]
  Lower 2 bits of pointers are used as tags in the following way.

     ...00: Variable Reference
     ...01: Pointer to a Cons Cell
     ...10: Atomic (integer, symbolic atom, etc)
     ...11: Pointer to a Functor Structure

  Variables without suspended goals are represented as
    self-referencing variables.
  Variables _with_ suspended goals are represented as
    a two-word pointer loop,
    the second word being the suspension record structure.

  [Tags for Atomic Data]
     ..0010: Integer (integers have 28 bits including sign)
     ..0110: Symbolic Atom
     ..1010: Reserved
     ..1110: Reserved
****************************************/

#define VALUEMASK	(~3L)
#define ADDRESSMASK	(~3L)

#define PTAGBITS	2L
#define PTAGMASK	0x3
#define	VARREF		0x0
#define CONS		0x1
#define ATOMIC		0x2
#define FUNCTOR		0x3

#define STRUCT		0x1
#define FUNCTNOTCONS	0x2
#define ATOMNOTREF	0x2

#define ptagof(x)	((unsigned long)(x) & PTAGMASK)
#define addressof(x)	((unsigned long)(x) & ADDRESSMASK)

#define isatomic(x)	(ptagof(x) == ATOMIC)
#define iscons(x)	(ptagof(x) == CONS)
#define isfunctor(x)	(ptagof(x) == FUNCTOR)
#define isref(x)	(ptagof(x) == VARREF)

#define isstruct(x)	((unsigned long)(x) & STRUCT)
#define functnotcons(x)	((unsigned long)(x) & FUNCTNOTCONS)
#define atomicnotref(x)	((unsigned long)(x) & ATOMNOTREF)

#define ATAGBITS	4
#define ATAGMASK	0xFL
#define INT		(ATOMIC+0x0)
#define SYM		(ATOMIC+0x4)

#define atagof(x)	((unsigned long)(x) & ATAGMASK)

#define isint(x)	(atagof(x) == INT)
#define issym(x)	(atagof(x) == SYM)

/****************************************
  DATA STRUCTURES

  All the data structures are to be allocated at 4 byte boundaries.
****************************************/

struct functor {
  q functor;			/* principal functor as atomic q object */
  q args[1];			/* arguments */
};

#define functor_of(s)		(functorp(s)->functor)
#define arg(s, k)		(functorp(s)->args[k])

struct cons {
  q cdr, car;
};

#define car_of(x)		(consp(x)->car)
#define cdr_of(x)		(consp(x)->cdr)

#define pcar_of(x)              (makeref(consp(x)+1))
#define pcdr_of(x)              (makeref(consp(x)))



#ifdef PFIRST
/*************************
	SCHEDULING RECORD
	***********************/
#define STAGMASK 0x80000000
#define SUNTAGMASK 0x7fffffff
#define SOBJMASK 0x80000003
#define SOBJTAG 0x80000000

#define tag_scheduling_record(var) (q)((unsigned long)(var) | STAGMASK)
#define untag_scheduling_record(var) (struct scheduling_record*)((unsigned long)(var) & SUNTAGMASK)
#define check_srec(var) ((unsigned long)(var) & STAGMASK)
#define issrec(var) (((unsigned long)(var) & SOBJMASK) == SOBJTAG)
#define smask(var) ((q)(((unsigned long)(var)) & SUNTAGMASK))
#define srecp(var) ((struct scheduling_record*)var)

#endif

/****************************************
  GOAL MANAGEMENT STRUCTURES
****************************************/

/* Predicate structures are constant structures outside of heap */
/* They are pointed to by atomic tags and treated as atomic in GC */

typedef char * (*module)();

struct predicate {
  module (*func)();		/* pointer to function */
  unsigned short int pred;	/* predicate number */
  unsigned short int arity;	/* number of args */
};

/* Goal or Thread records */

#ifdef SCHED
struct stackrec{
	struct stackrec *next;
	q *limit;
	q *bottom;
	q *top;
	long change_num;
/*	long size; */
};

struct goalrec {
  Const struct predicate *pred;	/* predicate descriptor of the goal */
  q args[6];			/* arguments; number 6 is meaningless */
};

struct generic_susp_stack{
	int flag;
	q  var;
	q* stp;
};

struct generic_susp_stack gs_stack;

/* Priority queue is currently implemented as a linear list of */
/* priority queue records, in descending order of precedence */

#ifdef PFIRST
struct scheduling_record {
	q var;
	struct threadrec *pthread;
};

struct request_record {
	struct request_record *next;
	struct susprec *reply_data;
	struct threadrec *wait_thread;
};

struct threadrec {
	struct threadrec *next;
	struct threadrec *before;
	struct stackrec *stack;
	struct request_record *rtop;
	struct request_record *rtail;
	long prio;
};


Extern struct request_record *current_request;
Extern struct request_record *no_request;

#define makenewrequest(allocp,newreq,susp,thread) \
{ \
	newreq = (struct request_record *)allocp; \
	newreq->reply_data = susp; \
	newreq->wait_thread = thread; \
	allocp += 3; \
}

#define NO_REQUEST  (struct susprec *)0
#define NO_WAIT_THREAD  (struct threadrec *)0

struct prioqrec {
  struct prioqrec *next;
  long prio;			/* signed, to use -1 as sentinel */
  struct threadrec *q;
	struct threadrec *last;
};

#else

struct threadrec {
	struct threadrec *next;
	struct stackrec *stack;
};

struct prioqrec {
  struct prioqrec *next;
  long prio;			/* signed, to use -1 as sentinel */
  struct threadrec *q;
};
#endif

struct thread_name_rec{
	struct thread_name_rec *next;
	char *thread_name;
	struct threadrec *addr;
};

#else
struct goalrec {
  struct goalrec *next;		/* pointer to next goal */
  Const struct predicate *pred;	/* predicate descriptor of the goal */
  q args[6];			/* arguments; number 6 is meaningless */
};
/* Priority queue is currently implemented as a linear list of */
/* priority queue records, in descending order of precedence */

struct prioqrec {
  struct prioqrec *next;
  long prio;			/* signed, to use -1 as sentinel */
  struct goalrec *q;
};
#endif


/****************************************
  DATA MANIPULATION MACROS
****************************************/

/* Type conversion */

#define refp(x)		((q *)((char *)(x) - VARREF))
#define consp(x)	((struct cons *)((char *)(x) - CONS))
#define functorp(x)	((struct functor *)((char *)(x) - FUNCTOR))
#define goalp(x)	((struct goalrec *)((char *)(x) - ATOMIC))
#define predp(x)	((struct predicate *)((char *)(x) - ATOMIC))

/* Variables */

#define makeref(x)	(q) ((char *)(x) + VARREF)
#define derefone(x)	(*refp(x))

/* Atomic Values */

#define intval(x)	((long)(x) >> ATAGBITS)
#define symval(x)	((unsigned long)(x) >> ATAGBITS)

#define makeint(n)	((q)(((long)(n) << ATAGBITS) + INT))
#define makecint(n)	((long)makeint(n))
#define makesym(n)	((q)(((unsigned long)(n) << ATAGBITS) + SYM))

#define eqatom(x, y)	((x) == (y))

/* for non-heap addr */
#define makeatomic(adr) ((q)((char *)(adr) + ATOMIC))

/* for principal functors */
#define makepf(x)	makesym(x)

/* Conses */
#define makecons(x)	((q)((char *)(x) + CONS))

/* Functors */

#define makefunctor(x) 	((q)((char *)(x) + FUNCTOR))

Extern struct funcdesc {
  q name;			/* atom */
  unsigned long arity;		/* number of args */
} functab[1];

/* Structures for linkage */

struct ext_atom {
  char *name;			/* name string */
  q value;			/* its value with symbol tag */
};

struct ext_funct {
  char *name;			/* name string */
  long arity;			/* arity */
  q value;			/* functor id with atomic tag */
};

struct ext_pred {
  char *module;			/* module name string */
  char *name;			/* predicate name */
  long arity;			/* arity */
  q value;			/* pred id with atomic tag */
};

/* Special Atoms */

#define NILATOM		makesym(0)
#define PERIODATOM	makesym(1)

/* Global Variables */

#define current_queue	(glbl->current_queue0)
#ifdef SCHED
#define resumed_threads	(glbl->resumed_threads0)
#define resume_flag (glbl->resume_flag0)
#define generic_suspend_threads (glbl->generic_suspend_threads0)
#ifdef PFIRST
#define prior_thread (glbl->prior_thread0)
#define throwing_thread (glbl->throwing_thread0)
#endif
#else
#define resumed_goals	(glbl->resumed_goals0)
#endif
#define prioq		(glbl->prioq0)
#define current_prio	(glbl->current_prio0)
#define top_prio	(glbl->top_prio0)
#define heapp		(glbl->heapp0)
#define heaptop		(glbl->heaptop0)
#define heaplimit	(glbl->heaplimit0)
#define real_heaplimit	(glbl->real_heaplimit0)
#define heapbottom	(glbl->heapbottom0)
#define heapsize	(glbl->heapsize0)
#define maxheapsize	(glbl->maxheapsize0)
#define real_heapbytesize (glbl->real_heapbytesize0)
#define incrementsize	(glbl->incrementsize0)
#define maxactiveratio	(glbl->maxactiveratio0)
#define new_space_top	(glbl->new_space_top0)
#define old_space_top	(glbl->old_space_top0)
#define new_space_size	(glbl->new_space_size0)
#define old_space_size	(glbl->old_space_size0)
#define this_more_space	(glbl->this_more_space0)
#define gcstack		(glbl->gcstack0)
#define gcsp		(glbl->gcsp0)
#define gcmax		(glbl->gcmax0)
#define gcstack_size	(glbl->gcstack_size0)
#ifdef PFIRST
#define srecstack		(glbl->srecstack0)
#define srecsp		(glbl->srecsp0)
#define srecmax		(glbl->srecmax0)
#define srecstack_size	(glbl->srecstack_size0)
#define gthreadstack		(glbl->gthreadstack0)
#define gthreadsp		(glbl->gthreadsp0)
#define gthreadmax		(glbl->gthreadmax0)
#define gthreadstack_size	(glbl->gthreadstack_size0)
#endif
#define interrupt_off   (glbl->interrupt_off0)
#define interrupt_qp    (glbl->interrupt_qp0)
#define program_name	(glbl->program_name0)
#define command_argc	(glbl->command_argc0)
#define command_argv	(glbl->command_argv0)
#define gc_hook_table	(glbl->gc_hook_table0)
#define gc_hooktab_size (glbl->gc_hooktab_size0)
#define num_gc_hooks	(glbl->num_gc_hooks0)
#define after_gc_hook_table (glbl->after_gc_hook_table0)
#define after_gc_hooktab_size (glbl->after_gc_hooktab_size0)
#define num_after_gc_hooks (glbl->num_after_gc_hooks0)
#define suspensions	(glbl->suspensions0)
#define	resumes		(glbl->resumes0)
#define copied_susp	(glbl->copied_susp0)
#define cum_susps	(glbl->cum_susps0)
#define	cum_resumps	(glbl->cum_resumps0)
#define suspended_goal_list (glbl->suspended_goal_list0)
#define postmortem_pred (glbl->postmortem_pred0)
#define postmortem_args (glbl->postmortem_args0)
#define generic_argc	(glbl->generic_argc0)
#define generic_arg	(glbl->generic_arg0)
#define reasons		(glbl->reasons0)
#define my_node		(glbl->par.my_num0)
#define total_node	(glbl->par.num_pes0)

#ifdef SCHED
#define current_thread (glbl->current_thread0)
#define stacksize (glbl->stacksize0)
#define number_of_stack (glbl->number_of_stack0)
#define stq    (glbl->stackq0)
#define suspended_thread_list (glbl->suspended_thread_list0)
#ifdef PDEBUG
#define trecqueue (glbl->trec_queue0)
#endif
#endif

#ifdef SCHED
Extern struct global_variables {
  /* First four items are accessed frequently */
  q *heapp0;			/* top of free area (when decached) */
  q * Volatile heaplimit0;	/* copy of real_heaplimit or 0 */
  struct threadrec *current_queue0; /* queue for current priority */
	struct threadrec *resumed_threads0;
	int resume_flag0;
	struct threadrec *generic_suspend_threads0;
#ifdef PFIRST
	struct threadrec *prior_thread0;
	struct threadrec *throwing_thread0;
#endif
  unsigned long current_prio0;	/* current priority */
  unsigned long top_prio0;	/* highest priority with ready goal */
  /* The rest are not accessed as often */
  struct prioqrec prioq0;	/* priority queue head */
  q *heaptop0;			/* top of the current heap */
  q *heapbottom0;		/* bottom of free area */
  q *real_heaplimit0;		/* limit of normal heap use */
  unsigned long heapsize0, maxheapsize0, incrementsize0;
  unsigned long real_heapbytesize0;
  double maxactiveratio0;
  unsigned long this_more_space0; /* at least this # of words needed after GC */
  q *new_space_top0, *old_space_top0;
  unsigned long new_space_size0, old_space_size0;
  q **gcstack0;
  q **gcsp0;
  q **gcmax0;
  unsigned long gcstack_size0;
#ifdef PFIRST
  struct scheduling_record **srecstack0;
  struct scheduling_record **srecsp0;
  struct scheduling_record **srecmax0;
  unsigned long srecstack_size0;
  struct threadrec ***gthreadstack0;
  struct threadrec ***gthreadsp0;
  struct threadrec ***gthreadmax0;
  unsigned long gthreadstack_size0;
#endif	
  Volatile long interrupt_off0;
  struct goalrec *interrupt_qp0;

  struct {
   /* parallel comm Imp. */
    long my_num0;
    long num_pes0;
    union {
      /* shared-memory Imp. */
      struct {
	long queued0;
	struct ex_goalrec* ex_qp0;
	long currid0;
	long oldid0;
	long shm_htop0;
	long shm_hbyte0;
	long dummy[10];
      } shm;
      /* dist-memory Imp. */

    } aux;
  } par;

  char *program_name0;
  int command_argc0;
  char **command_argv0;
  q* (**gc_hook_table0)();
  int gc_hooktab_size0;
  int num_gc_hooks0;
  q* (**after_gc_hook_table0)();
  int after_gc_hooktab_size0;
  int num_after_gc_hooks0;
  unsigned long
    suspensions0, resumes0, copied_susp0, cum_susps0, cum_resumps0;
  struct suspended_goal_rec *suspended_goal_list0;
  Const struct predicate *postmortem_pred0;
  q postmortem_args0;
  long generic_argc0;		/* number of arguments to generic methods */
  q generic_arg0[MAXGENERICARGS]; /* arguments of generic methods */
  q reasons0[MAXSUSPENSION];	/* suspension reasons */
/* AHOHOAHOHAOHDOAHSO */
	struct suspended_thread_rec *suspended_thread_list0;
	unsigned long stacksize0;
	unsigned long number_of_stack0;
	struct stackrec* stackq0;
	struct threadrec* current_thread0;
#ifdef PDEBUG
	struct thread_name_rec* trec_queue0;
#endif
} globals;
#else
Extern struct global_variables {
  /* First four items are accessed frequently */
  q *heapp0;			/* top of free area (when decached) */
  q * Volatile heaplimit0;	/* copy of real_heaplimit or 0 */
  struct goalrec *current_queue0; /* queue for current priority */
#ifdef SCHED
	struct threadrec *resumed_threads0;
#else
  struct goalrec *resumed_goals0; /* spontaneous goals during reduction */
#endif
  unsigned long current_prio0;	/* current priority */
  unsigned long top_prio0;	/* highest priority with ready goal */
  /* The rest are not accessed as often */
  struct prioqrec prioq0;	/* priority queue head */
  q *heaptop0;			/* top of the current heap */
  q *heapbottom0;		/* bottom of free area */
  q *real_heaplimit0;		/* limit of normal heap use */
  unsigned long heapsize0, maxheapsize0, incrementsize0;
  unsigned long real_heapbytesize0;
  double maxactiveratio0;
  unsigned long this_more_space0; /* at least this # of words needed after GC */
  q *new_space_top0, *old_space_top0;
  unsigned long new_space_size0, old_space_size0;
  q **gcstack0;
  q **gcsp0;
  q **gcmax0;
  unsigned long gcstack_size0;
  Volatile long interrupt_off0;
  struct goalrec *interrupt_qp0;

  struct {
   /* parallel comm Imp. */
    long my_num0;
    long num_pes0;
    union {
      /* shared-memory Imp. */
      struct {
	long queued0;
	struct ex_goalrec* ex_qp0;
	long currid0;
	long oldid0;
	long shm_htop0;
	long shm_hbyte0;
	long dummy[10];
      } shm;
      /* dist-memory Imp. */

    } aux;
  } par;

  char *program_name0;
  int command_argc0;
  char **command_argv0;
  q* (**gc_hook_table0)();
  int gc_hooktab_size0;
  int num_gc_hooks0;
  q* (**after_gc_hook_table0)();
  int after_gc_hooktab_size0;
  int num_after_gc_hooks0;
  unsigned long
    suspensions0, resumes0, copied_susp0, cum_susps0, cum_resumps0;
  struct suspended_goal_rec *suspended_goal_list0;
  Const struct predicate *postmortem_pred0;
  q postmortem_args0;
  long generic_argc0;		/* number of arguments to generic methods */
  q generic_arg0[MAXGENERICARGS]; /* arguments of generic methods */
  q reasons0[MAXSUSPENSION];	/* suspension reasons */
} globals;
#endif
Extern struct global_variables* My_glbl  Init(&globals);

#define declare_globals struct global_variables *glbl = My_glbl

/*
  memory block outside of heap.
*/

#ifdef SCHED
Extern long higher_priority_thread;
#else
Extern long higher_priority_goal;
#endif
/*
  variable for mutable objects
*/

Extern q suspension_reason;
Extern q rest_of_stream;
