% M2PL.pl
% Copyright (C) 1996 Kouichi Furukawa 
%                    (Read README.ps for detailed information.)

%%  m2pl130.pl: translates mg-clauses to pl-clauses for Ground-MGTP/pl

mgtp_version(
'Version: v1.3.0  fujita@sys.crl.melco.co.jp  ''95-12-18').

:- op(100, fx, (#)).

ioFile(Name,InF,OutF) :-
	name_concat([Name,'.m'],InF),
	name_concat([Name,'.mgp'],OutF).

save :- save(m2pl), unix(argv(Argv)), do(Argv).

do(Argv) :-
	init,
	( Argv=[F|Opt] -> set_options(Opt), do1(F) ; true ).
%        halt.

:- dynamic mgtp_option/2.

init :-
	abolish(mgtp_option,2),
	asserta(mgtp_option(delta_duplication,yes)).

set_options([f|Opt]) :- switch(delta_duplication,_,no), set_options(Opt).
set_options([d|Opt]) :- switch(delta_duplication,_,yes), set_options(Opt).
set_options([]).

switch(Type,Old,New) :-
	retract(mgtp_option(Type,Old)),
	asserta(mgtp_option(Type,New)).

%%  MGTP/pl SYNTAX
%%  ==============
%%  
%%  MGTP_Program ::= { MGTP_Clause }+ [ Aux_Clauses ]
%%  MGTP_Clause  ::= Negative_Clause
%%                |  Bipolar_Horn_Clause  | Bipolar_NonHorn_Clause
%%                |  Positive_Horn_Clause | Positive_NonHorn_Clause
%%
%%  Negative_Clause         ::= Antecedent        `-->' `false'           `.'
%%  Bipolar_Horn_Clause     ::= Antecedent        `-->' Consequent        `.'
%%  Bipolar_NonHorn_Clause  ::= Antecedent        `-->' Consequent        `.'
%%  Positive_Horn_Clause    ::= `true'            `-->' Consequent        `.'
%%  Positive_NonHorn_Clause ::= `true'            `-->' Consequent        `.'
%%  
%%  Antecedent         ::= Antecedent_Unit { `;' Antecedent_Unit }*
%%  Antecedent_Unit    ::= Antecedent_Atom { `,' Antecedent_Atom }*
%%  Antecedent_Atom    ::= MGTP_Atom
%%                      |  `{{' [ Aux_Literals ] `}}'
%%
%%  Consequent         ::= Plain_Conjunction
%%                      |  Disjunctive_Unit
%%                      |  `(' Disjunctive_Unit `)'
%%                         { `,' `(' Disjunctive_Unit `)' }+
%%  Disjunctive_Unit   ::= Plain_Conjunction { `;' Plain_Conjunction }+
%%
%%  Plain_Conjunction  ::= MGTP_Atom { `,' MGTP_Atom }*
%%  Plain_Disjunction  ::= MGTP_Atom { `;' MGTP_Atom }*
%%
%%  MGTP_Atom          ::= /* represented by a Prolog term */
%%  
%%  Aux_Literals       ::= Prolog_Call { `,' Prolog_Call }* 
%%  Aux_Clauses        ::= { Prolog_Clause }+

:- op( 800, xfx, (:)).

:- dynamic predicates/1, functions/1, pred_key/2, func_key/2.
:- dynamic t_call/2, h_call/2, nh_call/2,
	   t_call_number/2, h_call_number/2, nh_call_number/2.
:- dynamic auxCode/1, unknownTerms/1.

init1 :-
	abolish(predicates,1),
	abolish(functions,1),
	abolish(pred_key,2),
	abolish(func_key,2),
	abolish(t_call,2),         assert(t_call(0,0)),
	abolish(h_call,2),         assert(h_call(0,0)),
	abolish(nh_call,2),        assert(nh_call(0,0)),
	abolish(t_call_number,1),  assert(t_call_number(0)),
	abolish(h_call_number,1),  assert(h_call_number(0)),
	abolish(nh_call_number,1), assert(nh_call_number(0)),
	abolish(auxCode,1),        assert(auxCode(no)),
	abolish(unknownTerms,1),   assert(unknownTerms(no)).

do1(Name) :-
	init1,
	nofileerrors,
        ioFile(Name,InF,OutF), tell(OutF),
        outputBanner(OutF),
        see(InF), fileerrors,
        readCls(InF,Pcls,Tcls,Gcls),!,
	explosion_check(Gcls,Gcls0,[]),
        multi_entries(Tcls,Tcls1,[]),
        multi_entries(Gcls0,Gcls1,[]),
        multi_entries(Pcls,Pcls1,[]),
        remove_taut(Gcls1,Gcls2),
        review(Pcls1,Tcls1,Gcls2),
	newpage,
        outputHead,
        outputTM,
        newpage,
        firing_atoms(Pcls1,Pcls2,Tcls1,Gcls2),
        firing_atoms(Gcls2,Gcls3,Tcls1,Gcls2),
        output_firing_cls(Gcls2),
	nl,
	outputPcalls(Pcls2),
        newpage,
        outputCls(Tcls1),
        outputCls(Gcls3),
        outputCls(Pcls2),
	nl,
        copyVerbatim,
        outputFoot(OutF),
        told.

outputBanner(OutF) :-
        nl,write('%%  '),write(OutF),
	write('  Ground-MGTP/pl User-clauses'),nl,
	mgtp_version(Version),
	write('%%  '),write(Version),nl.

outputHead :-
        nl,write(':- op( 800, xfx, (:)).'),nl,nl.

outputFoot(OutF) :-
        write('%%  '),write(OutF),write('  EOF'),nl.

readCls(InF,Pcls,Tcls,Gcls) :-
        nl,write('%%  Given clauses:'),nl,write('%%'),nl,
        scan([InF],0,Pcls0,Ncls0,Gcls0),
        symbols(Pcls0,[],P2,[],F2),
        symbols(Ncls0,P2,P3,F2,F3),
        symbols(Gcls0,P3,Preds,F3,Fncts),
	abolish(predicates,1),
	sort_sym(Preds,SPreds,[]), assert(predicates(SPreds)),
	abolish(functions,1),
	sort_sym(Fncts,SFncts,[]), assert(functions(SFncts)),
        numbering(Pcls0,Pcls),
        numbering(Ncls0,Tcls),
        numbering(Gcls0,Gcls).

sort_sym([X|Rest],H,T) :-
	functor(X,_,A),
	split(A,Rest,Left,Right),
	sort_sym(Left,H,[X|M]),
	sort_sym(Right,M,T).
sort_sym([],H,H).

split(A,[Y|Rest],[Y|Left],Right) :- functor(Y,_,B), B<A,!,
	split(A,Rest,Left,Right).
split(A,[Y|Rest],Left,[Y|Right]) :-
	split(A,Rest,Left,Right).
split(_,[],[],[]).

scan(InF,CN,Pcls,Ncls,Gcls) :-
        read(Cls) ->
        scan1(InF,CN,Cls,Pcls,Ncls,Gcls).

scan1(InF,CN,Cls,Pcls,Ncls,Gcls) :-
	( Cls=(Ante-->false) ; Cls=(false:-Ante) ) ->
        CN1 is CN+1, name_concat(['c',CN1],Cid),
	write('%%   '), write(Cid), write(')	'),
        \+(\+(( numbervars(Ante,0,_),
                write((Ante-->false)),period,nl ))),
        splitAnte(CN1,0,_,Ante,false,Ncls,Ncls1),
        scan(InF,CN1,Pcls,Ncls1,Gcls).

scan1(InF,CN,Cls,Pcls,Ncls,Gcls) :-
	( Cls=(true-->Cnsq) ; Cls=(Cnsq:-true) ) ->
        CN1 is CN+1, name_concat(['c',CN1],Cid),
	write('%%   '), write(Cid), write(')	'),
        \+(\+(( numbervars(Cnsq,0,_),
                write((true-->Cnsq)),period,nl ))),
        standardCnsq(Cnsq,SCnsq),
        splitAnte(CN1,0,_,true,SCnsq,Pcls,Pcls1),
        scan(InF,CN1,Pcls1,Ncls,Gcls).

scan1(InF,CN,Cls,Pcls,Ncls,Gcls) :-
	( Cls=(true->G-->Cnsq) ; Cls=(Cnsq:-true) ) ->
        CN1 is CN+1, name_concat(['c',CN1],Cid),
	write('%%   '), write(Cid), write(')	'),
        \+(\+(( numbervars((G,Cnsq),0,_),
                write((true->G-->Cnsq)),period,nl ))),
        standardCnsq(Cnsq,SCnsq),
        splitAnte(CN1,0,_,(true->G),SCnsq,Pcls,Pcls1),
        scan(InF,CN1,Pcls1,Ncls,Gcls).

scan1(InF,CN,Cls,Pcls,Ncls,Gcls) :-
	( Cls=(Ante-->Cnsq) ; Cls=(Cnsq-->Ante) ) ->
        CN1 is CN+1, name_concat(['c',CN1],Cid),
	write('%%   '), write(Cid), write(')	'),
        \+(\+(( numbervars((Ante,Cnsq),0,_),
                write((Ante-->Cnsq)),period,nl ))),
        standardCnsq(Cnsq,SCnsq),
        splitAnte(CN1,0,_,Ante,SCnsq,Gcls,Gcls1),
        scan(InF,CN1,Pcls,Ncls,Gcls1).

scan1(InF,CN,(:-see(File)),Pcls,Ncls,Gcls) :- !,
        see(File),
        scan([File|InF],CN,Pcls,Ncls,Gcls).
scan1([_,File|InF],CN,end_of_file,Pcls,Ncls,Gcls) :- !,
        seen, see(File),
        scan([File|InF],CN,Pcls,Ncls,Gcls).

scan1([_],_,end_of_file,[],[],[]) :- !, seen.
scan1([_],_,(:- prolog),[],[],[]) :- !,
	abolish(auxCode,1), asserta(auxCode(yes)).

%scan1(InF,CN,(:- X),Pcls,Ncls,Gcls) :- !,
%        \+(\+(( numbervars(X,0,_),
%                write('%%  '),write((:-X)),nl ))),
%        scan(InF,CN,Pcls,Ncls,Gcls).

scan1(InF,CN,UserProg,Pcls,Ncls,Gcls) :- !,
        \+(\+(( numbervars(UserProg,0,_),
                write(UserProg),period,nl ))),
        scan(InF,CN,Pcls,Ncls,Gcls).

copyVerbatim :-	auxCode(yes) ->
	write('%%  Auxiliary Prolog clauses:'),nl,
	copyChars,nl ;
	true.

copyChars :- get0(Char) ->
        ( Char= -1 -> true ; put(Char), copyChars ).

symbols([mg_cl(_,(A-->C))|R],Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(C,P1,P2,F1,F2),
        symbols(R,P2,Po,F2,Fo).
symbols([],Pi,Pi,Fi,Fi) :- !.

symbols(true,Pi,Pi,Fi,Fi) :- !.
symbols(false,Pi,Pi,Fi,Fi) :- !.
symbols({_},Pi,Pi,Fi,Fi) :- !.

symbols((A->B),Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(B,P1,Po,F1,Fo).
symbols((A,B),Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(B,P1,Po,F1,Fo).
symbols((A;B),Pi,Po,Fi,Fo) :- !,
        symbols(A,Pi,P1,Fi,F1),
        symbols(B,P1,Po,F1,Fo).
symbols(Atom,Pi,Po,Fi,Fo) :-
        functor(Atom,Pred,Arity),
        functor(Atom1,Pred,Arity),
        enter(Atom1,Pi,Po),
        functors(1,Arity,Atom,Fi,Fo).

functors(Jth,Last,Term,Fi,Fo) :-
        Jth=<Last, arg(Jth,Term,Arg), var(Arg),!,
        J1th is Jth+1,
        functors(J1th,Last,Term,Fi,Fo).
functors(Jth,Last,Term,Fi,Fo) :-
        Jth=<Last, arg(Jth,Term,Arg), !,
        functor(Arg,Fnct,Arity),
        functor(Arg1,Fnct,Arity),
        enter(Arg1,Fi,F1),
        functors(1,Arity,Arg,F1,F2),
        J1th is Jth+1,
        functors(J1th,Last,Term,F2,Fo).
functors(Jth,Last,_,Fi,Fi) :- Jth>Last.

%enter((# _),X,X) :- !.
enter(Term,[Term|Rest],NewRest) :- !, NewRest=[Term|Rest].
enter(Term,[Term1|Rest],[Term1|NewRest]) :- enter(Term,Rest,NewRest).
enter(Term,[],[Term]).

vars_in(V,Vi,Vo) :- var(V),!, add_v(V,Vi,Vo).
vars_in({{P}},Vi,Vo) :- !, vars_in(P,Vi,Vo).
vars_in({{}},Vi,Vi) :- !.
vars_in([A|B],Vi,Vo) :- vars_in(A,Vi,Vm), vars_in(B,Vm,Vo).
vars_in([],Vi,Vi).
vars_in(Atom,Vi,Vi) :- atom(Atom),!.
vars_in(Int,Vi,Vi) :- integer(Int),!.
vars_in(Term,Vi,Vo) :- functor(Term,_,Arity),
        vars_in_args(1,Arity,Term,Vi,Vo).

vars_in_args(Jth,Last,Term,Vi,Vo) :-
        Jth<Last,
        arg(Jth,Term,Arg),
        vars_in(Arg,Vi,Vm),
        J1th is Jth+1,
        vars_in_args(J1th,Last,Term,Vm,Vo).
vars_in_args(Last,Last,Term,Vi,Vo) :-
        arg(Last,Term,Arg),
        vars_in(Arg,Vi,Vo).

add_v(V,[V1|Vi],[V1|Vi]) :- V==V1,!.
add_v(V,[V1|Vi],[V1|Vo]) :- add_v(V,Vi,Vo).
add_v(V,[],[V]).

numbering([mg_cl(CN,(A->G-->C))|Rest],
          [mg_cl(CN,(NewA->G-->NewC))|Nrest]) :-
        numbering1(A,ante,NewA,0,N),
        numbering1(C,cnsq,NewC,N,_),
        numbering(Rest,Nrest).
numbering([],[]).

numbering1((A,B),AC,(NewA,NewB),Ni,No) :- !,
        numbering1(A,AC,NewA,Ni,N1),
        numbering1(B,AC,NewB,N1,No).
numbering1((A;B),AC,(NewA;NewB),Ni,No) :- !,
        numbering1(A,AC,NewA,Ni,N1),
        numbering1(B,AC,NewB,N1,No).
numbering1(false,cnsq,No:false,Ni,No) :- !, No is Ni+1.
numbering1(Lit,cnsq,No:Lit,Ni,No) :- !, No is Ni+1.
numbering1(Lit,ante,No:Lit,Ni,No) :- !, No is Ni+1.

splitAnte(CN,Ni,No,(A;B),C,Hd,Tl) :- !,
        splitAnte1(CN,Ni,No,(A;B),C,Hd,Tl).
splitAnte(CN,Ni,Ni,ANT,C,[mg_cl(Cid,(A->G-->C))|Tl],Tl) :-
        splitAnte2(ANT,A,G),
	name_concat(['c',CN],Cid).

splitAnte1(CN,Ni,No,(A;B),C,Hd,Tl) :- !,
        splitAnte1(CN,Ni,Nm,A,C,Hd,Tl1),
        splitAnte1(CN,Nm,No,B,C,Tl1,Tl).
splitAnte1(CN,Ni,No,ANT,C,[mg_cl(Cid,(A->G-->C))|Tl],Tl) :-
        splitAnte2(ANT,A,G),
        No is Ni+1,
	name_concat(['c',CN,'a',No],Cid).

splitAnte2((P,Q),A,G) :- !,
        splitAnte2(P,PA,PG),
        splitAnte2(Q,QA,QG),
        makeConj(PA,QA,A),
        makeConj(PG,QG,G).
splitAnte2({{}},true,{{}}) :- !.
splitAnte2({{G}},true,{{G}}) :- !,
	abolish(unknownTerms,1), asserta(unknownTerms(yes)).
splitAnte2(P,P,{{}}) :- !.

makeConj({{}},{{}},{{}}) :- !.
makeConj({{P}},{{}},{{P}}) :- !.
makeConj({{}},{{Q}},{{Q}}) :- !.
makeConj({{P}},{{Q}},{{P,Q}}) :- !.
makeConj(true,true,true) :- !.
makeConj(P,true,P) :- !.
makeConj(true,Q,Q) :- !.
makeConj(P,Q,(P,Q)) :- !.

standardCnsq(C,C) :- checkCnsq(C).

checkCnsq(X) :-
	X=(_,_) -> ( plainConjunction(X) ; disjunctiveUnits(X) ) ;
	X=(_;_) -> disjunctiveUnit(X) ;
	true.

plainConjunction((A,B)) :- !, plainConjunction(A), plainConjunction(B).
plainConjunction((_;_)) :- !, fail.
plainConjunction(_).

disjunctiveUnits((A,B)) :- !, disjunctiveUnits(A), disjunctiveUnits(B).
disjunctiveUnits(A) :- disjunctiveUnit(A).

disjunctiveUnit((A;B)) :- !, disjunctiveUnit(A), disjunctiveUnit(B).
disjunctiveUnit(A) :- plainConjunction(A).

review(Pcls,Tcls,Gcls) :-
        write('
%%  After conversion...

%%  Negative clauses:
%%
'),
        review_cls(Tcls),
        write('
%%  Bipolar clauses:
%%
'),
        review_cls(Gcls),
        write('
%%  Positive clauses:
%%
'),
        review_cls(Pcls).

review_cls([G1|Gs]) :- review_cls_1([G1|Gs]).
review_cls([]).

write_ante(fire(Entry,OL,_)) :- write(Entry), write_OL(OL).

write_OL([{{}}]) :- !.
write_OL([{{}}|OL]) :- !, write_OL(OL).
write_OL([{{G}}]) :- !, comma, write({{G}}).
write_OL([{{G}}|OL]) :- !, comma, write({{G}}), write_OL(OL).
write_OL([[ID:A|B]]) :- !, write_OL([ID:A|B]).
write_OL([A]) :- !, comma, write(A).
write_OL([A1,A2|B]) :- !, comma, write(A1), write_OL([A2|B]).
write_OL([]).

single_lit([{_}]) :- !.
single_lit([{_}|OL]) :- !, single_lit(OL).
single_lit([]).

review_cls_1([mg_cl(N,(A->{{}}-->C))|Gs]) :- !, A=fire((Ent:_),OL,A0),
        \+(\+(( numbervars((A0,C),0,_),
        write('%%   '),write(N),
    	( single_lit(OL) -> true ; write('_'),write(Ent) ),
        write(')	'),write_ante(A), write(' -->'),
        write(' '),write(C),period,nl ))),
        review_cls_1(Gs).
review_cls_1([mg_cl(N,(A->G-->C))|Gs]) :- !, A=fire((Ent:_),OL,A0),
        \+(\+(( numbervars((A0,G,C),0,_),
        write('%%   '),write(N),
    	( single_lit(OL) -> true ; write('_'),write(Ent) ),
        write(')	'),write_ante(A), write(' -->'),nl,
        write('%%      '),write(C),period,nl ))),
        review_cls_1(Gs).
review_cls_1([]).

firing_atoms([mg_cl(N,(A->G-->C))|Gs],
             [mg_cl(N,(A->G-->NewC))|NewGs],Tcls,Gcls) :-
        firing_atoms1(C,NewC,Tcls,Gcls),
        firing_atoms(Gs,NewGs,Tcls,Gcls).
firing_atoms([],[],_,_).

firing_atoms1((A,B),(NewA,NewB),Tcls,Gcls) :- !,
        firing_atoms1(A,NewA,Tcls,Gcls),
        firing_atoms1(B,NewB,Tcls,Gcls).
firing_atoms1((A;B),(NewA;NewB),Tcls,Gcls) :- !,
        firing_atoms1(A,NewA,Tcls,Gcls),
        firing_atoms1(B,NewB,Tcls,Gcls).
firing_atoms1(AtomNbr:'X'(Atom),'ATOM'(Atom,AtomNbr,Tcall,Hcall,NHcall,explosive),
	      Tcls,Gcls) :- !,
        copy_term(Atom,TA),
	entry_atoms(Tcls,false,TA,Tentries,[]),
	( t_call(Tcall,Tentries) ;
          new_t_call_number(TN),
	  name_concat(['t_',TN],Tcall),
          assert(t_call(Tcall,Tentries)) ),
        copy_term(Atom,HA),
	entry_atoms(Gcls,horn,HA,Hentries,[]),
	( h_call(Hcall,Hentries) ;
          new_h_call_number(HN),
	  name_concat(['h_',HN],Hcall),
          assert(h_call(Hcall,Hentries)) ),
        copy_term(Atom,NHA),
	entry_atoms(Gcls,nonhorn,NHA,NHentries,[]),
	( nh_call(NHcall,NHentries) ;
          new_nh_call_number(NHN),
	  name_concat(['nh_',NHN],NHcall),
          assert(nh_call(NHcall,NHentries)) ).
firing_atoms1(AtomNbr:'R'(Atom),'ATOM'(Atom,AtomNbr,Tcall,Hcall,NHcall,reducing),
	      Tcls,Gcls) :- !,
        copy_term(Atom,TA),
	entry_atoms(Tcls,false,TA,Tentries,[]),
	( t_call(Tcall,Tentries) ;
          new_t_call_number(TN),
	  name_concat(['t_',TN],Tcall),
          assert(t_call(Tcall,Tentries)) ),
        copy_term(Atom,HA),
	entry_atoms(Gcls,horn,HA,Hentries,[]),
	( h_call(Hcall,Hentries) ;
          new_h_call_number(HN),
	  name_concat(['h_',HN],Hcall),
          assert(h_call(Hcall,Hentries)) ),
        copy_term(Atom,NHA),
	entry_atoms(Gcls,nonhorn,NHA,NHentries,[]),
	( nh_call(NHcall,NHentries) ;
          new_nh_call_number(NHN),
	  name_concat(['nh_',NHN],NHcall),
          assert(nh_call(NHcall,NHentries)) ).
firing_atoms1(AtomNbr:Atom,'ATOM'(Atom,AtomNbr,Tcall,Hcall,NHcall,normal),
	      Tcls,Gcls) :-
        copy_term(Atom,TA),
	entry_atoms(Tcls,false,TA,Tentries,[]),
	( t_call(Tcall,Tentries) ;
          new_t_call_number(TN),
	  name_concat(['t_',TN],Tcall),
          assert(t_call(Tcall,Tentries)) ),
        copy_term(Atom,HA),
	entry_atoms(Gcls,horn,HA,Hentries,[]),
	( h_call(Hcall,Hentries) ;
          new_h_call_number(HN),
	  name_concat(['h_',HN],Hcall),
          assert(h_call(Hcall,Hentries)) ),
        copy_term(Atom,NHA),
	entry_atoms(Gcls,nonhorn,NHA,NHentries,[]),
	( nh_call(NHcall,NHentries) ;
          new_nh_call_number(NHN),
	  name_concat(['nh_',NHN],NHcall),
          assert(nh_call(NHcall,NHentries)) ).

entry_atoms([mg_cl(CN,(A->_-->_))|Rest],false,Atom,Hd,Tl) :- !,
        entry_atoms1(CN,A,Atom,Hd,Tl1),
        entry_atoms(Rest,false,Atom,Tl1,Tl).
entry_atoms([mg_cl(CN,(A->_-->C))|Rest],horn,Atom,Hd,Tl) :- 
	\+nonhorn(C),!,
        entry_atoms1(CN,A,Atom,Hd,Tl1),
        entry_atoms(Rest,horn,Atom,Tl1,Tl).
entry_atoms([mg_cl(CN,(A->_-->C))|Rest],nonhorn,Atom,Hd,Tl) :-
	nonhorn(C),!,
        entry_atoms1(CN,A,Atom,Hd,Tl1),
        entry_atoms(Rest,nonhorn,Atom,Tl1,Tl).
entry_atoms([_|Rest],HNH,Atom,Hd,Tl) :- 
        entry_atoms(Rest,HNH,Atom,Hd,Tl).
entry_atoms([],_,_,Tl,Tl).

entry_atoms1(CN,fire(LN:Lit,OL,_),Atom,Hd,Tl) :-
        \+(\+(( unify(Lit,Atom) ))),!,
        ( single_lit(OL) -> Hd=[CN|Tl] ; Hd=[(CN-LN)|Tl] ).
entry_atoms1(_,_,_,Tl,Tl).

new_t_call_number(N1) :-
        retract(t_call_number(N)) -> N1 is N+1,
        assert(t_call_number(N1)).

new_h_call_number(N1) :-
        retract(h_call_number(N)) -> N1 is N+1,
        assert(h_call_number(N1)).

new_nh_call_number(N1) :-
        retract(nh_call_number(N)) -> N1 is N+1,
        assert(nh_call_number(N1)).

nonhorn((_;_)).
nonhorn(((_;_),_)).

output_firing_cls(_Gcls) :-
        write('
fire_testers(Index,Info,False) :-
    Call=..[Index,Info,False],!, call(Call).
'),
	write('
fire_generators(Index,Info,NewCls) :-
    findall(New,(Call=..[Index,Info,New], call(Call)),NewCls).
'),nl,
%	write('
%fire_generators(Index,(P,A,TM),NewCls) :-
%    findall(New,((Call=..[Index,(P,A,TM),New], call(Call));
%                 call(guard((true,(*),TM),New))),NewCls).
%'),nl,
	output_t_calls,nl,
	output_h_calls,nl,
%	output_guard_calls(Gcls),nl,
	output_nh_calls.

%output_guard_calls([mg_cl(CN,(fire(1:true,_,_)->_-->_))|Cls]) :-
%	Head=guard('X','Y'),
%	Body=..[CN,'X','Y'],
%        write((Head:-Body)),period,nl,!,
%	output_guard_calls(Cls).
%output_guard_calls([_|Cls]) :-
%	output_guard_calls(Cls).
%output_guard_calls([]).

outputPcalls(Cls) :-
	p_entries(Cls,PHentries,PNHentries),
        ( PHentries = [] ->
                write((positive_Horn('_','_'):-!,fail)),period,nl ;
          mk_fire1(PHentries,positive_Horn('X','Y')) ),
        ( PNHentries = [] ->
                write((positive_non_Horn('_','_'):-!,fail)),period,nl ;
          mk_fire1(PNHentries,positive_non_Horn('X','Y')) ).

p_entries([mg_cl(N,(_->_-->C))|Cls],H,NH) :-
	( nonhorn(C) -> H=H1, NH=[N|NH1] ; H=[N|H1], NH=NH1 ),
	p_entries(Cls,H1,NH1).
p_entries([],[],[]).

output_t_calls :-
	t_call(Tcall,Tentries), atom(Tcall),
        ( Tentries = [] ->
                Call=..[Tcall,'_','_'],
                write((Call:-!,fail)),period,nl ;
          Head=..[Tcall,'X','Y'],
          mk_fire1(Tentries,Head) ),fail.
output_t_calls.

output_h_calls :-
	h_call(Hcall,Hentries), atom(Hcall),
        ( Hentries = [] ->
                Call=..[Hcall,'_','_'],
                write((Call:-!,fail)),period,nl ;
          Head=..[Hcall,'X','Y'],
          mk_fire1(Hentries,Head) ),fail.
output_h_calls.

output_nh_calls :-
	nh_call(NHcall,NHentries), atom(NHcall),
        ( NHentries = [] ->
                Call=..[NHcall,'_','_'],
                write((Call:-!,fail)),period,nl ;
          Head=..[NHcall,'X','Y'],
          mk_fire1(NHentries,Head) ),fail.
output_nh_calls.

mk_fire1([CN-LN|Rest],Head) :- !,
	name_concat([CN,'_',LN],Pname),
	Body=..[Pname,'X','Y'],
        write((Head:-Body)),period,nl,
	mk_fire1(Rest,Head).
mk_fire1([CN|Rest],Head) :-
	Body=..[CN,'X','Y'],
        write((Head:-Body)),period,nl,
	mk_fire1(Rest,Head).
mk_fire1([],_).

explosion_check([mg_cl(CN,(A->G-->C))|Cls],Hd,Tl) :-
	explosion_check1(C,A,NewC),
	Hd=[mg_cl(CN,(A->G-->NewC))|Tl1],
	explosion_check(Cls,Tl1,Tl).
explosion_check([],Tl,Tl).

explosion_check1(C,_,C) :- !.
%%
explosion_check1(C,A,NewC) :-
	C=(_;_) -> NewC=C ;
	explosion_check2(C,A,NewC).

explosion_check2((C1,C2),A,(NewC1,NewC2)) :- !,
	explosion_check2(C1,A,NewC1), explosion_check2(C2,A,NewC2).
explosion_check2(N:C,A,NewC) :-
	( explosive(A,C) ->
		\+(\+((numbervars(C,0,_),
		       write(user_output,explosive(C)),ttynl))),
  	  NewC=(N:'X'(C)) ;
	  reducing(A,C) ->
		\+(\+((numbervars(C,0,_),
		       write(user_output,reducing(C)),ttynl))),
  	  NewC=(N:'R'(C)) ;
          NewC=(N:C) ).

explosive((A,B),C) :- !, ( explosive(A,C) ; explosive(B,C) ).
explosive(_:A,C) :-
	copy_term(A,AA),
	unify_once(AA,C),
	C=..[_|Cs], A=..[_|As],
	increasing(Cs,As).

reducing((A,B),C) :- !, ( reducing(A,C) ; reducing(B,C) ).
reducing(_:A,C) :-
	copy_term(A,AA),
	unify_once(AA,C),
	C=..[_|Cs], A=..[_|As],
	decreasing(Cs,As).

increasing([C|Cs],[A|As]) :- subterm(A,C) ; increasing(Cs,As). 

decreasing([C|Cs],[A|As]) :- subterm(C,A) ; decreasing(Cs,As). 

subterm(A,C) :- nonvar(C), C=..[_|Cs], subterm1(A,Cs).

subterm1(A,[B|_]) :- A==B,!.
subterm1(A,[B|C]) :- subterm(A,B) ; subterm1(A,C).

multi_entries(Cls,Hd,Tl) :-
        mgtp_option(delta_duplication,YN),
        multi_entries(Cls,Hd,Tl,YN).

multi_entries([mg_cl(CN,(A->G-->C))|Cls],Hd,Tl,YN) :-
        conj2list(A,AL,[]),
        findall(Entry,entry(CN,[],AL,A,G,{{}},C,Entry,YN),Entries),
%        merge_variants(Entries,Entries1),
%        multi_entries1(Entries1,Hd,Tl1),
        multi_entries1(Entries,Hd,Tl1),
        multi_entries(Cls,Tl1,Tl,YN).
multi_entries([],Tl,Tl,_).

merge_variants([E1|Entries1],NewEntries) :-
        merge_variants1(Entries1,E1,E2,Entries2),
        NewEntries=[E2|NewEntries1],
        merge_variants(Entries2,NewEntries1).
merge_variants([],[]).

merge_variants1([E|Entries1],E1,E2,Entries2) :-
        ( variant_entry(E,E1,NewE1) ->
                merge_variants1(Entries1,NewE1,E2,Entries2) ;
          Entries2=[E|Entries3],
          merge_variants1(Entries1,E1,E2,Entries3) ).
merge_variants1([],E1,E1,[]).

variant_entry(  mg_cl(CN,(fire(LN1:Lit1,OL1,_)->{{}}-->CSQ1)),
                mg_cl(CN,(fire(LN2:Lit2,OL2,A2)->{{}}-->CSQ2)),
                NewE1) :-
        ordered_ante(OL1,Ante1),
        ordered_ante(OL2,Ante2),
        copy_term((Lit1,Ante1,OL1,CSQ1),(Lit3,Ante3,OL3,CSQ3)),
        variant([Lit3|Ante3],[Lit2|Ante2]),
        (Lit3,Ante3)=(Lit2,Ante2),
	name_concat(['e',LN2,'e',LN1],LN3),
        merge_OL(OL3,OL2,NewOL),
        merge_cnsq(CSQ2,CSQ3,NewCSQ),
        NewE1=  mg_cl(CN,(fire(LN3:Lit2,NewOL,A2)->{{}}-->NewCSQ)).

merge_cnsq(CSQ1,CSQ2,CSQ1) :-
        \+(\+((numbervars(CSQ1,0,_),CSQ1=CSQ2))),!.
merge_cnsq(CSQ1,CSQ2,(CSQ1,CSQ2)).

ordered_ante([{{}}],[]) :- !.
ordered_ante([{{}}|OL],A) :- !, ordered_ante(OL,A).
ordered_ante([{{_}}],[]) :- !.
ordered_ante([{{_}}|OL],A) :- !, ordered_ante(OL,A).
ordered_ante([[_:A|B]],[A|Ante]) :- !, ordered_ante(B,Ante).
ordered_ante([_:A],[A]) :- !.
ordered_ante([_:A1,A2|B],[A1|Ante]) :- !, ordered_ante([A2|B],Ante).

merge_OL([{{}}],[{{}}],[{{}}]) :- !.
merge_OL([{{}}|OL1],[{{}}|OL2],[{{}}|OL3]) :- !, merge_OL(OL1,OL2,OL3).
merge_OL([[LN1:A1|B1]],[[LN2:_|B2]],[[LN3:A1|B3]]):- !,
	name_concat(['e',LN2,'e',LN1],LN3),
        merge_OL(B1,B2,B3).
merge_OL([LN1:A],[LN2:_],[LN3:A]) :- !,
	name_concat(['e',LN2,'e',LN1],LN3).
merge_OL([LN1:A1,A2|B1],[LN2:_,A3|B2],[LN3:A1|Ante]) :- !,
	name_concat(['e',LN2,'e',LN1],LN3),
        merge_OL([A2|B1],[A3|B2],Ante).

variant(X,Y) :-
        \+(\+(( numbervars(X,0,_), numbervars(Y,0,_), X=Y ))).

multi_entries1([X|Rest],[X|Hd],Tl) :- multi_entries1(Rest,Hd,Tl).
multi_entries1([],Tl,Tl).

entry(CN,Lits,[LN:Lit|Rest],A0,GD,Proc,CSQ,Entry,no) :-
        factoring(LN,Lit,Rest,Lits,(A0,GD,Proc,CSQ),
                  LN1,Lit1,Rest1,Lits1,(AL1,GD1,Proc1,CSQ1)),
	name_concat(['f',LN1],NewLN),
        rev_append(Lits1,Rest1,OL),
        continuation(Lit1,OL,GD1,Proc1,CSQ1,NewLit+NewOL),
        Entry= mg_cl(CN,(fire(NewLN:NewLit,NewOL,AL1)->GD1-->CSQ1)).
entry(CN,Lits,[LN:Lit|Rest],A0,GD,Proc,CSQ,Entry,_) :-
        rev_append(Lits,Rest,OL),
        continuation(Lit,OL,GD,Proc,CSQ,NewLit+NewOL),
        Entry= mg_cl(CN,(fire(LN:NewLit,NewOL,A0)->GD-->CSQ)).
entry(CN,Lits,[LN:Lit|Rest],A0,GD,Proc,CSQ,Entry,YN) :-
        entry(CN,[LN:Lit|Lits],Rest,A0,GD,Proc,CSQ,Entry,YN).

rev_append([X|L1],L2,L3) :- rev_append(L1,[X|L2],L3).
rev_append([],L2,L2).

factoring(LN,Lit,Rest,Lits,PP,LN1,Lit1,Rest1,Lits1,PP1) :-
        factoring1(LN,Lit,Rest,Lits,PP,LN2,Lit2,Rest2,Lits2,PP2),
        ( factoring(LN2,Lit2,Rest2,Lits2,PP2,LN1,Lit1,Rest1,Lits1,PP1) ;
          LN1=LN2, Lit1=Lit2, Rest1=Rest2, Lits1=Lits2, PP1=PP2 ).

factoring1(LN,Lit,[LN2:Lit2|Rest],Lits,PP,LN1,FLit,FRest,FLits,FPP) :-
        copy_term((Lit,Lit2,Rest,Lits,PP),(FLit,FLit2,FRest,FLits,FPP)),
        unify_once(FLit,FLit2),
        name_concat([LN,'f',LN2],LN1).
factoring1(LN,Lit,[LN2:Lit2|Rest],Lits,PP,LN1,Lit1,Rest1,Lits1,PP1) :-
        factoring1(LN,Lit,Rest,[LN2:Lit2|Lits],PP,LN1,Lit1,Rest1,Lits1,PP1).

unify_once(X,Y) :- unify(X,Y),!.

remove_taut([Cl|Cls1],Cls2) :- 
        \+(\+((numbervars(Cl,0,_),
               Cl=mg_cl(_,(fire(_:Atom,_,_)->_-->_:Atom))))),!,
        remove_taut(Cls1,Cls2).
remove_taut([Cl|Cls1],[Cl|Cls2]) :- remove_taut(Cls1,Cls2).
remove_taut([],[]).

unify(X,Y) :-
        var(X) ->
                (var(Y) ->
                        X = Y;
                %true ->
                        functor(Y,_,N),
                        (N = 0 -> true;
                        N = 1 ->
                                arg(1,Y,Y1),
                                not_occurs_in(X,Y1);
                        %true ->
                                not_occurs_in_args(X,Y,N)),
                        X = Y);
        var(Y) ->
                functor(X,_,N),
                (N = 0 ->
                        true;
                N = 1 ->
                        arg(1,X,X1),
                        not_occurs_in(Y,X1);
                %true ->
                        not_occurs_in_args(Y,X,N)),
                X = Y;
        %true ->
                functor(X,F,N),
                functor(Y,F,N),
                (N = 0 ->
                        true;
                N = 1 ->
                        arg(1,X,X1),
                        arg(1,Y,Y1),
                        unify(X1,Y1);
                %true ->
                        unify_args(X,Y,N)).

unify_args(X,Y,N) :-
        N = 1 ->
                arg(1,X,X1),
                arg(1,Y,Y1),
                unify(X1,Y1);
        %true ->
                arg(N,X,Xn),
                arg(N,Y,Yn),
                unify(Xn,Yn),
                N1 is N - 1,
                unify_args(X,Y,N1).

not_occurs_in(Var,Term) :-
        Var == Term ->
                fail;
        var(Term) ->
                true;
        %true ->
                functor(Term,_,N),
                (N = 0 ->
                        true;
                N = 1 ->
                        arg(1,Term,Arg1),
                        not_occurs_in(Var,Arg1);
                %true ->
                        not_occurs_in_args(Var,Term,N)).

not_occurs_in_args(Var,Term,N) :-
        N = 1 ->
                arg(1,Term,Arg1),
                not_occurs_in(Var,Arg1);
        %true ->
                arg(N,Term,Argn),
                not_occurs_in(Var,Argn),
                N1 is N - 1,
                not_occurs_in_args(Var,Term,N1).

outputTM :-
	predicates(Preds),
        length(Preds,PL),
        mk_new_vec(PL,PList),
        Root=..['''ROOT'''|PList],
        write(tm_new(Root)),period,nl,nl,
	functions(Fncts),
        outputTM_put(Preds,Fncts),
        newpage,
        outputTM_member(Preds,Fncts).

continuation(Lit,OL,GD,Proc,CSQ,Entry) :-
        vars_in(Lit,[],Lvars),
        vars_in((GD,Proc,CSQ),[],Qvars),
        reorder_lit(Lvars,Qvars,OL,OL1,GD),
        Entry= Lit+OL1,!.

reorder_lit(Lvars,Cvars,LL,[FixG|OL],GD) :-
        extract_tmpG(Lvars,GD,TmpG,RestG),
        check_lit(Lvars,LL,OL,TL,[],Ilit),
        reorder_lit1(Lvars,Cvars,Ilit,TL,RestG,TmpG,FixG).

reorder_lit1(_,_,[],[],{{}},LastG,LastG) :- !.
reorder_lit1(_,_,[],[],{{RestG}},{{}},{{RestG}}) :- !.
reorder_lit1(_,_,[],[],{{RestG}},{{TmpG}},{{TmpG,RestG}}) :- !.

reorder_lit1(Lvars,Cvars,[LN:Lit|Ilit],[[LN:Lit|OL]],RestG,TmpG,TmpG) :-
        vars_in(Lit,[],Tvars),
        union_vars(Lvars,Tvars,NewLvars),
        reorder_lit(NewLvars,Cvars,Ilit,OL,RestG).

extract_tmpG(_,{{}},{{}},{{}}) :- !.
extract_tmpG(Lvars,{{G1,G2}},TmpG,RestG) :- !,
        extract_tmpG(Lvars,{{G1}},TmpG1,RestG1),
        extract_tmpG(Lvars,{{G2}},TmpG2,RestG2),
        add_G(TmpG1,TmpG2,TmpG),
        add_G(RestG1,RestG2,RestG).
extract_tmpG(Lvars,{{G}},{{G}},{{}}) :-
        vars_in(G,[],Gvars),
        subset_vars(Gvars,Lvars),!.
extract_tmpG(_,{{G}},{{}},{{G}}).

add_G({{}},TmpG,TmpG) :- !.
add_G(TmpG,{{}},TmpG) :- !.
add_G({{A}},{{B}},{{A,B}}).

check_lit(Lvars,[LN:Lit|LL],[LN:Lit|OL],TL,Li,Lo) :-
        vars_in(Lit,[],Tvars),
        subset_vars(Tvars,Lvars),!,
        check_lit(Lvars,LL,OL,TL,Li,Lo).
check_lit(Lvars,[LN:Lit|LL],OL,TL,Li,Lo) :-
        vars_in(Lit,[],Tvars),
        intersection_vars(Lvars,Tvars,Ivars),
        length(Ivars,IV),
        insert_lit(Lvars,LN,Lit,IV,Li,Li1),
        check_lit(Lvars,LL,OL,TL,Li1,Lo).
check_lit(_,[],TL,TL,Li,Li).

insert_lit(_,LN,Lit,_,[],[LN:Lit]) :- !.
insert_lit(Lvars,LN,Lit,IV,[LN1:Lit1|Li],Lo) :-
        vars_in(Lit1,[],Tvars1),
        intersection_vars(Tvars1,Lvars,Ivars1),
        length(Ivars1,IV1),
        ( IV1 >= IV -> Lo=[LN1:Lit1|Lo1],
                      insert_lit(Lvars,LN,Lit,IV,Li,Lo1) ;
          IV1 < IV -> Lo=[LN:Lit,LN1:Lit1|Li] ). 

member_var(V,[VV|_]) :- V==VV,!.
member_var(V,[_|VV]) :- member_var(V,VV).

subset_vars([V|VV1],VV2) :- member_var(V,VV2),!,
        subset_vars(VV1,VV2).
subset_vars([],_).

union_vars(VV1,[],VV1) :- !.
union_vars([],VV2,VV2) :- !.
union_vars([V|VV1],VV2,UVV) :- member_var(V,VV2),!,
        union_vars(VV1,VV2,UVV).
union_vars([V|VV1],VV2,[V|UVV]) :- union_vars(VV1,VV2,UVV).

intersection_vars(_,[],[]) :- !.
intersection_vars([],_,[]) :- !.
intersection_vars([V|VV1],VV2,[V|UVV]) :- member_var(V,VV2),!,
         intersection_vars(VV1,VV2,UVV).
intersection_vars([_|VV1],VV2,UVV) :- intersection_vars(VV1,VV2,UVV).

vars_count(V,Vi,Vo) :- var(V),!, add_vc(V,Vi,Vo).
vars_count({{P}},Vi,Vo) :- !, vars_count(P,Vi,Vo).
vars_count([A|B],Vi,Vo) :- vars_count(A,Vi,Vm), vars_count(B,Vm,Vo).
vars_count([],Vi,Vi).
vars_count(Atom,Vi,Vi) :- atom(Atom),!.
vars_count(Int,Vi,Vi) :- integer(Int),!.
vars_count(Term,Vi,Vo) :- functor(Term,_,Arity),
        vars_count_args(1,Arity,Term,Vi,Vo).

vars_count_args(Jth,Last,Term,Vi,Vo) :-
        arg(Jth,Term,Arg),
        ( Jth < Last ->
	        J1th is Jth+1,
	        vars_count(Arg,Vi,Vm),
	        vars_count_args(J1th,Last,Term,Vm,Vo) ;
	  Jth= Last ->
	        vars_count(Arg,Vi,Vo) ).

add_vc(V,[V1=C|Vi],[V1=C1|Vi]) :- V==V1,!, C1 is C+1.
add_vc(V,[V1|Vi],[V1|Vo]) :- add_vc(V,Vi,Vo).
add_vc(V,[],[V=1]).

get_vc(V,[V1=C|_],C) :- V==V1,!.
get_vc(V,[_|Vi],C) :- get_vc(V,Vi,C).

mk_new_vec(0,[]) :- !.
mk_new_vec(N,['_'|L]) :- N1 is N-1, mk_new_vec(N1,L).

outputTM_member(Preds,Fncts) :-
        write('
tm_member(X,TM) :- tm_member0(X,TM,Leaf),!, nonvar(Leaf).
tm_non_member(X,TM) :- tm_member0(X,TM,Leaf),!, var(Leaf).

'),
        outputTM_member1(preds,Preds,1),write('
tm_member1(_,Node,_) :- var(Node),!.
'),
	( unknownTerms(yes) -> outputTM_member1(fncts,Fncts,2), write('
tm_member1(X,Node,NextNode) :- arg(1,Node,NodeList),
    tm_member_list(NodeList,X,NextNode).
'),write('
tm_member_list(NodeList,_,_) :- var(NodeList),!.
tm_member_list([(X->NextNode)|_],X,NextNode).
tm_member_list([_|NodeList],X,NextNode) :-
    tm_member_list(NodeList,X,NextNode).
') ; outputTM_member1(fncts,Fncts,1) ).

outputTM_member1(preds,[Term|Rest],Key) :- !,
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member0(Term,'Root','Leaf')),write(' :- !, '),
             write(arg(Key,'Root','Leaf')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member0(Term,'Root','Leaf')),write(' :- !, '),
             write(arg(Key,'Root','Node1')),comma,nl,
             outputTM_member2(Term,1,Arity,'Leaf') ))) ),
        Key1 is Key+1,
        outputTM_member1(preds,Rest,Key1).
outputTM_member1(fncts,[Term|Rest],Key) :- !,
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member1(Term,'Node','NextNode')),write(' :- '),
             write(arg(Key,'Node','NextNode')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(tm_member1(Term,'Node','NextNode')),write(' :- '),
             write(arg(Key,'Node','Node1')),comma,nl,
             outputTM_member2(Term,1,Arity,'NextNode') ))) ),
        Key1 is Key+1,
        outputTM_member1(fncts,Rest,Key1).
outputTM_member1(_,[],_).

outputTM_member2(Term,Jth,Last,LastNode) :-
        arg(Jth,Term,Arg),
        tab(4),write('tm_member1('),write(Arg),write(',Node'),
        ( Jth < Last -> J1th is Jth+1,
                write(Jth),write(',Node'),write(J1th),write('),'),nl,
                outputTM_member2(Term,J1th,Last,LastNode) ;
          Jth = Last ->
                write(Jth),write(','),write(LastNode),write(').'),nl ).

outputTM_put(Preds,Fncts) :-
        length(Fncts,FL),
        mk_new_vec(FL,FList),
	( unknownTerms(yes) ->
	        FreshNode=..['''NODE''','_'|FList],
		StartKey=2 ;
	  FreshNode=..['''NODE'''|FList],
	  StartKey=1 ),
        outputTM_put1(preds,Preds,1),
	write('
tm_put1(X,Node,NextNode) :- var(Node),!,
    Node='),write(FreshNode),write(',
    tm_put1(X,Node,NextNode).
'),
        outputTM_put1(fncts,Fncts,StartKey),
	( unknownTerms(yes) -> write('
tm_put1(X,Node,NextNode) :- arg(1,Node,NodeList),
    tm_put_list(NodeList,X,NextNode).
'),write('
tm_put_list(NodeList,X,NextNode) :- var(NodeList),!,
    NodeList=[(X->NextNode)|_].
tm_put_list([(X->NextNode)|_],X,NextNode) :- !.
tm_put_list([_|NodeList],X,NextNode) :-
    tm_put_list(NodeList,X,NextNode).
') ; true ).

outputTM_put1(preds,[Term|Rest],Key) :-
        write('tm_put('),
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),
             write(',Root,Leaf) :- !, '),
             write(arg(Key,'Root','Leaf')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),
             write(',Root,Leaf) :- !, '),
             write(arg(Key,'Root','Node1')),comma,nl,
             outputTM_put2(preds,Term,1,Arity,'Leaf') ))) ),
	assert(pred_key(Term,Key)),
        Key1 is Key+1,
        outputTM_put1(preds,Rest,Key1).
outputTM_put1(fncts,[Term|Rest],Key) :-
        write('tm_put1('),
        functor(Term,_,Arity),
        ( Arity = 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),write(',Node,NextNode) :- !, '),
             write(arg(Key,'Node','NextNode')),period,nl ))) ;
          Arity > 0 -> \+(\+(( numbervars(Term,0,_),
             write(Term),write(',Node,NextNode) :- !, '),
             write(arg(Key,'Node','Node1')),comma,nl,
             outputTM_put2(fncts,Term,1,Arity,'NextNode') ))) ),
	assert(func_key(Term,Key)),
        Key1 is Key+1,
        outputTM_put1(fncts,Rest,Key1).
outputTM_put1(_,[],_).

outputTM_put2(preds,Term,Jth,Last,LastNode) :-
        arg(Jth,Term,Arg),
        tab(4),write('tm_put1('),write(Arg),
        write(',Node'),write(Jth),
        ( Jth < Last -> J1th is Jth+1,
                write(',Node'),write(J1th),kokka,comma,nl,
                outputTM_put2(preds,Term,J1th,Last,LastNode) ;
          Jth = Last ->
                comma,write(LastNode),write(').'),nl ).

outputTM_put2(fncts,Term,Jth,Last,LastNode) :-
        arg(Jth,Term,Arg),
        tab(4),write('tm_put1('),write(Arg),
        write(',Node'),write(Jth),
        ( Jth < Last -> J1th is Jth+1,
                write(',Node'),write(J1th),kokka,comma,nl,
                outputTM_put2(fncts,Term,J1th,Last,LastNode) ;
          Jth = Last ->
                comma,write(LastNode),write(').'),nl ).

outputCls([C1|Cls]) :- outputCls1(C1),!, outputCls(Cls).
outputCls([]).

outputCls1(mg_cl(CN,(fire(LN:Lit,OL,ANT)->GD-->CSQ))) :-
	( single_lit(OL) -> Pname=CN ; name_concat([CN,'_',LN],Pname) ),
	refer_tm(OL,CSQ,TM),
	name_concat(['Ant',LN],AntN),
        \+(\+(( numbervars((ANT,GD,CSQ),0,_),
	        Call=..[Pname,(Lit,AntN,TM),'Res'],
	        nl,write(Call),if,nl,
	        output_ante(CN,LN,{LN:AntN},1,OL,CSQ) ))).

refer_tm([],_:false,'_') :- !.
refer_tm([{_}],_:false,'_') :- !.
refer_tm(_,_,'TM').

output_ante(CN,_,{Info},_,[],_:false) :- !,
	sort_info(Info,Sinfo),
	tab(4),write('Res'={CN:(Sinfo-->false)}),period,nl.
output_ante(CN,_,{Info},_,[],CNSQ) :- !,
	output_cnsq(CN,Info,CNSQ),period,nl.
output_ante(CN,LN,{Info},NN,[[LN1:Lit|OL]],CSQ) :- !,
	name_concat(['Ant',LN1],Ant1), NN1 is NN+1,
	tab(4),write(tm_member0(Lit,'TM',Ant1)),comma,
%	output_lit_member(Lit,Ant1),
	tab(4),write(nonvar(Ant1)),comma,nl,
    output_ante(CN,LN,{Info,LN1:Ant1},NN1,OL,CSQ).
output_ante(CN,LN,{Info},NN,[LN1:Lit|OL],CSQ) :- !,
	name_concat(['Ant',LN1],Ant1), NN1 is NN+1,
	tab(4),write(tm_member0(Lit,'TM',Ant1)),comma,
%	output_lit_member(Lit,Ant1),
	tab(4),write(nonvar(Ant1)),comma,nl,
    output_ante(CN,LN,{Info,LN1:Ant1},NN1,OL,CSQ).
output_ante(CN,LN,Info,NN,[{G}|OL],CSQ) :- !,
	output_guard({G}),
    output_ante(CN,LN,Info,NN,OL,CSQ).

output_lit_member(Lit,Target) :- atomic(Lit),!,
	pred_key(Lit,Key),
	tab(4),write(arg(Key,'TM',Target)),comma,nl.
output_lit_member(Lit,Target) :-
	pred_key(Lit,Key),
	name_concat([Target,'_1'],Target1),
	tab(4),write(arg(Key,'TM',Target1)),comma,nl,
	Lit=..[_|Args],
	output_lit_member_args(Args,Target,1,Target).

output_lit_member_args([A],UpNode,N,Target) :- A='$VAR'(_),!,
	name_concat([UpNode,'_',N],PrevNode),
	tab(4),write(tm_member1(A,PrevNode,Target)),comma,nl.
output_lit_member_args([A|Args],UpNode,N,Target) :- A='$VAR'(_),!,
	N1 is N+1,
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N1],ThisNode),
	tab(4),write(tm_member1(A,PrevNode,ThisNode)),comma,nl,
	output_lit_member_args(Args,UpNode,N1,Target).
output_lit_member_args([A],UpNode,N,Target) :- atomic(A),!,
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	tab(4),write(nonvar(PrevNode)),comma,nl,
	tab(4),write(arg(Key,PrevNode,Target)),comma,nl.
output_lit_member_args([A|Args],UpNode,N,Target) :- atomic(A),!,
	N1 is N+1,
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N1],ThisNode),
	tab(4),write(nonvar(PrevNode)),comma,nl,
	tab(4),write(arg(Key,PrevNode,ThisNode)),comma,nl,
	output_lit_member_args(Args,UpNode,N1,Target).
output_lit_member_args([A],UpNode,N,Target) :-
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N,'_1'],ThisNode),
	tab(4),write(nonvar(PrevNode)),comma,nl,
	tab(4),write(arg(Key,PrevNode,ThisNode)),comma,nl,
	A=..[_|B],
	output_lit_member_args(B,PrevNode,1,Target).
output_lit_member_args([A|Args],UpNode,N,Target) :-
	N1 is N+1,
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N1],MidNode),
%	name_concat([UpNode,'_',N1,'_1'],ThisNode),  % BUG!
	name_concat([UpNode,'_',N,'_1'],ThisNode),
	tab(4),write(nonvar(PrevNode)),comma,nl,
	tab(4),write(arg(Key,PrevNode,ThisNode)),comma,nl,
	A=..[_|B],
%	output_lit_member_args(B,MidNode,1,MidNode),  % BUG!
	output_lit_member_args(B,PrevNode,1,MidNode),
	output_lit_member_args(Args,UpNode,N1,Target).

output_csq_check(Lit,Target) :- atomic(Lit),!,
	pred_key(Lit,Key),
	write(arg(Key,'TM',Target)),comma,nl,
	tab(4),write(var(Target)).
output_csq_check(Lit,Target) :-
	pred_key(Lit,Key),
	name_concat([Target,'_1'],Target1),
	write(arg(Key,'TM',Target1)),comma,nl,
	Lit=..[_|Args],
	output_csq_check_args(Args,0,Kokka,Target,1,Target),comma,nl,
	tab(4),write(var(Target)),kokka(Kokka).

output_csq_check_args([A],Kakko,Kakko,UpNode,N,Target) :- A='$VAR'(_),!,
	name_concat([UpNode,'_',N],PrevNode),
	tab(4),write(tm_member1(A,PrevNode,Target)).
output_csq_check_args([A|Args],Kakko,Kokka,UpNode,N,Target) :- A='$VAR'(_),!,
	N1 is N+1,
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N1],ThisNode),
	tab(4),write(tm_member1(A,PrevNode,ThisNode)),comma,nl,
	output_csq_check_args(Args,Kakko,Kokka,UpNode,N1,Target).
output_csq_check_args([A],Kakko,Kokka,UpNode,N,Target) :- atomic(A),!,
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	tab(4),kakko,write(var(PrevNode)),then,write(true),semicolon,
	write(arg(Key,PrevNode,Target)),
	Kokka is Kakko+1.
output_csq_check_args([A|Args],Kakko,Kokka,UpNode,N,Target) :- atomic(A),!,
	N1 is N+1,
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N1],ThisNode),
	tab(4),kakko,write(var(PrevNode)),then,write(true),semicolon,
	write(arg(Key,PrevNode,ThisNode)),comma,nl,
	Kakko1 is Kakko+1,
	output_csq_check_args(Args,Kakko1,Kokka,UpNode,N1,Target).
output_csq_check_args([A],Kakko,Kokka,UpNode,N,Target) :-
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N,'_1'],ThisNode),
	tab(4),kakko,write(var(PrevNode)),then,write(true),semicolon,
	write(arg(Key,PrevNode,ThisNode)),comma,nl,
	A=..[_|B],
	Kakko1 is Kakko+1,
	output_csq_check_args(B,Kakko1,Kokka,PrevNode,1,Target).
output_csq_check_args([A|Args],Kakko,Kokka,UpNode,N,Target) :-
	N1 is N+1,
	func_key(A,Key),
	name_concat([UpNode,'_',N],PrevNode),
	name_concat([UpNode,'_',N1],MidNode),
%	name_concat([UpNode,'_',N1,'_1'],ThisNode),  % BUG!
	name_concat([UpNode,'_',N,'_1'],ThisNode),
	tab(4),kakko,write(var(PrevNode)),then,write(true),semicolon,
	write(arg(Key,PrevNode,ThisNode)),comma,nl,
	A=..[_|B],
	Kakko1 is Kakko+1,
%	output_csq_check_args(B,Kakko1,Kakko2,MidNode,1,MidNode),  % BUG!
	output_csq_check_args(B,Kakko1,Kakko2,PrevNode,1,MidNode),
	comma,nl,
	output_csq_check_args(Args,Kakko2,Kokka,UpNode,N1,Target).

output_cnsq(CN,Info,(A,B)) :-
	sort_info(Info,Ainfo),
        split_cnsq((A,B),Ulist,[],Dlist,[]),
	tab(2),kakko,space,
	( Ulist=[] -> output_cnsq_d(Dlist,CN,Ainfo) ;
	  Dlist=[] -> output_cnsq_u(Ulist,CN,Ainfo) ;
          output_cnsq_u(Ulist,CN,Ainfo),
	  space,semicolon,nl,tab(4),
          output_cnsq_d(Dlist,CN,Ainfo) ),
	space,kokka.
output_cnsq(CN,Info,(A;B)) :-
	sort_info(Info,Ainfo),
        disj2list((A;B),DL,[]),
	output_non_member(DL),
	tab(4),write('Res={[ '), output_disj2(DL,CN,Ainfo), write(' ]}').
output_cnsq(CN,Info,'ATOM'(U1,ID1,Tcall,Hcall,NHcall,Type)) :-
	sort_info(Info,Ainfo),
%	tab(4),write(tm_non_member(U1,'TM')),comma,nl,
	name_concat(['Csq',ID1],Target),
	tab(4),output_csq_check(U1,Target),
	comma,nl,tab(4),
	write('Res'='''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall,Type)).

sort_info(I,O) :- flatten_info(I,F,[]), sort_info1(F,O,[]).

flatten_info((A,B),H,T) :- !, flatten_info(A,H,M), flatten_info(B,M,T).
flatten_info(A,[A|T],T).

sort_info1([N:X|Rest],H,T) :-
	split_info(N,Rest,Low,High),
	sort_info1(Low,H,[X|M]),
	sort_info1(High,M,T).
sort_info1([],T,T).

split_info(N,[M:Y|Rest],[M:Y|Low],High) :- M<N,!,
	split_info(N,Rest,Low,High).
split_info(N,[M:Y|Rest],Low,[M:Y|High]) :-
	split_info(N,Rest,Low,High).
split_info(_,[],[],[]).

output_non_member([D]) :- !,
	tab(2),kakko,space,
	output_non_member1(D),space,kokka,comma,nl.
output_non_member([D|DL]) :-
	tab(2),kakko,space,
	output_non_member1(D),space,kokka,comma,nl,
	output_non_member(DL).

output_non_member1(['ATOM'(U1,_ID1,_,_,_,_)]) :- !,
	write(tm_non_member(U1,'TM')).
%	name_concat(['Csq',ID1],Target),
%	output_csq_check(U1,Target).
output_non_member1(['ATOM'(U1,_ID1,_,_,_,_)|UL]) :-
	write(tm_non_member(U1,'TM')),semicolon,nl,tab(4),
%	name_concat(['Csq',ID1],Target),
%	output_csq_check(U1,Target),
	semicolon,nl,tab(4),
	output_non_member(UL).

split_cnsq((A,B),Uh,Ut,Dh,Dt) :- !,
        split_cnsq(A,Uh,Um,Dh,Dm),
        split_cnsq(B,Um,Ut,Dm,Dt).
split_cnsq((A;B),Ut,Ut,[(A;B)|Dt],Dt) :- !.
split_cnsq(A,[A|Ut],Ut,Dt,Dt).

output_cnsq_u(['ATOM'(U1,ID1,Tcall,Hcall,NHcall,Type)],CN,Ainfo) :- !,
	write(tm_non_member(U1,'TM')),
%	name_concat(['Csq',ID1],Target),
%	output_csq_check(U1,Target),
	comma,nl,tab(4),
	write('Res'='''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall,Type)).
output_cnsq_u(['ATOM'(U1,ID1,Tcall,Hcall,NHcall,Type)|Us],CN,Ainfo) :-
	write(tm_non_member(U1,'TM')),
%	name_concat(['Csq',ID1],Target),
%	output_csq_check(U1,Target),
	comma,nl,tab(4),
	write('Res'='''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall,Type)),
	space,semicolon,nl,tab(4),
        output_cnsq_u(Us,CN,Ainfo).
output_cnsq_u([],_,_).

output_cnsq_d([D1],CN,Ainfo) :- !,
        disj2list(D1,DL,[]),
	output_non_member(DL),
	write('Res={[ '), output_disj2(DL,CN,Ainfo), write(' ]}').
output_cnsq_d([D1|Ds],CN,Ainfo) :-
        disj2list(D1,DL,[]),
	output_non_member(DL),
	write('Res={[ '), output_disj2(DL,CN,Ainfo), write(' ]}'),
	space,semicolon,nl,tab(4),
        output_cnsq_d(Ds,CN,Ainfo).
output_cnsq_d([],_,_).

output_disj2([D1],CN,Ainfo) :- !,
        write('[ '), output_disj3(D1,CN,Ainfo), write(' ]').
output_disj2([D1|Ds],CN,Ainfo) :- !,
        write('[ '), output_disj3(D1,CN,Ainfo), write(' ]'),
	comma,nl,tab(11),
        output_disj2(Ds,CN,Ainfo).

output_disj3(['ATOM'(U1,ID1,Tcall,Hcall,NHcall,Type)],CN,Ainfo) :- !,
        write('''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall,Type)).
output_disj3(['ATOM'(U1,ID1,Tcall,Hcall,NHcall,Type)|Us],CN,Ainfo) :-
        write('''ATOM'''(U1,{CN:(Ainfo-->ID1)},Tcall,Hcall,NHcall,Type)),
        comma,nl,tab(13),
        output_disj3(Us,CN,Ainfo).

conj2list((A,B),H,T) :- !, conj2list(A,H,M), conj2list(B,M,T).
conj2list(A,[A|T],T).

disj2list((A;B),H,T) :- !, disj2list(A,H,M), disj2list(B,M,T).
disj2list(A,[AL|T],T) :- conj2list(A,AL,[]).

output_guard({{}}) :- !.
output_guard({G}) :- output_guard1(G),comma,nl.

output_guard1({G1,G2}) :- !,
        output_guard1({G1}),comma,nl,
        output_guard1({G2}).
output_guard1({G}) :- tab(4),write(G).

name_concat(NameList,NewName) :- name_concat1(NameList,S), name(NewName,S).
name_concat1([H|L],S) :- name(H,SH), append(SH,SL,S), name_concat1(L,SL).
name_concat1([],[]).

append([H|X],Y,[H|Z]) :- append(X,Y,Z).
append([],Y,Y).

space :- write(' ').
comma :- write(',').
semicolon :- write(';').
period :- write('.').
kakko :- write('(').
kokka :- write(')').
if :- space,write(':-'),space.
then :- write('->').

kokka(N) :- N=<0 -> true ; write(')'), N1 is N-1, kokka(N1).

newpage :- nl,write('%%'),nl.

%%  m2pl130.pl  EOF

