// Copyright (C) 1996 Buntarou Shizuki(shizuki@is.titech.ac.jp)

#include "tracer-types.h"

RuleType
str_to_RuleType(char *name)
{
  if (strcmp(name, "network") == 0)
    return rule_network;
  if (strcmp(name, "vanishing") == 0)
    return rule_vanishing;
  if (strcmp(name, "continuous") == 0)
    return rule_continuous;

  return rule_unknown;
}

char *
RuleType_to_str(RuleType type)
{
  if (type == rule_network)
    return "rule_network";
  if (type == rule_vanishing)
    return "rule_vanishing";
  if (type == rule_continuous)
    return "rule_continuous";
  
  return "rule_unknown";
}

//
//
//
PortType
char_to_PortType(char io, char *name)
{
  if (strcmp(name, "single") == 0 && io == '+')
    return port_single_in;
  if (strcmp(name, "single") == 0 && io == '-')
    return port_single_out;
  if (strcmp(name, "stream") == 0 && io == '+')
    return port_stream_in;
  if (strcmp(name, "stream") == 0 && io == '-')
    return port_stream_out;
  
  return port_null;
}

char *
PortType_to_char(PortType type)
{
  if (type == port_single_in)
    return "+single";
  if (type == port_single_out)
    return "-single";
  if (type == port_stream_in)
    return "+stream";
  if (type == port_stream_out)
    return "-stream";

  return "NULL";
}

//
//
//
ActType str_to_ActType(char *name)
{
  if (strcmp(name, "CALL") == 0)
    return act_call;
  if (strcmp(name, "REDU") == 0)
    return act_redu;
  if (strcmp(name, "SUSP") == 0)
    return act_susp;
  if (strcmp(name, "PSUS") == 0)
    return act_psus;
  if (strcmp(name, "FAIL") == 0)
    return act_fail;

  return act_null;
}

char *ActType_to_str(ActType act)
{
  switch (act) {
  case act_call:
    return "CALL";
  case act_redu:
    return "REDU";
  case act_susp:
    return "SUSP";
  case act_psus:
    return "PSUS";
  case act_fail:
    return "FAIL";
  default:
    return "Unknown action type";
  }
}

#define UNPARSE_BUFFER_SIZE 1024*256
char rterm_unparse_buffer[UNPARSE_BUFFER_SIZE];
ostrstream buf(rterm_unparse_buffer, UNPARSE_BUFFER_SIZE);

int
rterm::isatom()
{
  rterm *term = deref(this);
  TermType type = term->Type();
  if (type == tt_integer || type == tt_real || type == tt_nil)
    return 1;
  if (type == tt_functor) {
    term_functor *func = (term_functor *)term;
    if (func->GetArity() == 0)
      return 1;
  }
  return 0;
}

int
rterm::isunbound()
{
  rterm *term = deref(this);
  TermType type = term->Type();
  if (type == tt_variable) {
    term_variable *var = (term_variable *)term;
    return var->GetValue() == NULL;
  }
  return 0;
}

//
//
//
rterm *
term_variable::GetValue()
{
  rterm *vp = content;

  if (vp == NULL)
    return NULL;		// unbound
  if (vp->Type() == tt_variable)
    return ((term_variable *) vp)->GetValue();
  return content;
}

int
term_variable::SetValue(rterm *t)
{
  if (content == NULL) {
    content = t;
    return 0;
  }
  if (content->Type() == tt_variable) {
    return ((term_variable *) content)->SetValue(t);
  }
  return 1;			// already bound
}

int
term_variable::isconcrete()
{
  if (content)
    return content->isconcrete();
  else
    return 0;
}

void
term_variable::Unparse()
{
  if (content)
    content->Unparse();
  else
    //buf << name;
    buf << '?';
}

void
term_variable::Print()
{
  if (content)
    content->Print();
  else
    printf(name);
}

term_functor::term_functor(char *n)
{
  char *fixed_n;
  fixed_n = strrchr(n, ':');
  if (fixed_n) {
    char new_n[strlen(n)+1];
    strcpy(new_n, n);
    n = new_n;
    fixed_n = strrchr(n, ':');
    *fixed_n = NULL;
    name = get_symbol(++fixed_n);
    module = get_symbol(n);
  } else {
    name = get_symbol(n);
    module = NULL;
  }
  tt = tt_functor;
  arity = 0;			// default
}

rterm *
term_functor::GetArg(int n)
{
  Pix i;
  int j = 0;

  for (i = args.first(); i != 0; args.next(i)) {
    if (j == n)
      return args(i);
    j++;
  }
  return NULL;
}

int
term_functor::isconcrete()
{
  Pix ti;
  for (ti = args.first(); ti != 0; args.next(ti)) {
    rterm *t = args(ti);
    if (! t->isconcrete)
      return 0;
  }
  return 1;
}

void
term_functor::Unparse()
{
  buf << name;
  if (arity == 0)
    return;

  buf << "(";
  Pix ti = args.first();
  args(ti)->Unparse();
  if (arity > 1) {
    buf << ",";
    for (args.next(ti); ti != 0; args.next(ti))
      args(ti)->Unparse();
  }      
  buf << ")";
}

void
term_functor::Print()
{
  printf(name);
  if (arity == 0)
    return;

  printf("(");
  Pix ti = args.first();
  args(ti)->Print();
  if (arity > 1) {
    printf(",");
    for (args.next(ti); ti != 0; args.next(ti))
      args(ti)->Print();
  }      
  printf(")");
}  

int
term_cons::isconcrete()
{
  return (car != NULL
	  && cdr != NULL
	  && car->isconcrete()
	  && cdr->isconcrete());
}

void
term_cons::Unparse()
{
  if (1) {
    buf << "[";
    rterm *t = this;
    for (;;) {
      ((term_cons *)t)->Car()->Unparse();
      t = ((term_cons *)t)->Cdr();
      t = deref(t);
      if (t->Type() == tt_nil)
	break;
      if (t->Type() == tt_cons) {
	buf << ",";
      } else if (t->Type() == tt_variable) {
	buf << "|";
	t->Unparse();
	break;
      } else {
	buf << ",";
	t->Unparse();
	break;
      }
    }
    buf << "]";
  } else {
    buf << "[";
    car->Unparse();
    buf << ",";
    cdr->Unparse();
    buf << "]";
  }

}

void
term_cons::Print()
{
  if (1) {
    printf("[");
    rterm *t = this;
    for (;;) {
      ((term_cons *)t)->Car()->Print();
      t = ((term_cons *)t)->Cdr();
      t = deref(t);
      if (t->Type() == tt_nil)
	break;
      if (t->Type() == tt_cons) {
	printf(",");
      } else if (t->Type() == tt_variable) {
	printf("|");
	t->Print();
	break;
      } else {
	printf(",");
	t->Print();
	break;
      }
    }
    printf("]");
  } else {
    printf("[");
    car->Print();
    printf(",");
    cdr->Print();
    printf("]");
  }
}

void
term_nil::Unparse()
{
  buf << "nil";
}

void
term_integer::Unparse()
{
  buf << value;
}

#define TERM_INTEGER_SIZE 1024
static term_integer *term_integer_list[TERM_INTEGER_SIZE];

void
init_term_integer()
{
  bzero(term_integer_list, sizeof(term_integer_list));
}

term_integer *
alloc_term_integer(int n)
{
  int i = n & TERM_INTEGER_SIZE;
  term_integer *t = term_integer_list[i];
  while (t != NULL) {
    if (t->GetValue() == n)
      return t;
    t = t->GetNext();
  }
  t = new term_integer(n, term_integer_list[i]);
  term_integer_list[i] = t;
  return t;
}

struct symbol_cell {
  char		*name;
  symbol_cell	*next;
};

#define SYMBOL_TABLE_SIZE 1021
static symbol_cell *symbol_table[SYMBOL_TABLE_SIZE];

inline int symbol_hash(char *name)
{
  int i = name[0]*(name[1]+1) % SYMBOL_TABLE_SIZE;
  //printf("symbol_hash: %s->%d\n", name, i);
  return i;
}

void
init_symbol()
{
  bzero(symbol_table, sizeof(symbol_table));
}

char *
get_symbol(char *name)
{
  if (name == NULL)
    return NULL;
  
  int i = symbol_hash(name);
  symbol_cell *cell = symbol_table[i];
  while (cell != NULL) {
    if (strcmp(cell->name, name) == 0)
      return cell->name;
    cell = cell->next;
  }
  symbol_cell *new_cell = new symbol_cell();
  new_cell->name = strdup(name);
  new_cell->next = symbol_table[i];
  symbol_table[i] = new_cell;

  return new_cell->name;
}

//
// mis-naming...
//
rterm *
deref(rterm *t)
{
  if (t == NULL)
    return NULL;		// something wrong
  if (t->Type() == tt_variable) {
    rterm *content = ((term_variable *) t)->GetContent();
    if (content)
      return deref(content);
  }
  return t;
}

/* eof */
