// -*- C++ -*-
// Copyright (C) 1996 Buntarou Shizuki(shizuki@is.titech.ac.jp)

#include "parse.h"

#ifndef _rproc_h
#define _rproc_h

#define RPROC_STATE_LEAF	0
#define RPROC_STATE_MAGINIFIED	1
#define RPROC_STATE_CLUSTER	2

#define pstore_init	rproc::Pstore_init
#define pstore_create	rproc::Pstore_create
#define pstore_get	rproc::Pstore_get
#define pstore_monitor	rproc::Pstore_monitor_ports

class rproc;
struct section {
  rproc *p;
  int orig;
  double scale;
  double doi;
  double start;
  double width;
};

#define MINIMUM_SIZE		10
#define EXPANSION_DEFAULT	0
#define EXPANSION_PROC		1

class DoublePointSize {
public:
  double		left, top, width, height;

  void			SetPoint(double L, double T) { left = L; top = T; }
  void			SetSize(double W, double H) { width = W; height = H; }

  void			SetLeft(double L) { left = L; }
  void			SetTop(double T) { top = T; }
  void			SetWidth(double W) { width = W; }
  void			SetHeight(double H) { height = H; }

  double		GetLeft() { return left; }
  double		GetTop() { return top; }
  double		GetWidth() { return width; }
  double		GetHeight() { return height; }

  double		GetRight() { return left+width; }
  double		GetBottom() { return top+height; }
};

class rproc : public DoublePointSize {
  enum rproc_state {
    rproc_dead,
    rproc_running,
    rproc_blocked,
  };
  struct link {
    int drawn;
    char *my_slot;
    char *to_slot;
    rproc *to_proc;
  };
  struct port {
    char     *name;
    int       monitored;
    PortType  type;
    rterm    *term;
    Am_Object image;
  };

  int			procid;
  int			subid;
  char			*name;
  Proc			*proc;
  rproc_state		state;
  int			creation_time;
  SubProc		*subproc;
  Rule			*rule;	// creator
  SLList<link *>	linklist;
  SLList<port *>	portlist;
  Am_Object		image;

  double		system_doi;
  double		doi;
  double		Xreq, Yreq;
  section		*x_sections, *y_sections;

  rproc			*parent, *leader;
  DLList<rproc *>	children, member;

  rproc			*next;	// for linked list in calc_min

  DoublePointSize	OldPoint, NewPoint;

  int			starting_marker;
  int			stop_marker;

  int			current_expansion;
  int			forced_expansion;
  int			future_expansion;

  int			current_visible; // set in rproc::draw()
  int			forced_visible;

  void			clear_forced_visible() { forced_visible = 0; }
  
  rproc			*get_list();
  void			set_marker() { starting_marker = 1; }
  void			clear_marker() { starting_marker = 0; }
  void			set_stop() { stop_marker = 1; }
  void			set_stops(rproc *p);
  void			clear_stop() { stop_marker = 0; }
  void			clear_stops(rproc *p);

  void			open() { current_expansion = 1; }
  void			close() { current_expansion = 0; }
  void			may_open() { future_expansion = 1; }
  void			may_close() { future_expansion = 0; }
  int			may_be_opened() { return future_expansion; }
  int			may_be_closed() { return !future_expansion; }

  void			calc_min_node();
  void			calc_min_cluster();
  void			create_section();
  void			resection(double new_x, double new_y,
				  double new_w, double new_h);

  void			reorder();
  void			reorder_downward();
  void			reorder_rightward();

  void			set_children_coordinate();
  void			set_member_coordinate();
  void			set_member_coordinate_h();
  void			set_member_coordinate_v();
  void			set_member_coordinate_default();

  rproc			*GetParent() { return parent; }
  DLList<rproc *>	GetChildren() { return children; }
  void			AddChild(rproc *p) { children.append(p); }
  DLList<rproc *>&	GetMember() { return member; }
  void			AddMember(rproc *p) { member.append(p); }

public:
			rproc(int id, char *n, int now, rproc *p_proc);
			~rproc() {}
  
  int			GetID() { return procid; }
  int			SetSubID(int id) { subid = id; }
  int			GetSubID() { return subid; }
  char			*GetName() { return name; }
  Proc			*GetProc() { return proc; }
  SubProc		*GetSubProc() { return subproc; }
  void			SetSubProc(SubProc *sb) { subproc = sb; }
  rproc			*GetChild(int n);
  rproc			*GetLeader() { return leader; }
  void			SetLeader(rproc *l) { leader = l; }
  rproc			*GetNext() { return next; }
  void			SetNext(rproc *p) { next = p; }
  Rule			*GetRule() { return rule; }
  void			SetRule(Rule *r) { rule = r; }
  void			SetLink(char *my_slot, char *to_slot, rproc *to_proc);
  SLList<link *>	GetLinks() { return linklist; }
  Am_Object		Image() { return image; }
  Am_Object		VisiblePart();

  int			isalive() { return state != rproc_dead; }
  int			isrunning() { return state == rproc_running; }
  int			issuspended() { return state == rproc_blocked; }
  int			isdead() { return state == rproc_dead; }

  int			is_opened() { return current_expansion; }
  int			is_closed() { return !current_expansion; }

  int			is_forced() { return forced_expansion != 0; }
  int			is_force_opened() { return forced_expansion == 1; }
  int			is_force_closed() { return forced_expansion == 2; }
  void			force_open() { forced_expansion = 1; }
  void			force_close() { forced_expansion = 2; }
  void			clear_forced() { forced_expansion = 0; }
  void			force_visible() { forced_visible = 1; }
  void			force_unvisible() { forced_visible = 2; }
  int			is_forced_visible() { return forced_visible == 1; }
  int			is_forced_unvisible() { return forced_visible == 2; }

  void			all_stop();
  void			all_clear();

  void			bind(SLList<rterm *> terms);
  void			bind_cont(SLList<rterm *> terms);
  SLList<port *>	get_ports() { return portlist; }
  rproc::port		*get_port(char *name);
  void			monitor_enable(char *name);
  void			monitor_disable(char *name);
  void			monitor_update();
  
  void			update();
  void			run();
  void			suspend();
  void			die();

  void			calc_init();
  void			calc_min(double allowed_w, double allowed_h);
  double		get_x_min() { return Xreq; }
  double		get_y_min() { return Yreq; }
  void			mod_min(double new_x, double new_y, double new_w, double new_h);
  void			set_min() { Xreq = 10; Yreq = 10; }
  void			commit_expand();

  void			desection();

  void			update_system_doi();
  void			propagate_system_doi(double new_doi);

  double		getdoi();
  void			update_doi(double diff);
  void			update_doi_downward(double diff);

  void			draw(int count, int limit, int visible);
  void			draw_links();

  int			do_grow();;
  int			do_expand();
  int			do_expand_auto();
  int			do_shrink();
  double		cur_left() { return OldPoint.GetLeft(); }
  double		cur_top() { return OldPoint.GetTop(); }
  double		cur_width() { return OldPoint.GetWidth(); }
  double		cur_height() { return OldPoint.GetHeight(); }

  rproc			*pstore_next; // for linked list in Pstore
  static int		Pstore_init();
  static rproc		*Pstore_create(char *name, rproc *parent, Rule *creator,
				       int now);
  static rproc		*Pstore_get(int id);
  static void		Pstore_monitor_ports();

  // for debuggin
  void			dump_member(int level);
  void			dump_children(int level);

private:
  double		Xscale, Yscale;
  void			unvisible();
public:
  int			isvisible();
  int			haschildren() { return children.length(); }
  int			hasmember() { return member.length(); }
  int			expanding();
};

#endif

/* eof */
