// Copyright (C) 1996 Buntarou Shizuki(shizuki@is.titech.ac.jp)

#include <amulet/am_io.h>
  
#include <amulet/formula_advanced.h>
#include <amulet/standard_slots.h>
#include <amulet/opal_advanced.h>
#include <amulet/value_list.h>
#include <amulet/inter_advanced.h> // to get Am_Initialize_Interactors
#include <amulet/widgets_advanced.h>

#include <amulet/object_advanced.h>
#include <amulet/gdefs.h>
#include <amulet/gem.h>
#include "rproc.h"

extern Am_Object Am_Rproc;
extern Am_Object Am_Rlink;
extern Am_Object rgroup;
extern Am_Object rlinks;
extern Am_Object Stream_Monitor;

extern Am_Slot_Key RPROC;
extern Am_Slot_Key RPROC_MOVER_PART;
extern Am_Slot_Key RPROC_NORMAL;
extern Am_Slot_Key RPROC_SMALL;
extern Am_Slot_Key RPROC_LITTLE;
extern Am_Slot_Key RPROC_STATE;
extern Am_Slot_Key RPROC_SELECT;
extern Am_Slot_Key RPROC_BACK;
extern Am_Slot_Key PROC_PART;
extern Am_Slot_Key BL_SRC_NAME;
extern Am_Slot_Key BL_DEST_NAME;
extern Am_Slot_Key CLICK_GROW;
extern Am_Slot_Key CLICK_TOGGLE;
extern Am_Slot_Key CLICK_SHRINK;
extern Am_Slot_Key RPROC_POPUP;
extern Am_Slot_Key MONITOR_GROUP;
extern Am_Slot_Key MONITOR_DISMISS;
extern Am_Slot_Key MONITOR_CHECK;
extern Am_Slot_Key MONITOR_CONTENT;
extern Am_Slot_Key MONITOR_SCROLL;
extern Am_Slot_Key MONITOR_TEXT;

Am_Declare_Formula(bool, rproc_normal_visible);

static rproc* merge_list(rproc *list1, rproc *list2);
static int compare_section(section *s1, section *s2);
static void print_list(rproc *p);
static void dump_sections(char *message, section *sections, int number);

rproc::rproc(int id, char *n, int now, rproc *p_proc)
{
  Am_Object normal_image;

  procid = id;
  name = get_symbol(n);
  proc = getProc(name, 0);
  parent = p_proc;
  normal_image = proc->Image()
    .Set(Am_VISIBLE, Am_Formula::Create(rproc_normal_visible))
    ;
  state = rproc_running;
  creation_time = now;
  image = Am_Rproc.Create(name)
    .Set(RPROC, (void *)this)
    .Set(PROC_NAME, name)
    .Add_Part(RPROC_NORMAL, normal_image)
    ;
  //  printf("new rproc(this=%x)\n", this);

  clear_forced_visible();
  clear_marker();
  set_stop();
  close();
  may_close();
  clear_forced();
  set_min();

  if (parent)
    doi = parent->getdoi();	// inherit parent's doi
  else
    doi = 1;			// for main process(default doi)
  system_doi = exp(0);		// XXX
  Xscale = 1;
  Yscale = 1;
  x_sections = NULL;
  y_sections = NULL;
}
/*
rproc::rproc(int id, char *n, int now)
{
  Am_Object normal_image;

  procid = id;
  name = get_symbol(n);
  proc = getProc(name, 0);
  normal_image = proc->Image()
    .Set(Am_VISIBLE, Am_Formula::Create(rproc_normal_visible))
    ;
  state = rproc_running;
  creation_time = now;
  image = Am_Rproc.Create(name)
    .Set(RPROC, (void *)this)
    .Set(PROC_NAME, name)
    .Add_Part(RPROC_NORMAL, normal_image)
    ;
  //  printf("new rproc(this=%x)\n", this);

  clear_forced_visible();
  clear_marker();
  clear_stop();
  open();
  may_open();
  clear_forced();
  set_min();

  doi = 1;
  system_doi = exp(0);		// XXX
  Xscale = 1;
  Yscale = 1;
  x_sections = NULL;
  y_sections = NULL;
}
*/

#define PSTORE_HASH_SIZE 1021
static rproc *proc_table[PSTORE_HASH_SIZE];

int
rproc::Pstore_init()
{
  int i;
  for (i = 0; i < PSTORE_HASH_SIZE; i++) {
    proc_table[i] = NULL;
  }
  return 0;
}

rproc *
rproc::Pstore_create(char *name, rproc *p_proc, Rule *creator,
		     int now)
{
  static int next_id = 0;
  int i = next_id % PSTORE_HASH_SIZE;

  rproc *p = new rproc(next_id, name, now, p_proc);
  if (debug_rproc)
    printf("Rproc created %s(%d)\n", name, next_id);
  next_id++;
  if (p == NULL)
    return NULL;

  if (p_proc)
    p_proc->AddChild(p);
  p->SetRule(creator);

  p->pstore_next = proc_table[i];
  proc_table[i] = p;

  rgroup.Add_Part(p->Image());
  return p;
}

rproc *
rproc::Pstore_get(int id)
{
  rproc *p;
  int i = id % PSTORE_HASH_SIZE;

  // Oh..., My!!
  for (p = proc_table[i]; p != NULL; p = p->pstore_next) {
    if (p->GetID() == id)
      return p;
  }
  return NULL;
}

void
rproc::Pstore_monitor_ports()
{
  if (! mode_update_monitor)
    return;

  int i;
  rproc *process;
  if (debug_stream_monitor)
    printf("Monitoring...\n");
  
  for (i = 0; i < PSTORE_HASH_SIZE; i++) {
    for (process = proc_table[i]; process != NULL; process = process->pstore_next) {
      process->monitor_update();
    }
  }
}

static void
draw_link(rproc *src_proc, char *src_name, rproc *dst_proc, char *dst_name)
{
  if (src_proc == dst_proc)
    return;			// XXX

  if (dst_proc->isvisible()) {
    if (! mode_draw_fixed_links) {
      rproc::port *p = dst_proc->get_port(dst_name);
      if (p->term->isconcrete())
	return;
    }
    
    if (debug_link)
      printf("       %s to %s(%s,%d)\n",
	     src_name, dst_name, dst_proc->GetName(), dst_proc->GetID());

    Am_Object new_line = Am_Rlink.Create()
      .Set(BL_SRC, Proc_Get_Port(src_proc->Image().Get_Part(RPROC_NORMAL),
				 src_name))
      .Set(BL_SRC_NAME, src_name)
      .Set(BL_DEST, Proc_Get_Port(dst_proc->Image().Get_Part(RPROC_NORMAL),
				  dst_name))
      .Set(BL_DEST_NAME, dst_name)
      ;
    rlinks.Add_Part(new_line);
  } else {			// XXX: more checks should be needed
    Pix li;
    SLList<rproc::link *> linklist = dst_proc->GetLinks();
    for (li = linklist.first(); li != 0; linklist.next(li)) {
      rproc::link *link = linklist(li);
      if (strcmp(dst_name, link->my_slot) == 0)
	draw_link(src_proc, src_name, link->to_proc, link->to_slot);
    }
  }
}

void
rproc::draw_links()
{
  if (this == NULL)
    return;

  if (debug_link)
    printf("rproc::draw_links(%s,%d)\n", name, procid);

  if (isvisible()) {
    //
    // draw links emitting from this process
    //
    Pix i;
    for (i = linklist.first(); i != 0; linklist.next(i)) {
      rproc::link *link = linklist(i);
      char *src_name    = link->my_slot;
      draw_link(this, src_name, link->to_proc, link->to_slot);
    }
  }
  if (isvisible() && is_opened()) {
    //
    // traverse process tree
    //
    Pix pi;
    for (pi = member.first(); pi != 0; member.next(pi))
      member(pi)->draw_links();
  }
}

void
rproc::run()
{
  state = rproc_running;
  image.Set(Am_FILL_STYLE, COLOR_RUNNING);
  if (leader && this != leader)	// XXX
    leader->run();
}

void
rproc::suspend()
{
  // check whether all live children are blocked or not
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    if (member(pi)->isrunning()) // XXX
      return;
  }
  state = rproc_blocked;
  image.Set(Am_FILL_STYLE, COLOR_SUSPENDED);
  if (leader)
    leader->suspend();
}

void
rproc::die()
{
  int all_dead = 1;

  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    if (p->isrunning())
      return;
    if (p->issuspended())
      all_dead = 0;
  }

  if (all_dead && procid == 0) {
    // main process is dead...
    state = rproc_dead;
    doi = 0;			// XXX
//    update_doi(0);		// XXX
    image.Set(Am_FILL_STYLE, COLOR_DEAD);
    //window_current_state = WINDOW_STATE_DEAD; // XXX

  } else if (all_dead) {
    state = rproc_dead;
    doi = 0;			// XXX
    leader->update_doi(0);	// XXX
    image.Set(Am_FILL_STYLE, COLOR_DEAD);
    if (leader)
      leader->die();
  } else {
    state = rproc_blocked;
    image.Set(Am_FILL_STYLE, COLOR_SUSPENDED);
    if (leader)
      leader->suspend();
  }
}

Am_Object
rproc::VisiblePart()
{
  Am_Object normal_image = image.Get_Part(RPROC_NORMAL);
  if (normal_image.Get(Am_VISIBLE))
    return normal_image;
  Am_Object small_image = image.Get_Part(RPROC_SMALL);
  if (small_image.Get(Am_VISIBLE))
    return small_image;
  Am_Object little_image = image.Get_Part(RPROC_LITTLE);
  if (little_image.Get(Am_VISIBLE))
    return little_image;
  return NULL;
}

void
rproc::bind(SLList<rterm *> terms)
{
  SLList<Port *> proc_ports = proc->GetPorts();
  Pix pi = proc_ports.first();
  Pix ti;
  for (ti = terms.first(); ti != 0; terms.next(ti)) {
    Port *proc_port = proc_ports(pi);
    port *p = new port();
    p->name = proc_port->name;
    if (proc_port->type==port_single_in || proc_port->type==port_single_out)
      p->monitored = 1;		// for single ports
    else
      p->monitored = 0;		// for stream ports
    p->type = proc_port->type;
    p->term = terms(ti);
    p->image = NULL;		// XXX
    portlist.append(p);
    proc_ports.next(pi);
  }
}

void
rproc::bind_cont(SLList<rterm *> terms)
{
  Pix pi = portlist.first();
  Pix ti;
  for (ti = terms.first(); ti != 0; terms.next(ti)) {
    port *p = portlist(pi);
    // update single ports only
    if (p->type==port_single_in || p->type==port_single_out)
      p->term = terms(ti);
    portlist.next(pi);
  }
}

rproc::port *
rproc::get_port(char *name)
{
  Pix pi;
  for (pi = portlist.first(); pi != 0; portlist.next(pi)) {
    port *p = portlist(pi);
    if (strcmp(p->name, name) == 0)
      return p;
  }
  return NULL;
}

void
rproc::monitor_enable(char *name)
{
  Pix pi;
  for (pi = portlist.first(); pi != 0; portlist.next(pi)) {
    port *p = portlist(pi);
    if (strcmp(p->name, name) == 0) {
      p->monitored = 1;
      monitor_update();
      return;
    }
  }
}

void
rproc::monitor_disable(char *name)
{
  Pix pi;
  for (pi = portlist.first(); pi != 0; portlist.next(pi)) {
    port *p = portlist(pi);
    if (strcmp(p->name, name) == 0) {
      p->monitored = 0;
      monitor_update();
      return;
    }
  }
  // something should be done for single ports
}

#ifndef DEMO
extern Am_Cursor my_cursor;
Am_Drawonable* Get_a_drawonable (Am_Object obj);
#endif

void
rproc::monitor_update()
{
  if (debug_stream_monitor)
    printf("        %s(pid=%d)\n", name, procid);
  Pix pi;
  for (pi = portlist.first(); pi != 0; portlist.next(pi)) {
    port *p = portlist(pi);
    if (p->monitored) {
      rterm *t = p->term;
      // in texts
      if (debug_stream_monitor)
	printf("                %s: %s\n", p->name, t->Get());

      // in graphics
      if  (p->type == port_stream_in || p->type == port_stream_out) {
	if (! p->image.Valid()) {
	  p->image = Stream_Monitor.Create()
	    .Set(RPROC, this)
	    .Set(PORT_NAME, p->name)
	    ;
	  Am_Screen.Add_Part(p->image);
/*
	  Am_Drawonable *r = Get_a_drawonable(p->image);
	  printf("drawonable: %x\n", r);
	  r->Set_Cursor(my_cursor);
*/
	}
	p->image.Set(MONITOR_TEXT, t->Get());
	p->image.Set(Am_VISIBLE, true);
      } else if (p->type == port_single_in || p->type == port_single_out) {
	Am_Object normal_image = image.Get_Part(RPROC_NORMAL);
	Am_Object single_port = Proc_Get_Port(normal_image, p->name);
	if (! p->image.Valid()) {
	  p->image = Atom_Proto.Create();
	  Port_Add(single_port, p->image);
	}
	if (t->isatom()) {
	  char *content = t->Get();
	  p->image.Get_Part(CONTENTS_PART).Set(Am_TEXT, content);
	  // overlapping items must be moved a little
	} else if (t->isunbound()) {
	  p->image.Get_Part(CONTENTS_PART).Set(Am_TEXT, "?");
	  // overlapping items must be moved a little
	} else {
	  p->image.Get_Part(CONTENTS_PART).Set(Am_TEXT, "*");
	  //
	}
      }
    } else {
      //
      // monitoring of this port is disabled
      //
      if (p->type == port_single_in || p->type == port_single_out) {
	if (p->image.Valid()) {
	  Am_Object content = p->image.Get_Part(CONTENTS_PART);
	  Port_Remove_Value(content);
	}
      }
    }
  }
}

//
// update parent-child tree
//
void
rproc::update()
{
  {
    int i = 0;
    Pix pi = children.first();
    Rule *creator = children(pi)->GetRule(); // XXX
    for (pi = children.first(); pi != 0; children.next(pi)) {
      rproc *child = children(pi);
      SubProc *sp = creator->GetBody(i);
      child->SetSubID(i);
      child->SetSubProc(sp);
      i++;
    }
  }
  
  set_children_coordinate();

  // re-order children based on this process's arrangement requirement
  reorder();
  
  // update leader-member tree
  if (children.length() > 0) {
    if (parent && strcmp(name, parent->GetName()) == 0) {
      DLList<rproc *> &leader_member = leader->GetMember();

      // Pix j = leader_member.seek(this);
      Pix j;
      for (j = leader_member.first(); j != 0; leader_member.next(j)) {
	if (this == leader_member(j))
	  break;
      }
      if (j == 0) {
	printf("rproc::update: Something is wrong!!!\n");
	// exit(1);
      }
      Pix pi;
      for (pi = children.first(); pi != 0; children.next(pi)) {
	rproc *p = children(pi);
	leader_member.ins_before(j, p);
	p->SetLeader(leader);
	if (mode_creation_time)
	  p->update_system_doi();
      }

      leader_member(j)->force_unvisible(); // XXX
      leader_member.del(j);

      // reset members' coordinates
      leader->set_member_coordinate();
    } else {
      Pix pi;
      for (pi = children.first(); pi != 0; children.next(pi)) {
	rproc *p = children(pi);
	this->AddMember(p);
	p->SetLeader(this);
	if (mode_creation_time)
	  p->update_system_doi();
      }
      // set children's coordinates
      set_member_coordinate();
    }
  } else {
    // ???
  }
}

void
rproc::reorder()
{
  // re-order children based on this process's arrangement requirement
  ExpandMethod m = proc->GetExpand();
  if (m == expand_downward)
    reorder_downward();
  else if (m == expand_rightward)
    reorder_rightward();
}

void
rproc::reorder_downward()
{
  int i, j;
  int length = children.length();
  rproc *plist[length];

  Pix pi = children.first();
  for (i = 0; i < length; i++) {
    plist[i] = children(pi);
    children.next(pi);
  }

  for (j = 0; j < length; j++) {
    for (i = j+1; i < length; i++) {
      if (plist[j]->GetTop() > plist[i]->GetTop()) {
	// swap plist[j] and plist[i]
	rproc *tmp = plist[j];
	plist[j] = plist[i];
	plist[i] = tmp;
      }
    }
  }

  pi = children.first();
  for (i = 0; i < length; i++) {
    children(pi) = plist[i];
    children.next(pi);
  }
}

void
rproc::reorder_rightward()
{
  int i, j;
  int length = children.length();
  rproc *plist[length];

  Pix pi = children.first();
  for (i = 0; i < length; i++) {
    plist[i] = children(pi);
    children.next(pi);
  }

  for (j = 0; j < length; j++) {
    for (i = j+1; i < length; i++) {
      if (plist[j]->GetLeft() > plist[i]->GetLeft()) {
	// swap plist[j] and plist[i]
	rproc *tmp = plist[j];
	plist[j] = plist[i];
	plist[i] = tmp;
      }
    }
  }

  pi = children.first();
  for (i = 0; i < length; i++) {
    children(pi) = plist[i];
    children.next(pi);
  }
}

rproc *
rproc::GetChild(int n)
{
  Pix c;
  int i = 0;
  for (c = children.first(); c != 0; children.next(c)) {
//    if (n == i)
//      return children(c);
    if (children(c)->GetSubID() == n)
      return children(c);
    i++;
  }
  return NULL;
}

void
rproc::SetLink(char *my_slot, char *to_slot, rproc *to_proc)
{
  // check whether the same link was stored or not
  Pix li;
  for (li = linklist.first(); li != 0; linklist.next(li)) {
    link *l = linklist(li);
    if (strcmp(l->my_slot, my_slot) == 0
	&& strcmp(l->to_slot, to_slot) == 0
	&& l->to_proc == to_proc)
      return;
  }

  // this is a new link
  link *new_link = new link();

  new_link->drawn = 0;
  new_link->my_slot = my_slot;
  new_link->to_slot = to_slot;
  new_link->to_proc = to_proc;

  linklist.append(new_link);

  if (debug_trace)
    printf("Link: %s(%d)::%s ---> %s(%d)::%s\n",
	   name, procid, my_slot,
	   to_proc->GetName(), to_proc->GetID(), to_slot);
}

//
// all memebers have same size horizontally aligned
//
void
rproc::set_member_coordinate()
{
  ExpandMethod m = proc->GetExpand();
  if (m == expand_downward)
    set_member_coordinate_v();
  else if (m == expand_rightward)
    set_member_coordinate_h();
  else
    set_member_coordinate_default();
}

#define PROC_MARGIN 4
void
rproc::set_member_coordinate_h()
{
  int m_length = member.length();
  if (m_length < 1) {
    printf("rproc::set_member_coodinate_h: something wrong\n");
    return;
  }
  Pix pi = member.first();
  double new_T = member(pi)->GetTop();
  double new_W = width/(2*m_length + 1);
  double new_H = member(pi)->GetHeight();
  double new_L = new_W + left;

  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    p->SetPoint(new_L, new_T);
    p->SetSize(new_W, new_H);
    new_L = new_L + 2*new_W;
  }
}

void
rproc::set_member_coordinate_v()
{
  int m_length = member.length();
  if (m_length < 1) {
    printf("rproc::set_member_coodinate_v: something wrong\n");
    return;
  }
  Pix pi = member.first();
  double new_L = member(pi)->GetLeft();
  double new_W = member(pi)->GetWidth();
  double new_H = height / (2*m_length + 1);
  double new_T = new_H + top;

  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    p->SetPoint(new_L, new_T);
    p->SetSize(new_W, new_H);
    new_T = new_T + 2*new_H;
  }
}

#undef PROC_MARGIN

void
rproc::set_member_coordinate_default()
{
  return;
}

void
rproc::calc_init()
{
  if (1) {
    // new process is appearing from the center of parent process
    double x = parent->cur_left() + parent->cur_width()/2;
    double y = parent->cur_top() + parent->cur_height()/2;
  
    OldPoint.SetPoint(x, y);
  } else {
    // new process is appearing from heaven
    OldPoint.SetPoint(0, 0);
  }
  OldPoint.SetSize(0, 0);

/*
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    member(pi)->calc_init();
  }
*/
}

void
rproc::all_stop()
{
  set_stop();
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi))
    member(pi)->all_clear();
}  

void
rproc::all_clear()
{
  clear_stop();
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi))
    member(pi)->all_clear();
}

// proc_main->calc_min(screen size)
void
rproc::calc_min(double allowed_w, double allowed_h)
{
  // first, close me and all children
  may_close();
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi))
    member(pi)->may_close();

  if (isdead() || is_force_closed() || member.length() == 0) // XXX
    return;

  //proc_main->clear_stops(GetLeader());
  proc_main->clear_stops(this);
  
  if (! is_force_opened()) {
    double x_min = proc_main->get_x_min();
    double y_min = proc_main->get_y_min();
    if (x_min > allowed_w || y_min > allowed_h)
      return;

    may_open();
    proc_main->calc_min_node();
    x_min = proc_main->get_x_min();
    y_min = proc_main->get_y_min();
    if (x_min > allowed_w || y_min > allowed_h) {
      // failed to open this cluster
      may_close();
      proc_main->calc_min_node();
      proc_main->set_stops(this);
      return;
    }
  } else {
    may_open();
    proc_main->calc_min_node();
  }
  
  // hear this cluster node is opened
  // sort all member based on those DOI and requirement for open
  int length = member.length();
  rproc *plist[length];

  int i, j;
  pi = member.first();
  for (i = 0; i < length; i++) {
    plist[i] = member(pi);
    member.next(pi);
  }

  // based on force-opened
  int limit = 0;
  for (j = 0; j < length; j++)
    if (plist[j]->is_force_opened())
      swap(plist[limit++], plist[j]);
  for (j = 0; j < limit; j++)
    for (i = j+1; i < length; i++)
      if (plist[j]->getdoi() < plist[i]->getdoi())
	swap(plist[i], plist[j]);
  // based on DOI
  for (j = limit; j < length; j++)
    for (i = j+1; i < length; i++)
      if (plist[j]->getdoi() < plist[i]->getdoi())
	swap(plist[i], plist[j]);

  for (i = 0; i < length; i++)
    plist[i]->calc_min(allowed_w, allowed_h);

  proc_main->set_stops(this);
}

/*
void
rproc::calc_min_old(double allowed_w, double allowed_h)
{
  rproc *list = get_list();
  rproc *head = list;

  //
  // assume that all nodes below "this" are initially unexpanded
  // and marked
  //
  while (head != NULL) {
    rproc *cur_leader = head->GetLeader();
    if (head != this) {
      if (cur_leader->may_be_closed()) {
	cur_leader->may_open();
	clear_stops(cur_leader);
	calc_min_node();
	if (!cur_leader->is_force_opened()
	    && (Xreq > allowed_w ||Yreq > allowed_h)) {
	  // spaced allocation failed
	  cur_leader->may_close();
	  calc_min_node();
	  set_stops(cur_leader);
	  break;
	}
	set_stops(cur_leader);
      }
    } else {
      // head is a top process
      head->clear_stop();
      head->calc_min_node();
      //

    }
    head = head->GetNext();
  }
  //
  rproc *p;
  for (p = list; p != head; p = p->GetNext()) {
    rproc *q = p->GetLeader();
    if (q != NULL)
      q->clear_stop();
  }
  return;
}
*/

void
rproc::calc_min_node()
{
  if (starting_marker || stop_marker)
    return;
  if (isdead() || member.length() == 0 || !may_be_opened()) {
    set_min();
    return;
  }
  calc_min_cluster();
}

//
// calculate minimum reqested size as a cluster
//
void
rproc::calc_min_cluster()
{
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi))
    member(pi)->calc_min_node();
  create_section();
  
  //
  // calculate this cluster's minimum requested size
  //
  int limit = member.length()*2+2;
  Xreq = 0; Yreq = 0;
  int i;
  for (i = 0; i < limit-1; i++) {
    Xreq += x_sections[i].scale * x_sections[i].width;
    Yreq += y_sections[i].scale * y_sections[i].width;
  }
}

void
rproc::create_section()
{
  int proc_number = member.length();
  int limit = proc_number*2+2;
  
  free(x_sections);
  free(y_sections);
  x_sections = (section *)malloc(sizeof(section)*limit);
  y_sections = (section *)malloc(sizeof(section)*limit);
  bzero(x_sections, sizeof(section)*limit);
  bzero(y_sections, sizeof(section)*limit);
  
  x_sections[0].p = NULL;
  x_sections[0].orig = 0;
  x_sections[0].start = left;
  x_sections[limit-1].p = NULL;
  x_sections[limit-1].orig = limit;
  x_sections[limit-1].scale = -1;
  x_sections[limit-1].start = left+width;
  x_sections[limit-1].width = -1;
  
  y_sections[0].p = NULL;
  y_sections[0].orig = 0;
  y_sections[0].start = top;
  y_sections[limit-1].p = NULL;
  y_sections[limit-1].orig = limit;
  y_sections[limit-1].scale = -1;
  y_sections[limit-1].start = top+height;
  y_sections[limit-1].width = -1;
  
  //
  // breaks up into intervals
  //
  Pix pi;
  int i = 1;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    
    x_sections[i].p = p;
    x_sections[i].orig = i;
    x_sections[i].start = p->GetLeft();
    x_sections[i+1].p = p;
    x_sections[i+1].orig = i+1;
    x_sections[i+1].start = p->GetRight();
    y_sections[i].p = p;
    y_sections[i].orig = i;
    y_sections[i].start = p->GetTop();
    y_sections[i+1].p = p;
    y_sections[i+1].orig = i+1;
    y_sections[i+1].start = p->GetBottom();
    
    i = i + 2;
  }
  qsort(x_sections, limit, sizeof(section), compare_section);
  qsort(y_sections, limit, sizeof(section), compare_section);
  
  for (i = 0; i < limit-1; i++) {
    x_sections[i].width = x_sections[i+1].start - x_sections[i].start;
    y_sections[i].width = y_sections[i+1].start - y_sections[i].start;
  }  
  
  //
  // set scale factor of each section
  //
  for (i = 0; i < limit-1; i++) {
    x_sections[i].scale = -1;
    y_sections[i].scale = -1;
    x_sections[i].doi = 0;
    y_sections[i].doi = 0;
    for (pi = member.first(); pi != 0; member.next(pi)) {
      rproc *p = member(pi);
      double p_doi = p->getdoi();
      if (x_sections[i].start >= p->GetLeft()
	  && p->GetRight() > x_sections[i].start) {
	if (x_sections[i].width > 0) {
	  double x_scale = p->get_x_min() / x_sections[i].width;
	  if (x_sections[i].scale < x_scale) {
	    x_sections[i].scale = x_scale;
	  }
	  if (x_sections[i].doi < p_doi) {
	    x_sections[i].doi = p_doi;
	  }
	}
      }
      if (y_sections[i].start >= p->GetTop()
	  && p->GetBottom() > y_sections[i].start) {
	if (y_sections[i].width > 0) {
	  double y_scale = p->get_y_min() / y_sections[i].width;
	  if (y_sections[i].scale < y_scale) {
	    y_sections[i].scale = y_scale;
	  }
	  if (y_sections[i].doi < p_doi) {
	    y_sections[i].doi = p_doi;
	  }
	}
      }
    }
  }
  
  //
  // set minimum requirement for blank sections
  //
#define BLANK_REQ 12
  for (i = 0; i < limit-1; i++) {
    if (x_sections[i].scale < 0 && x_sections[i].width > 0)
      x_sections[i].scale = BLANK_REQ / x_sections[i].width;
    if (y_sections[i].scale < 0 && y_sections[i].width > 0)
      y_sections[i].scale = BLANK_REQ / y_sections[i].width;
  }
  if (debug_czoom_section) {
    dump_sections("xsections: ", x_sections, limit);
    dump_sections("ysections: ", y_sections, limit);
  }
}

void
rproc::mod_min(double new_x, double new_y, double new_w, double new_h)
{
  NewPoint.SetPoint(new_x, new_y);
  NewPoint.SetSize(new_w, new_h);
  
  if (starting_marker || stop_marker)
    return;
  if (member.length() == 0 || !may_be_opened())
    return;
  resection(new_x, new_y, new_w, new_h);

  if (debug_czoom_resection) {
    int proc_number = member.length();
    int limit = proc_number*2+2;
    dump_sections("xsections: ", x_sections, limit);
    dump_sections("ysections: ", y_sections, limit);
  }
}

void
rproc::resection(double new_x, double new_y, double new_w, double new_h)
{
  double XLfs = (double)(int)(new_w - Xreq); // XXX
  double YLfs = (double)(int)(new_h - Yreq); // XXX
  
  if (debug_czoom_resection) {
    printf("resection: %s(%d) Lfs=(%f,%f)\n",
	   name, procid, XLfs, YLfs);
  }

  if (XLfs < 0 || YLfs < 0) {
    printf("Warning: space requirement not satisfied(X=%f,Y=%f)\n",
	   XLfs, YLfs);
    if (XLfs < 0)
      XLfs = 0;
    if (YLfs < 0)
      YLfs = 0;
  }
  
  int limit = member.length()*2+2;
  double XDOIsum = 0;
  double YDOIsum = 0;
  int i;
  for (i = 0; i < limit-1; i++) {
    XDOIsum += x_sections[i].doi * x_sections[i].width;
    YDOIsum += y_sections[i].doi * y_sections[i].width;
  }
  for (i = 0; i < limit-1; i++) {
    double XDOIf = x_sections[i].doi*x_sections[i].width / XDOIsum;
    double YDOIf = y_sections[i].doi*y_sections[i].width / YDOIsum;
    // set width destructively
    x_sections[i].width = x_sections[i].width*x_sections[i].scale + XDOIf*XLfs;
    y_sections[i].width = y_sections[i].width*y_sections[i].scale + YDOIf*YLfs;
  }
  
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    int i;
    i = 0;
    double L = new_x;
    double W = 0;
    while (x_sections[i].p != p) {
      L += x_sections[i].width;
      i++;
    }
    do {
      W += x_sections[i].width;
      i++;
    } while (x_sections[i].p != p);
    
    i = 0;
    double T = new_y;
    double H = 0;
    while (y_sections[i].p != p) {
      T += y_sections[i].width;
      i++;
    }
    do {
      H += y_sections[i].width;
      i++;
    } while (y_sections[i].p != p);
    p->mod_min(L, T, W, H);
  }
}

void
rproc::draw(int count, int limit, int visible)
{
  int prev_visible = current_visible;
  current_visible = !is_forced_unvisible() && visible;

  if (!prev_visible && current_visible && leader) {
    double x = leader->cur_left() + leader->cur_width()/2;
    double y = leader->cur_top() + leader->cur_height()/2;
    OldPoint.SetPoint(x, y);
    OldPoint.SetSize(0, 0);
  }

  double p = limit-count;
  double q = count;
  
  int L = (int)((p*OldPoint.GetLeft()   + q*NewPoint.GetLeft())   / limit);
  int T = (int)((p*OldPoint.GetTop()    + q*NewPoint.GetTop())    / limit);
  int W = (int)((p*OldPoint.GetWidth()  + q*NewPoint.GetWidth())  / limit);
  int H = (int)((p*OldPoint.GetHeight() + q*NewPoint.GetHeight()) / limit);
  
  int s = RPROC_STATE_LEAF;
  if (visible && is_opened())
    s = RPROC_STATE_CLUSTER;

/*
  image
    .Set(Am_LEFT, L)
    .Set(Am_TOP, T)
    .Set(Am_WIDTH, W)
    .Set(Am_HEIGHT, H)
    .Set(Am_VISIBLE, current_visible)
    .Set(RPROC_STATE, s)
    ;
*/
  // is this coding meaningful?
  if ((int)image.Get(Am_LEFT) != L)
    image.Set(Am_LEFT, L);
  if ((int)image.Get(Am_TOP) != T)
    image.Set(Am_TOP, T);
  if ((int)image.Get(Am_WIDTH) != W)
    image.Set(Am_WIDTH, W);
  if ((int)image.Get(Am_HEIGHT) != H)
    image.Set(Am_HEIGHT, H);
  if ((bool)image.Get(Am_VISIBLE) != current_visible)
    image.Set(Am_VISIBLE, current_visible);
  if ((int)image.Get(RPROC_STATE) != s)
    image.Set(RPROC_STATE, s);
  
  if (count == limit)
    OldPoint = NewPoint;
  
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi))
    member(pi)->draw(count, limit, isalive() && is_opened() && visible);
}

int
rproc::do_grow()
{
  if (leader == NULL)
    // top level process can't grow any more
    return 1;

  update_doi_downward(+1);
  leader->update_doi(0);
  return 0;
}

#define MIN_LIMIT 0.97

int
rproc::do_expand()
{
  if (is_closed())
    force_open();
  else
    force_close();
  
  return 0;
}

/*
int
rproc::do_expand()
{
  if (is_opened()) {
    // treat this node as leaf node and make its size minimum
    force_close();
    set_marker();
    set_min();

    main_draw();
    clear_marker();
    return 0;
  } else {
    rproc *Croot = this;
    while (Croot != NULL) {
      Croot->force_open();
      Croot->calc_min(Croot->cur_width(), Croot->cur_height());
      double root_x_min = Croot->get_x_min();
      double root_y_min = Croot->get_y_min();
      double root_x_size = Croot->cur_width();
      double root_y_size = Croot->cur_height();
    
      if (root_x_min > root_x_size*MIN_LIMIT
	  || root_y_min > root_y_size*MIN_LIMIT) {
	Croot = Croot->GetLeader();
      } else {
	Croot->commit_expand();
	break;
      }
    }

    rproc *p;
    for (p = this; p != NULL; p = p->GetLeader())
      p->clear_forced();

    if (Croot == NULL) {
      // when failed

      return 1;			// XXX
    } else {
      // when succeeded
      Croot->mod_min(Croot->cur_left(), Croot->cur_top(),
		     Croot->cur_width(), Croot->cur_height());
      Croot->commit_expand();
      calc_min(NewPoint.GetWidth(), NewPoint.GetHeight());
      commit_expand();
      mod_min(NewPoint.GetLeft(), NewPoint.GetTop(),
	      NewPoint.GetWidth(), NewPoint.GetHeight());
      return 0;
    }
  }
}
*/

int
rproc::do_expand_auto()
{
  clear_forced();
  return 0;
}

int
rproc::do_shrink()
{
  update_doi_downward(-1);
  leader->update_doi(0);
  main_draw();
  return 0;
}

void
rproc::commit_expand()
{
  current_expansion = future_expansion;
  if (starting_marker || stop_marker)
    return;
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi))
    member(pi)->commit_expand();
}

rproc *
rproc::get_list()
{
  set_stop();
  next = NULL;

  if (starting_marker) {
    if (is_force_closed())	// XXX
      may_close();		// XXX
    return this;
  }
  if (isdead() || is_force_closed()) {
    may_close();
    set_min();
    return this;
  }

  may_close();
  set_min();
  rproc *head1 = this;
  rproc *head2 = NULL;

  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    head2 = member(pi)->get_list();
    head1 = merge_list(head1, head2);
  }
  return head1;
}

void
rproc::set_stops(rproc *p)
{
  if (p == NULL)
    return;
  for (;;) {
    p->set_stop();
    if (p == this)
      return;
    p = p->GetLeader();
  }
}

void
rproc::clear_stops(rproc *p)
{
  if (p == NULL)
    return;
  for (;;) {
    p->clear_stop();
    if (p == this)
      return;
    p = p->GetLeader();
  }
}

static void
dump_sections(char *message, section *sections, int number)
{
  int i;
  
  printf("%s:\n", message);
  for (i = 0; i < number; i++) {
    printf("\t[%d]: start=%f, width=%f, scale=%f, proc=%s(%x)\n",
	   i,
	   sections[i].start,
	   sections[i].width,
	   sections[i].scale,
	   sections[i].p ? sections[i].p->GetName() : "NULL",
	   sections[i].p);
  }
}

void
rproc::update_system_doi()
{
  if (state == rproc_dead) {
    system_doi = 0;
    leader->propagate_system_doi(0);
  } else if (member.length() == 0) {
    system_doi = exp(creation_time - global_time);
    leader->propagate_system_doi(system_doi);
  } else {
    system_doi = 0;
    Pix pi;
    for (pi = member.first(); pi != 0; member.next(pi)) {
      rproc *p = member(pi);
      p->update_system_doi();
    }
  }
}

void
rproc::propagate_system_doi(double new_doi)
{
  if (this == NULL)
    return;

  if (system_doi < new_doi)
    system_doi = new_doi;
  leader->propagate_system_doi(system_doi);
}

double
rproc::getdoi()
{
  if (mode_creation_time)
    return system_doi;		// XXX
  return doi;
}

void
rproc::update_doi(double diff)
{
  if (this == NULL)
    return;

  if (mode_creation_time)
    return;			// XXX
  
/*
  doi = doi + diff;
  if (isalive() && doi < 1)
    doi = 1;
  if (doi < 0)
    doi = 0;
*/
  doi = 0;
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    if (p->getdoi() > doi)
      doi = p->getdoi();
  }

  if (leader)
    leader->update_doi(0);
}

void
rproc::update_doi_downward(double diff)
{
  doi = doi + diff;
  if (isalive() && doi < 1)
    doi = 1;
  if (doi < 0)
    doi = 0;

  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    p->update_doi_downward(diff);
  }
}

int
rproc::isvisible()
{
  return current_visible && !is_forced_unvisible(); // XXX
//  return current_visible;
}

void
rproc::unvisible()
{
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    p->Image().Set(Am_VISIBLE, false);
    p->unvisible();
  }
}

void
rproc::desection()
{
  free(x_sections); x_sections = NULL;
  free(y_sections); y_sections = NULL;

  Pix pi;
  for (pi = children.first(); pi != 0; children.next(pi)) {
    children(pi)->desection();
  }
}

//
// default method for expansion
// must be called BEFORE set_children_coordinate()
//
void
rproc::set_children_coordinate()
{
  int c = 0;
  if (children.length() != 0) {
    Pix i = children.first();

    Rule *creator = children(i)->GetRule(); // XXX
/*
    double scale_w = width  / creator->GetWidth(); // XXX Danger!!!
    double scale_h = height / creator->GetHeight(); // XXX Danger!!!
*/
    while (i != 0) {
      rproc *child = children(i);
//      SubProc *sp = creator->GetBody(c);
      SubProc *sp = child->GetSubProc();
      // fixed algoritm...
      double new_L = left + sp->GetLeft()*width  / creator->GetWidth();
      double new_T = top  + sp->GetTop() *height / creator->GetHeight();
      double new_W = sp->GetWidth()  * width / creator->GetWidth();
      double new_H = sp->GetHeight() * height / creator->GetHeight();
      child->SetPoint(new_L, new_T);
      child->SetSize(new_W, new_H);
//      printf("Process %d (%d, %d) %dx%d(%fx%f)\n",
//	     child->GetID(), new_L, new_T, new_W, new_H, scale_w, scale_h);
      children.next(i);
      c++;
    }
  }  
}

// these should be moved to some header file
extern Am_Slot_Key CO_X;
extern Am_Slot_Key CO_Y;
#define DEFAULT_CO_X ((double)0.20)
#define DEFAULT_CO_Y ((double)0.10)

int
rproc::expanding()
{
  Am_Object pnode = proc->Pnode();
  double x;
  if (pnode.Valid())
    x = pnode.Get(CO_X);
  else
    x = DEFAULT_CO_X;

  return x > 0.05;		// XXX
}

void
rproc::dump_member(int level)
{
  int i;
  for (i = 0; i < level; i++)
    printf(" ");
  printf("pid=%d(%s(%d,%s),%x): %f(%f,%f)(%f,%f)(%f,%f)\n",
	 procid,
	 name,
	 state,			// XXX
	 is_opened() ? "expand" : "unexpand",
	 this,
	 doi,
	 left, top,
	 width, height,
	 Xreq, Yreq);
  
  Pix pi;
  for (pi = member.first(); pi != 0; member.next(pi)) {
    rproc *p = member(pi);
    p->dump_member(level + 1);
  }
}

void
rproc::dump_children(int level)
{
  int i;
  for (i = 0; i < level; i++)
    printf(" ");
  printf("pid=%d(%s(%d,%s),%x): %f(%f,%f)(%f,%f)(%f,%f)\n",
	 procid,
	 name,
	 state,			// XXX
	 is_opened() ? "expand" : "unexpand",
	 this,
	 doi,
	 left, top,
	 width, height,
	 Xreq, Yreq);
  
  Pix pi;
  for (pi = children.first(); pi != 0; children.next(pi)) {
    rproc *p = children(pi);
    p->dump_children(level + 1);
  }
}

static rproc*
merge_list(rproc *list1, rproc *list2)
{
  if (list2 == NULL)
    return list1;
  if (list1 == NULL)
    return list2;
  
  rproc *head, *cur, *p, *q;
  if (list1->getdoi() < list2->getdoi()) {
    head = cur = list2;
    p = list1;
    q = list2->GetNext();
  } else {
    head = cur = list1;
    p = list1->GetNext();
    q = list2;
  }
  
  while (p != NULL && q != NULL) {
    if (p->getdoi() >= q->getdoi()) {
      cur->SetNext(p);
      cur = p;
      p = p->GetNext();
    } else {
      cur->SetNext(q);
      cur = q;
      q = q->GetNext();
    }
  }
  if (p != NULL)
    cur->SetNext(p);
  if (q != NULL)
    cur->SetNext(q);
  
  return head;
}

static int
compare_section(section *s1, section *s2)
{
  if (s1->start > s2->start)
    return 1;
  if (s1->start < s2->start)
    return -1;
  return 0;
}

static void
print_list(rproc *p)
{
  while (p != NULL) {
    printf("%s(id=%d,doi=%f) ", p->GetName(), p->GetID(), p->getdoi());
    p = p->GetNext();
  }
  printf("\n");
}

/* eof */
