// Copyright (C) 1996  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet/amulet.h>
#include <amulet/object_advanced.h>
#include <amulet/formula_advanced.h>
#include <amulet/inter_advanced.h>
#include <amulet/widgets_advanced.h>
#include <amulet/debugger.h>
#include <strstream.h>
#include "my_amulet.h"
#include "kl_widgets.h"

/* ---------------- Margins ----------------- */ 

int proc_margin = 8;
int slot_width_margin = 8;
int slot_height_margin = 4;
int atom_width_margin = 8;
int atom_height_margin = 2;
int structure_width_margin = 8;
int structure_height_margin = 4;
int guard_width_margin = 8;
int guard_height_margin = 6;
int bind_margin = 2;

/* ---------------- Name List ---------------- */

// ---- slot keys ----
Am_Slot_Key NAME_LIST  	    = Am_Register_Slot_Name("NAME_LIST");

// ---- operations ----

void Add_Name(Am_Object proc, char* name)
{
  proc.Make_Unique(NAME_LIST);
  Am_Value_List name_list = (Am_Value_List)proc.Get(NAME_LIST);
  if (name_list.Empty()) {
    proc.Set(NAME_LIST, name_list.Add(name));
  } else {
    name_list.Add(name, Am_TAIL, false);
    proc.Note_Changed(NAME_LIST);
  }
}

void Remove_Name(Am_Object proc, char* name)
{
  proc.Make_Unique(NAME_LIST);
  Am_Value_List name_list = (Am_Value_List)proc.Get(NAME_LIST);
  name_list.Start();
  if (name_list.Member(name)) {
    name_list.Delete(false);
    proc.Note_Changed(NAME_LIST);
  } else {
    cerr << "No " << name << "in " << proc << endl;
  }
  name_list.Start();
}
  
bool Member_Name(Am_Object proc, char* name)
{
  Am_Value_List name_list = (Am_Value_List)proc.Get(NAME_LIST);
  name_list.Start();
  bool result = name_list.Member(name);
  name_list.Start();
  return result;
}

Am_String Next_Name(Am_Object proc, char* base)
// base $B$r(Bprefix$B$H$7$?%W%m%;%9Cf$G%f%K!<%/$JL>A0$rJV$9!%(B
{
  char name[1000];
  ostrstream oname(name, 1000);
  Am_Value_List name_list = (Am_Value_List)proc.Get(NAME_LIST);
  name_list.Start();
  if (!name_list.Member(base)) {
    oname << base << '\0';
  } else {
    int i = 0; do {
      i = i + 1;
      oname.seekp(0, ios::beg);
      oname << base << i << '\0';
    } while (name_list.Start(), name_list.Member(name));
  }
  name_list.Start();
  return (Am_String)name;
}

/* ---------------- Module ---------------- */

// ---- global objects ----
Am_Object Module_Win;
Am_Object Process_Agg;

// ---- slot keys ----
Am_Slot_Key MODULE_WINS     = Am_Register_Slot_Name("MODULE_WINS");
Am_Slot_Key MODULE_NUM      = Am_Register_Slot_Name("MODULE_NUM");
Am_Slot_Key MODULE_NAME     = Am_Register_Slot_Name("MODULE_NAME");

// ---- operations ----
void Module_Rename(Am_Object win, char* name)
{
  win.Set(MODULE_NAME, name);
}

void Modules_Add(Am_Object modules, Am_Object win, char* name)
{
  Am_String new_name = Next_Name(modules, name);
  Module_Rename(win, new_name);
  modules
    .Set(MODULE_NUM, (int)Am_Screen.Get(MODULE_NUM) + 1)
    .Add_Part(win);
}

// ---- formulas ----
Am_Define_Value_List_Formula(module_wins_form) {
  self.GV(MODULE_NUM);
  Am_Value_List wins = Am_Value_List();
  Am_Part_Iterator gparts = self;
  for (gparts.Start(); !gparts.Last(); gparts.Next()) {
    Am_Object obj = gparts.Get();
    if (obj.Is_Instance_Of(Module_Win))
      wins.Add(obj);
  }
  return wins;
}

Am_Define_Value_List_Formula(module_name_list_form) {
  Am_Value_List names = Am_Value_List();
  Am_Value_List wins = (Am_Value_List)self.GV(MODULE_WINS);
  for (wins.Start(); !wins.Last(); wins.Next()) {
    Am_Object win = (Am_Object)wins.Get();
    names.Add((char*)(Am_String)win.GV(MODULE_NAME));
  }
  return names;
}

void Initialize_Module(void)
{
  Am_Screen
    .Set(MODULE_NUM, 0)
    .Set(MODULE_WINS, Am_Formula::Create(module_wins_form))
    .Set(NAME_LIST, Am_Formula::Create(module_name_list_form));

  Module_Win = Am_Window.Create("Module_Win")
    .Set(MODULE_NAME, "")
    ;

  Am_Object_Advanced obj_adv = (Am_Object_Advanced&)Module_Win;
  obj_adv.Get_Slot(MODULE_NAME)->Set_Single_Constraint_Mode(false);

  Process_Agg = Am_Group.Create("Process_Agg")
    .Set(MODULE_NAME, "")
    .Set(NAME_LIST, Am_Value_List())
    ;
}

/* ---------------- Process, Port, Atoms and Structures ---------------- */

// ---- common slot keys ----
Am_Slot_Key FRAME_PART    = Am_Register_Slot_Name("FRAME_PART");
Am_Slot_Key CONTENTS_PART = Am_Register_Slot_Name("CONTENTS_PART");
Am_Slot_Key NAME_PART     = Am_Register_Slot_Name("NAME_PART");

// ---- common formulas ----

// Frame width
Am_Define_Formula(int, frame_width_proc) {
  return (int)self.GV_Sibling(CONTENTS_PART).GV(Am_WIDTH) +(int)cc.Get_Data();
}
Am_Constraint* KL_Frame_Width(int margin) {
  Am_Formula_Advanced* formula =
    (Am_Formula_Advanced*)Am_Formula::Create(frame_width_proc);
  formula->Set_Data((void*)margin, NULL);
  return formula;
}

// Frame height
Am_Define_Formula(int, frame_height_proc) {
  return (int)self.GV_Sibling(CONTENTS_PART).GV(Am_HEIGHT)+ (int)cc.Get_Data();
}
Am_Constraint* KL_Frame_Height(int margin) {
  Am_Formula_Advanced* formula =
    (Am_Formula_Advanced*)Am_Formula::Create(frame_height_proc);
  formula->Set_Data((void*)margin, NULL);
  return formula;
}

// Frame width + margin
Am_Define_Formula(int, frame_width_plus_proc) {
  return (int)self.GV_Part(FRAME_PART).GV(Am_WIDTH)+ (int)cc.Get_Data();
}
Am_Constraint* KL_Frame_Width_Plus(int margin) {
  Am_Formula_Advanced* formula =
    (Am_Formula_Advanced*)Am_Formula::Create(frame_width_plus_proc);
  formula->Set_Data((void*)margin, NULL);
  return formula;
}

// Frame height + margin
Am_Define_Formula(int, frame_height_plus_proc) {
  return (int)self.GV_Part(FRAME_PART).GV(Am_HEIGHT)+ (int)cc.Get_Data();
}
Am_Constraint* KL_Frame_Height_Plus(int margin) {
  Am_Formula_Advanced* formula =
    (Am_Formula_Advanced*)Am_Formula::Create(frame_height_plus_proc);
  formula->Set_Data((void*)margin, NULL);
  return formula;
}

/* ---------------- Process, Rule, Goal ---------------- */

// ---- global objects ----
Am_Object Process_Proto;
Am_Object Creator_Proto;
Am_Object Append_Proto;
Am_Object Merge_Proto;
Am_Object Rule_Proto;
Am_Object Vanishing_Rule_Proto;
Am_Object Continuous_Rule_Proto;

// ---- slot keys ----

// Process
Am_Slot_Key PROC_NAME       = Am_Register_Slot_Name("PROC_NAME");
Am_Slot_Key PROC_MODULE     = Am_Register_Slot_Name("PROC_MODULE");
Am_Slot_Key PROC_TYPE       = Am_Register_Slot_Name("PROC_TYPE");
#define KL_Creator_Proc 0
#define KL_Normal_Proc  1
Am_Slot_Key PROC_RULES      = Am_Register_Slot_Name("PROC_RULES");


// Rule
Am_Slot_Key RULE_TYPE       = Am_Register_Slot_Name("RULE_TYPE");
#define KL_Transition_Rule 0
#define KL_Network_Rule    1
Am_Slot_Key PARENT_PROC     = Am_Register_Slot_Name("PARENT_PROC");
//Am_Slot_Key PORT_ID_LIST  = Am_Register_Slot_Name("PORT_ID_LIST");
//Am_Slot_Key RULE_NUMBER = Am_Register_Slot_Name("RULE_NUMBER");

// Icon
Am_Slot_Key IS_ICON	    = Am_Register_Slot_Name("IS_ICON");
Am_Slot_Key ICON_TOP        = Am_Register_Slot_Name("ICON_TOP");
Am_Slot_Key ICON_LEFT       = Am_Register_Slot_Name("ICON_LEFT");
Am_Slot_Key ORIG_TOP        = Am_Register_Slot_Name("ORIG_TOP");
Am_Slot_Key ORIG_LEFT       = Am_Register_Slot_Name("ORIG_LEFT");

// Graphical parts
Am_Slot_Key PORTS_PART      = Am_Register_Slot_Name("PORTS_PART");
Am_Slot_Key GOALS_PART      = Am_Register_Slot_Name("GOALS_PART");
Am_Slot_Key GUARD_PART      = Am_Register_Slot_Name("GUARD_PART");
Am_Slot_Key BINDER_PART     = Am_Register_Slot_Name("BINDER_PART");
Am_Slot_Key PARTS_MOVER     = Am_Register_Slot_Name("PARTS_MOVER");
Am_Slot_Key TEXT_INTER_PART = Am_Register_Slot_Name("TEXT_INTER_PART");
Am_Slot_Key SUB_FRAME_PART  = Am_Register_Slot_Name("SUB_FRAME_PART");

// ---- operations ----

////// Process operations //////

bool is_before_port(Am_Object port1, Am_Object port2)
{
  int mode1 = port1.Get(PORT_MODE);
  int mode2 = port2.Get(PORT_MODE);
  if (mode1 == KL_Input && mode2 == KL_Output) return true;
  if (mode1 == KL_Output && mode2 == KL_Input) return false;

  int type1 = port1.Get(PORT_TYPE);
  int type2 = port2.Get(PORT_TYPE);  
  if (type1 == KL_Stream && type2 == KL_Single) return true;
  if (type1 == KL_Single && type2 == KL_Stream) return false;

  Am_String name1 = (Am_String)port1.Get(PORT_NAME);
  Am_String name2 = (Am_String)port2.Get(PORT_NAME);
  if (strcmp(name1, name2) < 0) return true;
  else return false;
}

Am_Value_List sort_ports(Am_Value_List ports)
{
  Am_Value_List new_ports;
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    for (new_ports.Start(); ; new_ports.Next()) {
      if (new_ports.Last()) {
	new_ports.Add(port);
	break;
      }
      if (is_before_port(port, new_ports.Get())) {
	new_ports.Insert(port, Am_BEFORE, false);
	break;
      }
    }
  }
  return new_ports;
}

void Add_Process(Am_Object proc_agg, Am_Object proc)
{
  Am_String proc_name = (Am_String)proc.Get(PROC_NAME);
  if (Member_Name(proc_agg, proc_name)) {
    proc_name = Next_Name(proc_agg, proc_name);
    Proc_Rename(proc, (char *)proc_name);
  }
  proc_agg.Add_Part(proc);
  Add_Name(proc_agg, proc_name);
}

void Remove_Process(Am_Object proc_agg, Am_Object proc)
{
  proc_agg.Remove_Part(proc);
  Remove_Name(proc_agg, (Am_String)proc.Get(PROC_NAME));
}

bool Proc_Rename(Am_Object proc, char* name)
{
  Am_Object proc_agg = proc.Get_Owner();
  if (proc_agg.Valid()) {
    if (Member_Name(proc_agg, name)) return false;
    Remove_Name(proc_agg, (Am_String)proc.Get(PROC_NAME));
    Add_Name(proc_agg, name);
  }
  proc.Get_Part(CONTENTS_PART)
    .Get_Part(NAME_PART)
    .Set(Am_TEXT, name);
  return true;
}

void Rule_Add_Old_Port(Am_Object rule, Am_Object port);
void Rule_Remove_Old_Port(Am_Object rule, Am_Object port);

void Goal_Add_Port(Am_Object proc, Am_Object port)
{
  Am_Object port_agg = proc.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART);
  Add_Name(proc, (Am_String)port.Get(PORT_NAME));
  port_agg.Add_Part(port).Remove_Part(port); // $B:G8e$N0l8DLdBj$N$$$$2C8:$JBP1~!%(B
  port_agg.Add_Part(port);
}

void Goal_Remove_Port(Am_Object proc, Am_Object port)
{
  Am_Object port_agg = proc.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART);
  port_agg.Remove_Part(port);
  Remove_Name(proc, (Am_String)port.Get(PORT_NAME));
}

void Proc_Add_Port(Am_Object proc, Am_Object port, int x, int y, Am_Object ref_obj)
{
  Am_Object port_agg = proc.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART);

  Am_Translate_Coordinates(ref_obj, x, y, port_agg, x, y); // $B:BI8$NJQ49!%(B
  x = imax(0, x);
  y = imax(0, y);
  port.Set(Am_LEFT, x).Set(Am_TOP, y);

  Am_String port_name = (Am_String)port.Get(PORT_NAME); // $BL>A0$N%A%'%C%/$*$h$SEPO?!%(B
  if (Member_Name(proc, port_name)) {
    port_name = Next_Name(proc, port_name);
    Port_Rename(port, port_name);
  }
  Add_Name(proc, port_name);

  port_agg.Add_Part(port).Remove_Part(port); // $B:G8e$N0l8DLdBj$N$$$$2C8:$JBP1~!%(B
  port_agg.Add_Part(port);

  Am_Value_List rules = (Am_Value_List)proc.Get(PROC_RULES);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = (Am_Object)rules.Get();
    Rule_Add_Old_Port(rule, port.Create());
  }
  Am_Instance_Iterator goals = proc;
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = goals.Get();
    Goal_Add_Port(goal, port.Create());
  }
}

void Proc_Remove_Port(Am_Object proc, Am_Object port)
{
  Am_Object port_agg = proc.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART);
  port_agg.Remove_Part(port);
  Remove_Name(proc, (Am_String)port.Get(PORT_NAME));

  Am_Value_List rules = (Am_Value_List)proc.Get(PROC_RULES);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = (Am_Object)rules.Get();
    Am_Value_List ports = Rule_Ports(rule);
    for (ports.Start(); !ports.Last(); ports.Next()) {
      Am_Object pt = (Am_Object)ports.Get();
      if (pt.Is_Instance_Of(port)) {
	Remove_All_Binder(pt);
	Rule_Remove_Old_Port(rule, pt);
      }
    }
  }
  Am_Instance_Iterator goals = proc;
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = goals.Get();
    Am_Value_List ports = Proc_Ports(goal);
    for (ports.Start(); !ports.Last(); ports.Next()) {
      Am_Object pt = (Am_Object)ports.Get();
      if (pt.Is_Instance_Of(port)) {
	Remove_All_Binder(pt);
	Goal_Remove_Port(goal, pt);
      }
    }
  }
}

Am_Object Proc_Ports_Part(Am_Object process)
{
  return process.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART);
}

Am_Value_List Proc_Ports(Am_Object process)
{
  return (Am_Value_List)process.Get_Part(CONTENTS_PART)
                          .Get_Part(PORTS_PART).Get(Am_GRAPHICAL_PARTS);
}

Am_Object Proc_Get_Port(Am_Object process, char* name)
{
  Am_Value_List ports = Proc_Ports(process);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    Am_String port_name = (Am_String)pt.Get(PORT_NAME);
    if (port_name == name) {
      return pt;
    }
  }
  return 0;
}

////// Rule operations //////

/*
// Am_Value_List Index operations
int Object_List_Index(Am_Value_List list, Am_Object value)
{
  int i = 1;
  for (list.Start(); !list.Last(); list.Next(), ++i) {
    if ((Am_Object)list.Get() == value) return i;
  }
  return 0;
}

int String_List_Index(Am_Value_List list, Am_String value)
{
  int i = 1;
  for (list.Start(); !list.Last(); list.Next(), ++i) {
    if (strcmp((Am_String)list.Get(), value) == 0) return i;
  }
  return 0;
}

Am_Define_Formula(int, rule_number_form) {
  Am_Object parent = (Am_Object)self.GV(PARENT_PROC);
  Am_Value_List rules = (Am_Value_List)parent.GV(PROC_RULES);
  return Object_List_Index(rules, self);
}

// Port ID formulas
Am_Define_Formula(int, old_port_id_form) { // old port $B$N(BID
  Am_Object rule = self.GV_Owner().GV_Owner().GV_Owner();
  if (!rule.Valid()) return 0;
  Am_Value_List ports = (Am_Value_List)rule.GV_Part(CONTENTS_PART)
                                           .GV_Part(PORTS_PART).GV(Am_GRAPHICAL_PARTS);
  ports = sort_ports(ports);
  return Object_List_Index(ports, self);
}

Am_Define_Formula(int, out_port_id_form) {
  Am_Object rule;
  for (rule = self; !rule.Is_Instance_Of(Rule_Proto); ) {
    if (!(rule = rule.GV_Owner().GV_Owner().GV_Owner()).Valid()) return 0;
  }
  Am_Value_List ports = (Am_Value_List)rule.GV_Part(CONTENTS_PART)
                                           .GV_Part(PORTS_PART).GV(Am_GRAPHICAL_PARTS);
  int old_ports = ports.Length();
  Am_Value_List names = (Am_Value_List)rule.GV(NAME_LIST);
  return old_ports + String_List_Index(names, (Am_String)self.Get(PORT_NAME));
}

Am_Define_Formula(int, in_port_id_form) { // $B%P%$%s%@$,B8:_$9$l$P$=$3$+$i(BID$B$r0z$/(B
  return 0;
  Am_Value_List binders = (Am_Value_List)self.GV(PORT_BINDERS);
  binders.Start();
  if (binders.Empty()) return 0;
  else return ((Am_Object)binders.Get()).GV(BL_ID);
}
*/
// formulas for Iconify

Am_Object parent_rule(Am_Object obj)
{
  Am_Object parent = obj;
  while((parent = parent.Get_Owner()).Valid()) {
    if (parent.Is_Instance_Of(Rule_Proto))
      return parent;
  }
  return 0;
}

Am_Define_Formula(bool, icon_visible_form)
{
  Am_Object rule = 0;
  Am_Object parent = self;
  while((parent = parent.GV_Owner()).Valid()) {
    if (parent.Is_Instance_Of(Rule_Proto)) {
      rule = parent;
      break;
    }
  }
  if (!rule.Valid()) return true;
// $BFbMF$,%"%$%3%s;~$K8+$($J$$!%(B
  if (rule.GV(IS_ICON)) return false;
  else return true;
}

void Rule_Add_Old_Port(Am_Object rule, Am_Object port)
{
  Am_Object dist_agg = Proc_Ports_Part(rule);
  Am_Object p = port.Create().Set(PORT_AGE, KL_Old_Port);
//  if (s.Get(PORT_TYPE) == KL_Stream)
//    Am_To_Bottom(s.Get_Part(CONTENTS_PART).Get_Part(VALUE_PART)); // marker$B$r0\F0(B
  int top  = p.Get(Am_TOP);
  int left = p.Get(Am_LEFT);
  dist_agg.Add_Part(p.Set(PORT_OPEN, KL_Close)
//		     .Set(PORT_ID,   Am_Formula::Create(old_port_id_form))
		     .Set(Am_TOP,   top).Set(Am_LEFT,   left)
		     .Set(ORIG_TOP, top).Set(ORIG_LEFT, left)
		     .Set(ICON_TOP, top).Set(ICON_LEFT, left))
    .Remove_Part(p);
  dist_agg.Add_Part(p);
  if ((int)rule.Get(RULE_TYPE) == KL_Transition_Rule)
    p.Set(PORT_OPEN, KL_Open);
  // $B=PNO%]!<%H$O%"%$%3%s;~$K$O8+$($J$$!%(B
  if ((int)p.Get(PORT_MODE) == KL_Output) {
    p.Set(Am_VISIBLE, Am_Formula::Create(icon_visible_form));
  }      
}

void Rule_Remove_Old_Port(Am_Object rule, Am_Object port)
{
  Am_Object dist_agg = Proc_Ports_Part(rule);
  dist_agg.Remove_Part(port);
}
  
Am_Object Rule_Create(Am_Object process, Am_Object rule_proto) {
  Am_Object rule;
  // $B%W%m%H%?%$%W7hDj!%(B
  if ((int)rule_proto.Get(RULE_TYPE) == KL_Transition_Rule)
    rule = Continuous_Rule_Proto.Create();
  else
    rule =  Vanishing_Rule_Proto.Create();
  // $B?F%W%m%;%9;XDj!%(B
  rule.Set(PARENT_PROC, process);
  // $B%]!<%HDI2C!%(B
  Am_Object dist_agg = Rule_Ports_Part(rule);
  Am_Value_List ports = Proc_Ports(process);
  Am_Object pt;
  for (ports.Start(); !ports.Last(); ports.Next()) {
    pt = ports.Get();
    int left = pt.Get(Am_LEFT);
    int top = pt.Get(Am_TOP);
    Rule_Add_Old_Port(rule, pt);
    if (rule.Is_Instance_Of(Continuous_Rule_Proto) &&
	(int)pt.Get(PORT_TYPE) == KL_Single && (int)pt.Get(PORT_MODE) == KL_Input) {
      dist_agg.Add_Part(pt.Create()
			.Set(PORT_AGE, KL_New_Port)
			.Set(Am_VISIBLE, Am_Formula::Create(icon_visible_form))
			.Set(ORIG_TOP, top).Set(ORIG_LEFT, left)
			.Set(ICON_TOP, top).Set(ICON_LEFT, left));
    }
  }
  ports.Start();
  return rule;
}

void Rule_Toggle_Icon(Am_Object rule)
{
//  cout << "start Rule_Toggle_Icon on " << rule << endl;
  bool is_icon = (bool)rule.Get(IS_ICON);
  Am_Value_List ports = Proc_Ports(rule);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    int left = pt.Get(Am_LEFT);
    int top = pt.Get(Am_TOP);
    if (is_icon) {
      pt.Set(ICON_LEFT, left)
	.Set(ICON_TOP,  top)
	.Set(Am_LEFT, (int)pt.Get(ORIG_LEFT))
	.Set(Am_TOP,  (int)pt.Get(ORIG_TOP));
//      cout << "icon -> rule" << endl;
    } else {
      pt.Set(ORIG_LEFT, left)
	.Set(ORIG_TOP,  top)
	.Set(Am_LEFT, (int)pt.Get(ICON_LEFT))
	.Set(Am_TOP,  (int)pt.Get(ICON_TOP));
//      cout << "rule -> icon" << endl;
    }
  }
  Am_Object guard_part = Rule_Guard_Part(rule);
  Am_Value_List guards = (Am_Value_List)guard_part.Get(Am_GRAPHICAL_PARTS);
  for (guards.Start(); !guards.Last(); guards.Next()) {
    Am_Object gd = (Am_Object)guards.Get();
    int left = gd.Get(Am_LEFT);
    int top = gd.Get(Am_TOP);
    if (is_icon) {
      gd.Set(ICON_LEFT, left)
	.Set(ICON_TOP,  top)
	.Set(Am_LEFT, (int)gd.Get(ORIG_LEFT))
	.Set(Am_TOP,  (int)gd.Get(ORIG_TOP));
//      cout << "icon -> rule" << endl;
    } else {
      gd.Set(ORIG_LEFT, left)
	.Set(ORIG_TOP,  top)
	.Set(Am_LEFT, (int)gd.Get(ICON_LEFT))
	.Set(Am_TOP,  (int)gd.Get(ICON_TOP));
//      cout << "rule -> icon" << endl;
    }
  }
  rule.Set(IS_ICON, !is_icon); 
}

void Add_Rule(Am_Object rules_agg, Am_Object rule)
{
  Am_Object process = (Am_Object)rule.Get(PARENT_PROC);
  process.Make_Unique(PROC_RULES);
  Am_Value_List rule_list = (Am_Value_List)process.Get(PROC_RULES);
  if (rule_list.Empty()) {
    process.Set(PROC_RULES, rule_list.Add(rule, Am_TAIL, false));
  } else {
    rule_list.Add(rule, Am_TAIL, false);
    process.Note_Changed(PROC_RULES);
  }
  rules_agg.Add_Part(rule);
}
  
void Remove_Rule(Am_Object rules_agg, Am_Object rule)
{
  rules_agg.Remove_Part(rule);
  Am_Object proc = (Am_Object)rule.Get(PARENT_PROC);
  proc.Make_Unique(PROC_RULES);
  Am_Value_List rule_list = (Am_Value_List)proc.Get(PROC_RULES);
  rule_list.Start();
  rule_list.Member(rule);
  rule_list.Delete(false);
  proc.Note_Changed(PROC_RULES);
  rule_list.Start();
}

void Rule_Add_Guard(Am_Object rule, Am_Object guard, int x, int y, Am_Object ref_obj)
{
  Am_Object guard_agg = Rule_Guard_Part(rule);
  Am_Translate_Coordinates(ref_obj, x, y, guard_agg, x, y);
  x = imax(0, x); y = imax(0, y);
  guard_agg.Add_Part(guard.Set(Am_LEFT, x)  .Set(Am_TOP, y)
                          .Set(ORIG_LEFT, x).Set(ORIG_TOP, y)
                          .Set(ICON_LEFT, x).Set(ICON_TOP, y));
  guard_agg.Remove_Part(guard);
  guard_agg.Add_Part(guard);
}

void Rule_Add_Goal(Am_Object rule, Am_Object goal, int x, int y, Am_Object ref_obj)
{
  Am_Object goals_agg = Rule_Goals_Part(rule);
  Am_Translate_Coordinates(ref_obj, x, y, goals_agg, x, y);
  x = imax(0, x); y = imax(0, y);
  goals_agg.Add_Part(goal.Set(Am_LEFT, x).Set(Am_TOP, y));
  goals_agg.Remove_Part(goal);  // $B:G8e$N%4!<%k$r>C$7$F$+$i2C$($k$HI=<($5$l$J$$LdBj$N(B
  goals_agg.Add_Part(goal);     // $BD6%"%I%[%C%/$J2r7h!%(B
}

void Rule_Add_Inner_Port(Am_Object rule, Am_Object port)
{
  Am_String port_name = (Am_String)port.Get(PORT_NAME);
  if (Member_Name(rule, port_name)) {
    port_name = Next_Name(rule, port_name);
    Port_Rename(port, port_name);
  }
  Add_Name(rule, port_name);
  if ((int)port.Get(PORT_MODE) == KL_Input) {
    port // .Set(PORT_ID, Am_Formula::Create(in_port_id_form))
        .Get_Part(CONTENTS_PART)
	  .Get_Part(VALUE_PART)
            .Set(Am_VISIBLE, Am_Formula::Create(icon_visible_form));
                             // $BF~NO%]!<%H$NFbMF$O%"%$%3%s;~$K$O8+$($J$$!%(B
  } else {
//    port.Set(PORT_ID, Am_Formula::Create(out_port_id_form));
  }
}

void Rule_Remove_Inner_Port(Am_Object rule, Am_Object port)
{
//  port.Set(PORT_ID, 0);
  Remove_Name(rule, (Am_String)port.Get(PORT_NAME));
}

Am_Object Rule_Ports_Part(Am_Object rule)
{
  return rule.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART);
}

Am_Value_List Rule_Ports(Am_Object rule)
{
  return (Am_Value_List)rule.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART).Get(Am_GRAPHICAL_PARTS);
}

Am_Object Rule_Binder_Part(Am_Object rule)
{
  return rule.Get_Part(CONTENTS_PART).Get_Part(BINDER_PART);
}

Am_Value_List Rule_Binder(Am_Object rule)
{
  return (Am_Value_List)rule.Get_Part(CONTENTS_PART).Get_Part(BINDER_PART).Get(Am_GRAPHICAL_PARTS);
}

Am_Object Rule_Guard_Part(Am_Object rule)
{
  return rule.Get_Part(CONTENTS_PART).Get_Part(GUARD_PART);
}

Am_Value_List Rule_Guard(Am_Object rule)
{
  return (Am_Value_List)rule.Get_Part(CONTENTS_PART).Get_Part(GUARD_PART).Get(Am_GRAPHICAL_PARTS);
}

Am_Object Rule_Goals_Part(Am_Object rule)
{
  return rule.Get_Part(CONTENTS_PART).Get_Part(GOALS_PART);
}

Am_Value_List Rule_Goals(Am_Object rule)
{
  return (Am_Value_List)rule.Get_Part(CONTENTS_PART).Get_Part(GOALS_PART).Get(Am_GRAPHICAL_PARTS);
}


/* -----------------------------------------------------------------------
void Add_Port_ID(Am_Object rule, int id)
{
  rule.Make_Unique(PORT_ID_LIST);
  Am_Value_List name_list = (Am_Value_List)rule.Get(PORT_ID_LIST);
  if (name_list.Empty()) {
    rule.Set(PORT_ID_LIST, name_list.Add(id));
  } else {
    name_list.Add(id, Am_TAIL, false);
    rule.Note_Changed(PORT_ID_LIST);
  }
}

void Remove_Port_ID(Am_Object rule, int id)
{
  rule.Make_Unique(PORT_ID_LIST);
  Am_Value_List name_list = (Am_Value_List)rule.Get(PORT_ID_LIST);
  name_list.Start();
  name_list.Member(id);
  name_list.Delete(false);
  rule.Note_Changed(PORT_ID_LIST);
  name_list.Start();
}
p  
bool Member_Port_ID(Am_Object rule, int id)
{
  Am_Value_List name_list = (Am_Value_List)rule.Get(PORT_ID_LIST);
  name_list.Start();
  bool result = name_list.Member(id);
  name_list.Start();
  return result;
}

int Next_Port_ID(Am_Object rule)
{
  Am_Value_List id_list = (Am_Value_List)rule.Get(PORT_ID_LIST);
  id_list.Start();
  int id;
  for (id = 1; id_list.Start(), id_list.Member(id); ++id)
    ;
  id_list.Start();
  return id;
}
---------------------------------------------------------------------- */

// Goal operations

Am_Define_Formula(bool, goal_contents_visible_form) {
  Am_Object goal = self.GV_Owner().GV_Owner();
  if (goal.Valid()) {
    if ((bool)goal.GV(IS_ICON)) return false;
  }
  return true;
}

Am_Define_String_Formula(goal_module_form) {
  Am_Object parent = (Am_Object)self.GV(PARENT_PROC);
  if (!parent.Valid()) return (Am_String)"";
  return (Am_String)parent.GV(PROC_MODULE);
}

Am_Define_String_Formula(goal_name_form) {
  char name[1000];
  ostrstream oname(name, 1000);
  // $B?F%W%m%;%9$NL>A0(B
  Am_Object parent = (Am_Object)self.GV_Owner().GV_Owner().GV(PARENT_PROC);
  if (!parent.Valid()) return (Am_String)"";
  Am_String parent_name = (Am_String)parent.GV(PROC_NAME);
  // $BAk$,F1$8$+$I$&$+$G%b%8%e!<%k$,0c$&$+H=Dj!%$-!$1x$$!%!%!%(B
  Am_Object parent_win = (Am_Object)parent.GV(Am_WINDOW);
  Am_Object my_win = (Am_Object)self.GV(Am_WINDOW);
  if (my_win != parent_win) {
    oname << (Am_String)self.GV_Owner().GV_Owner().GV(PROC_MODULE) << ":";
  }
  oname << parent_name << '\0';
  return (Am_String) name;
}

Am_Object Goal_Create(Am_Object process) {
  Am_Object goal = process.Create();
  int left = goal.Get(Am_LEFT);
  int top = goal.Get(Am_TOP);
  goal
    .Set(PARENT_PROC, process)
    .Set(PROC_MODULE, Am_Formula::Create(goal_module_form))
    .Set(IS_ICON, false)
    .Set(ORIG_TOP, top).Set(ORIG_LEFT, left)
    .Set(ICON_TOP, top).Set(ICON_LEFT, left)
    .Get_Part(CONTENTS_PART)
      .Get_Part(NAME_PART)
        .Set(Am_TEXT, Am_Formula::Create(goal_name_form))
      .Get_Owner()
      .Get_Part(PORTS_PART)
        .Set(Am_VISIBLE, Am_Formula::Create(goal_contents_visible_form))
      .Get_Owner();
  Am_Object dist_agg = Proc_Ports_Part(goal);
  Am_Value_List slots = Proc_Ports(process);
  Am_Object port;
  for (slots.Start(); !slots.Last(); slots.Next()) {
    port = slots.Get();
    port = port.Create()
      .Set(Am_TOP, (int)port.Get(Am_TOP))
      .Set(Am_LEFT, (int)port.Get(Am_LEFT));
    if ((int)port.Get(PORT_MODE) == KL_Input) // Input port $B$OJD$8$F$*$/!%(B
      port.Set(PORT_OPEN, KL_Close);
    if ((int)port.Get(PORT_MODE) == KL_Input) {
      dist_agg.Add_Part(port /* .Set(PORT_ID, Am_Formula::Create(in_port_id_form)) */);
    } else {
      dist_agg.Add_Part(port /* .Set(PORT_ID, Am_Formula::Create(out_port_id_form)) */);
    }
  }
  slots.Start();
  return goal;
}

void Goal_Toggle_Icon(Am_Object goal)
{
  goal.Set(IS_ICON, !(bool)goal.Get(IS_ICON));
}

// ---- formulas ----
Am_Define_Formula(int, under_process_name) {
  Am_Object name = self.GV_Sibling(NAME_PART);
  if (name.Valid())
    return (int)name.GV(Am_HEIGHT) + 1;
  else
    return 0;
}

Am_Define_String_Formula (process_name_formula) {
  Am_Object name = self.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  if (name.Valid())
    return (Am_String)name.GV(Am_TEXT);
  else
    return (Am_String)"";
}

Am_Define_String_Formula (process_module_formula) {
  Am_Object proc_agg = self.GV_Owner();
  if (proc_agg.Valid() && proc_agg.Is_Instance_Of(Process_Agg))
    return (Am_String)proc_agg.GV(MODULE_NAME);
  else
    return (Am_String)"";
}

Am_Define_String_Formula (rule_name_for_text) {
  Am_Object rule = self.GV_Owner().GV_Owner();
  if (rule.Valid()) 
    return (Am_String)rule.GV(PROC_NAME);
  else
    return (Am_String)"";
}

Am_Define_String_Formula (rule_name_formula) {
  Am_Object parent = (Am_Object)self.GV(PARENT_PROC);
  if (parent.Valid()) 
    return (Am_String)parent.GV(PROC_NAME);
  else
    return (Am_String)"";
}

Am_Define_Formula (int, rule_type_formula) {
  Am_Object parent = (Am_Object)self.GV(PARENT_PROC);
  if (parent.Valid()) 
    return parent.GV(PROC_TYPE);
  else
    return KL_Normal_Proc;
}

Am_Define_Formula(bool, creator_visible) {
  if ((int)self.GV_Owner().GV(PROC_TYPE) == KL_Creator_Proc)
    return true;
  else
    return false;
}

Am_Define_Formula(bool, normal_proc_visible) {
  if ((int)self.GV_Owner().GV(PROC_TYPE) == KL_Creator_Proc)
    return false;
  else
    return true;
}

Am_Define_Formula(int, creator_frame_radius_form) {
  Am_Object parent = (Am_Object)self.GV_Owner().GV(PARENT_PROC);
  if (parent.Valid())
    return (int)parent.GV_Part(SUB_FRAME_PART).GV(Am_RADIUS);
  else
    return 0;
}

void Initialize_Process(void)
{
  Am_Object parts_mover = Raise_Move_Grow_Interactor.Create("parts_mover")
    .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&Am_Inter_In_Part)
    .Set(Am_GRID_PROC,        (Am_Object_Proc*)&Grid_Positive)
    ;
  // Process
  Am_Object process_frame = Am_Border_Rectangle.Create("process_frame")
    .Set(Am_VISIBLE,    Am_Formula::Create(normal_proc_visible))
    .Set(Am_WIDTH,      KL_Frame_Width(proc_margin))
    .Set(Am_HEIGHT,     KL_Frame_Height(proc_margin))
    .Set(Am_FILL_STYLE, Am_Motif_Light_Gray)
    ;
  Am_Object creator_frame = Am_Roundtangle.Create("creator_frame")
    .Set(Am_VISIBLE,    Am_Formula::Create(creator_visible))
    .Set(Am_WIDTH,      KL_Frame_Width(proc_margin))
    .Set(Am_HEIGHT,     KL_Frame_Height(proc_margin))
    .Set(Am_RADIUS,	0)
    .Set(Am_FILL_STYLE, Am_No_Style)
    .Set(Am_LINE_STYLE, Am_Line_2)
    ;
  Am_Object process_name = Am_Text.Create("process_name")
    .Set(Am_TEXT, "process")
    .Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--12-*"))
    ;
  Am_Object parts_group = Am_Group.Create("parts_group")
    .Set(Am_TOP,    Am_Formula::Create(under_process_name))
    .Set(Am_WIDTH,  Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    .Add_Part(PARTS_MOVER, parts_mover)
    ;
  Am_Object process_contents = Am_Group.Create("process_contents")
    .Set(Am_LEFT,   proc_margin /2)
    .Set(Am_TOP,    proc_margin /2)
    .Set(Am_WIDTH,  Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    .Add_Part(PORTS_PART, parts_group.Create("ports_part"))
    .Add_Part(NAME_PART,  process_name)
    ;

  Process_Proto = Am_Group.Create("Process_Proto")
    .Set(PROC_NAME,   Am_Formula::Create(process_name_formula))
    .Set(PROC_MODULE, Am_Formula::Create(process_module_formula))
    .Set(PROC_TYPE,   KL_Normal_Proc)
    .Set(PROC_RULES,  Am_Value_List())
    .Set(NAME_LIST,   Am_Value_List())
    .Set(Am_WIDTH,    Am_From_Part(FRAME_PART, Am_WIDTH))
    .Set(Am_HEIGHT,   Am_From_Part(FRAME_PART, Am_HEIGHT))
    .Add_Part(FRAME_PART,     process_frame)
    .Add_Part(SUB_FRAME_PART, creator_frame)
    .Add_Part(CONTENTS_PART,  process_contents)
    ;

  Creator_Proto = Process_Proto.Create("Creator_Proto")
    .Set(PROC_TYPE, KL_Creator_Proc)
    ;
  Proc_Rename(Creator_Proto, "creator");

  Append_Proto = Process_Proto.Create("Append_Proto");
  Merge_Proto = Process_Proto.Create("Merge_Proto");

  // Rule
  Am_Object rule_frame = Am_Rectangle.Create("rule_frame")
    .Set(Am_VISIBLE,    Am_Formula::Create(normal_proc_visible))
    .Set(Am_WIDTH,      KL_Frame_Width(proc_margin))
    .Set(Am_HEIGHT,     KL_Frame_Height(proc_margin))
    .Set(Am_FILL_STYLE, Am_Motif_Light_Gray)
    ;
  Am_Object cont_rule_frame = Am_Border_Rectangle.Create("cont_rule_frame")
    .Set(Am_VISIBLE,    Am_Formula::Create(normal_proc_visible))
    .Set(Am_WIDTH,      KL_Frame_Width(proc_margin))
    .Set(Am_HEIGHT,     KL_Frame_Height(proc_margin))
    .Set(Am_FILL_STYLE, Am_Motif_Light_Gray)
    ;
  Am_Object cre_rule_frame = Am_Roundtangle.Create("cre_rule_frame")
    .Set(Am_VISIBLE,    Am_Formula::Create(creator_visible))
    .Set(Am_WIDTH,      KL_Frame_Width(proc_margin))
    .Set(Am_HEIGHT,     KL_Frame_Height(proc_margin))
    .Set(Am_RADIUS,	Am_Formula::Create(creator_frame_radius_form))
    .Set(Am_FILL_STYLE, Am_No_Style)
    .Set(Am_LINE_STYLE,
	 Am_Style (0, 0, 0, 2, Am_CAP_BUTT, Am_JOIN_MITER,
		   Am_LINE_ON_OFF_DASH, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_SOLID, Am_FILL_POLY_EVEN_ODD, Am_Image_Array(50)) )
    ;
  Am_Object cre_cont_rule_frame = Am_Rectangle.Create("cre_cont_rule_frame")
    .Set(Am_VISIBLE,    Am_Formula::Create(creator_visible))
    .Set(Am_WIDTH,      KL_Frame_Width(proc_margin))
    .Set(Am_HEIGHT,     KL_Frame_Height(proc_margin))
    .Set(Am_FILL_STYLE, Am_No_Style)
    .Set(Am_LINE_STYLE,
	 Am_Style (0, 0, 0, 2, Am_CAP_BUTT, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_SOLID, Am_FILL_POLY_EVEN_ODD, Am_Image_Array(50)) )
    ;

  Am_Object rule_name = Am_Text.Create("rule_name")
    .Set(Am_TEXT, Am_Formula::Create(rule_name_for_text))
    .Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--12-*"))
    ;
  Am_Object rule_contents = Am_Group.Create("rule_contents")
    .Set(Am_LEFT,   proc_margin / 2)
    .Set(Am_TOP,    proc_margin / 2)
    .Set(Am_WIDTH,  Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    .Add_Part(GOALS_PART,  parts_group.Create("goals_part")
	      .Set(Am_VISIBLE, Am_Formula::Create(icon_visible_form)))
    .Add_Part(PORTS_PART,  parts_group.Create("ports_part"))
    .Add_Part(GUARD_PART,  parts_group.Create("guard_part"))
    .Add_Part(BINDER_PART, parts_group.Create("binder_part")
	      .Set(Am_VISIBLE, Am_Formula::Create(icon_visible_form)))
    .Add_Part(NAME_PART,   rule_name)
    ;
  rule_contents.Get_Part(PORTS_PART).Remove_Part(PARTS_MOVER);
  rule_contents.Get_Part(BINDER_PART).Remove_Part(PARTS_MOVER);

  Rule_Proto = Am_Group.Create("Rule_Proto")
    .Set(RULE_TYPE,    KL_Transition_Rule)
    .Set(IS_ICON, false)
    .Set(NAME_LIST,    Am_Value_List())
//    .Set(PORT_ID_LIST, Am_Value_List())
//    .Set(RULE_NUMBER,  Am_Formula::Create(rule_number_form))
    .Set(PARENT_PROC,  Process_Proto)
    .Set(PROC_NAME,    Am_Formula::Create(rule_name_formula))
    .Set(PROC_TYPE,    Am_Formula::Create(rule_type_formula))
    ;
  Vanishing_Rule_Proto = Rule_Proto.Create("Vanishing_Rule_Proto")
    .Set(RULE_TYPE, KL_Network_Rule)
    .Set(Am_WIDTH,  Am_From_Part(FRAME_PART, Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Part(FRAME_PART, Am_HEIGHT))
    .Add_Part(FRAME_PART,     rule_frame)
    .Add_Part(SUB_FRAME_PART, cre_rule_frame)
    .Add_Part(CONTENTS_PART,  rule_contents.Create());
  Continuous_Rule_Proto = Rule_Proto.Create("Continuous_Rule_Proto")
    .Set(RULE_TYPE, KL_Transition_Rule)
    .Set(Am_WIDTH,  Am_From_Part(FRAME_PART, Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Part(FRAME_PART, Am_HEIGHT))
    .Add_Part(FRAME_PART,     cont_rule_frame)
    .Add_Part(SUB_FRAME_PART, cre_cont_rule_frame)
    .Add_Part(CONTENTS_PART,  rule_contents.Create());
}

/* ---------------- Port ---------------- */

// ---- global objects ----
Am_Object Port_Proto;
Am_Object In_Single_Proto;
Am_Object Out_Single_Proto;
Am_Object In_Stream_Proto;
Am_Object Out_Stream_Proto;

// ---- Port's slot keys ----
Am_Slot_Key VALUE_PART = Am_Register_Slot_Name("VALUE_PART");
Am_Slot_Key MARKER_PART = Am_Register_Slot_Name("MARKER_PART");

Am_Slot_Key PORT_NAME = Am_Register_Slot_Name("PORT_NAME");
Am_Slot_Key PORT_AGE = Am_Register_Slot_Name("PORT_AGE");
#define KL_New_Port	0
#define KL_Old_Port	1
Am_Slot_Key PORT_TYPE = Am_Register_Slot_Name("PORT_TYPE");
#define KL_Single	0
#define KL_Stream	1
Am_Slot_Key PORT_MODE = Am_Register_Slot_Name("PORT_MODE");
#define KL_Output	0
#define KL_Input	1
Am_Slot_Key PORT_OPEN = Am_Register_Slot_Name("PORT_OPEN");
#define KL_Open		0
#define KL_Close	1
Am_Slot_Key PORT_DIR = Am_Register_Slot_Name("PORT_DIR");
Am_Slot_Key PORT_DIR_DAEMON = Am_Register_Slot_Name("PORT_DIR_DAEMON");
// Am_Slot_Key PORT_DIR_TOGGLER = Am_Register_Slot_Name("PORT_DIR_TOGGLER");
#define KL_Left		0
#define KL_Right	1
// Am_Slot_Key PORT_ID = Am_Register_Slot_Name("PORT_ID");
Am_Slot_Key PORT_BINDERS = Am_Register_Slot_Name("PORT_BINDERS");

// ---- operations ----

bool Port_Rename(Am_Object port, char* name)
{
  Am_Object parent = Port_Parent_Rule(port);
  if (!parent.Valid()) parent = Port_Parent_Proc(port);
  if (parent.Valid()) {
    if (Member_Name(parent, name)) return false;
    Remove_Name(parent, (Am_String)port.Get(PORT_NAME));
    Add_Name(parent, name);
  }
  port.Get_Part(CONTENTS_PART)
    .Get_Part(NAME_PART)
    .Set(Am_TEXT, name);
  return true;
}

void Port_Add(Am_Object port, Am_Object data)
{
  if (data.Is_Instance_Of(Port_Proto)) {
    if ((int)data.Get(PORT_TYPE) == KL_Stream &&
	(int)data.Get(PORT_MODE) == KL_Input)
      data.Set(PORT_OPEN, KL_Close);
  }
  Am_Object values_agg = port.Get_Part(CONTENTS_PART).Get_Part(VALUE_PART);
  values_agg.Add_Part(data).Remove_Part(data);
  values_agg.Add_Part(data);
}

void Port_Remove(Am_Object port, Am_Object data)
{
  Am_Object values_agg = port.Get_Part(CONTENTS_PART).Get_Part(VALUE_PART);
  values_agg.Remove_Part(data);
}

void Port_Remove_Value(Am_Object value)
{
  value.Remove_From_Owner();
}

Am_Object Port_Parent(Am_Object port)
{
  Am_Object parent = port.Get_Owner().Get_Owner().Get_Owner();
  if (parent.Valid()) return parent;
  else return 0;
}

Am_Object Port_Top_Parent(Am_Object port)
{
  Am_Object parent = port;
  Am_Object tmp;
  while((tmp = Port_Parent(parent)).Valid()) {
    if (!tmp.Is_Instance_Of(Port_Proto) && !tmp.Is_Instance_Of(Structure_Proto))
      return parent;
    parent = tmp;
  }
  return 0;
}

Am_Object Port_Parent_Proc(Am_Object port) // $B0lHV6a$$?F%W%m%;%9(B,$B%4!<%k(B,$B%k!<%k$rJV$9!%(B
{
  Am_Object parent = port;
  while((parent = Port_Parent(parent)).Valid()) {
    if (!parent.Is_Instance_Of(Port_Proto) && !parent.Is_Instance_Of(Structure_Proto))
      return parent;
  }
  return 0;
}

Am_Object Port_Parent_Rule(Am_Object port) // $B?F%k!<%k$rJV$9!%(B
{
  Am_Object parent = port;
  while((parent = Port_Parent(parent)).Valid()) {
    if (parent.Is_Instance_Of(Rule_Proto))
      return parent;
  }
  return 0;
}

Am_Object Port_Values_Part(Am_Object port)
{
  return port.Get_Part(CONTENTS_PART).Get_Part(VALUE_PART);
}

Am_Value_List Port_Values(Am_Object port)
{
  return (Am_Value_List)port.Get_Part(CONTENTS_PART).Get_Part(VALUE_PART).Get(Am_GRAPHICAL_PARTS);
}

bool Port_Has_Value(Am_Object port)
{
  Am_Value_List values = Port_Values(port);
  return !values.Empty();
}

bool Port_Is_Bound(Am_Object port)
{
  Am_Value_List binder_list = (Am_Value_List)port.Get(PORT_BINDERS);
  return !binder_list.Empty();
}

bool Port_Add_Allowed(Am_Object src_port, Am_Object dest_port)
{
  // Valid $B%A%'%C%/(B
  if (!(src_port.Valid() && dest_port.Valid())) return false;

  // atom $B$K$O(B add $B$G$-$J$$!%(Bport $B$+(B structure $B$G$"$k$3$H$rJ]>Z$9$k!%(B
  if (dest_port.Is_Instance_Of(Atom_Proto)) return false;

  int dest_type = dest_port.Get(PORT_TYPE);
  int dest_age  = dest_port.Get(PORT_AGE);
  int dest_mode = dest_port.Get(PORT_MODE);

  // single port $B$K$O0l$D$7$+F~$i$J$$!%(B
  if ((dest_type == KL_Single) && (Port_Has_Value(dest_port) || Port_Is_Bound(dest_port)))
    return false;

  // atom, structure $B$NDI2C!%(B
  if (src_port.Is_Instance_Of(Atom_Proto)||src_port.Is_Instance_Of(Structure_Proto)) {
    // dest_port $B$,(B structure $B$N$H$-$b$3$l$K4^$^$l$k!%(B
    if (dest_mode == KL_Input || dest_type == KL_Stream) return true;
    // single output port $B$O(B old $B$N$_5v$9!%(B
    if (dest_type == KL_Single && dest_age == KL_Old_Port) return true;
  } else if (src_port.Is_Instance_Of(Port_Proto)) {
    int src_type  =  src_port.Get(PORT_TYPE);
    int src_age   =  src_port.Get(PORT_AGE);
    int src_mode  =  src_port.Get(PORT_MODE);
    bool dest_is_goal_port = Port_Parent(dest_port).Is_Instance_Of(Process_Proto);
    bool dest_is_rule_port = Port_Parent(dest_port).Is_Instance_Of(Rule_Proto);
    // single port $B$NDI2C!%(B
    if (src_type == KL_Single) {
      if (dest_type == KL_Stream) {
	// $B%4!<%k$N=PNO(B stream$B0J30$O(BOK.
	if (dest_is_goal_port && dest_mode == KL_Output) return false;
	else return true;
      }
      if (dest_type == KL_Single) {
	if (src_mode == KL_Output && dest_is_goal_port && dest_mode == KL_Output)
	  return true;
	if (src_mode == KL_Output && dest_is_rule_port && dest_mode == KL_Input)
	  return true;
	return false;
      }
    }
    // stream port $B$NDI2C!%(B
    if (src_type == KL_Stream) {
      if (dest_is_goal_port && dest_mode == KL_Output) return false; // goal $B$O$@$a!%(B
      if (dest_type != KL_Single) return true; // structure $B$O(B Stream$B07$$$J$N$G(Bok.
    }
  }
  return false;
}

bool Port_Move_Allowed(Am_Object src_port, Am_Object dest_port)
{
  // Valid $B%A%'%C%/(B
  if (!(src_port.Valid() && dest_port.Valid())) return false;
  // $BF1$8%]!<%H$O$@$a!%(B
  if (src_port == dest_port) return false;
  // $BF1$8%k!<%k$NCf$K$"$k$+$N%A%'%C%/!%(B
  if (Port_Parent_Rule(src_port) != Port_Parent_Rule(dest_port)) return false;
  // atom $B$K$O(B add $B$G$-$J$$!%(Bport $B$+(B structure $B$G$"$k$3$H$rJ]>Z$9$k!%(B
  if (dest_port.Is_Instance_Of(Atom_Proto)) return false;

  int dest_type = dest_port.Get(PORT_TYPE);
  int dest_age  = dest_port.Get(PORT_AGE);
  int dest_mode = dest_port.Get(PORT_MODE);

  // single port $B$K$O0l$D$7$+F~$i$J$$!%(B
  if (dest_type == KL_Single && (Port_Has_Value(dest_port) || Port_Is_Bound(dest_port)))
    return false;

  // atom, structure $B$NDI2C!%(B
  if (src_port.Is_Instance_Of(Atom_Proto)||src_port.Is_Instance_Of(Structure_Proto)) {
    // dest_port $B$,(B structure $B$N$H$-$b$3$l$K4^$^$l$k!%(B
    if (dest_mode = KL_Input || dest_type == KL_Stream) return true;
    // single output port $B$O(B old $B$N$_5v$9!%(B
    if (dest_type == KL_Single || dest_age == KL_Old_Port) return true;
  } else if (src_port.Is_Instance_Of(Port_Proto)) {
    int src_type  =  src_port.Get(PORT_TYPE);
    int src_age   =  src_port.Get(PORT_AGE);
    int src_mode  =  src_port.Get(PORT_MODE);
    bool dest_is_goal_port = Port_Parent(dest_port).Is_Instance_Of(Process_Proto);
    bool dest_is_rule_port = Port_Parent(dest_port).Is_Instance_Of(Rule_Proto);
    // single port $B$NDI2C!%(B
    if (src_type == KL_Single) {
      if (dest_type == KL_Stream) {
	// $B%4!<%k$N=PNO(B stream$B0J30$O(BOK.
        //	if (dest_is_goal_port && dest_mode == KL_Output) return false;
        //	else
	  return true;
      }
      if (dest_type == KL_Single && src_mode == KL_Output) {
	if (/* dest_is_goal_port && */dest_mode == KL_Input && dest_age == KL_New_Port) return true;
	if (dest_is_rule_port && dest_mode == KL_Output) return true;
	if (dest_is_rule_port && dest_mode == KL_Input && dest_age == KL_Old_Port)
	  return true;
	return false;
      }
    }
    // stream port $B$NDI2C!%(B
    if (src_type == KL_Stream) {
      if (dest_is_goal_port && dest_mode == KL_Output) return false; // goal $B$O$@$a!%(B
      if (dest_type != KL_Single) return true; // structure $B$O(B Stream$B07$$$J$N$G(Bok.
    }
  }
  return false;
}

bool Port_Unify_Allowed(Am_Object src_port, Am_Object dest_port)
{
  // Valid $B%A%'%C%/(B
  if (!(src_port.Valid() && dest_port.Valid())) return false;
  // $BF1$8%]!<%H$O$@$a!%(B
  if (src_port == dest_port) return false;
  // $BF1$8%k!<%k$NCf$K$"$k$+$N%A%'%C%/!%(B
  if (Port_Parent_Rule(src_port) != Port_Parent_Rule(dest_port)) return false;
  // Transition Rule $B$G$O(BUnify$B$OIT2D!%(B
  if ((int)Port_Parent_Rule(src_port).Get(RULE_TYPE) == KL_Transition_Rule) return false;
  // $BF1$8%?%$%W(B(stream, single)$B$+$I$&$+$N%A%'%C%/!%(B
  int src_type = (int)src_port.Get(PORT_TYPE);
  int dest_type = (int)dest_port.Get(PORT_TYPE);
  if (src_type != dest_type) return false;
  // $BF1$8%4!<%kCf$N%K%e!<%]!<%HF1;N$O$@$a!%(B
  int src_age = src_port.Get(PORT_AGE);
  int dest_age = dest_port.Get(PORT_AGE);
//   if (Port_Parent_Proc(src_port) == Port_Parent_Proc(dest_port)
//       && src_age == KL_New_Port && dest_age == KL_New_Port) return false;
  // $B%b!<%I$,5U$K$J$C$F$$$k$+$I$&$+$N%A%'%C%/!%(B
  int src_mode = src_port.Get(PORT_MODE); // $B$3$3$G$3$s$J$3$H$9$k$N$O$d$J$s$@$1$I!%!%(B
  if (src_age == KL_New_Port) src_mode = !src_mode;
  int dest_mode = dest_port.Get(PORT_MODE);
  if (dest_age == KL_New_Port) dest_mode = !dest_mode;
  if (src_mode == dest_mode) return false;
  // $BC10lF~NO%]!<%H$O$R$H$D$7$+<u$1<h$l$J$$!%(B
  if (dest_type == KL_Single
      && (Port_Is_Bound(dest_port) || Port_Has_Value(dest_port))) return false;
  // $B8E$$C10lF~NO%]!<%H$K(B,$B$9$G$KCM$,F~$C$F$$$k!%(B
  if (src_age == KL_Old_Port && src_type == KL_Single && Port_Has_Value(src_port))
    return false;
  // $B$9$G$K7k9g$5$l$F$$$k$+$I$&$+!%(B
  Am_Value_List binders = (Am_Value_List)src_port.Get(PORT_BINDERS);
  for (binders.Start(); !binders.Last(); binders.Next()) {
    Am_Object line = (Am_Object)binders.Get();
    if ((Am_Object)line.Get(BL_DEST) == dest_port) return false;
  }
  return true;
}

void Add_Binder(Am_Object port, Am_Object binder)
{
  port.Make_Unique(PORT_BINDERS);
  Am_Value_List binder_list = (Am_Value_List)port.Get(PORT_BINDERS);
  if (binder_list.Empty()) {
    port.Set(PORT_BINDERS, binder_list.Add(binder));
  } else {
    binder_list.Add(binder, Am_TAIL, false);
    port.Note_Changed(PORT_BINDERS);
  }
}

void Remove_Binder(Am_Object port, Am_Object binder)
{
  port.Make_Unique(PORT_BINDERS);
  Am_Value_List binder_list = (Am_Value_List)port.Get(PORT_BINDERS);
  binder_list.Start();
  binder_list.Member(binder);
  binder_list.Delete(false);
  binder_list.Start();
  port.Note_Changed(PORT_BINDERS);
  return;
}

void Remove_All_Binder(Am_Object port)
{
  Am_Value_List binders = (Am_Value_List)port.Get(PORT_BINDERS);
  for (binders.Start(); !binders.Last(); binders.Next()) {
    Am_Object line = (Am_Object)binders.Get();
    Binding_Line_Remove(line);
  }
  Am_Value_List values = Port_Values(port);
  for (values.Start(); !values.Last(); values.Next()) {
    Am_Object value = (Am_Object)values.Get();
    if (!value.Is_Instance_Of(Atom_Proto)) {
      Remove_All_Binder(value);
    }
  }
}
  

// ---- formulas ----
Am_Define_Formula(int, port_type_to_radius) {
  if ((int)self.GV_Owner().GV(PORT_TYPE) == KL_Single) 
    return 7;
  else
    return 0;
}

Am_Define_Formula(int, head_dir_daemon) {
  Am_Object slot = self.GV_Owner();
  if (!slot.Valid()) return 0;
  int head_dir = slot.GV(PORT_DIR);
  if (head_dir == KL_Left) {
    self.Set(Am_LAYOUT, Am_Horizontal_Layout);
    self.Get_Part(VALUE_PART).Set(Am_LAYOUT, Am_Horizontal_Layout);
  } else {
    self.Set(Am_LAYOUT, Am_Horizontal_Rev_Layout);
    self.Get_Part(VALUE_PART).Set(Am_LAYOUT, Am_Horizontal_Rev_Layout);
  }
  return 0;
}

void head_dir_toggle(Am_Object cmd) {
  Am_Object slot = cmd.Get_Owner().Get_Owner().Get_Owner().Get_Owner().Get_Owner();
//  cout << slot << endl;
//  cout << (int)slot.Get(HEAD_DIR) << endl;
  if (slot.Valid()) 
    slot.Set(PORT_DIR, 1 - (int)slot.Get(PORT_DIR));
//  cout << (int)slot.Get(HEAD_DIR) << endl;
//  Am_Call(Am_Object_Proc, Am_Choice_Command, Am_DO_ACTION, (cmd));
}

Am_Define_String_Formula (port_name_form) {
//  cout << self << self.GV_Owner() << self.GV_Owner().GV_Owner() << endl;
  Am_Object port_name = self.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  if (port_name.Valid()) 
    return (Am_String)port_name.GV(Am_TEXT);
  else
    return (Am_String)"";
}

Am_Define_Style_Formula(port_fill_style) {
  int age = self.GV_Owner().GV(PORT_AGE);
  if (age == KL_New_Port)
    return Am_Motif_Light_Gray;
  else
    return Am_Motif_Green;
}

Am_Define_Style_Formula(slot_name_color) {
  Am_Object slot = self.GV_Owner().GV_Owner();
  if (slot.Valid()) {
    int age = slot.GV(PORT_AGE);
    if (age == KL_New_Port)
      return Am_Black;
    else
      return Am_White;
  }
  return Am_Black;
}

// marker's formula
Am_Define_Formula(int, marker_height_form) { // $BL>A0$HF1$89b$5!%(B
  Am_Object owner = self.GV_Owner();
  if (!owner.Valid()) return 0;
  Am_Object name_part = owner.GV_Part(NAME_PART);
  if (name_part.Valid())
    return (int)name_part.GV(Am_HEIGHT);
  else
    return 0;
}

Am_Define_Formula(int, marker_width_form) { // Open$B;~$OL>A0$N9b$5$N(B2/3, Close$B;~$OF1$8!%(B
  Am_Object port = self.GV_Owner().GV_Owner();
  if (port.Valid()) {
    if ((int)port.GV(PORT_OPEN) == KL_Open)
      return (int)self.GV(Am_HEIGHT) / 2;
    else
      return (int)self.GV(Am_HEIGHT);
  }
}

Am_Define_Point_List_Formula(marker_arrow_point_list_form) {
  Am_Point_List tri;
  Am_Object port = self.GV_Owner().GV_Owner().GV_Owner();
  if (!port.Valid()) return tri;
  
  int left   = (int)self.GV(Am_LEFT);
  int width  = (int)self.GV(Am_WIDTH);
  int height = (int)self.GV(Am_HEIGHT);
  int dir    = (int)port.GV(PORT_DIR);
  int mode   = (int)port.GV(PORT_MODE);

//  if (mode == KL_Input && dir == KL_Left || mode == KL_Output && dir == KL_Right) {
  if (dir == KL_Right) {
    tri.Add_Point(1 + left,		2,		0);
    tri.Add_Point(1 + left,		height - 2,	1);
    tri.Add_Point(width - 1 + left,	height / 2,	2);
    tri.Add_Point(1 + left,		2,		3);
  } else {
    tri.Add_Point(width - 1 + left,	2,		0);
    tri.Add_Point(width - 1 + left,	height - 2,	1);
    tri.Add_Point(1 + left,		height / 2,	2);
    tri.Add_Point(width - 1 + left,	2,		3);
  }
  return tri;
}

Am_Define_Formula(int, marker_arrow_width_form) {
  return (int)self.GV(Am_HEIGHT) / 2;
}

Am_Define_Formula(int, marker_arrow_left_form) {
  Am_Object port = self.GV_Owner().GV_Owner().GV_Owner();
  if (port.Valid()) {
    int dir  = (int)port.GV(PORT_DIR);
    int mode = (int)port.GV(PORT_MODE);
    int open = (int)port.GV(PORT_OPEN);
//    if (open == KL_Open ||
//	mode == KL_Input && dir == KL_Left || mode == KL_Output && dir == KL_Right)
    if (dir == KL_Right)
      return 0;
    else
      return (int)self.GV_Owner().GV(Am_WIDTH) - (int)self.GV(Am_WIDTH) - 1;
  }
  return 0;
}

Am_Define_Style_Formula(marker_arrow_style_form) {
  Am_Object port = self.GV_Owner().GV_Owner().GV_Owner();
  if (port.Valid()) {
    int open = port.GV(PORT_OPEN);
    int age = port.GV(PORT_AGE);
    if (open == KL_Close || age == KL_Old_Port)
      return Am_White;
    else
      return Am_Black;
  }
  return Am_Black;
}

Am_Define_Formula(int, close_rect_visible_form) {
  Am_Object slot = self.GV_Owner().GV_Owner().GV_Owner();
  if (slot.Valid()) {
    return slot.GV(PORT_OPEN);
  }
  return 0;
}

/* ----------------------------------------
Am_Define_Formula(int, marker_arrow_dir) {
  Am_Object slot = self.GV_Owner().GV_Owner().GV_Owner();
  if (slot.Valid()) {
    int head_dir = (int)slot.GV(PORT_DIR);
    int in_out = (int)slot.GV(PORT_MODE);
    if (head_dir == KL_Left  && in_out == KL_Input ||
	head_dir == KL_Right && in_out == KL_Output)
      return Am_SCROLL_ARROW_RIGHT;
    else
      return Am_SCROLL_ARROW_LEFT;
  }
  return Am_SCROLL_ARROW_RIGHT;
}

Am_Define_Formula(int, marker_arrow_in_out) {
  Am_Object slot = self.GV_Owner().GV_Owner().GV_Owner();
  if (slot.Valid())
    return !(int)slot.GV(PORT_MODE);
  else
    return KL_Input;
}

Am_Define_Style_Formula(marker_fill_style) {
  Am_Object slot = self.GV_Owner().GV_Owner().GV_Owner();
  if (slot.Valid()) {
    int open = slot.GV(PORT_OPEN);
    if (open == KL_Open)
      return Am_Motif_Light_Gray;
    else
      return Am_Motif_Gray;
  }
  return Am_Motif_Gray;
}

Am_Define_Formula(int, close_rect_left) {
//  return self.GV_Sibling(CONTENTS_PART).GV(Am_WIDTH) - 2;
  return 4;
}

Am_Define_String_Formula(port_name_form) {
  if (self.GV(PORT_AGE) == KL_Old_Port) return (Am_String)self.GV(PORT_NAME);
  if (self.GV(PORT_MODE) == KL_Input) {
    Am_Value_List binders = self.GV(PORT_BINDERS);
    binders.Start();
    if (binders.Empty()) return (Am_String)"";
    else return ((Am_Object)binders.Get()).GV(BL_NAME);
  } else {
    Am_Object rule = self;
    while((rule = rule.GV_Owner().GV_Owner().GV_Owner()).Valid())
      if (rule.Is_Instance_Of(Rule_Proto)) break;
    if (rule.Valid()) {
      Am_Value_List names = rule.GV(NAME_LIST);
      return Next_Name(rule, "*VAR*");
    }
    return (Am_String)"";
  }
}
---------------------------------------- */

void Initialize_Port(void)
{
  Am_Object port_frame = Am_Border_Roundtangle.Create("port_frame")
    .Set(Am_WIDTH,      KL_Frame_Width(slot_width_margin))
    .Set(Am_HEIGHT,     KL_Frame_Height(slot_height_margin))
    .Set(Am_FILL_STYLE, Am_Formula::Create(port_fill_style))
    .Set(Am_RADIUS,     Am_Formula::Create(port_type_to_radius))
    .Set(Am_SELECTED,   Am_From_Owner(PORT_MODE))
    ;

  Am_Object port_contents = Am_Group.Create("port_contents")
    .Set(Am_LEFT,      slot_width_margin / 2)
    .Set(Am_TOP,       slot_height_margin / 2)
    .Set(Am_WIDTH,     Am_Width_Of_Parts)
    .Set(Am_HEIGHT,    Am_Height_Of_Parts)
    .Set(Am_LAYOUT,    Am_Horizontal_Layout)
    .Set(Am_H_SPACING, 2)
    ;

  Am_Object port_name = Am_Text.Create("port_name")
//  .Set(Am_FONT, Am_Font(Am_FONT_SANS_SERIF, true, false, false, Am_FONT_MEDIUM))
    .Set(Am_FONT,       Am_Font("-*-helvetica-*-r-*--10-*"))
    .Set(Am_LINE_STYLE, Am_Formula::Create(slot_name_color))
    ;

  Am_Object port_values = Am_Group.Create("port_values")
    .Set(Am_WIDTH,     Am_Width_Of_Parts)
    .Set(Am_HEIGHT,    Am_Height_Of_Parts)
    .Set(Am_LAYOUT,    Am_Horizontal_Layout)
    .Set(Am_H_SPACING, 1)
    ;

  Am_Object port_marker = Am_Group.Create("port_marker")
    .Set(Am_WIDTH,  Am_Formula::Create(marker_width_form))
    .Set(Am_HEIGHT, Am_Formula::Create(marker_height_form))
    .Add_Part(FRAME_PART, Am_Rectangle.Create()
 	      .Set(Am_WIDTH,      Am_From_Owner(Am_WIDTH))
 	      .Set(Am_HEIGHT,     Am_From_Owner(Am_HEIGHT))
 	      .Set(Am_LINE_STYLE, Am_White)
 	      .Set(Am_VISIBLE,    Am_Formula::Create(close_rect_visible_form))
 	      .Set(Am_FILL_STYLE, Am_Black)
      )
    .Add_Part(CONTENTS_PART, Am_Polygon.Create()
	      .Set(Am_LEFT,       Am_Formula::Create(marker_arrow_left_form))
	      .Set(Am_HEIGHT,     Am_From_Owner(Am_HEIGHT))
	      .Set(Am_WIDTH,      Am_Formula::Create(marker_arrow_width_form))
	      .Set(Am_POINT_LIST, Am_Formula::Create(marker_arrow_point_list_form))
	      .Set(Am_LINE_STYLE, Am_Formula::Create(marker_arrow_style_form))
	      .Set(Am_FILL_STYLE, Am_Formula::Create(marker_arrow_style_form))
      )
    ;

/* ----------------------------------------
  Am_Object port_marker = Am_Group.Create("port_maker")
    .Set(Am_WIDTH, Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    .Add_Part(FRAME_PART, Am_Rectangle.Create()
 	      .Set(Am_WIDTH, 16)
 	      .Set(Am_HEIGHT, 12)
 	      .Set(Am_LINE_STYLE, Am_White)
 	      .Set(Am_VISIBLE, Am_Formula::Create(close_rect_visible))
 	      .Set(Am_FILL_STYLE, Am_Black)
      )
    .Add_Part(CONTENTS_PART, Am_Scroll_Arrow.Create()
 	      .Set(Am_TOP, 1)
 	      .Set(Am_LEFT,3)
 	      .Set(Am_FILL_STYLE, Am_Formula::Create(marker_fill_style))
 	      .Set(Am_SCROLL_ARROW_DIRECTION, Am_Formula::Create(marker_arrow_dir))
 	      .Set(Am_INTERIM_SELECTED, Am_Formula::Create(marker_arrow_in_out))
 	      .Add_Part(PORT_DIR_TOGGLER,
 			Am_One_Shot_Interactor.Create("head_dir_toggler")
 			.Set(Am_PRIORITY, 5)
 			.Get_Part(Am_COMMAND)
 			.Set(Am_DO_ACTION, (Am_Object_Proc*)&head_dir_toggle)
 			.Get_Owner())
      )
    ;
---------------------------------------- */

  Port_Proto = Am_Group.Create("Port_Proto")
    .Set(PORT_TYPE,    KL_Single)
    .Set(PORT_AGE,     KL_New_Port)
    .Set(PORT_MODE,    KL_Input)
    .Set(PORT_OPEN,    KL_Open)
    .Set(PORT_DIR,     KL_Right)
//    .Set(PORT_ID,      0)
    .Set(PORT_BINDERS, Am_Value_List())
    .Set(Am_WIDTH,     KL_Frame_Width_Plus(0))
    .Set(Am_HEIGHT,    KL_Frame_Height_Plus(0))
    .Add_Part(FRAME_PART,    port_frame)
    .Add_Part(CONTENTS_PART, port_contents)
    ;

  In_Single_Proto = Port_Proto.Create("In_Single_Proto")
    .Set(PORT_NAME, Am_Formula::Create(port_name_form))
    .Set(PORT_TYPE, KL_Single)
    .Set(PORT_MODE, KL_Input)
    .Get_Part(CONTENTS_PART)
      .Add_Part(NAME_PART,  port_name.Create().Set(Am_TEXT, "In"))
      .Add_Part(VALUE_PART, port_values.Create())
    .Get_Owner()
    ;
  Out_Single_Proto = Port_Proto.Create("Out_Single_Proto")
    .Set(PORT_NAME, Am_Formula::Create(port_name_form))
    .Set(PORT_TYPE, KL_Single)
    .Set(PORT_MODE, KL_Output)
    .Get_Part(CONTENTS_PART)
      .Add_Part(NAME_PART,  port_name.Create().Set(Am_TEXT, "Out"))
      .Add_Part(VALUE_PART, port_values.Create())
    .Get_Owner()
    ;

  In_Stream_Proto = Port_Proto.Create("In_Stream_Proto")
    .Set(PORT_NAME, Am_Formula::Create(port_name_form))
    .Set(PORT_TYPE, KL_Stream)
    .Set(PORT_MODE, KL_Input)
    .Get_Part(CONTENTS_PART)
      .Set(PORT_DIR_DAEMON, Am_Formula::Create(head_dir_daemon))
      .Add_Part(VALUE_PART,  port_values.Create())
      .Add_Part(MARKER_PART, port_marker.Create())
      .Add_Part(NAME_PART,   port_name.Create().Set(Am_TEXT, "In Stream"))
    .Get_Owner()
    ;

  Out_Stream_Proto = Port_Proto.Create("Out_Stream_Proto")
    .Set(PORT_NAME, Am_Formula::Create(port_name_form))
    .Set(PORT_TYPE, KL_Stream)
    .Set(PORT_MODE, KL_Output)
    .Get_Part(CONTENTS_PART)
      .Set(PORT_DIR_DAEMON, Am_Formula::Create(head_dir_daemon))
      .Add_Part(VALUE_PART,  port_values.Create())
      .Add_Part(MARKER_PART, port_marker.Create())
      .Add_Part(NAME_PART,   port_name.Create().Set(Am_TEXT, "Out Stream"))
    .Get_Owner()
    ;
}

/* ---------------- Atom data ---------------- */

// ---- global objects ----
Am_Object Atom_Proto;

void Initialize_Atom(void)
{
  Am_Object atom_frame = Am_Roundtangle.Create("atom_frame")
    .Set(Am_WIDTH, KL_Frame_Width(atom_width_margin))
    .Set(Am_HEIGHT, KL_Frame_Height(atom_height_margin))
    .Set(Am_FILL_STYLE, Am_White)
    .Set(Am_RADIUS, 8)
    ;
  Am_Object atom_content = Am_Text.Create("atom_content")
    .Set(Am_TOP, atom_height_margin /2)
    .Set(Am_LEFT, atom_width_margin / 2)
    .Set(Am_TEXT, "0")
    .Add_Part(TEXT_INTER_PART, Am_Text_Edit_Interactor.Create()
	      .Set(Am_PRIORITY, 5)
	      .Set(Am_START_WHEN, "DOUBLE_LEFT_DOWN")
      )
    ;

  Atom_Proto = Am_Group.Create("Atom_Proto")
    .Set(PORT_MODE, KL_Input)
    .Set(PORT_TYPE, KL_Single)
    .Set(PORT_AGE, KL_New_Port)
    .Set(Am_LEFT, 0)
    .Set(Am_TOP, 0)
    .Set(Am_WIDTH, KL_Frame_Width_Plus(1))
    .Set(Am_HEIGHT, KL_Frame_Height_Plus(1))
    .Add_Part(FRAME_PART, atom_frame)
    .Add_Part(CONTENTS_PART, atom_content)
    ;
  ((Am_Object_Advanced&)Atom_Proto).Set_Default_Inherit_Rule(Am_COPY);
}

/* ---------------- Structure ---------------- */

// ---- global objects ----
Am_Object Structure_Proto;

void Initialize_Structure(void)
{
  Am_Object structure_frame = Am_Roundtangle.Create("structure_frame")
    .Set(Am_WIDTH, KL_Frame_Width(structure_width_margin))
    .Set(Am_HEIGHT, KL_Frame_Height(structure_height_margin))
    .Set(Am_FILL_STYLE, Am_Motif_Light_Gray)
    .Set(Am_RADIUS, 8)
    ;
  Am_Object structure_contents = Am_Group.Create("structure_contents")
    .Set(Am_LEFT, structure_width_margin / 2)
    .Set(Am_TOP, structure_height_margin / 2)
    .Set(Am_WIDTH, Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    .Set(Am_LAYOUT, Am_Horizontal_Layout)
    .Set(Am_H_SPACING, 2)
    ;

  Am_Object structure_slots = Am_Group.Create("structure_slots")
    .Set(Am_WIDTH, Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    .Set(Am_LAYOUT, Am_Horizontal_Layout)
    .Set(Am_H_SPACING, 2)
    ;

  Structure_Proto = Am_Group.Create("Structure_Proto")
    .Set(PORT_TYPE,    KL_Stream)  // $B$4$^$+$7!"%9%H%j!<%`$@$H;W$C$F$7$^$&!%(B
    .Set(PORT_MODE,    KL_Output)  // output $B$G$J$$$HF0$+$;$J$$!%(B
    .Set(PORT_AGE,     KL_Old_Port)
    .Set(PORT_BINDERS, Am_Value_List())
    .Set(PORT_NAME,    Am_Formula::Create(port_name_form))
    .Set(Am_WIDTH,     KL_Frame_Width_Plus(1))
    .Set(Am_HEIGHT,    KL_Frame_Height_Plus(1))
    .Add_Part(FRAME_PART, structure_frame)
    .Add_Part(CONTENTS_PART, structure_contents
	      .Add_Part(NAME_PART, Am_Text.Create().Set(Am_TEXT, "message")
			.Add_Part(TEXT_INTER_PART, Am_Text_Edit_Interactor.Create()
				  .Set(Am_PRIORITY, 5)
				  .Set(Am_START_WHEN, "DOUBLE_LEFT_DOWN")))
	      .Add_Part(VALUE_PART, structure_slots))
    ;
}

/* ---------------- Guard ---------------- */

// ---- global objects ----
Am_Object Guard_Proto;

Am_Define_Point_List_Formula(guard_point_list) {
  Am_Point_List hex;
  int width = (int)self.GV(Am_WIDTH);
  int height = (int)self.GV(Am_HEIGHT);
  int radius = (int)self.GV(Am_RADIUS);

  hex.Add_Point(0,		height / 2,	0);
  hex.Add_Point(radius,		0,		1);
  hex.Add_Point(width - radius, 0,		2);
  hex.Add_Point(width - 1,	height / 2,	3);
  hex.Add_Point(width - radius, height - 1,	4);
  hex.Add_Point(radius,		height - 1,	5);
  hex.Add_Point(0,		height / 2,	6);
  
  return hex;
}

void Initialize_Guard(void)
{
  Am_Object guard_frame = Am_Polygon.Create("guard_frame")
    .Set(Am_WIDTH, KL_Frame_Width(guard_width_margin))
    .Set(Am_HEIGHT, KL_Frame_Height(guard_height_margin))
    .Set(Am_POINT_LIST, Am_Formula::Create(guard_point_list))
    .Set(Am_FILL_STYLE, Am_White)
    .Set(Am_RADIUS, 4)
    ;

  Am_Object guard_content = Am_Text.Create("guard_content")
    .Set(Am_TOP, guard_height_margin /2)
    .Set(Am_LEFT, guard_width_margin / 2)
    .Set(Am_TEXT, "")
    .Add_Part(TEXT_INTER_PART, Am_Text_Edit_Interactor.Create()
	      .Set(Am_PRIORITY, 5)
	      .Set(Am_START_WHEN, "DOUBLE_LEFT_DOWN")
      )
    ;

  Guard_Proto = Am_Group.Create("Guard_Proto")
    .Set(Am_LEFT, 0)
    .Set(Am_TOP, 0)
    .Set(Am_WIDTH, KL_Frame_Width_Plus(0))
    .Set(Am_HEIGHT, KL_Frame_Height_Plus(0))
    .Add_Part(FRAME_PART, guard_frame)
    .Add_Part(CONTENTS_PART, guard_content)
    ;
}

/* ---------------- Binding Line ---------------- */

// ---- global objects ----
Am_Object Binding_Line;
Am_Object Unify_Line;
Am_Object Append_Line;
Am_Object Merge_Line;

// ---- slot keys ----
Am_Slot_Key BINDING_LINE = Am_Register_Slot_Name("BINDING_LINE");
// Am_Slot_Key BL_ID = Am_Register_Slot_Name("BL_ID");
Am_Slot_Key BL_SRC = Am_Register_Slot_Name("BL_SRC");
Am_Slot_Key BL_DEST = Am_Register_Slot_Name("BL_DEST");
Am_Slot_Key LT_LINE = Am_Register_Slot_Name("LT_LINE");
Am_Slot_Key RT_LINE = Am_Register_Slot_Name("RT_LINE");
Am_Slot_Key LB_LINE = Am_Register_Slot_Name("LB_LINE");
Am_Slot_Key RB_LINE = Am_Register_Slot_Name("RB_LINE");

// ---- formulas ----
Am_Define_Formula(int, binding_line_x1) {
  Am_Object bl_src = (Am_Object)self.GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  if (!(src_top.Valid() && dest_top.Valid() && src_proc.Valid() && dest_proc.Valid()))
    return 0;
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.Get_Part(BINDER_PART).Valid()) src_top = src_top.Get_Owner();
  src_top = src_top.Get_Part(BINDER_PART);
  Am_Translate_Coordinates(bl_src, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(bl_dest, 0, 0, src_top, dest_x, dest_y);

  int src_w = bl_src.GV(Am_WIDTH);
  int src_h = bl_src.GV(Am_HEIGHT);
  int x, y;
  if ((int)bl_src.GV(PORT_TYPE) == KL_Stream) {
    x = ((int)bl_src.GV(PORT_DIR) == KL_Left)
      ? src_x + bind_margin : src_x + src_w - bind_margin;
    y = src_y + (src_h / 2);
  } else {
    x = src_x + (src_w /2);
    y = src_y + (src_h /2);
  }
  self.Set(Am_Y1, y);
  return x;
}

Am_Define_Formula(int, binding_line_x2) {
  Am_Object bl_src = (Am_Object)self.GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  if (!(src_top.Valid() && dest_top.Valid() && src_proc.Valid() && dest_proc.Valid()))
    return 0;
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.Get_Part(BINDER_PART).Valid()) src_top = src_top.Get_Owner();
  src_top = src_top.Get_Part(BINDER_PART);
  Am_Translate_Coordinates(bl_src, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(bl_dest, 0, 0, src_top, dest_x, dest_y);

  int dest_w = bl_dest.GV(Am_WIDTH);
  int dest_h = bl_dest.GV(Am_HEIGHT);
  int x, y;
  if ((int)bl_dest.GV(PORT_TYPE) == KL_Stream) {
    x = ((int)bl_dest.GV(PORT_DIR) == KL_Left)
         ? dest_x + bind_margin : dest_x + dest_w - bind_margin;
    y = dest_y + (dest_h / 2);
  } else {
    x = dest_x + (dest_w /2);
    y = dest_y + (dest_h /2);
  }
  self.Set(Am_Y2, y);
  return x;
}

Am_Define_Formula(int, binding_arrow_x1) {
  Am_Object bl_src = (Am_Object)self.GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.GV_Part(BINDER_PART).Valid()) src_top = src_top.GV_Owner();
  src_top = src_top.GV_Part(BINDER_PART);
  Am_Object src_name = bl_src.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  Am_Object dest_name = bl_dest.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  if (src_proc.GV(IS_ICON)) src_name = src_proc; // Icon$B$J$i!$H"$K9g$o$;$k!%(B
  if (dest_proc.GV(IS_ICON)) dest_name = dest_proc; // Icon$B$J$i!$H"$K9g$o$;$k!%(B
  Am_Translate_Coordinates(src_name, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(dest_name, 0, 0, src_top, dest_x, dest_y);

  int src_w = src_name.GV(Am_WIDTH);   int src_h = src_name.GV(Am_HEIGHT);
  int dest_w = dest_name.GV(Am_WIDTH); int dest_h = dest_name.GV(Am_HEIGHT);

  int x = src_x + (src_w / 2);
  int y = src_y + (src_h / 2);
  if (src_x - (dest_x + dest_w) > 0) x = src_x; // + bind_margin; 
  if ((src_x + src_w) - dest_x < 0) x = src_x + src_w; // - bind_margin;
  if (src_y - (dest_y + dest_h) > 0) y = src_y; // + bind_margin;
  if ((src_y + src_h) - dest_y < 0) y = src_y + src_h - bind_margin;

  self.Set(Am_Y1, y);
  return x;
}

Am_Define_Formula(int, binding_arrow_x2) {
  Am_Object bl_src = (Am_Object)self.GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.Get_Part(BINDER_PART).Valid()) src_top = src_top.Get_Owner();
  src_top = src_top.Get_Part(BINDER_PART);
  Am_Object src_name = bl_src.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  Am_Object dest_name = bl_dest.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  if (src_proc.GV(IS_ICON)) src_name = src_proc; // Icon$B$J$i!$H"$K9g$o$;$k!%(B
  if (dest_proc.GV(IS_ICON)) dest_name = dest_proc; // Icon$B$J$i!$H"$K9g$o$;$k!%(B
  Am_Translate_Coordinates(src_name, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(dest_name, 0, 0, src_top, dest_x, dest_y);

  int src_w = src_name.GV(Am_WIDTH);   int src_h = src_name.GV(Am_HEIGHT);
  int dest_w = dest_name.GV(Am_WIDTH); int dest_h = dest_name.GV(Am_HEIGHT);

  int x = dest_x + (dest_w / 2);
  int y = dest_y + (dest_h / 2);
  if (dest_x - (src_x + src_w) > 0) x = dest_x; // + bind_margin;
  if ((dest_x + dest_w) - src_x < 0) x = dest_x + dest_w; // - bind_margin;
  if (dest_y - (src_y + src_h) > 0) y = dest_y; // + bind_margin;
  if ((dest_y + dest_h) -src_y < 0) y = dest_y + dest_h - bind_margin;

  self.Set(Am_Y2, y);
  return x;
}

#define center_of_object(obj, cx, cy) {\
  int x = obj.GV(Am_LEFT);\
  int y = obj.GV(Am_TOP);\
  int width = obj.GV(Am_WIDTH);\
  int height = obj.GV(Am_HEIGHT);\
  cx = x + width  / 2;\
  cy = y + height / 2;\
}

Am_Define_Formula(int, unify_line_x1) {
  Am_Object bl_src = (Am_Object)self.GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  if (!(src_top.Valid() && dest_top.Valid() && src_proc.Valid() && dest_proc.Valid()))
    return 0;
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.Get_Part(BINDER_PART).Valid()) src_top = src_top.Get_Owner();
  src_top = src_top.Get_Part(BINDER_PART);
  Am_Translate_Coordinates(bl_src, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(bl_dest, 0, 0, src_top, dest_x, dest_y);

  Am_Object src_marker = bl_src.GV_Part(CONTENTS_PART).GV_Part(MARKER_PART);
  if (!src_marker.Valid()) src_marker = bl_src.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  int src_cx, src_cy;
  center_of_object(src_marker, src_cx, src_cy);
  
  int x, y;
  x = src_x + src_cx;
  y = src_y + src_cy;

  self.Set(Am_Y1, y);
  return x;
}

Am_Define_Formula(int, unify_line_x2) {
  Am_Object bl_src = (Am_Object)self.GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  if (!(src_top.Valid() && dest_top.Valid() && src_proc.Valid() && dest_proc.Valid()))
    return 0;
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.Get_Part(BINDER_PART).Valid()) src_top = src_top.Get_Owner();
  src_top = src_top.Get_Part(BINDER_PART);
  Am_Translate_Coordinates(bl_src, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(bl_dest, 0, 0, src_top, dest_x, dest_y);

  Am_Object dest_marker = bl_dest.GV_Part(CONTENTS_PART).GV_Part(MARKER_PART);
  if (!dest_marker.Valid()) dest_marker = bl_src.GV_Part(CONTENTS_PART).GV_Part(NAME_PART);
  int dest_cx, dest_cy;
  center_of_object(dest_marker, dest_cx, dest_cy);
  
  int x, y;
  x = dest_x + dest_cx;
  y = dest_y + dest_cy;

  self.Set(Am_Y2, y);
  return x;
}

Am_Define_Formula(int, lt_line_x1) {
  Am_Object bl_src = (Am_Object)self.GV_Owner().GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV_Owner().GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  if (!(src_top.Valid() && dest_top.Valid() && src_proc.Valid() && dest_proc.Valid()))
    return 0;
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.Get_Part(BINDER_PART).Valid()) src_top = src_top.Get_Owner();
  src_top = src_top.Get_Part(BINDER_PART);
  Am_Translate_Coordinates(bl_src, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(bl_dest, 0, 0, src_top, dest_x, dest_y);

  int margin = 0;
  if ((int)bl_src.GV(PORT_TYPE) == KL_Single) margin = 3; else margin = 1;
  int src_w  = bl_src.GV(Am_WIDTH);  int src_h  = bl_src.GV(Am_HEIGHT);
  int dest_w = bl_dest.GV(Am_WIDTH); int dest_h = bl_dest.GV(Am_HEIGHT);
  
  self.GV_Sibling(RT_LINE).Set(Am_X1, src_x + src_w - margin).Set(Am_Y1, src_y + margin);
  self.GV_Sibling(LB_LINE).Set(Am_X1, src_x + margin).Set(Am_Y1, src_y + src_h - margin);
  self.GV_Sibling(RB_LINE).Set(Am_X1, src_x + src_w - margin).Set(Am_Y1, src_y + src_h - margin);
  self.Set(Am_Y1, src_y + margin);
  return src_x + margin;
}

Am_Define_Formula(int, lt_line_x2) {
  Am_Object bl_src = (Am_Object)self.GV_Owner().GV(BL_SRC);
  Am_Object bl_dest = (Am_Object)self.GV_Owner().GV(BL_DEST);
  if (!(bl_src.Valid() && bl_dest.Valid())) return 0;

  Am_Object src_top = Port_Top_Parent(bl_src);
  Am_Object dest_top = Port_Top_Parent(bl_dest);
  Am_Object src_proc = Port_Parent_Proc(bl_src);
  Am_Object dest_proc = Port_Parent_Proc(bl_dest);
  if (!(src_top.Valid() && dest_top.Valid() && src_proc.Valid() && dest_proc.Valid()))
    return 0;
  src_top.GV(Am_LEFT);  src_top.GV(Am_TOP);  dest_top.GV(Am_LEFT);  dest_top.GV(Am_TOP);
  src_proc.GV(Am_LEFT); src_proc.GV(Am_TOP); dest_proc.GV(Am_LEFT); dest_proc.GV(Am_TOP);

  int src_x, src_y, dest_x, dest_y;
  while(!src_top.Get_Part(BINDER_PART).Valid()) src_top = src_top.Get_Owner();
  src_top = src_top.Get_Part(BINDER_PART);
  Am_Translate_Coordinates(bl_src, 0, 0, src_top, src_x, src_y);
  Am_Translate_Coordinates(bl_dest, 0, 0, src_top, dest_x, dest_y);

  int margin = 0;
  if ((int)bl_src.GV(PORT_TYPE) == KL_Single) margin = 3; else margin = 1;
  int src_w  = bl_src.GV(Am_WIDTH);  int src_h  = bl_src.GV(Am_HEIGHT);
  int dest_w = bl_dest.GV(Am_WIDTH); int dest_h = bl_dest.GV(Am_HEIGHT);
  
  self.GV_Sibling(RT_LINE).Set(Am_X2, dest_x+dest_w - margin).Set(Am_Y2, dest_y + margin);
  self.GV_Sibling(LB_LINE).Set(Am_X2, dest_x + margin).Set(Am_Y2, dest_y+dest_h - margin);
  self.GV_Sibling(RB_LINE).Set(Am_X2, dest_x + dest_w - margin).Set(Am_Y2, dest_y + dest_h - margin);
  self.Set(Am_Y2, dest_y + margin);
  return dest_x + margin;
}

// ---- operations ----
Am_Object Binding_Line_Create(Am_Object src_port, Am_Object dest_port)
{
  int src_age = src_port.Get(PORT_AGE);
  int src_mode = src_port.Get(PORT_MODE);
  int dest_age = dest_port.Get(PORT_AGE);
  int dest_mode = dest_port.Get(PORT_MODE);
  if (src_age == KL_New_Port && src_mode == KL_Input ||
      src_age == KL_Old_Port && src_mode == KL_Output) {
    Am_Object tmp = src_port; src_port = dest_port; dest_port = tmp;
  }
  if (Port_Unify_Allowed(src_port, dest_port)) {
    src_port.Set(PORT_OPEN, KL_Open);
    dest_port.Set(PORT_OPEN, KL_Open);
    Am_Object rule = Port_Parent_Rule(src_port);
    Am_Object new_line;
    if (src_age == KL_New_Port && dest_age == KL_New_Port) {
      new_line = Binding_Line.Create()
	.Set(BL_SRC, src_port)
	.Set(BL_DEST, dest_port)
	.Set(Am_VISIBLE, true);
      Add_Binder(src_port, new_line);
      Add_Binder(dest_port, new_line);
    } else {
      new_line = Unify_Line.Create()
	.Set(BL_SRC, src_port)
	.Set(BL_DEST, dest_port);
      Add_Binder(src_port, new_line);
      Add_Binder(dest_port, new_line);
    }
    Rule_Binder_Part(rule).Add_Part(new_line).Remove_Part(new_line);
    Rule_Binder_Part(rule).Add_Part(new_line);    //$B:G8e$N(BRemove$BLdBj!%(B
  }
  return 0;
}

void Binding_Line_Remove(Am_Object line)
{
  Am_Object src_port = (Am_Object)line.Get(BL_SRC);
  Am_Object dest_port = (Am_Object)line.Get(BL_DEST);
  line.Set(BL_SRC, 0);
  line.Set(BL_DEST, 0);
  Remove_Binder(src_port, line);
  Remove_Binder(dest_port, line);
  if (!Port_Is_Bound(src_port) && (int)src_port.Get(PORT_AGE) == KL_Old_Port)
    src_port.Set(PORT_OPEN, KL_Close);
  if (!Port_Is_Bound(dest_port)) dest_port.Set(PORT_OPEN, KL_Close);
//  Rule_Binder_Part(rule).Remove_Part(line);
  line.Remove_From_Owner();
}

void Initialize_Binding_Line()
{
  Binding_Line = Am_Arrow.Create("Binding_Line")
    .Set(BL_SRC, 0)
    .Set(BL_DEST, 0)
    .Set(Am_VISIBLE, false)
    .Set(Am_ARROW_LENGTH, 14)
    .Set(Am_LINE_STYLE,
	 Am_Style (0.1, 0.1, 0.1, 6, Am_CAP_ROUND, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(50)) )
    .Set(Am_LINE_STYLE,
	 Am_Style (0, 0, 0, 4, Am_CAP_BUTT, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(60)) )
    .Set(Am_FILL_STYLE,
	 Am_Style (0.1, 0.1, 0.1, 4, Am_CAP_BUTT, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(50)) )
    .Set(Am_X1, Am_Formula::Create(unify_line_x1))
    .Set(Am_X2, Am_Formula::Create(unify_line_x2))
    .Set(Am_X1, Am_Formula::Create(binding_arrow_x1))
    .Set(Am_X2, Am_Formula::Create(binding_arrow_x2))
    ;

  Unify_Line = Am_Arrow.Create("Unify_Line")
    .Set(BL_SRC,  0)
    .Set(BL_DEST, 0)
//    .Set(Am_VISIBLE, false)
    .Set(Am_LINE_STYLE,
	 Am_Style (0.27, 0.52, 0.53, 8, Am_CAP_ROUND, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(50)) )
    .Set(Am_LINE_STYLE,
	 Am_Style (0.27, 0.52, 0.53, 4, Am_CAP_BUTT, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(60)) )
    .Set(Am_FILL_STYLE,
	 Am_Style (0.27, 0.52, 0.53, 4, Am_CAP_BUTT, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(50)) )
    .Set(Am_X1, Am_Formula::Create(unify_line_x1))
    .Set(Am_X2, Am_Formula::Create(unify_line_x2))
    .Set(Am_X1, Am_Formula::Create(binding_arrow_x1))
    .Set(Am_X2, Am_Formula::Create(binding_arrow_x2))
    ;

  static char unify_dash_list[2] = {1, 2};
  Am_Object Old_Unify_Line = Am_Group.Create("Unify_Line")
//    .Set(BL_ID,   Am_Formula::Create(line_id_form))
    .Set(BL_SRC,  0)
    .Set(BL_DEST, 0)
    .Set(Am_WIDTH, Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    .Set(Am_LINE_STYLE,
	 Am_Style (0.1, 0.1, 0.1, 1, Am_CAP_ROUND, Am_JOIN_MITER,
		   Am_LINE_ON_OFF_DASH, unify_dash_list,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_SOLID, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(50)) )
    .Add_Part(LT_LINE, Am_Line.Create()
	      .Set(Am_LINE_STYLE, Am_From_Owner(Am_LINE_STYLE))
	      .Set(Am_X1, Am_Formula::Create(lt_line_x1))
	      .Set(Am_X2, Am_Formula::Create(lt_line_x2)))
    .Add_Part(RT_LINE, Am_Line.Create().Set(Am_LINE_STYLE, Am_From_Owner(Am_LINE_STYLE)))
    .Add_Part(LB_LINE, Am_Line.Create().Set(Am_LINE_STYLE, Am_From_Owner(Am_LINE_STYLE)))
    .Add_Part(RB_LINE, Am_Line.Create().Set(Am_LINE_STYLE, Am_From_Owner(Am_LINE_STYLE)))
    ;

  Append_Line = Am_Line.Create("Append_Line")
    .Set(Am_VISIBLE, false)
    .Set(Am_LINE_STYLE,
	 Am_Style ("white", 8, Am_CAP_ROUND, Am_JOIN_MITER,
		   Am_LINE_ON_OFF_DASH, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_SOLID, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array(50)) )
    ;

  Merge_Line = Am_Line.Create("Merge_Line")
    .Set(Am_VISIBLE, false)
    .Set(Am_LINE_STYLE,
	 Am_Style ("white", 8, Am_CAP_ROUND, Am_JOIN_MITER,
		   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
		   Am_DEFAULT_DASH_LIST_LENGTH,
		   Am_FILL_STIPPLED, Am_FILL_POLY_EVEN_ODD,
		   Am_Image_Array::Make(Am_Image_Array("/home/nsl/toyoda/hoge"))))
    ;
}

/* ---------------- KL_Widgets Initialize routine ---------------- */

void Initialize_KL_Widgets(void)
{
  Initialize_Module();
  Initialize_Binding_Line();
  Initialize_Process();
  Initialize_Port(); 
  Initialize_Atom();
  Initialize_Structure();
  Initialize_Guard();
}
