// Copyright (C) 1996  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <fstream.h>
#include <strstream.h>

#include <amulet/amulet.h>
#include <amulet/formula_advanced.h>
#include <amulet/inter_advanced.h>
#include <amulet/widgets_advanced.h>
#include <amulet/debugger.h>

#include "my_amulet.h"
#include "kl_widgets.h"
#include "kl_pattern.h"
#include "translator.h"

Am_Slot_Key GEN_ORDER = Am_Register_Slot_Name("GEN_ORDER");

// ----------------------------------------------------------------------
//		ID Management 
// ----------------------------------------------------------------------

Am_Slot_Key GOAL_PORTS = Am_Register_Slot_Name("GOAL_PORTS");
Am_Slot_Key PROC_ID = Am_Register_Slot_Name("PROC_ID");
Am_Slot_Key RULE_ID = Am_Register_Slot_Name("RULE_ID");
Am_Slot_Key PORT_ID = Am_Register_Slot_Name("PORT_ID");
Am_Slot_Key PORT_NEXT_ID = Am_Register_Slot_Name("PORT_NEXT_ID");
Am_Slot_Key BL_ID = Am_Register_Slot_Name("BL_ID");
Am_Slot_Key BL_INNER_NAME = Am_Register_Slot_Name("BL_INNER_NAME");
Am_Slot_Key PORT_INNER_NAME = Am_Register_Slot_Name("PORT_INNER_NAME");
Am_Slot_Key PORT_INNER_NEXT_NAME = Am_Register_Slot_Name("PORT_INNER_NEXT_NAME");

// Process ID formula -------------------------
Am_Define_Formula(int, proc_id_form) {
  Am_Object procs = self.GV_Owner();
  if (procs.Valid()) {
    if (procs.Is_Instance_Of(Process_Agg)) {
      Am_Value_List proc_list = (Am_Value_List)procs.GV(Am_GRAPHICAL_PARTS);
      return Object_List_Index(proc_list, self);
    } else {
      return ((Am_Object)self.GV(PARENT_PROC)).GV(PROC_ID);
    }
  }
  return 0;
}

// Rule ID formula --------------------------
Am_Define_Formula(int, rule_id_form) {
  Am_Object parent = (Am_Object)self.GV(PARENT_PROC);
  Am_Value_List rules = (Am_Value_List)parent.GV(PROC_RULES);
  return Object_List_Index(rules, self) - 1;
}

// Port ID formulas -----------------------------
#define object_list_max(max, list, key) \
{ \
  max = 0; \
  for (list.Start(); !list.Last(); list.Next()) { \
    Am_Object item = (Am_Object)list.Get(); \
    int value = item.GV(key); \
    if (value > max) max = value; \
  } \
}

// -------------------------------------------------------------------
// |  Old Ports   | Goal Ports                         | Inner Ports |
// |--------------+------------------------------------+-------------|
// | i1, i2,...,i | (i+1, i+2), (i+3, i+4)... (j, j+1) | j+2, j+3,...|
// -------------------------------------------------------------------
//   ^ Pattern$B$N$H$-$O(B     ^ Pattern $B$N:F5"$N$H$-$N(Bnext_id
//  $B$=$N?F$N%4!<%k%]!<%H$N(Bid
Am_Define_Formula(int, port_id_form) {
  int id;
  Am_Object rule = 0;
  Am_Object parent = self.GV_Owner().GV_Owner().GV_Owner();
  if (KL_Type(parent) == KL_Pattern) {
    rule = parent.GV_Owner().GV_Owner().GV_Owner();
  }
  if (KL_Type(rule) != KL_Rule && KL_Type(rule) != KL_Pattern) {
    for (rule = self; !rule.Is_Instance_Of(Rule_Proto); ) {
      if (!(rule = rule.GV_Owner().GV_Owner().GV_Owner()).Valid()) {
//        cerr << "This port: " << self << " isn't in rule" << endl;
	return 0;
      }
    }
  }
  Am_Value_List ports = (Am_Value_List)rule.GV_Part(CONTENTS_PART).GV_Part(PORTS_PART)
                                                   .GV(Am_GRAPHICAL_PARTS);
  Am_Value_List goal_ports = (Am_Value_List)rule.GV(GOAL_PORTS);
  Am_Value_List new_ports = (Am_Value_List)rule.GV(INNER_PORT_NAME_LIST);

  char buff[1000];
  ostrstream name(buff, 1000);
  name.seekp(0, ios::beg);

  // Old Ports
  ports = sort_ports(ports);
  id = Object_List_Index(ports, self);
  if (id != 0) {
    name << /* (Am_String)self.GV(PORT_NAME) << */ "_" << id << "_" << ends;
    self.Set(PORT_INNER_NAME, (char*)Next_Name_At(rule, INNER_PORT_NAME_LIST, buff));
    name.seekp(0, ios::beg);
    name << /* (Am_String)self.GV(PORT_NAME) << */ "_" << id << "__" << ends;
    self.Set(PORT_INNER_NEXT_NAME, (char*)Next_Name_At(rule, INNER_PORT_NAME_LIST, buff));
    return id;
  }
  // Goal Ports
  int old_ports_max; object_list_max(old_ports_max, ports, PORT_ID);
  id = Object_List_Index(goal_ports, self);
  if (id != 0) {
    id = old_ports_max + id * 2;
    name << /* (Am_String)self.GV(PORT_NAME) << */ "_" << id << "_" << ends;
    self.Set(PORT_INNER_NAME, (char*)Next_Name_At(rule, INNER_PORT_NAME_LIST, buff));
    name.seekp(0, ios::beg);
    name << /* (Am_String)self.GV(PORT_NAME) << */ "_" << id << "__" << ends;
    self.Set(PORT_INNER_NEXT_NAME, (char*)Next_Name_At(rule, INNER_PORT_NAME_LIST, buff));
    return id;
  }
  // New Ports
//  int goal_ports_length = goal_ports.Length();
  int goal_ports_max; object_list_max(goal_ports_max, goal_ports, PORT_ID);
  id = String_List_Index(new_ports, (Am_String)self.GV(PORT_NAME));
  if (id != 0) {
    id = goal_ports_max + id + 1;
    self.Set(PORT_INNER_NAME, (char*)(Am_String)self.GV(PORT_NAME));
    name << "_" << id << "_" << (Am_String)self.GV(PORT_NAME) << '\0';
    self.Set(PORT_INNER_NEXT_NAME, (char*)Next_Name_At(rule, INNER_PORT_NAME_LIST, buff));
    return id;
  }
//  cerr << "This port: " << self << " isn't in rule" << endl;
  return 0;
}  

Am_Define_Formula(int, port_next_id_form) {
  int id;
  Am_Object rule = 0;
  Am_Object parent = self.GV_Owner().GV_Owner().GV_Owner();
  if (KL_Type(parent) == KL_Pattern) {
    rule = parent.GV_Owner().GV_Owner().GV_Owner();
  }
  if (KL_Type(rule) != KL_Rule && KL_Type(rule) != KL_Pattern) {
    for (rule = self; !rule.Is_Instance_Of(Rule_Proto); ) {
      if (!(rule = rule.GV_Owner().GV_Owner().GV_Owner()).Valid()) {
//        cerr << "This port: " << self << " isn't in rule" << endl;
	return 0;
      }
    }
  }
  Am_Value_List ports = (Am_Value_List)rule.GV_Part(CONTENTS_PART).GV_Part(PORTS_PART)
                                                   .GV(Am_GRAPHICAL_PARTS);
  Am_Value_List goal_ports = (Am_Value_List)rule.GV(GOAL_PORTS);

  // Goal Ports $B$K$7$+(Bnext id$B$O$D$+$J$$!%(B
  int old_ports_max; object_list_max(old_ports_max, ports, PORT_ID);
  id = Object_List_Index(goal_ports, self);
  if (id != 0) {
    id = old_ports_max + id * 2 + 1;
    return id;
  }
  return 0;
}  

/*
Am_Define_Formula(int, in_port_id_form) { // $B%P%$%s%@$,B8:_$9$l$P$=$3$+$i(BID$B$r0z$/(B
  return 0;
  Am_Value_List binders = (Am_Value_List)self.GV(PORT_BINDERS);
  binders.Start();
  if (binders.Empty()) return 0;
  else return ((Am_Object)binders.Get()).GV(BL_ID);
}
*/

// Binding Line ID formula ----------------------

Am_Define_Formula(int, line_id_form) {
  Am_Object src_port = (Am_Object)self.GV(BL_SRC);
  if (src_port.Valid()) {
    self.Set(BL_INNER_NAME, (char*)(Am_String)src_port.GV(PORT_INNER_NAME));
    return (int)src_port.GV(PORT_ID);
  }
  return 0;
}

// Goal Port ID formula --------------------------

Am_Define_Value_List_Formula(goal_ports_form)
{
  Am_Value_List out_ports = Am_Value_List();
  Am_Value_List goals = (Am_Value_List)self.GV_Part(CONTENTS_PART).GV_Part(GOALS_PART)
                                                   .GV(Am_GRAPHICAL_PARTS);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object gl = (Am_Object)goals.Get();
    Am_Value_List ports = (Am_Value_List)gl.GV_Part(CONTENTS_PART).GV_Part(PORTS_PART)
                                                   .GV(Am_GRAPHICAL_PARTS);
    for (ports.Start(); !ports.Last(); ports.Next()) {
      Am_Object pt = (Am_Object)ports.Get();
      out_ports.Add(pt);
    }
  }
  return out_ports;
}

// Initialize Routine ----------------------------

void initialize_id(void)
{
  Process_Proto
    .Set(PROC_ID, 0)
    .Set(PROC_ID, Am_Formula::Create(proc_id_form));
  Creator_Proto.Set(PROC_ID, Am_Formula::Create(proc_id_form));
  
  In_Single_Proto
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  In_Stream_Proto
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Out_Single_Proto
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Out_Stream_Proto
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Pat_In_Single_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Pat_In_Stream_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Pat_Out_Single_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Pat_Out_Stream_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Map_In_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Map_Out_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Recursive_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Broadcast_Single_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Broadcast_Stream_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");
  Merge_Port
    .Set(PORT_ID, Am_Formula::Create(port_id_form))
    .Set(PORT_NEXT_ID, Am_Formula::Create(port_next_id_form))
    .Set(PORT_INNER_NAME, "")
    .Set(PORT_INNER_NEXT_NAME, "");

  Binding_Line.Set(BL_ID, Am_Formula::Create(line_id_form));
  Dashed_Binding_Line.Set(BL_ID, Am_Formula::Create(line_id_form));
  Unify_Line  .Set(BL_ID, Am_Formula::Create(line_id_form));

  Vanishing_Rule_Proto
    .Set(RULE_ID, Am_Formula::Create(rule_id_form))
    .Set(GOAL_PORTS, Am_Formula::Create(goal_ports_form));
  Continuous_Rule_Proto
    .Set(RULE_ID, Am_Formula::Create(rule_id_form))
    .Set(GOAL_PORTS, Am_Formula::Create(goal_ports_form));
  Pattern_Proto
    .Set(GEN_ORDER, 1)
    .Set(RULE_ID, 0)
    .Set(GOAL_PORTS, Am_Formula::Create(goal_ports_form));

  Sequence_Pattern_Proto
    .Set(GEN_ORDER, 1)
    .Set(RULE_ID, 0)
    .Set(GOAL_PORTS, Am_Formula::Create(goal_ports_form));
}

// ----------------------------------------------------------------------
//		Translator
// ----------------------------------------------------------------------

#define Mode_Str(mode)	((mode == KL_Input) ? " +" : " -")
#define Type_Str(type)	((type == KL_Single) ? " single" : " stream")
#define Dir_Str(dir)	((dir == KL_Left) ? " left" : " right")
#define Open_Str(open)	((open == KL_Open) ? "open" : "close")

#define Rule_Type_Str(type) ((type == KL_Network_Rule) ? "network" : "transition")


Am_Slot_Key KL_TRANS_METHOD = Am_Register_Slot_Name("KL_TRANS_METHOD");
typedef int KL_Trans_Method(Am_Object item, int flag, ofstream& ofile);
#define KL_Translate_Call(item, err, flag, ofile) \
        Am_Function_Call(KL_Trans_Method, item, KL_TRANS_METHOD, err, (item, flag, ofile))
#define HEAD_PHASE       1
#define GOAL_PHASE       2
#define LINK_PHASE       3
#define UNIFY_PHASE      4
#define TERM_HEAD_PHASE  5
#define TERM_UNIFY_PHASE 6
#define REC_GOAL_PHASE   7

int translate_module(Am_Object proc_agg, ofstream& ofile);
int translate_patterns(Am_Object rules_agg, ofstream& ofile);
int translate_process(Am_Object process, ofstream& ofile);
int translate_port_in_proc(Am_Object port, ofstream& ofile);

int translate_rule(Am_Object rule, ofstream& ofile);
int translate_head_part(Am_Object rule, ofstream& ofile);
int translate_guard_part(Am_Object rule, ofstream& ofile);
int translate_goal_part(Am_Object rule, ofstream& ofile);
int translate_unify_part(Am_Object rule, ofstream& ofile);
int translate_transition_body(Am_Object rule, ofstream& ofile);

int translate_port(Am_Object port, int flag, ofstream& ofile);
int translate_atom(Am_Object atom, int flag, ofstream& ofile);
int translate_structure(Am_Object structure, int flag, ofstream& ofile);

// Utility ------------------------------------------

int output_point_size(Am_Object obj, ofstream& ofile)
{
  int x = (int)obj.Get(Am_LEFT) + 1;
  int y = (int)obj.Get(Am_TOP) + 1;
  int w = obj.Get(Am_WIDTH);
  int h = obj.Get(Am_HEIGHT);
  ofile << " point " << x << " " << y
	<< " size "  << w << " " << h;
  return 1;
}

// Translate Module ----------------------------------

int translate_module(Am_Object module_win, ofstream& ofile)
{
  Am_String module_name = (Am_String)module_win.Get(MODULE_NAME);
  Am_Object proc_agg = Module_Process_Part(module_win);
  Am_Object rules_agg = Module_Rules_Part(module_win);
  ofile << "/* start of module */" << endl;

  ofile << ":- module " << /* module_name */ "main" << "." << endl << endl;
  translate_patterns(rules_agg, ofile);
  Am_Value_List procs = (Am_Value_List)proc_agg.Get(Am_GRAPHICAL_PARTS);
  for (procs.Start(); !procs.Last(); procs.Next()) {
    translate_process(procs.Get(), ofile);
  }
  ofile << "/* end of module */" << endl;
  return 1;
}

// Translate Process header --------------------------

int translate_process(Am_Object process, ofstream& ofile)
{
  Am_String proc_name = (Am_String)process.Get(PROC_NAME);
  char* proc_type = ((int)process.Get(PROC_TYPE) == KL_Normal_Proc)? " normal" : " creator";
  /* translate spec */
  ofile << "% define '" << proc_name << proc_type; 
  output_point_size(process, ofile);
  ofile << endl;
  Am_Value_List ports = sort_ports(Proc_Ports(process));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    translate_port_in_proc(port, ofile);
  }
  ofile << "% begin" << endl;
  /* translate rules */
  Am_Value_List rules = (Am_Value_List)process.Get(PROC_RULES);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = (Am_Object)rules.Get();
    translate_rule(rule, ofile);
  }
  ofile << "c_" << proc_name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}

int translate_port_in_proc(Am_Object port, ofstream& ofile)
{
  Am_String port_name = (Am_String)port.Get(PORT_NAME);
  char* port_mode = Mode_Str((int)port.Get(PORT_MODE));
  char* port_type = Type_Str((int)port.Get(PORT_TYPE));
  char* port_dir  = Dir_Str((int)port.Get(PORT_DIR));
  int x = port.Get(Am_LEFT);  int y = port.Get(Am_TOP);
  int w = port.Get(Am_WIDTH); int h = port.Get(Am_HEIGHT);
  ofile << "%	port " << port_name << port_mode << port_type << port_dir;
  output_point_size(port, ofile);
  ofile << endl;
  return 1;
}

// Translate Rule -------------------------------------

int translate_rule(Am_Object rule, ofstream& ofile)
{
  Am_String rule_name = (Am_String)rule.Get(PROC_NAME);
  int rule_id = rule.Get(RULE_ID);
  int type = rule.Get(RULE_TYPE);
  char* rule_type = Rule_Type_Str(type);
  if (type == KL_Network_Rule && Rule_Goals(rule).Empty()) { // $B$H$F$b1x$$%Q%C%A!%(B
    rule_type = "vanishing";
  }
  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << rule_type;
  output_point_size(rule, ofile);
  ofile << endl;
  // Head$BIt(B 
  translate_head_part(rule, ofile);
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(rule, ofile);
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  if ((int)rule.Get(RULE_TYPE) == KL_Network_Rule) {
    translate_goal_part(rule, ofile);
    ofile << "% other_part" << endl;
    translate_unify_part(rule, ofile);
  } else if ((int)rule.Get(RULE_TYPE) == KL_Transition_Rule) {
    translate_transition_body(rule, ofile);
  }
  ofile << "." << endl;
  return 1;
}

int translate_head_part(Am_Object rule, ofstream& ofile)
{
  Am_String rule_name = (Am_String)rule.Get(PROC_NAME);
  ofile << rule_name;
  Am_Value_List ports = sort_ports(Proc_Ports(rule));
  int first = true;
  for (ports.Start(); !ports.Last(); ports.Next(),first = false) {
    Am_Object pt = (Am_Object)ports.Get();
    if ((int)rule.Get(RULE_TYPE) != KL_Transition_Rule ||
	(int)pt.Get(PORT_AGE) == KL_Old_Port) {
      if (first) ofile << "( ";
      else       ofile << ", ";
      int err;
      Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err, (pt,HEAD_PHASE,ofile));
      if (err == 0) {
	cerr << "error in port:" << pt << endl;
	return 0;
      }
      output_point_size(pt, ofile);
      ofile << endl;
    }
  }
  if (!first) ofile << ") ";
  return 1;
}

int translate_head_part_by(Am_Object rule, int flag, ofstream& ofile)
{
  Am_String rule_name = (Am_String)rule.Get(PROC_NAME);
  ofile << rule_name;
  Am_Value_List ports = sort_ports(Proc_Ports(rule));
  int first = true;
  for (ports.Start(); !ports.Last(); ports.Next(),first = false) {
    Am_Object pt = (Am_Object)ports.Get();
    if ((int)rule.Get(RULE_TYPE) != KL_Transition_Rule ||
	(int)pt.Get(PORT_AGE) == KL_Old_Port) {
      if (first) ofile << "( ";
      else       ofile << ", ";
      int err;
      Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err, (pt, flag, ofile));
      if (err == 0) {
	cerr << "error in port:" << pt << endl;
	return 0;
      }
      output_point_size(pt, ofile);
      ofile << endl;
    }
  }
  if (!first) ofile << ") ";
  return 1;
}

int translate_guard_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List guards = Rule_Guard(rule);
  int first = true;
  for (guards.Start(); !guards.Last(); guards.Next(), first = false) {
    if (!first) ofile << ", ";
    Am_Object gd = (Am_Object)guards.Get();
    ofile << (Am_String)gd.Get_Part(CONTENTS_PART).Get(Am_TEXT);
    ofile << "\t % guard" << endl;
  }
  if (first == true) {
    ofile << "true ";
    ofile << "\t % guard" << endl;
  }
  return 1;
}

int translate_goal(Am_Object goal, ofstream& ofile)
{
  Am_String name = (Am_String)goal.Get(PROC_NAME);
  ofile << ", " << name;
  Am_Value_List ports = sort_ports(Proc_Ports(goal));
  int first = true;
  for (ports.Start(); !ports.Last(); ports.Next(), first = false) {
    if (first) ofile << "( ";
    else       ofile << ", ";
    Am_Object pt = (Am_Object)ports.Get();
    int err;
    Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err,(pt, GOAL_PHASE, ofile));
    if (err == 0) {
      cerr << "error in goal: " << goal << " port: " << pt << endl;
      return 0;
    }
    output_point_size(pt, ofile);
    ofile << endl;
  }
  if (!first) ofile << ")";
  ofile << "\t % proc '" << name;
  output_point_size(goal, ofile);
  ofile << endl;
  return 1;
}

int translate_goal_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List goals = Rule_Goals(rule);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = (Am_Object)goals.Get();
    if (translate_goal(goal, ofile) == 0) return 0;
  }
  return 1;
}

int translate_unify_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List goals = Rule_Goals(rule);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object gl = (Am_Object)goals.Get();
    Am_Value_List ports = Proc_Ports(gl);
    for (ports.Start(); !ports.Last(); ports.Next()) {
      Am_Object pt = (Am_Object)ports.Get();
      int err;
      Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err,(pt,LINK_PHASE,ofile));
      if (err == 0) {
	cerr << "error in goal: " << gl << " port: " << pt << endl;
	return 0;
      }
//      ofile << endl;
    }
  }
  Am_Value_List ports = sort_ports(Proc_Ports(rule));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    int err;
    Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err, (pt, UNIFY_PHASE, ofile));
    if (err == 0) {
      cerr << "error in port:" << pt << endl;
      return 0;
    }
//    ofile << endl;
  }
  return 1;
}

int translate_transition_body(Am_Object rule, ofstream& ofile)
{
  ofile << ", ";
  Am_String rule_name = (Am_String)rule.Get(PROC_NAME);
  ofile << rule_name;
  Am_Value_List ports = sort_ports(Proc_Ports(rule));
  int first = true;
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    int id = pt.Get(PORT_ID);
    Am_String name = (Am_String)pt.Get(PORT_NAME);
    Am_String inner_name = (Am_String)pt.Get(PORT_INNER_NAME);
    Am_String next_name = (Am_String)pt.Get(PORT_INNER_NEXT_NAME);
    int type = pt.Get(PORT_TYPE);
    int mode = pt.Get(PORT_MODE);
    int age = pt.Get(PORT_AGE);
    int open = pt.Get(PORT_OPEN);
    if (type == KL_Single && age == KL_Old_Port) continue;
    if (first) ofile << "( ";
    else       ofile << ", ";
    if (type == KL_Stream) {
      if (mode == KL_Input) {
	if (open == KL_Close) ofile << "[ ]";
	else ofile << inner_name;
	ofile << "\t % port " << name << " " << id << Mode_Str(mode) << Type_Str(type);
	output_point_size(pt, ofile);
	ofile << endl;
      } else if (mode == KL_Output) {
	if (!Port_Has_Value(pt)) ofile << inner_name;
	else ofile << next_name;
	ofile << "\t % port " << name << " " << id << Mode_Str(mode) << Type_Str(type);
	output_point_size(pt, ofile);
	ofile << endl;
      }
    } else if (type == KL_Single && age == KL_New_Port) {
      int err;
      Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD,err,(pt,GOAL_PHASE,ofile));
      if (err == 0) {
	cerr << "error in port:" << pt << endl;
	return 0;
      }
//      ofile << "\t % port " << name << " " << id << Mode_Str(mode) << Type_Str(type);
      output_point_size(pt, ofile);
      ofile << endl;
    }
    first = false;
  }
  if (!first) ofile << ") ";
  ofile << "\t % proc '" << rule_name << endl;
  ofile << "% other_part" << endl;
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    int mode = pt.Get(PORT_MODE);
    int open = pt.Get(PORT_OPEN);
    if (mode == KL_Output) {
      if (Port_Has_Value(pt) || open == KL_Close) {
	int err;
	Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD,err,(pt,UNIFY_PHASE,ofile));
	if (err == 0) {
	  cerr << "error in port:" << pt << endl;
	  return 0;
	}
      }
/*      else {
	int id = pt.Get(PORT_ID);
	Am_String inner_name = pt.Get(PORT_INNER_NAME);
	Am_String next_name = pt.Get(PORT_INNER_NEXT_NAME);
	ofile << ", " << next_name << " = " << inner_name << "\t% unify " << id << endl;
      }
*/
    }
  }
  return 1;
}

int translate_atom(Am_Object atom, int flag, ofstream& ofile)
{
  if (flag == UNIFY_PHASE || flag == LINK_PHASE) return 1;
  Am_Object text_part = atom.Get_Part(CONTENTS_PART);
  Am_String name = (Am_String)text_part.Get(Am_TEXT);
  ofile << "~(" << name << ")" << "\t % atom '" << name << "'"; 
  return 1;
}

int translate_structure(Am_Object structure, int flag, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(structure);
  Am_String name = (Am_String)structure.Get(PORT_NAME);
  Am_Value_List values = Port_Values(structure);

  if (flag == UNIFY_PHASE || flag == LINK_PHASE) {
    for (values.Start(); !values.Last(); values.Next()) {
      Am_Object value = (Am_Object)values.Get();
      int err;
      Am_Function_Call(KL_Trans_Method, value, KL_TRANS_METHOD, err, (value,flag,ofile));
      if (err == 0) {
	cerr << "error in structure:" << structure << " value: " << value << endl;
	return 0;
      }
    }
    return 1;
  }
  ofile << name;
  int first = true;
  for (values.Start(); !values.Last(); values.Next(), first = false) {
    if (first) ofile << "( ";
    else       ofile << ", ";
    Am_Object value = (Am_Object)values.Get();
    int err;
    Am_Function_Call(KL_Trans_Method, value, KL_TRANS_METHOD, err, (value, flag, ofile));
    if (err == 0) {
      cerr << "error in structure:" << structure << " value: " << value << endl;
      return 0;
    }
    ofile << endl;
  }
  if (!first) ofile << ")";
  ofile << "\t % mesg '" << name;
  return 1;
}

int link_input_stream(Am_Object port, ofstream& ofile)
{
  int id   = port.Get(PORT_ID);
  int age  = port.Get(PORT_AGE);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  Am_Value_List binders;
  if (age == KL_Old_Port && KL_Type(Port_Parent(port)) == KL_Pattern) {
    binders = port.Get(PORT_INNER_BINDERS);
  } else {
    binders = port.Get(PORT_BINDERS);
  }
  binders.Start();
  if (!binders.Empty()) {
    if (binders.Length() == 1) {
      Am_Object bd = (Am_Object)binders.Get();
      ofile << ", " << inner_name << " = " << (Am_String)bd.Get(BL_INNER_NAME);
      ofile << "\t % link " << id << " " << (int)bd.Get(BL_ID) << endl;
    } else {
      ofile << ", generic:new(merge, {";
      int first = true;
      for (binders.Start(); !binders.Last(); binders.Next(), first = false) {
	if (!first) ofile << ", ";
	Am_Object bd = (Am_Object)binders.Get();
	ofile << (Am_String)bd.Get(BL_INNER_NAME);
      }
      ofile << "}, " << inner_name << ")";
      ofile << "\t % merge ";
      ofile << "into " << id << " from ";
      for (binders.Start(); !binders.Last(); binders.Next()) {
	Am_Object bd = (Am_Object)binders.Get();
	ofile << (int)bd.Get(BL_ID) << " ";
      }
      ofile << endl;
    }
  }
  return 1;
}

int link_phase(Am_Object port, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  Am_String name = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  int type = port.Get(PORT_TYPE);
  int id   = port.Get(PORT_ID);
  int mode = port.Get(PORT_MODE);
  int open = port.Get(PORT_OPEN);
  int age  = port.Get(PORT_AGE);
  Am_Value_List values = Port_Values(port); values.Start();
  Am_Value_List binders;
  if (age == KL_Old_Port && KL_Type(Port_Parent(port)) == KL_Pattern) {
    binders = port.Get(PORT_INNER_BINDERS);
  } else {
    binders = port.Get(PORT_BINDERS);
  }
  binders.Start();
  Am_String mode_str = Mode_Str(mode);
  if ((int)port.Get(PORT_AGE) == KL_Old_Port) {
    mode = !mode;
  }    
  if (mode == KL_Input && open == KL_Open &&
      !Port_Has_Value(port) && binders.Empty()) {
    cerr << "Port: " << port << "has no value or binder." << endl;
    return 0;
  }
  if (mode == KL_Input) {
    switch (type) {
    case KL_Single:
      if (!binders.Empty()) {
	Am_Object bd = (Am_Object)binders.Get();
	ofile << ", " << inner_name << " = " << (Am_String)bd.Get(BL_INNER_NAME);
	ofile << "\t % link " << id << " " << (int)bd.Get(BL_ID) << endl;
      } else if (Port_Has_Value(port)) {
	Am_Object val = (Am_Object)values.Get();
	int err;
	Am_Function_Call(KL_Trans_Method, val, KL_TRANS_METHOD,
			 err, (val, LINK_PHASE, ofile));
      } else {
	cerr << "Port: " << port << "has neither binder nor value." << endl;
	return 0;
      }
      return 1;
    case KL_Stream:
      link_input_stream(port, ofile);
      for (values.Start(); !values.Last(); values.Next()) {
	Am_Object val = (Am_Object)values.Get();
	int err;
	KL_Translate_Call(val, err, LINK_PHASE, ofile);
      }
      return 1;
    default:
      return 0;
    }
  }
  if (mode == KL_Output) {
    if (Port_Has_Value(port)) {
      for(values.Start(); !values.Last(); values.Next()) {
	Am_Object val = (Am_Object)values.Get();
	int err;
	Am_Function_Call(KL_Trans_Method, val,
			 KL_TRANS_METHOD, err, (val, LINK_PHASE, ofile));
      }
    }
    return 1;
  }
  return 0;
}  

int unify_phase(Am_Object port, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  Am_String name = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  Am_String next_name = (Am_String)port.Get(PORT_INNER_NEXT_NAME);
  int type = port.Get(PORT_TYPE);
  int id = port.Get(PORT_ID);
  int mode = port.Get(PORT_MODE);
  int open = port.Get(PORT_OPEN);
  Am_Value_List values = Port_Values(port); values.Start();
  Am_Value_List binders;
  int paren_type = KL_Type(Port_Parent(port));
  if (paren_type == KL_Pattern) binders = port.Get(PORT_INNER_BINDERS);
                           else binders = port.Get(PORT_BINDERS);
  binders.Start();
  Am_String mode_str = Mode_Str(mode);
  if ((int)port.Get(PORT_AGE) == KL_Old_Port) mode = !mode;
  int err;
  if (mode == KL_Input) {
    switch (type) {
    case KL_Single:
      if (Port_Has_Value(port)) {
	ofile << ", ";
	Am_Object val = (Am_Object)values.Get();
	KL_Translate_Call(val, err, GOAL_PHASE, ofile);
	ofile << endl << " = " << inner_name << "\t % unify " << id << endl;
	KL_Translate_Call(val, err, LINK_PHASE, ofile); // $BCf?H$N%j%s%/$rD%$j$K9T$/!%(B
      } else if (!binders.Empty()) {
	Am_Object bd = (Am_Object)binders.Get();
	int src_id = bd.Get(BL_ID);
	ofile << ", " << inner_name << " = " << (Am_String)bd.Get(BL_INNER_NAME);
	ofile << "\t % link " << id << " " << src_id << endl;
      } else {
	cerr << "unify_phase(), port: "<< port <<" has neither binder nor value" << endl;
	return 0;
      }
      return 1;
    case KL_Stream:
      if (Port_Has_Value(port) || open == KL_Close) {
	ofile << ", [ ";
	int first = true;
	for (values.Start(); !values.Last(); values.Next(), first = false) {
	  if (!first) ofile << ", ";
	  Am_Object value = (Am_Object)values.Get();
	  KL_Translate_Call(value, err, GOAL_PHASE, ofile);
	  ofile << endl;
	  if (err == 0) {cerr <<"unify_phase(): error in port"<< port << endl; return 0;}
	}
	if (open == KL_Open) ofile << "| " << next_name << " ]";
	                else ofile << "]";
	ofile << " = " << inner_name << "\t % unify " << id << endl;
	port.Set(PORT_INNER_NAME, (char*)next_name); // $B0J9_$O?7$7$$JQ?t$K%j%s%/$9$k!%(B
	KL_Translate_Call(port, err, LINK_PHASE, ofile);
	if (err == 0) {cerr <<"unify_phase(): error in port"<< port << endl; return 0;}
	port.Set(PORT_INNER_NAME, (char*)inner_name);

      } else if (!binders.Empty()) {
	KL_Translate_Call(port, err, LINK_PHASE, ofile);
      } else {
	cerr << "unify_phase(), port: "<< port <<" has neither binder nor value" << endl;
	return 0;
      }
      return 1;
    }
  }
  if (mode == KL_Output) {
    if (Port_Has_Value(port)) {
      for(values.Start(); !values.Last(); values.Next()) {
	Am_Object val = (Am_Object)values.Get();
	KL_Translate_Call(val, err, UNIFY_PHASE, ofile);
      }
    }
    return 1;
  }
  return 0;
}  

int translate_port(Am_Object port, int flag, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  Am_String name = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  int type = port.Get(PORT_TYPE);
  int id = port.Get(PORT_ID);
  int mode = port.Get(PORT_MODE);
  int open = port.Get(PORT_OPEN);
  Am_Value_List values = Port_Values(port); values.Start();
  Am_Value_List binders = (Am_Value_List)port.Get(PORT_BINDERS); binders.Start();
  Am_String mode_str = Mode_Str(mode);
  if ((int)port.Get(PORT_AGE) == KL_Old_Port) {
    mode = !mode;
  }    
  if (flag == LINK_PHASE) {
    if (link_phase(port, ofile) == 1) return 1;
    return 0;
  }
  if (flag == UNIFY_PHASE) {
    if (unify_phase(port, ofile) == 1) return 1;
    return 0;
  }
  // Head$BIt=hM}Cf$N(BInput Port
  if (mode == KL_Input && flag == HEAD_PHASE) {
    ofile << inner_name;
    ofile << "\t % port " << name << " " << id << mode_str << Type_Str(type);
//    output_point_size(port, ofile);
    return 1;
  } if (KL_Type(Port_Parent(port)) == KL_Pattern && flag == GOAL_PHASE) {
    ofile << inner_name;
    ofile << "\t % port " << name << " " << id << mode_str << Type_Str(type);
    return 1;
  } else {
  // $B$=$l0J30!%(B
    switch (type) {
    case KL_Single:
      if (values.Empty()) {
	ofile << inner_name;
	ofile << "\t % port " << name << " " << id << mode_str << Type_Str(type);	
	return 1;
      } else {
	values.Start();
	Am_Object value = (Am_Object)values.Get();
	int err;
	Am_Function_Call(KL_Trans_Method,
			 value, KL_TRANS_METHOD, err, (value, flag, ofile));
	if (err == 0) {
	  cerr << "error in port: " << port << " value: " << value << endl;
	  return 0;
	}
        // value $B$NF~$C$F$$$k%]!<%H$N>pJs!%(B
	ofile << " port " << name << " " << mode_str << Type_Str(type);
//	ofile << endl;
	return 1;
      }
      break;
    case KL_Stream:
      if (values.Empty() && open == KL_Open) {
	ofile << inner_name;
      } else {
	ofile << "[ ";
	int first = true;
	for (values.Start(); !values.Last(); values.Next(), first = false) {
	  if (!first) ofile << ", ";
	  Am_Object value = (Am_Object)values.Get();
	  int err;
	  Am_Function_Call(KL_Trans_Method,
			   value, KL_TRANS_METHOD, err, (value, flag, ofile));
	  if (err == 0) {
	    cerr << "error in port:" << port << endl;
	    return 0;
	  }
	  ofile << endl;
	}
	if (open == KL_Open)
	  ofile << "| " << inner_name << " ]";
	else
	  ofile << "]";
      }
      break;
    default:
      cerr << "Top Rule Port: " << name << " has invalid type: " << type << endl;
      return 0;
    }
  }
  ofile << "\t % port " << name << " " << id << mode_str << Type_Str(type);
//  output_point_size(port, ofile);
  return 1;
}

//
// ------------------- Pattern Translator ------------------------------
//
#define Def_Phase  10
#define Goal_Phase 11

int translate_patterns_iter(Am_Object rule, ofstream& ofile)
{
  Am_Value_List goals = Rule_Goals(rule);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = (Am_Object)goals.Get();
    if (KL_Type(goal) == KL_Pattern) {
      if ((Am_Object)goal.Get(PARENT_LINK) == goal) { // $B%3%T!<$G$O$J$$$3$H!%(B
	int err;
	KL_Translate_Call(goal, err, Def_Phase, ofile);
	if (err == 0) {
	  cerr << "translate_patterns_iter(): failure in "
	       << (Am_String)goal.Get(PROC_NAME) << endl;
	  return 0;
	}
	translate_patterns_iter(goal, ofile);
      }
    }
  }
}

int translate_patterns(Am_Object rules_agg, ofstream& ofile)
{
  Am_Value_List rules = (Am_Value_List)rules_agg.Get(Am_GRAPHICAL_PARTS);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = (Am_Object)rules.Get();
    if (KL_Type(rule) == KL_Rule) {
      if (!translate_patterns_iter(rule, ofile)) {
	cerr << "translate_patterns(): failure in "
	     << (Am_String)rule.Get(PROC_NAME) << endl;
	return 0;
      }
    }
  }
  return 1;
}

// -------- Free Format Pattern --------

int freeform_pattern_def_phase(Am_Object pattern, ofstream& ofile)
{
  Am_String name = (Am_String)pattern.Get(PROC_NAME);
  /* translate spec */
  ofile << "% define '" << name << " creator ";
  output_point_size(pattern, ofile);
  ofile << endl;
  Am_Value_List ports = sort_ports(Proc_Ports(pattern));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    translate_port_in_proc(port, ofile);
    port.Set(PORT_AGE, KL_Old_Port); // $B%k!<%k$H$7$F$_$k$H$-$O(B,Old_Port.$B$H$F$b1x$$!%(B
  }
  ofile << "% begin" << endl;
  /* translate rules */
  translate_rule(pattern, ofile);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    port.Set(PORT_AGE, KL_New_Port); // $B85$KLa$9!%(B
  }
  ofile << "c_" << name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}

int freeform_pattern_goal_phase(Am_Object pattern, ofstream& ofile)
{
  return 1;
}

int translate_freeform_pattern(Am_Object pattern, int flag, ofstream& ofile)
{
  if (flag == Def_Phase)  return freeform_pattern_def_phase(pattern, ofile);
  if (flag == Goal_Phase) return freeform_pattern_goal_phase(pattern, ofile);
  return 0;
}
    
// -------- Sequence Pattern --------

// Utility
Am_Object bound_port_at(Am_Object port, Am_Slot_Key key)
{
  Am_Value_List binders = (Am_Value_List)port.Get(key); binders.Start();// $B30$O4X78$J$$!%(B
  if (binders.Empty()) return 0;
  Am_Object bd = (Am_Object)binders.Get(); // $B0lK\$@$1$r2>Dj!%$H$j$"$($:!%(B
  Am_Object src = (Am_Object)bd.Get(BL_SRC);
  Am_Object des = (Am_Object)bd.Get(BL_DEST);
  Am_Object return_port = (src == port) ? des : src; // $B9T@h$O<+J,$G$O$J$$$[$&!%(B
  cerr << "bound_port: " << (Am_String)port.Get(PORT_NAME) << " "
       << (Am_String)return_port.Get(PORT_NAME) << endl;
  return return_port;
}

Am_Object inner_bound_port(Am_Object port)
{
  return bound_port_at(port, PORT_INNER_BINDERS);
}

Am_Object bound_port(Am_Object port)
{
  return bound_port_at(port, PORT_BINDERS);
}

int bound_goal_index_at(Am_Object port, Am_Slot_Key key)
{
  Am_Object dest_port = bound_port_at(port, key);
  if (!dest_port.Valid()) return 0;
  Am_Object goal = Port_Parent_Proc(dest_port);
  if (!goal.Valid()) return 0;
  return Group_Index(goal);
}

int bound_same_level_goal_index_at(Am_Object port, Am_Slot_Key key)
{
  Am_Object dest_port = bound_port_at(port, key);
  if (!dest_port.Valid()) return 0;
  Am_Object goal = Port_Parent_Proc(dest_port);
  if (!goal.Valid()) return 0;
  if (goal.Get_Owner() != Port_Parent_Proc(port).Get_Owner()) return 0;
  return Group_Index(goal);
}

int bound_goal_index(Am_Object port)
{
  return bound_goal_index_at(port, PORT_INNER_BINDERS);
}

Am_Object first_port(Am_Object port)
{
  //                 first_goal          second_goal         third_goal
  // return---[first_src, first_des]---[second_src, _ ]---[ _, third_des]--- port
  cerr << "first_port()" << endl;
  Am_Object third_des = inner_bound_port(port);        if (!third_des.Valid()) return 0;
  Am_Object third_goal = Port_Parent_Proc(third_des);  if (!third_goal.Valid()) return 0;
  cerr << "third_des: " << third_des << " third_goal: " << third_goal << endl;
  Am_Value_List third_des_pos = Position_Get(third_des, third_goal);
  Am_Object first_goal = Group_Get_By_Index(third_goal.Get_Owner(), 1);
                                                       if (!first_goal.Valid()) return 0;
  Am_Object first_des = Position_To_Obj(first_goal, third_des_pos); 
                                                       if (!first_des.Valid()) return 0;
  cerr << "first_des: " << first_des << " first_goal: " << first_goal << endl;
  Am_Object second_src = bound_port(first_des);        if (!second_src.Valid()) return 0;
  Am_Object second_goal = Port_Parent_Proc(second_src);if (!second_goal.Valid())return 0;
  cerr << "second_src: " << second_src << " second_goal: " << second_goal << endl;
  Am_Value_List second_src_pos = Position_Get(second_src, second_goal);
  Am_Object first_src = Position_To_Obj(first_goal, second_src_pos);
                                                       if (!first_src.Valid()) return 0;
  
  return bound_port(first_src);
}

Am_Object next_link_port(Am_Object port)
{
  // port---[first_src, return]---[srcond_src, _ ]---...
  cerr << "next_link_port()";
  Am_Object first_src = inner_bound_port(port);         if (!first_src.Valid()) return 0;
  cerr << (Am_String)first_src.Get(PORT_NAME) << " in ";
  Am_Object first_goal = Port_Parent_Proc(first_src);   if (!first_goal.Valid()) return 0;
  cerr << (Am_String)first_goal.Get(PROC_NAME) << endl;
  Am_Value_List first_src_pos = Position_Get(first_src, first_goal);
  Am_Object second_goal = Group_Get_By_Index(first_goal.Get_Owner(), 2);
                                                        if (!second_goal.Valid())return 0;
  cerr << (Am_String)second_goal.Get(PROC_NAME) << " " ;
  Am_Object second_src = Position_To_Obj(second_goal, first_src_pos);
                                                        if (!second_src.Valid()) return 0;
  cerr << (Am_String)second_src.Get(PORT_NAME) << endl;
  return bound_port(second_src);
}

int translate_stream(Am_Object port, int flag, char* name, ofstream& ofile)
{
  ofile << "[ ";
  int first = true;
  Am_Value_List values = Port_Values(port); values.Start();
  for (values.Start(); !values.Last(); values.Next(), first = false) {
    if (!first) ofile << ", ";
    Am_Object value = (Am_Object)values.Get();
    int err;
    KL_Translate_Call(value, err, flag, ofile);
    ofile << endl;
    if (err == 0) {cerr <<"translate_stream(): error in port"<< port << endl; return 0;}
  }
  if ((int)port.Get(PORT_OPEN) == KL_Open)
    ofile << "| " << name << " ]";
  else
    ofile << "]";
  return 1;
}

// translate recursion rule
int translate_recursion(Am_Object pattern, ofstream& ofile)
{
  Am_String name = (Am_String)pattern.Get(PROC_NAME);
  ofile << ", " << name;
  Am_Value_List ports = sort_ports(Proc_Ports(pattern));
  int first = true;
  for (ports.Start(); !ports.Last(); ports.Next(), first = false) {
    if (first) ofile << "( ";
    else       ofile << ", ";
    Am_Object pt = (Am_Object)ports.Get();
    int err;
    Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err,(pt,REC_GOAL_PHASE,ofile));
    if (err == 0) {
      cerr << "error in pattern: " << pattern << " port: " << pt << endl;
      return 0;
    }
    ofile << " point " << 0 << " " << 0 // $B:BI8$O#0!$#0$K$9$k!JBg$-$9$.$J$$$h$&!K!%(B
	  << " size " << (int)pt.Get(Am_WIDTH) << " " << (int)pt.Get(Am_HEIGHT) << endl;
  }
  if (!first) ofile << ")";
  ofile << "\t % proc '" << name;
  Am_Object second_goal = Object_List_Get(Rule_Goals(pattern), 2);
  output_point_size(second_goal, ofile); // $B#2HVL\$N%4!<%k$HF1$8Bg$-$5!%(B
  ofile << endl;
  return 1;
}

int translate_rec_unify_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List goals = Rule_Goals(rule);
  if (goals.Empty()) return 0;
  goals.Start();
  Am_Object gl = (Am_Object)goals.Get(); // $B:G=i$N%4!<%k$N$_!%(B
  Am_Value_List ports = Proc_Ports(gl);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    int err;
    if (bound_same_level_goal_index_at(pt, PORT_BINDERS) != 2) {//$B#2HVL\$K$N$S$k@~$OL5;k!%(B
      Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err,(pt,LINK_PHASE,ofile));
      if (err == 0) {
        cerr << "error in goal: " << gl << " port: " << pt << endl;
        return 0;
      }
    }
  }
  ports = sort_ports(Proc_Ports(rule));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    int err;
    Am_Function_Call(KL_Trans_Method, pt, KL_TRANS_METHOD, err, (pt, UNIFY_PHASE, ofile));
    if (err == 0) {
      cerr << "error in port:" << pt << endl;
      return 0;
    }
  }
  return 1;
}

int sequence_recursive_rule(Am_Object pattern, ofstream& ofile)
{
  Am_String rule_name = (Am_String)pattern.Get(PROC_NAME);
  int rule_id = 0;
  char* rule_type = Rule_Type_Str((int)pattern.Get(RULE_TYPE));
  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << rule_type;
  output_point_size(pattern, ofile);
  ofile << endl;
  // Head$BIt(B 
  translate_head_part(pattern, ofile); //$B$=$N$^$^$D$+$&!%(B
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(pattern, ofile); // $B<h$j$"$($:;D$7$F$*$/!%$I$&$;6u!%(B
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  Am_Value_List goals = Rule_Goals(pattern); goals.Start();
  if (goals.Empty()) return 0;
  int order = pattern.Get(GEN_ORDER);
  if (order == 0) {
    translate_goal((Am_Object)goals.Get(), ofile); // $B%4!<%k$O$=$N$^$^%H%i%s%9%l!<%H!%(B
    translate_recursion(pattern, ofile); // $B:F5"!%(B
  } else {
    translate_recursion(pattern, ofile); // $B:F5"!%(B
    translate_goal((Am_Object)goals.Get(), ofile); // $B%4!<%k$O$=$N$^$^%H%i%s%9%l!<%H!%(B
  }
  ofile << "% other_part" << endl;
  translate_rec_unify_part(pattern, ofile);
  ofile << "." << endl;
  return 1;
}

int translate_term_unify_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List ports = sort_ports(Proc_Ports(rule));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    int err;
    KL_Translate_Call(pt, err, TERM_UNIFY_PHASE, ofile);
    if (err == 0) {
      cerr << "error in port:" << pt << endl;
      return 0;
    }
  }
  return 1;
}

int sequence_terminate_rule(Am_Object pattern, ofstream& ofile)
{
  Am_String rule_name = (Am_String)pattern.Get(PROC_NAME);
  int rule_id = 1;
  char* rule_type = Rule_Type_Str((int)pattern.Get(RULE_TYPE));
  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << "vanishing"; // rule_type;$B%"!<%-%?%J%$%Q%C%A!%(B
  output_point_size(pattern, ofile);
  ofile << endl;
  // Head$BIt(B 
  translate_head_part_by(pattern, TERM_HEAD_PHASE, ofile);
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(pattern, ofile); // $B<h$j$"$($:;D$7$F$*$/!%$I$&$;6u!%(B
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  ofile << "% other_part" << endl;
  translate_term_unify_part(pattern, ofile);
  ofile << "." << endl;
  return 1;
}

char* Seq_Dir_Str[4] = {" rightward ", " downward ", " rightward ", " downward "};
//char* Seq_Dir_Str[4] = {" rightward ", " downward ", " leftward ", " upward "};

/*
int translate_port_in_rec_proc(Am_Object port, ofstream& ofile)
{
  Am_String port_name = (Am_String)port.Get(PORT_NAME);
  char* port_mode = Mode_Str((int)port.Get(PORT_MODE));
  char* port_type = Type_Str((int)port.Get(PORT_TYPE));
  char* port_dir  = Dir_Str((int)port.Get(PORT_DIR));
  int x = port.Get(Am_LEFT);  int y = port.Get(Am_TOP);
  int w = port.Get(Am_WIDTH); int h = port.Get(Am_HEIGHT);
  ofile << "%	port " << port_name << port_mode << port_type << port_dir;
  ofile << " point " << 0 << " " << 0 << " size " << 0 << " " << 0;
//  output_point_size(port, ofile);
  ofile << endl;
  return 1;
}
*/

int sequence_pattern_def_phase(Am_Object pattern, ofstream& ofile)
{
  Am_String name = (Am_String)pattern.Get(PROC_NAME);
  int dir = pattern.Get(SEQ_DIR);
  /* translate spec */
  ofile <<"% define '"<< name <<" creator arrange "<< Seq_Dir_Str[dir]; // << " invisible";
  output_point_size(pattern, ofile);
  ofile << endl;
  Am_Value_List ports = sort_ports(Proc_Ports(pattern));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    translate_port_in_proc(port, ofile);
    port.Set(PORT_AGE, KL_Old_Port); // $B%k!<%k$H$7$F$_$k$H$-$O(B,Old_Port.$B$H$F$b1x$$!%(B
  }
  ofile << "% begin" << endl;
  /* translate rules */
  sequence_recursive_rule(pattern, ofile);
  sequence_terminate_rule(pattern, ofile);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    port.Set(PORT_AGE, KL_New_Port); // $B85$KLa$9!%(B
  }
  ofile << "c_" << name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}

int sequence_pattern_goal_phase(Am_Object pattern, ofstream& ofile)
{
  return 1;
}

int translate_sequence_pattern(Am_Object pattern, int flag, ofstream& ofile)
{
  if (flag == Def_Phase)  return sequence_pattern_def_phase(pattern, ofile);
  if (flag == Goal_Phase) return sequence_pattern_goal_phase(pattern, ofile);
  return 0;
}

// ---------- translate special ports ----------

int trans_normal_port(Am_Object port, int flag, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  if ((int)rule.Get(PATTERN_TYPE) == KL_Free_Pattern)
    return translate_port(port, flag, ofile);
  Am_String name       = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  Am_String next_name  = (Am_String)port.Get(PORT_INNER_NEXT_NAME);
  int age  = port.Get(PORT_AGE);
  int id   = port.Get(PORT_ID);
  int next_id = port.Get(PORT_NEXT_ID);
  int open = port.Get(PORT_OPEN);
  int mode = port.Get(PORT_MODE);
  Am_String mode_str = Mode_Str(mode);
  mode = !mode;  // $B%k!<%k$J$N$G5UE>!%(B
  Am_Value_List values = Port_Values(port); values.Start();
  int err;
  switch (flag) {
  case HEAD_PHASE: {
    if (values.Empty()) {
      ofile << inner_name;
    } else {
      if (translate_stream(port, HEAD_PHASE, inner_name, ofile) == 0) return 0;
    }
    break;
  }
  case GOAL_PHASE:
    ofile << inner_name;  break;
  case REC_GOAL_PHASE:
    if (bound_goal_index(port) == 1) ofile << next_name;
                                else ofile << inner_name;
    ofile << "\t % port " << name << " " << next_id << mode_str << Type_Str(KL_Stream);
    return 1;
  case LINK_PHASE:
    if (mode == KL_Output) {
      link_input_stream(port, ofile);
    }
    return 1;
  case UNIFY_PHASE:
    if (bound_goal_index(port) == 1) {
      Am_Object pt = inner_bound_port(port);
      if (!pt.Valid()) return 0;
      if (mode == KL_Input) {
        ofile << (Am_String)pt.Get(PORT_INNER_NAME) << " = " << inner_name;
        ofile << "\t% link " << next_id << " " << (int)pt.Get(PORT_ID) << endl;
      }
      Am_Object next_port = next_link_port(port);
      if (!next_port.Valid()) return 0;
      ofile << ", " << (Am_String)next_port.Get(PORT_INNER_NAME) << " = ";
      ofile << next_name;
      if (mode == KL_Input)
	ofile << "\t% link " << (int)next_port.Get(PORT_ID) << " " << next_id << endl;
      else
	ofile << "\t% link " << next_id << " " << (int)next_port.Get(PORT_ID) << endl;
    }
    return 1;
  case TERM_HEAD_PHASE:
    if (values.Empty()) {
      ofile << inner_name;
    } else {
      ofile << "[]";
    }
    break;
  case TERM_UNIFY_PHASE:
    if (bound_goal_index(port) == 3) {
      Am_Object first_pt = first_port(port);
      if (!first_pt.Valid()) return 0;
      int first_id = first_pt.Get(PORT_ID);
      Am_String first_name = (Am_String)first_pt.Get(PORT_INNER_NAME);
      ofile << ", " << first_name << " = " << inner_name
	    << "\t% link " << first_id << " " << id << endl;
    }
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  ofile << "\t % port " << name << " " << id << mode_str << Type_Str(KL_Stream);
  return 1;
}

int trans_map_in_port(Am_Object port, int flag, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  Am_String name       = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  int id   = port.Get(PORT_ID);
  int next_id = port.Get(PORT_NEXT_ID);
  int open = port.Get(PORT_OPEN);
  int mode = KL_Output; // $B%k!<%k$J$N$G5UE>!%(B
  Am_String mode_str = Mode_Str(KL_Input);
  Am_Value_List values = Port_Values(port); values.Start();
  if (values.Empty()) {
    cerr << "trans_map_in_port(): no values " << name <<" in "<< (Am_String)rule.Get(PROC_NAME) <<endl;
    return 0;
  }
  int err;
  switch (flag) {
  case HEAD_PHASE: {
    ofile << "[ ";
    Am_Object value = (Am_Object)values.Get(); // $B6u$G$O$J$$$O$:!%(B
    KL_Translate_Call(value, err, flag, ofile); ofile << endl;
    if (err == 0) return 0;
    ofile << "| " << inner_name << " ]";
    break;
  }
  case GOAL_PHASE:
    ofile << inner_name;
    break;
  case REC_GOAL_PHASE:
    ofile << inner_name;
    ofile << "\t % port " << name << " " << next_id << mode_str << Type_Str(KL_Stream);
    return 1;
  case LINK_PHASE:
    link_input_stream(port, ofile);
    return 1;
  case UNIFY_PHASE: {
    Am_Object value = (Am_Object)values.Get();
    KL_Translate_Call(value, err,  LINK_PHASE, ofile);
    if (err == 0) {cerr <<"trans_map_in_port(): error in port"<<value<<endl; return 0;}
    ofile << ", " << inner_name << " = " << inner_name
          << "\t% link " << next_id << " " << id << endl;
    return 1;
  }
  case TERM_HEAD_PHASE:
    ofile << "[]";
    break;
  case TERM_UNIFY_PHASE:
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  ofile << "\t % port " << name << " " << id << mode_str << Type_Str(KL_Stream);
  return 1;
}

int trans_map_out_port(Am_Object port, int flag, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  Am_String name       = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  Am_String next_name  = (Am_String)port.Get(PORT_INNER_NEXT_NAME);
  int id   = port.Get(PORT_ID);
  int next_id = port.Get(PORT_NEXT_ID);
  int open = port.Get(PORT_OPEN);
  int mode = KL_Input; // $B%k!<%k$J$N$G5UE>!%(B
  Am_String mode_str = Mode_Str(KL_Output);
  Am_Value_List values = Port_Values(port); values.Start();
  if (values.Empty()) {
    cerr << "trans_map_out_port(): no values " << name <<" in "<< (Am_String)rule.Get(PROC_NAME) <<endl;
    return 0;
  }
  int err;
  switch (flag) {
  case HEAD_PHASE:
    ofile << inner_name;  break;
  case GOAL_PHASE:
    ofile << inner_name;  break;
  case REC_GOAL_PHASE:
    ofile << next_name;
    ofile << "\t % port " << name << " " << next_id << mode_str << Type_Str(KL_Stream);
    return 1;
  case UNIFY_PHASE: {
    ofile << ", [ ";
    Am_Object value = (Am_Object)values.Get(); // $B6u$G$O$J$$$O$:!%(B
    KL_Translate_Call(value, err, GOAL_PHASE, ofile); ofile << endl;
    if (err == 0) return 0;
    ofile << "| " << next_name << " ] = " << inner_name << "\t % unify " << id << endl;
    ofile << ", " << next_name << " = " << next_name
	  << "\t% link " << id << " " << next_id << endl;
    KL_Translate_Call(value, err, LINK_PHASE, ofile); // $B@~$rD%$k!%(B
    return 1;
  }
  case LINK_PHASE:
    return 1;
  case TERM_HEAD_PHASE:
    ofile << inner_name; break;
  case TERM_UNIFY_PHASE:
    ofile << ", [] = " << inner_name << "\t % unify " << id << endl;
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  ofile << "\t % port " << name << " " << id << mode_str << Type_Str(KL_Stream);
  return 1;
  
}

int trans_merge_port(Am_Object port, int flag, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  Am_String name       = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  Am_String next_name  = (Am_String)port.Get(PORT_INNER_NEXT_NAME);
  int id   = port.Get(PORT_ID);
  int next_id = port.Get(PORT_NEXT_ID);
  int open = port.Get(PORT_OPEN);
  int mode = KL_Input; // $B%k!<%k$J$N$G5UE>!%(B
  Am_String mode_str = Mode_Str(KL_Output);
  Am_Value_List binders = (Am_Value_List)port.Get(PORT_INNER_BINDERS); //$B%Q%?!<%s$N30It=PNO8N!%(B
  binders.Start();
  if (binders.Empty()) {
    cerr << "trans_merge_port(): no binders " << endl;
    return 0;
  }
  int err;
  switch (flag) {
  case HEAD_PHASE:
    ofile << inner_name;  break;
  case GOAL_PHASE:
    ofile << inner_name;  break;
  case REC_GOAL_PHASE:
    ofile << next_name;
    ofile << "\t % port " << name << " " << next_id << mode_str << Type_Str(KL_Stream);
    return 1;
  case UNIFY_PHASE: {
    Am_Object binder = (Am_Object)binders.Get(); // $B6u$G$O$J$$$O$:!%(B
    ofile << ", generic:new(merge, {" << (Am_String)binder.Get(BL_INNER_NAME);
    ofile << ", " << next_name << "}, " << inner_name << ")";
    ofile << "\t % merge ";
    ofile << "into " << id << " from ";
    ofile << (int)binder.Get(BL_ID) << " " <<  next_id << endl;
    ofile << ", " << next_name << " = " << next_name
	  << "\t% link " << id << " " << next_id << endl;
    return 1;
  }
  case LINK_PHASE:
    return 1;
  case TERM_HEAD_PHASE:
    ofile << inner_name; break;
  case TERM_UNIFY_PHASE:
    ofile << ", [] = " << inner_name << "\t % unify " << id << endl;
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  ofile << "\t % port " << name << " " << id << mode_str << Type_Str(KL_Stream);
  return 1;
}

int trans_broadcast_port(Am_Object port, int flag, ofstream& ofile)
{
  Am_Object rule = Port_Parent_Rule(port);
  Am_String name       = (Am_String)port.Get(PORT_NAME);
  Am_String inner_name = (Am_String)port.Get(PORT_INNER_NAME);
  int id   = port.Get(PORT_ID);
  int next_id = port.Get(PORT_NEXT_ID);
  int open = port.Get(PORT_OPEN);
  int type = port.Get(PORT_TYPE);
  int mode = KL_Output; // $B%k!<%k$J$N$G5UE>!%(B
  Am_String mode_str = Mode_Str(KL_Input);
  Am_Value_List values = Port_Values(port); values.Start();
  int err;
  switch (flag) {
  case HEAD_PHASE:
    if (!values.Empty() && type == KL_Single) {
      Am_Object val = (Am_Object)values.Get();
      KL_Translate_Call(val, err, HEAD_PHASE, ofile);
      ofile << " port " << name << " " << mode_str << Type_Str(type);
      return 1;
    } else {
      ofile << inner_name; break;
    }
  case GOAL_PHASE:
    ofile << inner_name; break;
  case REC_GOAL_PHASE:
    if (!values.Empty() && type == KL_Single) {
      Am_Object val = (Am_Object)values.Get();
      KL_Translate_Call(val, err, HEAD_PHASE, ofile);
      ofile << " port " << name << " " << mode_str << Type_Str(type);
      return 1;
    } else {
      ofile << inner_name;
      ofile << "\t % port " << name << " " << next_id << mode_str << Type_Str(type);
      return 1;
    }
  case LINK_PHASE:
    link_input_stream(port, ofile);
    return 1;
  case UNIFY_PHASE:
    if (values.Empty()) {
      ofile << ", " << inner_name << " = " << inner_name
	    << "\t% link " << next_id << " " << id << endl;
    }
    return 1;
  case TERM_HEAD_PHASE:
    ofile << inner_name; break;
  case TERM_UNIFY_PHASE:
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  ofile << "\t % port " << name << " " << id << mode_str << Type_Str(type);
  return 1;
}

void Initialize_Translator(void)
{
  initialize_id();
  Atom_Proto           .Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_atom);
  Structure_Proto      .Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_structure);
  In_Single_Proto      .Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_port);
  In_Stream_Proto      .Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_port);
  Out_Single_Proto     .Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_port);
  Out_Stream_Proto     .Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_port);
  Pat_In_Single_Port   .Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_normal_port);
  Pat_In_Stream_Port   .Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_normal_port);
  Pat_Out_Single_Port  .Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_normal_port);
  Pat_Out_Stream_Port  .Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_normal_port);
  Map_In_Port          .Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_map_in_port);
  Map_Out_Port         .Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_map_out_port);
  Recursive_Port       .Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_port); 
  Broadcast_Single_Port.Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_broadcast_port);
  Broadcast_Stream_Port.Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_broadcast_port);
  Merge_Port           .Set(KL_TRANS_METHOD, (Am_Object_Proc*)trans_merge_port);

  Pattern_Proto.Set(KL_TRANS_METHOD, (Am_Object_Proc*)translate_freeform_pattern);
  Sequence_Pattern_Proto.Set(KL_TRANS_METHOD,(Am_Object_Proc*)translate_sequence_pattern);
}       
