// Copyright (C) 1996  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <fstream.h>
#include <strstream.h>

#include <amulet/amulet.h>
#include <amulet/formula_advanced.h>
#include <amulet/inter_advanced.h>
#include <amulet/widgets_advanced.h>
#include <amulet/debugger.h>

#include "my_amulet.h"
#include "kl_widgets.h"
#include "kl_pattern.h"
#include "translator.h"

// ---- Tags ----
#define Beg_Module 'M'
#define End_Module 'm'

#define Beg_Procs  'P'
#define End_Procs  'p'
#define Beg_Proc   'C'
#define End_Proc   'c'

#define Beg_Rules  'R'
#define End_Rules  'r'
#define Beg_Rule   'L'
#define End_Rule   'l'
#define Beg_Patt   'E'
#define End_Patt   'e'
#define Beg_Goals  'O'
#define End_Goals  'o'
#define Beg_Goal   'N'
#define End_Goal   'n'
#define Beg_Holes  'H'
#define End_Holes  'h'
#define Beg_Hole   'I'
#define End_Hole   'i'
#define Beg_Gurds  'G'
#define End_Gurds  'g'
#define Beg_Gurd   'F'
#define End_Gurd   'f'
#define Beg_Bndrs  'B'
#define End_Bndrs  'b'
#define Beg_Bndr   'D'
#define End_Bndr   'd'


#define Beg_Ports  'Q'
#define End_Ports  'q'
#define Beg_Port   'T'
#define End_Port   't'
#define Beg_Vals   'V'
#define End_Vals   'v'
#define Beg_Stru   'S'
#define End_Stru   's'
#define Beg_Atom   'A'
#define End_Atom   'a'


// ---- Global ----
Am_Object Process_Table[1024];
Am_Object Inner_Port_Table[1024];

// ---- Utility ----
int save_position(Am_Object obj, ofstream& ofile)
{
  ofile << " " << (int)obj.Get(Am_LEFT) << " " << (int)obj.Get(Am_TOP) << " ";
  return 1;
}

int load_position(Am_Object obj, ifstream& ifile)
{
  int left, top;
  ifile >> left >> top;
  obj.Set(Am_LEFT, left)
     .Set(Am_TOP, top);
  return 1;
}

int save_position_with_icon_info(Am_Object obj, ofstream& ofile)
{
  save_position(obj, ofile);
  ofile << (bool)obj.Get(IS_ICON) << " "
	<< (int)obj.Get(ORIG_LEFT) << " " << (int)obj.Get(ORIG_TOP) << " "
	<< (int)obj.Get(ICON_LEFT) << " " << (int)obj.Get(ICON_TOP) << " ";
  return 1;
}

int load_position_with_icon_info(Am_Object obj, ifstream& ifile)
{
  load_position(obj, ifile);
  int is_icon, orig_left, orig_top, icon_left, icon_top;
  ifile >> is_icon >> orig_left >> orig_top >> icon_left >> icon_top;
  obj.Set(IS_ICON, is_icon)
    .Set(ORIG_LEFT, orig_left)
    .Set(ORIG_TOP, orig_top)
    .Set(ICON_LEFT, icon_left)
    .Set(ICON_TOP, icon_top);
  return 1;
}

// -----------------------------------
int save_module(Am_Object Window, ofstream& ofile);
int load_module(Am_Object window, ifstream& ifile);
int   save_proc_part(Am_Object proc_agg, ofstream& ofile);
int   load_proc_part(Am_Object proc_agg, ifstream& ifile);
int     save_process(Am_Object process, ofstream& ofile);
int     load_process(Am_Object proc_agg, ifstream& ifile);
int   save_rules_part(Am_Object rule_agg, ofstream& ofile);
int   load_rules_part(Am_Object rule_agg, ifstream& ifile);
int     save_rule(Am_Object rule, ofstream& ofile);
int     load_rule(Am_Object rules_agg, ifstream& ifile);
int     load_really_rule(Am_Object rules_agg, ifstream& ifile);
int       save_goals_part(Am_Object goals_agg, ofstream& ofile);
int       load_goals_part(Am_Object rule, ifstream& ifile);
int         save_goal(Am_Object goal, ofstream& ofile);
int         load_goal(Am_Object rule, ifstream& ifile);
int         load_really_goal(Am_Object rule, ifstream& ifile);
int         save_binder_part(Am_Object rule, ofstream& ofile);
int       load_binder_part(Am_Object rule, ifstream& ifile);
int       save_binder(Am_Object rule, Am_Object binder, ofstream& ofile);
int       load_binder(Am_Object rule, ifstream& ifile);
int       save_guard_part(Am_Object rule, ofstream& ofile);
int       load_guard_part(Am_Object rule, ifstream& ifile);
int       save_guard(Am_Object guard, ofstream& ofile);
int       load_guard(Am_Object rule, ifstream& ifile);
int     save_pattern(Am_Object pattern, ofstream& ofile);
int     load_pattern(Am_Object patt_agg, ifstream& ifile);
int       save_holes_part(Am_Object holes_agg, ofstream& ofile);
int       load_holes_part(Am_Object patt, ifstream& ifile);
int       save_hole(Am_Object hole, ofstream& ofile);
int       load_hole(Am_Object patt, ifstream& ifile);

int save_port_part(Am_Object port_agg, ofstream& ofile);
int load_port_part(Am_Object proc_agg, ifstream& ifile);
int load_exist_port_part(Am_Object process, ifstream& ifile);
int save_port(Am_Object port, ofstream& ofile);
int load_port(Am_Object port, ifstream& ifile);
int load_exist_port(Am_Object port, ifstream& ifile);

int save_value_part(Am_Object value_agg, ofstream& ofile);
int load_value_part(Am_Object port, ifstream& ifile);
int save_value(Am_Object value, ofstream& ofile);
int load_value(Am_Object port, ifstream& ifile);

int save_atom(Am_Object value, ofstream& ofile);
int load_atom(Am_Object port, ifstream& ifile);
int save_structure(Am_Object value, ofstream& ofile);
int load_structure(Am_Object port, ifstream& ifile);

// -------- Module --------
// Beg_Module Name
// ... processes ...
// ... rules ...
// End_Module
int save_module(Am_Object window, ofstream& ofile)
{
  ofile << Beg_Module << " " << (Am_String)window.Get(MODULE_NAME) << endl;
  save_proc_part(Module_Process_Part(window), ofile);
  save_rules_part(Module_Rules_Part(window), ofile);
  ofile << End_Module << endl;
  return 1;
}

int load_module(Am_Object window, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Module) return 0;
  char name[1000];
  ifile >> name; if (!Module_Rename(window, (char*)name)) return 0;
  load_proc_part(Module_Process_Part(window), ifile);
  load_rules_part(Module_Rules_Part(window), ifile);
  ifile >> tag; if (tag != End_Module) return 0;
}

// -------- Processes --------
// Beg_Procs
// ... process ...
// End_Procs
int save_proc_part(Am_Object proc_agg, ofstream& ofile)
{
  ofile << Beg_Procs << endl;
  Am_Value_List procs = (Am_Value_List)proc_agg.Get(Am_GRAPHICAL_PARTS);
  for (procs.Start(); !procs.Last(); procs.Next()) {
    Am_Object proc = (Am_Object)procs.Get();
    save_process(proc, ofile);
  }
  ofile << End_Procs << endl;
}

int load_proc_part(Am_Object proc_agg, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Procs) return 0;
  while (load_process(proc_agg, ifile));
  ifile >> tag; if (tag != End_Procs) return 0;
}

// -------- Process --------
// Beg_Proc Id Name Type Left Top Icon OLeft OTop ILeft ITop
// ... ports ... 
// End_Proc
int save_process(Am_Object process, ofstream& ofile)
{
  ofile << Beg_Proc << " ";
  int Id         = process.Get(PROC_ID);
  Am_String Name = (Am_String)process.Get(PROC_NAME);
  int Type       = process.Get(PROC_TYPE);
  ofile << Id << " " << Name << " " << Type;
  save_position_with_icon_info(process, ofile);
  ofile << endl;
  save_port_part(Proc_Ports_Part(process), ofile);
  ofile << End_Proc << endl;
}

int load_process(Am_Object proc_agg, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Proc) {ifile.putback(tag); return 0;}
  Am_Object new_proc = Process_Proto.Create();
  int Id, Type;
  char Name[1000];
  ifile >> Id >> Name >> Type;
  new_proc.Set(PROC_NAME, Name)
          .Set(PROC_TYPE, Type);
  load_position_with_icon_info(new_proc, ifile);
  load_port_part(new_proc, ifile);
  Process_Table[Id] = new_proc; // global table $B$KEPO?!%(B
  Add_Process(proc_agg, new_proc);
  ifile >> tag; if (tag != End_Proc) return 0;
  return 1;
}

// -------- Rules Part --------
// Beg_Rules
// ... rules ...
// End_Rules
int save_rules_part(Am_Object rules_agg, ofstream& ofile)
{
  ofile << Beg_Rules << endl;
  Am_Value_List rules = (Am_Value_List)rules_agg.Get(Am_GRAPHICAL_PARTS);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = (Am_Object)rules.Get();
    if (KL_Type(rule) == KL_Rule)    save_rule(rule, ofile);
    if (KL_Type(rule) == KL_Pattern) save_pattern(rule, ofile);
  }
  ofile << End_Rules << endl;
}

int load_rules_part(Am_Object rules_agg, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Rules) return 0;
  while (load_rule(rules_agg, ifile));
  ifile >> tag; if (tag != End_Rules) return 0;
}

// -------- Rule --------
// Beg_Rule Proc Type Left Top Icon OLeft OTop ILeft ITop
// ... ports ...
// ... goals ...
// ... binders...
// End_Rule
int save_rule(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Rule << " ";
  int Proc = (int)((Am_Object)rule.Get(PARENT_PROC)).Get(PROC_ID);
  int Type = rule.Get(RULE_TYPE);
  ofile << Proc << " " << Type;
  save_position_with_icon_info(rule, ofile);
  ofile << endl;
  save_port_part(Rule_Ports_Part(rule), ofile);
  save_goals_part(Rule_Goals_Part(rule), ofile);
  save_binder_part(rule, ofile);
  save_guard_part(rule, ofile);
  ofile << End_Rule << endl;
}

int load_rule(Am_Object rules_agg, ifstream& ifile)
{
  char tag;
  ifile >> tag;
  if (tag == Beg_Rule) return load_really_rule(rules_agg, ifile);
  if (tag == Beg_Patt) return load_pattern(rules_agg, ifile);
  ifile.putback(tag);
  return 0;
}

int load_really_rule(Am_Object rules_agg, ifstream& ifile)
{
  char tag;
//  ifile >> tag; if (tag != Beg_Rule) {ifile.putback(tag); return 0;}
  int Proc, Type;
  ifile >> Proc >> Type;
  Am_Object rule;
  if   (Type == KL_Transition_Rule) rule = Continuous_Rule_Proto;
  else if (Type == KL_Network_Rule) rule = Vanishing_Rule_Proto;
  else return 0;
  rule = Rule_Create(Process_Table[Proc], rule);

  load_position_with_icon_info(rule, ifile);
  load_exist_port_part(rule, ifile);
  load_goals_part(rule, ifile);
  load_binder_part(rule, ifile);
  load_guard_part(rule, ifile);
  Add_Rule(rules_agg, rule);
  ifile >> tag; if (tag != End_Rule) return 0;
  return 1;
}

// -------- Goals Part --------
// Beg_Goals
// ... goals ...
// End_Goals
int save_goals_part(Am_Object goals_agg, ofstream& ofile)
{
  ofile << Beg_Goals << endl;
  Am_Value_List goals = (Am_Value_List)goals_agg.Get(Am_GRAPHICAL_PARTS);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = (Am_Object)goals.Get();
    if (KL_Type(goal) == KL_Process)    save_goal(goal, ofile);
    if (KL_Type(goal) == KL_Pattern) save_pattern(goal, ofile);
  }
  ofile << End_Goals << endl;
}

int load_goals_part(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Goals) return 0;
  while (load_goal(rule, ifile));
  ifile >> tag; if (tag != End_Goals) return 0;
}

// -------- Goal --------
// Beg_Goal Id Parent Hole position_with_icon
// ... ports ...
// End_Goal
int save_goal(Am_Object goal, ofstream& ofile)
{
  ofile << Beg_Goal << " ";
  int Id = goal.Get(PROC_ID);
  int Parent = 0; // Pattern$B$N$H$-$NBP1~!%(B
  Am_Object parent = (Am_Object)goal.Get(PARENT_LINK);
  if (parent != goal) { // $B%3%T!<85$N$H$-$O(B,$B#0(B
    Parent = Group_Index(parent);
  }
  int Hole = 0; // Pattern$B$N$H$-$NBP1~!%(B
  Am_Object hole = (Am_Object)goal.Get(HOLE_LINK);
  if (hole.Valid()) {
    Hole = Group_Index(hole);
  }
  ofile << Id << " " << Parent << " " << Hole;
  save_position_with_icon_info(goal, ofile); ofile << endl;
  save_port_part(Proc_Ports_Part(goal), ofile);
  ofile << End_Goal << endl;
}

int load_goal(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag;
  if (tag == Beg_Goal) return load_really_goal(rule, ifile);
  if (tag == Beg_Patt) return load_pattern(Rule_Goals_Part(rule), ifile);
  ifile.putback(tag);
  return 0;
}

int load_really_goal(Am_Object rule, ifstream& ifile)
{
  char tag;
  // tag check omitted by load_goal()
  int Id, Parent, Hole;
  ifile >> Id >> Parent >> Hole;
  Am_Object goal;
  if (Parent == 0) {
    goal = Goal_Create(Process_Table[Id]);
  } else {
    Am_Object parent = Object_List_Get(Rule_Goals(rule), Parent);
    Am_Function_Call(Copy_Method, parent, PARENT_COPY, goal, (parent));
  }
  if (Hole != 0) {
    Am_Object hole = Object_List_Get(Rule_Holes(rule), Hole);
    goal.Set(HOLE_LINK, hole); // kitanai
    hole.Set(HOLE_LINK, goal);
  }
  load_position_with_icon_info(goal, ifile);
  Rule_Add_Goal(rule, goal);
  load_exist_port_part(goal, ifile);
  ifile >> tag; if (tag != End_Goal) return 0;
  return 1;
}

// -------- Binder Part --------
// Beg_Bndrs
// ... binders ...
// End_Bndrs
int save_binder_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Bndrs << endl;
  Am_Value_List binders = Rule_Binder(rule);
  for (binders.Start(); !binders.Last(); binders.Next()) {
    Am_Object binder = (Am_Object)binders.Get();
    save_binder(rule, binder, ofile);
  }
  ofile << End_Bndrs << endl;
}

int load_binder_part(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Bndrs) return 0;
  while (load_binder(rule, ifile));
  ifile >> tag; if (tag != End_Bndrs) return 0;
}

// -------- Binder --------
// Beg_Bndr Visible
// Pos1.1 Pos1.2 ... -1
// Pos2.1 Pos2.2 ... -1
// End_Bndr
int save_binder(Am_Object rule, Am_Object binder, ofstream& ofile)
{
  ofile << Beg_Bndr << (bool)binder.Get(Am_VISIBLE) << endl;
  Am_Object pair[2];
  pair[0] = binder.Get(BL_SRC);
  pair[1] = binder.Get(BL_DEST);
  int i;
  for (i = 0; i < 2; ++i) {
    Am_Value_List positions = Position_Get(pair[i], rule);
    for (positions.Start(); !positions.Last(); positions.Next()) {
      int pos = positions.Get();
      ofile << pos << " ";
    }
    ofile << -1 << endl;
  }
  ofile << End_Bndr << endl;
}

int load_binder(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Bndr) {ifile.putback(tag); return 0;}
  int visible;
  ifile >> visible;
  int pos;
  Am_Value_List positions[2] = {Am_Value_List(), Am_Value_List()};
  int i;
  for (i = 0; i < 2; ++i) {
    while (ifile >> pos, pos >= 0) {
      positions[i].Add(pos);
    }
  }
  Am_Object src_port = Position_To_Obj(rule, positions[0]);
  Am_Object des_port = Position_To_Obj(rule, positions[1]);
  Am_Object new_line =Single_Binding_Line_Create(src_port, des_port);
//  new_line.Set(Am_VISIBLE, visible);
  ifile >> tag; if (tag != End_Bndr) return 0;
}
  
// -------- Guard Part --------
// Beg_Gurds
// ... gurads ...
// End_Gurds
int save_guard_part(Am_Object rule, ofstream& ofile)
{
  ofile << Beg_Gurds << endl;
  Am_Value_List guards = Rule_Guard(rule);
  for (guards.Start(); !guards.Last(); guards.Next()) {
    Am_Object guard = (Am_Object)guards.Get();
    save_guard(guard, ofile);
  }
  ofile << End_Gurds << endl;
}

int load_guard_part(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Gurds) return 0;
  while (load_guard(rule, ifile));
  ifile >> tag; if (tag != End_Gurds) return 0;
}

// -------- Guard --------
// Beg_Gurd position_with_icon_info Contents
// End_Gurd
int save_guard(Am_Object guard, ofstream& ofile)
{
  ofile << Beg_Gurd << " ";
  save_position_with_icon_info(guard, ofile);
  ofile << (Am_String)guard.Get_Part(CONTENTS_PART).Get(Am_TEXT) << endl;
  ofile << End_Gurd << endl;
}

int load_guard(Am_Object rule, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Gurd) {ifile.putback(tag); return 0;}
  Am_Object new_guard = Guard_Proto.Create();
  load_position_with_icon_info(new_guard, ifile);
  char Contents[1000];
  ifile >> ws;
  ifile.getline(Contents, 1000);
  new_guard
    .Get_Part(CONTENTS_PART)
    .Set(Am_TEXT, Contents)
    .Get_Owner();
  Rule_Add_Guard(rule, new_guard);
  ifile >> tag; if (tag != End_Gurd) return 0;
  return 1;
}

// -------- Pattern --------
// Beg_Patt Type Name Dir Parent Hole position_with_icon_info
// ... ports ...
// ... holes ...
// ... goals ...
// ... binders ...
// End_Patt
int save_pattern(Am_Object pattern, ofstream& ofile)
{
  ofile << Beg_Patt << " ";
  Am_String Name = (Am_String)pattern.Get(PROC_NAME);
  int Type = pattern.Get(PATTERN_TYPE);
  int Dir  = pattern.Get(SEQ_DIR);
  int Parent = 0;
  Am_Object parent = (Am_Object)pattern.Get(PARENT_LINK);
  if (parent != pattern) { // $B%3%T!<85$N$H$-$O(B,$B#0(B
    Parent = Group_Index(parent);
  }
  int Hole = 0; // Pattern$B$N$H$-$NBP1~!%(B
  Am_Object hole = (Am_Object)pattern.Get(HOLE_LINK);
  if (hole.Valid()) {
    Hole = Group_Index(hole);
  }
  ofile << Type << " " << Name << " " << Dir << " " << Parent << " " << Hole;
  save_position_with_icon_info(pattern, ofile); ofile << endl;
//  save_port_part(Rule_Ports_Part(pattern), ofile);
  if (Parent == 0) {
    save_port_part(Rule_Ports_Part(pattern), ofile);
    save_holes_part(Rule_Holes_Part(pattern), ofile);
    save_goals_part(Rule_Goals_Part(pattern), ofile);
    save_binder_part(pattern, ofile);
  }
  ofile << End_Patt << endl;
}

int load_pattern(Am_Object patt_agg, ifstream& ifile)
{
  char tag;
//  ifile >> tag; $B%?%0%A%'%C%/$O$$$i$J$$!%(B
  int Type, Dir, Parent, Hole;
  char Name[1000];
  ifile >> Type >> Name >> Dir >> Parent >> Hole;
  Am_Object new_patt;
  if (Parent == 0) {
    if (Type == KL_Free_Pattern) new_patt = Pattern_Proto.Create();
    if (Type == KL_Seq_Pattern) new_patt = Sequence_Pattern_Proto.Create();
    new_patt.Set(PROC_NAME, Name)
            .Set(SEQ_DIR, Dir);
  } else {
    Am_Object parent = Group_Get_By_Index(patt_agg, Parent);
    Am_Function_Call(Copy_Method, parent, PARENT_COPY, new_patt, (parent));
  }
  if (Hole != 0) {
    Am_Object hole = Group_Get_By_Index(patt_agg.Get_Sibling(HOLES_PART), Hole);
    new_patt.Set(HOLE_LINK, hole); // kitanai
    hole.Set(HOLE_LINK, new_patt);
  }
  load_position_with_icon_info(new_patt, ifile);
  Add_Pattern(patt_agg, new_patt);
  if (Parent == 0) {
    load_port_part(new_patt, ifile);
    load_holes_part(new_patt, ifile);
    load_goals_part(new_patt, ifile);
    load_binder_part(new_patt, ifile);
  } else {
//    load_exist_port_part(new_patt, ifile);
  }
  ifile >> tag; if (tag != End_Patt) return 0;
  return 1;
}

// -------- Hole Part --------
// Beg_Holes
// ... holes ...
// End_Holes
int save_holes_part(Am_Object holes_agg, ofstream& ofile)
{
  ofile << Beg_Holes << endl;
  Am_Value_List holes = (Am_Value_List)holes_agg.Get(Am_GRAPHICAL_PARTS);
  for (holes.Start(); !holes.Last(); holes.Next()) {
    Am_Object hole = (Am_Object)holes.Get();
    save_hole(hole, ofile);
  }
  ofile << End_Holes << endl;
}

int load_holes_part(Am_Object patt, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Holes) return 0;
  while (load_hole(patt, ifile));
  ifile >> tag; if (tag != End_Holes) return 0;
}

// -------- Hole --------
// Beg_Hole Name Parent Visible position_with_icon_info
// ... ports ...
// End_Hole
// $B?F$OC/$+!)(B
int save_hole(Am_Object hole, ofstream& ofile)
{
  ofile << Beg_Hole << " ";
  Am_String Name = (Am_String)hole.Get(PROC_NAME);
  int Parent = 0;
  Am_Object parent = (Am_Object)hole.Get(PARENT_LINK);
  if (parent != hole) { // $B%3%T!<85$N$H$-$O(B,$B#0(B
    Parent = Group_Index(parent);
  }
  int Visible = (bool)hole.Get(Am_VISIBLE);
  ofile << Name << " " << Parent << " " << Visible;
  save_position_with_icon_info(hole, ofile); ofile << endl;
  save_port_part(Proc_Ports_Part(hole), ofile);
  ofile << End_Hole << endl;
}

int load_hole(Am_Object patt, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Hole) {ifile.putback(tag); return 0;}
  char Name[1000];
  int Parent, Visible;
  ifile >> Name >> Parent >> Visible;
  Am_Object hole;
  if (Parent == 0) {
    hole = Hole_Proto.Create();
    hole.Set(PROC_NAME, Name);
//        .Set(Am_VISIBLE, Visible);
  } else {
    Am_Object parent = Object_List_Get(Rule_Holes(patt), Parent);
    Am_Function_Call(Copy_Method, parent, PARENT_COPY, hole, (parent));
  }
  load_position_with_icon_info(hole, ifile);
  Pattern_Add_Hole(patt, hole);
  if (Parent == 0) load_port_part(hole, ifile);
  else load_exist_port_part(hole, ifile);
  ifile >> tag; if (tag != End_Hole) return 0;
  return 1;
}

// -------- Port Part --------
// Beg_Ports
// ... ports ...
// End_Ports
int save_port_part(Am_Object port_agg, ofstream& ofile)
{
  ofile << Beg_Ports << endl;
  Am_Value_List ports = (Am_Value_List)port_agg.Get(Am_GRAPHICAL_PARTS);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    save_port(port, ofile);
  }
  ofile << End_Ports << endl;
}

int load_port_part(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Ports) return 0;
  while (load_port(process, ifile));
  ifile >> tag; if (tag != End_Ports) return 0;
}

int load_exist_port_part(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Ports) return 0;
  Am_Value_List ports = Proc_Ports(process);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    if (!load_exist_port(port, ifile)) return 0;
  }
  ifile >> tag; if (tag != End_Ports) return 0;
}

// -------- Port --------
// Beg_Port Id Name Type Age Mode Open Dir Spec Left Top Icon OLeft OTop ILeft ITop
// ... Values ...
// End_Port
int save_port(Am_Object port, ofstream& ofile)
{
  ofile << Beg_Port  << " ";
  int Id   = port.Get(PORT_ID);
  Am_String Name = (Am_String)port.Get(PORT_NAME);
  int Type = port.Get(PORT_TYPE);
  int Age  = port.Get(PORT_AGE);
  int Mode = port.Get(PORT_MODE);
  int Open = port.Get(PORT_OPEN);
  int Dir  = port.Get(PORT_DIR);
  int Spec = port.Get(PORT_SPECIAL);
  ofile << Id <<" "<< Name <<" "<< Type <<" "<< Age <<" "<< Mode <<" "<< Open <<" "<< Dir <<" "<< Spec;
  save_position_with_icon_info(port, ofile);
  ofile << endl;
  save_value_part(Port_Values_Part(port), ofile);
  ofile << End_Port << endl;
}

int load_port(Am_Object process, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int Id, Type, Age, Mode, Open, Dir, Spec;
  char Name[1000];
  ifile >> Id >> Name >> Type >> Age >> Mode >> Open >> Dir >> Spec;
  Am_Object new_port;
  if (Spec == KL_Normal) {
    if (Mode == KL_Input  && Type == KL_Single) new_port = In_Single_Proto.Create();
    if (Mode == KL_Output && Type == KL_Single) new_port = Out_Single_Proto.Create();
    if (Mode == KL_Input  && Type == KL_Stream) new_port = In_Stream_Proto.Create();
    if (Mode == KL_Output && Type == KL_Stream) new_port = Out_Stream_Proto.Create();
  } else {
    if (Spec == KL_Pat_Single && Mode == KL_Input) new_port = Pat_In_Single_Port.Create();
    if (Spec == KL_Pat_Single && Mode == KL_Output) new_port = Pat_Out_Single_Port.Create();    
    if (Spec == KL_Pat_Stream && Mode == KL_Input) new_port = Pat_In_Stream_Port.Create();
    if (Spec == KL_Pat_Stream && Mode == KL_Output) new_port = Pat_Out_Stream_Port.Create();    
    if (Spec == KL_Map && Mode == KL_Input) new_port = Map_In_Port.Create();
    if (Spec == KL_Map && Mode == KL_Output) new_port = Map_Out_Port.Create();
    if (Spec == KL_Broadcast && Type == KL_Single) new_port = Broadcast_Single_Port.Create();
    if (Spec == KL_Broadcast && Type == KL_Stream) new_port = Broadcast_Stream_Port.Create();
    if (Spec == KL_Merge) new_port = Merge_Port.Create();
  }
  new_port.Set(PORT_NAME, Name)
    .Set(PORT_TYPE, Type)
    .Set(PORT_AGE,  Age)
    .Set(PORT_MODE, Mode)
    .Set(PORT_OPEN, Open)
    .Set(PORT_DIR,  Dir)
    ;
  load_position_with_icon_info(new_port, ifile);
  Am_Call(Add_Port_Method, process, ADD_PORT, (process, new_port));
  load_value_part(new_port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

int load_exist_port(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int Id, Type, Age, Mode, Open, Dir, Spec;
  char Name[1000];
  ifile >> Id >> Name >> Type >> Age >> Mode >> Open >> Dir >> Spec;
  port.Set(PORT_OPEN, Open)
      .Set(PORT_DIR,  Dir)
    ;
  load_position_with_icon_info(port, ifile);
  load_value_part(port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

int load_val_port(Am_Object port, ifstream& ifile)
{
  char tag;
//  ifile >> tag; if (tag != Beg_Port) {ifile.putback(tag); return 0;}
  int Id, Type, Age, Mode, Open, Dir, Spec;
  char Name[1000];
  ifile >> Id >> Name >> Type >> Age >> Mode >> Open >> Dir >> Spec;

  Am_Object rule = Port_Parent_Rule(port);
  if (!rule.Valid()) return 0;
  Am_Object old_port = 0;
  if (rule.Valid()) {
    Am_Value_List inner_ports = (Am_Value_List)rule.Get(INNER_PORTS);
    old_port = Object_List_Get_By_Key(inner_ports, PORT_NAME, Name);
  }
  Am_Object new_port;
  if (old_port.Valid()) {
    new_port = Port_Create(old_port);
    load_position_with_icon_info(new_port, ifile);
    Port_Add(port, new_port);
  } else if (Spec == KL_Normal) {
    if (Mode == KL_Input  && Type == KL_Single) new_port = In_Single_Proto.Create();
    if (Mode == KL_Output && Type == KL_Single) new_port = Out_Single_Proto.Create();
    if (Mode == KL_Input  && Type == KL_Stream) new_port = In_Stream_Proto.Create();
    if (Mode == KL_Output && Type == KL_Stream) new_port = Out_Stream_Proto.Create();
    new_port.Set(PORT_NAME, Name)
      .Set(PORT_TYPE, Type)
      .Set(PORT_AGE,  Age)
      .Set(PORT_MODE, Mode)
      .Set(PORT_OPEN, Open)
      .Set(PORT_DIR,  Dir)
      ;
    load_position_with_icon_info(new_port, ifile);
    new_port = Rule_Add_Inner_Port(rule, port, new_port);
  } else return 0;
  load_value_part(new_port, ifile);
  ifile >> tag; if (tag != End_Port) return 0;
  return 1;
}

// -------- Values Part --------
// Beg_Vals
// ... values ...
// End_Vals
int save_value_part(Am_Object value_agg, ofstream& ofile)
{
  ofile << Beg_Vals << endl;
  Am_Value_List vals = (Am_Value_List)value_agg.Get(Am_GRAPHICAL_PARTS);
  for (vals.Start(); !vals.Last(); vals.Next()) {
    Am_Object val = (Am_Object)vals.Get();
    save_value(val, ofile);
  }
  ofile << End_Vals << endl;
}

int load_value_part(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag; if (tag != Beg_Vals) return 0;
  while (load_value(port, ifile));
  ifile >> tag; if (tag != End_Vals) return 0;
}

// -------- Value --------
int save_value(Am_Object value, ofstream& ofile)
{
  int type = KL_Type(value);
  if (type == KL_Atom) return save_atom(value, ofile);
  if (type == KL_Structure) return save_structure(value, ofile);
  if (type == KL_Port) return save_port(value, ofile);
  return 0;
}

int load_value(Am_Object port, ifstream& ifile)
{
  char tag;
  ifile >> tag;
  if (tag == Beg_Atom) return load_atom(port, ifile);
  if (tag == Beg_Stru) return load_structure(port, ifile);
  if (tag == Beg_Port) return load_val_port(port, ifile);
  ifile.putback(tag);
  return 0;
}

// -------- Atom --------
// Beg_Atom Contents
// End_Atom
int save_atom(Am_Object value, ofstream& ofile)
{
  ofile << Beg_Atom << " ";
  ofile << (Am_String)value.Get_Part(CONTENTS_PART).Get(Am_TEXT) << endl;
  ofile << End_Atom << endl;
}

int load_atom(Am_Object port, ifstream& ifile)
{
  char tag;
  char Contents[1000];
  ifile >> ws;
  ifile.getline(Contents, 1000);
  Am_Object new_atom = Atom_Proto.Create()
    .Get_Part(CONTENTS_PART)
    .Set(Am_TEXT, Contents)
    .Get_Owner();
  Port_Add(port, new_atom);
  ifile >> tag; if (tag != End_Atom) return 0;
  return 1;
}

// -------- Structure --------
// Beg_Stru Name
// ... values ...
// End_Stru
int save_structure(Am_Object value, ofstream& ofile)
{
  ofile << Beg_Stru << " ";
  ofile << (Am_String)value.Get(PORT_NAME) << endl;
  save_value_part(Port_Values_Part(value), ofile);
  ofile << End_Stru << endl;
}

int load_structure(Am_Object port, ifstream& ifile)
{
  char tag;
  char Name[1000];
  ifile >> Name;
  Am_Object new_str = Structure_Proto.Create()
    .Get_Part(CONTENTS_PART).Get_Part(NAME_PART)
    .Set(Am_TEXT, Name)
    .Get_Owner().Get_Owner();
  Port_Add(port, new_str);
  load_value_part(new_str, ifile);
  ifile >> tag; if (tag != End_Stru) return 0;
  return 1;
}

  

