// Copyright (C) 1996  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet/am_io.h>

#if defined(SHORT_NAMES)
#include <amulet/widgts_a.h>
#include <amulet/std_slot.h>
#include <amulet/val_lst.h>
#include <amulet/object_a.h>  // for Am_Slot_Advanced
#include <amulet/opal_a.h>  // for Am_DRAWONABLE, Am_Window_Coordinate
#else
#include <amulet/widgets_advanced.h>
#include <amulet/standard_slots.h>
#include <amulet/value_list.h>
#include <amulet/object_advanced.h>  // for Am_Slot_Advanced
#include <amulet/opal_advanced.h>  // for Am_DRAWONABLE, Am_Window_Coordinate
#endif

#include <amulet/gem.h>
#include <amulet/inter.h>
#include <amulet/inter_advanced.h>
#include <amulet/opal.h>
#include <amulet/widgets.h>

// #include <amulet/amulet.h>
#include <math.h>
#include "my_amulet.h"

/* ---------------- Useful functions ------------------ */

void Value_List_Slot_Add(Am_Object parent, Am_Slot_Key list_slot, Am_Object obj)
{
  parent.Make_Unique(list_slot);
  Am_Value_List obj_list = (Am_Value_List)parent.Get(list_slot);
  if (obj_list.Empty()) {
    parent.Set(list_slot, obj_list.Add(obj));
  } else {
    obj_list.Add(obj, Am_TAIL, false);
    parent.Note_Changed(list_slot);
  }
}

// Am_Value_List Index operations
int Object_List_Index(Am_Value_List list, Am_Object value)
{
  int i = 1;
  for (list.Start(); !list.Last(); list.Next(), ++i) {
    if ((Am_Object)list.Get() == value) return i;
  }
  return 0;
}

Am_Object Object_List_Get(Am_Value_List list, int i)
{
  int j = 1;
  for (list.Start(); !list.Last(); list.Next(), ++j) {
    if (i == j) return (Am_Object)list.Get();
  }
  return 0;
}

int Object_List_Max(Am_Value_List list, Am_Slot_Key key)
{
  int max = 0;
  for (list.Start(); !list.Last(); list.Next()) {
    Am_Object item = (Am_Object)list.Get();
    int value = item.Get(key);
    if (value > max) max = value;
  }
  return max;
}

int Group_Index(Am_Object obj)
{
  Am_Object owner = obj.Get_Owner();
  if (owner.Valid()) {
    Am_Value_List parts = (Am_Value_List)owner.Get(Am_GRAPHICAL_PARTS);
    return Object_List_Index(parts, obj);
  }
  return 0;
}

Am_Object Group_Get_By_Index(Am_Object group, int idx)
{
  if (!group.Valid()) return 0;
  Am_Value_List parts = (Am_Value_List)group.Get(Am_GRAPHICAL_PARTS);
  return Object_List_Get(parts, idx);
}

Am_Object Object_List_Get_By_Key(Am_Value_List list, Am_Slot_Key key, Am_String s)
{
  for (list.Start(); !list.Last(); list.Next()) {
    Am_Object item = (Am_Object)list.Get();
    if ((Am_String)item.Get(key) == s) {
      return item;
    }
  }
  return 0;
}  

int String_List_Index(Am_Value_List list, Am_String value)
{
  int i = 1;
  for (list.Start(); !list.Last(); list.Next(), ++i) {
    if (strcmp((Am_String)list.Get(), value) == 0) return i;
  }
  return 0;
}


Am_Define_Object_Formula(loopback_form) {
  return self;
}

Am_Object My_Add_Part(Am_Object parent, Am_Object part)
{
  parent.Add_Part(part).Remove_Part(part); // $B:G8e$N0l8DLdBj$N$$$$2C8:$JBP1~!%(B
  parent.Add_Part(part);
  return parent;
}

void Set_Slot_Inherit_Rule(Am_Object obj, Am_Slot_Key key, Am_Inherit_Rule rule)
{
  Am_Object_Advanced obj_adv = (Am_Object_Advanced&)obj;
  Am_Slot_Advanced *slot = obj_adv.Get_Slot(key);
  slot->Set_Inherit_Rule(rule);
}

void My_Set_Single_Constraint_Mode(Am_Object obj, Am_Slot_Key key, bool mode)
{
  Am_Object_Advanced obj_adv = (Am_Object_Advanced&)obj;
  obj_adv.Get_Slot(key)->Set_Single_Constraint_Mode(mode);
}

void Am_Center_Of_Object(Am_Object obj, int& cx, int& cy)
{
  int x = obj.Get(Am_LEFT);
  int y = obj.Get(Am_TOP);
  int width = obj.Get(Am_WIDTH);
  int height = obj.Get(Am_HEIGHT);

  cx = x + width  / 2;
  cy = y + height / 2;
}
  
int Am_Distance_Of_Objects(Am_Object obj1, Am_Object obj2)
{
  int x1 = obj1.Get(Am_LEFT);
  int y1 = obj1.Get(Am_TOP);
  int x2 = obj2.Get(Am_LEFT);
  int y2 = obj2.Get(Am_TOP);
  int dx = x2 - x1;
  int dy = y2 - y1;
  return (int)floor(sqrt(dx * dx + dy * dy));
}  


/* ---------------- Opal ---------------- */

Am_Define_Formula (int, my_width_of_parts_procedure) {
  int max_x = 0, comp_right;
  Am_Value_List components;
  Am_Object comp;
  if ((bool)self.GV(Am_VISIBLE) == false) return 0;
  components = self.GV(Am_GRAPHICAL_PARTS);
  for (components.Start(); !components.Last(); components.Next()) {
    comp = components.Get ();
    if ((bool)comp.GV(Am_VISIBLE) == true) {
      // compute how much of the component extends right of the origin
      comp_right = ((int)(comp.GV(Am_LEFT)) + (int)(comp.GV(Am_WIDTH)));
      max_x = imax (max_x, comp_right);
    }
  }
  return max_x; // always >=0 since it's initialized to 0
}

Am_Define_Formula (int, my_height_of_parts_procedure) {
  int max_y = 0, comp_bottom;
  Am_Value_List components;
  Am_Object comp;
  if ((bool)self.GV(Am_VISIBLE) == false) return 0;
  components = self.GV(Am_GRAPHICAL_PARTS);
  for (components.Start(); !components.Last(); components.Next()) {
    comp = components.Get ();
    if ((bool)comp.GV(Am_VISIBLE) == true) {
      // compute how much of the component extends below the origin
      comp_bottom = ((int)(comp.GV(Am_TOP)) + (int)(comp.GV(Am_HEIGHT)));
      max_y = imax (max_y, comp_bottom);
    }
  }
  return max_y; // always >=0 since it's initialized to 0
}


Am_Constraint* Am_Horizontal_Rev_Layout;

void get_fixed_sizes (Am_Object self, Am_Value_List& components,
		      Am_Constraint_Context& cc,
		      int& fixed_width, int& fixed_height);

void get_max_rank_and_size (Am_Object self, Am_Constraint_Context& cc,
			    int& max_rank, int& max_size);

void find_line_size_and_rank (Am_Value_List& components, int indent,
			      int fixed_primary, Am_Slot_Key primary_slot,
			      int primary_spacing, Am_Slot_Key secondary_slot,
			      int max_rank, int max_size,
			      int& rank, int& max_secondary);

void my_find_line_size_and_rank (Am_Value_List& components, int indent,
			      int fixed_primary, Am_Slot_Key primary_slot,
			      int primary_spacing, Am_Slot_Key secondary_slot,
			      int max_rank, int max_size,
			      int& rank, int& max_secondary)
{
  Am_Object start_item;
  start_item = components.Get ();
  rank = 0;
  max_secondary = 0;
  int position = indent;
  while (!components.First ()) {
    Am_Object item;
    item = components.Get ();
    if ((bool)item.Get (Am_VISIBLE)) {
      int primary = item.Get (primary_slot);
      int secondary = item.Get (secondary_slot);
      if (rank &&
	  ((max_rank && (rank == max_rank)) ||
	   (max_size &&
	    ((position + (fixed_primary ? fixed_primary : primary)) >=
	     max_size))))
	break;
      ++rank;
      position += (fixed_primary ? fixed_primary : primary) + primary_spacing;
      if (secondary > max_secondary)
        max_secondary = secondary;
    }
    components.Prev ();
  }
  components.Start ();
  components.Member (start_item);
}

Am_Define_Formula (Am_Ptr, my_horizontal_layout_procedure)
{
  Am_Value_List components;
  components = self.GV (Am_GRAPHICAL_PARTS);
  int fixed_width, fixed_height;
  get_fixed_sizes (self, components, cc, fixed_width, fixed_height);
  int max_rank, max_size;
  get_max_rank_and_size (self, cc, max_rank, max_size);
  int x_offset = self.GV (Am_X_OFFSET);
  int y_offset = self.GV (Am_Y_OFFSET);
  int left = x_offset;
  int top = y_offset;
  int h_spacing = self.GV (Am_H_SPACING);
  int v_spacing = self.GV (Am_V_SPACING);
  int h_align = self.GV (Am_H_ALIGN);
  int v_align = self.GV (Am_V_ALIGN);
  int indent = self.GV (Am_INDENT);
  components.Start ();
  while (!components.Last ()) {
    int line_rank;
    int line_height;
    find_line_size_and_rank (components, left, fixed_width, Am_WIDTH,
			     h_spacing, Am_HEIGHT, max_rank, max_size,
			     line_rank, line_height);
    int rank = 0;
    while (rank < line_rank) {
      Am_Object item;
      item = components.Get ();
      if ((bool)item.GV (Am_VISIBLE)) {
        int width = item.GV (Am_WIDTH);
        int height = item.GV (Am_HEIGHT);
	if (fixed_width) {
	  switch (h_align) {
	  case Am_LEFT_ALIGN:
            item.Set (Am_LEFT, left);
	    break;
	  case Am_RIGHT_ALIGN:
            item.Set (Am_LEFT, left + fixed_width - width);
	    break;
	  default:
            item.Set (Am_LEFT, left + (fixed_width - width) / 2);
	    break;
	  }
	}
	else
	  item.Set (Am_LEFT, left);
        switch (v_align) {
	case Am_TOP_ALIGN:
          item.Set (Am_TOP, top);
	  break;
	case Am_BOTTOM_ALIGN:
          item.Set (Am_TOP, top + (fixed_height ? fixed_height : line_height)
		     - height);
	  break;
	default:
          item.Set (Am_TOP, top + ((fixed_height ? fixed_height: line_height)
				    - height) / 2);
	  break;
	}
        left += (fixed_width ? fixed_width : width) + h_spacing;
        ++rank;
      }
      components.Next ();
    }
    while(!components.Last()) {
      Am_Object item = (Am_Object)components.Get();
      if (!(bool)item.GV(Am_VISIBLE)) {
	components.Next();
      } else
	break;
    }
    top += (fixed_height ? fixed_height : line_height) + v_spacing;
    left = x_offset + indent;
  }
  return NULL;
}

Am_Define_Formula (Am_Ptr, rev_horizontal_layout_procedure)
{
  if (!self.GV(Am_VISIBLE)) return NULL;
  Am_Value_List components;
  components = self.GV (Am_GRAPHICAL_PARTS);
  int fixed_width, fixed_height;
  get_fixed_sizes (self, components, cc, fixed_width, fixed_height);
  int max_rank, max_size;
  get_max_rank_and_size (self, cc, max_rank, max_size);
  int x_offset = self.GV (Am_X_OFFSET);
  int y_offset = self.GV (Am_Y_OFFSET);
  int left = x_offset;
  int top = y_offset;
  int h_spacing = self.GV (Am_H_SPACING);
  int v_spacing = self.GV (Am_V_SPACING);
  int h_align = self.GV (Am_H_ALIGN);
  int v_align = self.GV (Am_V_ALIGN);
  int indent = self.GV (Am_INDENT);
  components.End ();
  while (!components.First ()) {
    int line_rank;
    int line_height;
    my_find_line_size_and_rank (components, left, fixed_width, Am_WIDTH,
			     h_spacing, Am_HEIGHT, max_rank, max_size,
			     line_rank, line_height);
    int rank = 0;
    while (rank < line_rank) {
      Am_Object item;
      item = components.Get ();
      if ((bool)item.GV (Am_VISIBLE)) {
        int width = item.GV (Am_WIDTH);
        int height = item.GV (Am_HEIGHT);
	if (fixed_width) {
	  switch (h_align) {
	  case Am_LEFT_ALIGN:
            item.Set (Am_LEFT, left);
	    break;
	  case Am_RIGHT_ALIGN:
            item.Set (Am_LEFT, left + fixed_width - width);
	    break;
	  default:
            item.Set (Am_LEFT, left + (fixed_width - width) / 2);
	    break;
	  }
	}
	else
	  item.Set (Am_LEFT, left);
        switch (v_align) {
	case Am_TOP_ALIGN:
          item.Set (Am_TOP, top);
	  break;
	case Am_BOTTOM_ALIGN:
          item.Set (Am_TOP, top + (fixed_height ? fixed_height : line_height)
		     - height);
	  break;
	default:
          item.Set (Am_TOP, top + ((fixed_height ? fixed_height: line_height)
				    - height) / 2);
	  break;
	}
        left += (fixed_width ? fixed_width : width) + h_spacing;
        ++rank;
      }
      components.Prev ();
    }
    while(!components.First()) {
      Am_Object item = (Am_Object)components.Get();
      if (!(bool)item.GV(Am_VISIBLE)) {
	components.Prev();
      } else
	break;
    }
    top += (fixed_height ? fixed_height : line_height) + v_spacing;
    left = x_offset + indent;
  }
  return NULL;
}

Am_Define_Formula (Am_Ptr, my_vertical_layout_procedure)
{
  Am_Value_List components;
  components = self.GV (Am_GRAPHICAL_PARTS);
  int fixed_width, fixed_height;
  get_fixed_sizes (self, components, cc, fixed_width, fixed_height);
  int max_rank, max_size;
  get_max_rank_and_size (self, cc, max_rank, max_size);
  int x_offset = self.GV (Am_X_OFFSET);
  int y_offset = self.GV (Am_Y_OFFSET);
  int top = y_offset;
  int left = x_offset;
  int h_spacing = self.GV (Am_H_SPACING);
  int v_spacing = self.GV (Am_V_SPACING);
  int h_align = self.GV (Am_H_ALIGN);
  int v_align = self.GV (Am_V_ALIGN);
  int indent = self.GV (Am_INDENT);
  components.Start ();
  while (!components.Last ()) {
    int line_rank;
    int line_width;
    find_line_size_and_rank (components, top, fixed_height, Am_HEIGHT,
			     v_spacing, Am_WIDTH, max_rank, max_size,
			     line_rank, line_width);
    int rank = 0;
    while (rank < line_rank) {
      Am_Object item;
      item = components.Get ();
      if ((bool)item.GV (Am_VISIBLE)) {
        int width = item.GV (Am_WIDTH);
        int height = item.GV (Am_HEIGHT);
	if (fixed_height) {
	  switch (v_align) {
	  case Am_TOP_ALIGN:
            item.Set (Am_TOP, top);
	    break;
	  case Am_BOTTOM_ALIGN:
            item.Set (Am_TOP, top + fixed_height - height);
	    break;
	  default:
            item.Set (Am_TOP, top + (fixed_height - height) / 2);
	    break;
	  }
	}
	else
	  item.Set (Am_TOP, top);
        switch (h_align) {
	case Am_LEFT_ALIGN:
          item.Set (Am_LEFT, left);
	  break;
	case Am_RIGHT_ALIGN:
          item.Set (Am_LEFT, left + (fixed_width ? fixed_width : line_width)
		     - width);
	  break;
	default:
          item.Set (Am_LEFT, left + ((fixed_width ? fixed_width : line_width)
				    - width) / 2);
	  break;
	}
        top += (fixed_height ? fixed_height : height) + v_spacing;
        ++rank;
      }
      components.Next ();
    }
    while(!components.Last()) {
      Am_Object item = (Am_Object)components.Get();
      if (!(bool)item.GV(Am_VISIBLE)) {
	components.Next();
      } else
	break;
    }
    left += (fixed_width ? fixed_width : line_width) + h_spacing;
    top = y_offset + indent;
  }
  return NULL;
}
     
//Am_Object Am_Arrow = 0;

Am_Slot_Key Am_ARROW_LENGTH = Am_Register_Slot_Name("Am_ARROW_LENGTH");
Am_Slot_Key Am_ARROW_WIDTH = Am_Register_Slot_Name("Am_ARROW_WIDTH");

Am_Define_Point_List_Formula(arrow_point_list)
{
  int x1 = self.GV(Am_X1);
  int y1 = self.GV(Am_Y1);
  int x2 = self.GV(Am_X2);
  int y2 = self.GV(Am_Y2);
  int dx = x2 - x1;
  int dy = y2 - y1;
  int al = self.GV(Am_ARROW_LENGTH);
  int aw = self.GV(Am_ARROW_WIDTH);
  double dl = sqrt(dx * dx + dy * dy);
  double sin_d = 0;
  double cos_d = 0;
  if (dl != 0) {
    sin_d = dy / dl;
    cos_d = dx / dl;
  }
  int arx = (int)floor(x2 - al * cos_d);
  int ary = (int)floor(y2 - al * sin_d);
  int ax1 = (int)floor(arx - aw * sin_d);
  int ay1 = (int)floor(ary + aw * cos_d);
  int ax2 = (int)floor(arx + aw * sin_d);
  int ay2 = (int)floor(ary - aw * cos_d);

  Am_Point_List pts = (Am_Point_List)self.GV(Am_POINT_LIST);
  pts.Add_Point(x1, y1, 0);
  pts.Add_Point(arx, ary, 1);
  pts.Add_Point(ax1, ay1, 2);
  pts.Add_Point(x2, y2, 3);
  pts.Add_Point(ax2, ay2, 4);
  pts.Add_Point(arx, ary, 5);
  return pts;
}


void My_Opal_Initialize(void)
{
  Am_Horizontal_Layout =
    Am_Formula::Create(my_horizontal_layout_procedure)->Note_Reference ();
  Am_Horizontal_Rev_Layout =
    Am_Formula::Create(rev_horizontal_layout_procedure)->Note_Reference ();
  Am_Vertical_Layout =
    Am_Formula::Create(my_vertical_layout_procedure)->Note_Reference ();
  Am_Width_Of_Parts =
    Am_Formula::Create (my_width_of_parts_procedure)->Note_Reference ();
  Am_Height_Of_Parts =
    Am_Formula::Create (my_height_of_parts_procedure)->Note_Reference ();

  Am_Arrow = Am_Polygon.Create("Am_Arrow")
    .Set(Am_X1, 0)
    .Set(Am_Y1, 0)
    .Set(Am_X2, 0)
    .Set(Am_Y2, 0)
    .Set(Am_ARROW_LENGTH, 8)
    .Set(Am_ARROW_WIDTH, 4)
    .Set(Am_POINT_LIST, Am_Formula::Create(arrow_point_list))
    ;
}

/* ---------------- Interactor ----------------*/

// ---- Raise_Move_Grow_Interactor ----

Am_Object Raise_Move_Grow_Command;
Am_Object Raise_Move_Grow_Interactor;
Am_Object My_Move_Grow_Interactor;

extern void set_initial_values (Am_Object inter, Am_Object obj,
				Am_Object command_obj,
				int first_x, int first_y, Am_Object ref_obj);
extern void calc_all (Am_Object inter, Am_Object obj,
		      int x, int y, Am_Object ref_obj, Am_Four_Ints_Data* data);
extern void Am_Get_Filtered_Input(Am_Object inter, int x, int y,
				  int& out_x, int & out_y);
extern void move_grow_interim_val(Am_Object command_obj, Am_Object object_to_move,
			      Am_Object inter, int x, int y, Am_Object ref_obj);

// Grid procedure
void Grid_Positive(Am_Object inter, int x, int y, int &out_x, int &out_y)
{
  Am_Object window = (Am_Object)inter.Get(Am_WINDOW);
  Am_Object owner = ((Am_Object)inter.Get(Am_START_OBJECT)).Get_Owner();

  Am_Translate_Coordinates(window, x, y, owner, x, y);
  x = imax(0 + (int)inter.Get(Am_X_OFFSET), x);
  y = imax(0 + (int)inter.Get(Am_Y_OFFSET), y);
  Am_Translate_Coordinates(owner, x, y, window, out_x, out_y);
//  out_x = imax(0, x);
//  out_y = imax(0, y);
}

void my_move_grow_interim_val(Am_Object command_obj, Am_Object object_to_move,
			      Am_Object inter, int x, int y, Am_Object ref_obj) {
  Am_Four_Ints_Data* data =
    Am_Four_Ints_Data::Narrow(command_obj.Get(Am_INTERIM_VALUE));
//  Am_Get_Filtered_Input(inter, x, y, x, y); // set x,y based on gridding
  data = (Am_Four_Ints_Data*)data->Make_Unique();
  calc_all (inter, object_to_move, x, y, ref_obj, data);
  command_obj.Set (Am_INTERIM_VALUE, data);
  Am_Call (Am_Object_Proc, command_obj, Am_INTERIM_DO_ACTION, (command_obj));
}

void my_Raise_Move_Grow_Start_Action (Am_Object inter, Am_Object object,
				Am_Object event_window,
				Am_Input_Event *ev) {
  int x = ev->x;
  int y = ev->y;
  //first transform x and y to be in coordinate system of object's owner
  int x1, y1;
  Am_Object owner = object.Get_Owner();
  if (owner.Valid()) {
    Am_Translate_Coordinates(event_window, x, y, owner, x1, y1);
  }
  if (Am_Inter_Tracing(inter))
    cout << "Move_Grow starting over " << object
	 << " translated coordinates " << x1 << "," << y1 << endl;

  // first, call the prototype's method
  ( (Am_Action_Function*)(Am_Ptr) (Am_Interactor.Get(Am_START_ACTION)))
    (inter, object, event_window, ev);

  // Now, call the correct Command method
  Am_Object command_obj;
  command_obj = inter.Get(Am_COMMAND);
  if (command_obj.Valid ()) {
    //sets Am_OLD_VALUE of inter with orig_points and initializes
    //command_obj Am_OBJECT_MODIFIED slot
    set_initial_values (inter, object, command_obj, x1, y1, owner);
    Am_Call(Am_Object_Proc, command_obj, Am_START_ACTION, (command_obj));
    // now call command_obj routine
//    move_grow_interim_val(command_obj, object, inter, x1, y1, owner); 
// $B%M%9%H$7$?%*%V%8%'%/%H$rF0$+$9$H$-(B,$B:G=i$@$1JQ49$7$?:BI8$G8F$S=P$7$F$$$k!%(B
    if (!(bool)inter.Get(Am_CONTINUOUS)) // not continuous, call stop
      Am_Call(Am_Object_Proc, command_obj, Am_DO_ACTION, (command_obj));
  }

  owner.Remove_Part(object); // raise the moving object.
  owner.Add_Part(object);
}

void my_Move_Grow_Start_Action (Am_Object inter, Am_Object object,
				Am_Object event_window,
				Am_Input_Event *ev) {
  int x = ev->x;
  int y = ev->y;
  //first transform x and y to be in coordinate system of object's owner
  int x1, y1;
  Am_Object owner = object.Get_Owner();
  if (owner.Valid()) {
    Am_Translate_Coordinates(event_window, x, y, owner, x1, y1);
  }
  if (Am_Inter_Tracing(inter))
    cout << "Move_Grow starting over " << object
	 << " translated coordinates " << x1 << "," << y1 << endl;

  // first, call the prototype's method
  ( (Am_Action_Function*)(Am_Ptr) (Am_Interactor.Get(Am_START_ACTION)))
    (inter, object, event_window, ev);

  // Now, call the correct Command method
  Am_Object command_obj;
  command_obj = inter.Get(Am_COMMAND);
  if (command_obj.Valid ()) {
    //sets Am_OLD_VALUE of inter with orig_points and initializes
    //command_obj Am_OBJECT_MODIFIED slot
    set_initial_values (inter, object, command_obj, x1, y1, owner);
    Am_Call(Am_Object_Proc, command_obj, Am_START_ACTION, (command_obj));
    // now call command_obj routine
//    /* my_ */ move_grow_interim_val(command_obj, object, inter, x1, y1, owner); 
// $B%M%9%H$7$?%*%V%8%'%/%H$rF0$+$9$H$-(B,$B:G=i$@$1JQ49$7$?:BI8$G8F$S=P$7$F$$$k!%(B
    if (!(bool)inter.Get(Am_CONTINUOUS)) // not continuous, call stop
      Am_Call(Am_Object_Proc, command_obj, Am_DO_ACTION, (command_obj));
  }
}

void my_Move_Grow_Back_Inside_Action (Am_Object inter, Am_Object /*object*/,
				      Am_Object event_window,
				      Am_Input_Event *ev) {
  int x = ev->x;
  int y = ev->y;
  // ignore object parameter, used saved object
  Am_Object object;
  object = inter.Get(Am_START_OBJECT);
  if (Am_Inter_Tracing(inter))
    cout << "Move_Grow back inside over " << object << endl;
  // first, call the prototype's method
  ( (Am_Action_Function*)(Am_Ptr) (Am_Interactor.Get(Am_BACK_INSIDE_ACTION)))
    (inter, object, event_window, ev);

  Am_Object command_obj;
  command_obj = inter.Get(Am_COMMAND);
  if (command_obj.Valid ()) {
    Am_Call(Am_Object_Proc, command_obj, Am_START_ACTION, (command_obj));
    // now call command_obj routine
    my_move_grow_interim_val(command_obj, object, inter, x, y, event_window);
  }
}

void my_Move_Grow_Running_Action (Am_Object inter, Am_Object /* object */,
				  Am_Object event_window,
				  Am_Input_Event *ev) {
  int x = ev->x;
  int y = ev->y;
  // ignore object parameter, used saved object
  Am_Object object;
  object = inter.Get(Am_START_OBJECT);

  if (Am_Inter_Tracing(inter))
    cout << "Move_Grow running over " << object << endl;

  // first, call the prototype's method
  ( (Am_Action_Function*)(Am_Ptr) (Am_Interactor.Get(Am_RUNNING_ACTION)))
    (inter, object, event_window, ev);

  Am_Object command_obj;
  command_obj = inter.Get(Am_COMMAND);
  if (command_obj.Valid ())
    my_move_grow_interim_val(command_obj, object, inter, x, y, event_window);
}

void my_Move_Grow_Stop_Action (Am_Object inter, Am_Object /* object */,
			       Am_Object event_window,
			       Am_Input_Event *ev) {
  int x = ev->x;
  int y = ev->y;
  // ignore object parameter, used saved object
  Am_Object object;
  object = inter.Get(Am_START_OBJECT);
  if (Am_Inter_Tracing(inter))
    cout << "Move_Grow stopping over " << object << endl;


  if (Am_Inter_Tracing(inter)) cout << "Move_Grow stopping over "
				    << object << endl;
  // first, call the prototype's method
  ( (Am_Action_Function*)(Am_Ptr) (Am_Interactor.Get(Am_STOP_ACTION)))
    (inter, object, event_window, ev);

  Am_Object command_obj;
  command_obj = inter.Get(Am_COMMAND);
  if (command_obj.Valid ()) {
    Am_Four_Ints_Data* old = Am_Four_Ints_Data::Narrow(command_obj.Get(Am_OLD_VALUE));
    Am_Four_Ints_Data* dest =Am_Four_Ints_Data::Narrow(command_obj.Get(Am_INTERIM_VALUE));
    if (abs(old->data.rect.left - dest->data.rect.left) < (int)inter.Get(Am_MINIMUM_WIDTH)
	&& abs(old->data.rect.top - dest->data.rect.top) < (int)inter.Get(Am_MINIMUM_HEIGHT)) {
      Am_Call(Am_Object_Proc, command_obj, Am_ABORT_ACTION, (command_obj));
    } else {
      /* my_ */move_grow_interim_val(command_obj, object, inter, x, y, event_window);
      Am_Call(Am_Object_Proc, command_obj, Am_DO_ACTION, (command_obj));
    }
  }
}

void raise_move_interm_do_action(Am_Object cmd)
{
  Am_Four_Ints_Data* interim = Am_Four_Ints_Data::Narrow(cmd.Get(Am_INTERIM_VALUE));
  interim = (Am_Four_Ints_Data*)interim->Make_Unique();
  int x = interim->data.rect.left;
  int y = interim->data.rect.top;
  Am_Object inter = cmd.Get_Owner();
  int out_x, out_y;
  Am_Get_Filtered_Input(inter, x, y, out_x, out_y);
  interim->data.rect.left = out_x;
  interim->data.rect.top = out_y;
  cmd.Set(Am_INTERIM_VALUE, interim);
  Am_Call (Am_Object_Proc, Am_Move_Grow_Command, Am_INTERIM_DO_ACTION, (cmd));
}

void My_Inter_Initialize(void)
{
  Raise_Move_Grow_Command = Am_Move_Grow_Command.Create()
    .Set(Am_INTERIM_DO_ACTION, (Am_Object_Proc*)&raise_move_interm_do_action)
    ;
  Raise_Move_Grow_Interactor =
    Am_Move_Grow_Interactor.Create("Raise_Move_Grow_Interactor")
    .Set(Am_START_ACTION, (Am_Object_Proc*)&my_Raise_Move_Grow_Start_Action)
//    .Set(Am_BACK_INSIDE_ACTION, (Am_Object_Proc*)&my_Move_Grow_Back_Inside_Action)
//    .Set(Am_RUNNING_ACTION, (Am_Object_Proc*)&my_Move_Grow_Running_Action)
//    .Set(Am_STOP_ACTION, (Am_Object_Proc*)&my_Move_Grow_Stop_Action)
    .Set(Am_MINIMUM_WIDTH, 0)
    .Set(Am_MINIMUM_HEIGHT, 0)
//    .Add_Part(Am_COMMAND, Raise_Move_Grow_Command.Create())
    ;

  My_Move_Grow_Interactor =
    Am_Move_Grow_Interactor.Create("My_Move_Grow_Interactor")
    .Set(Am_START_ACTION, (Am_Object_Proc*)&my_Move_Grow_Start_Action)
//    .Set(Am_BACK_INSIDE_ACTION, (Am_Object_Proc*)&my_Move_Grow_Back_Inside_Action)
//    .Set(Am_RUNNING_ACTION, (Am_Object_Proc*)&my_Move_Grow_Running_Action)
    .Set(Am_STOP_ACTION, (Am_Object_Proc*)&my_Move_Grow_Stop_Action)
    .Set(Am_MINIMUM_WIDTH, 0)
    .Set(Am_MINIMUM_HEIGHT, 0)
//    .Add_Part(Am_COMMAND, Raise_Move_Grow_Command.Create())
    ;
}

/* ---------------- Widgets ---------------- */

Am_Declare_Formula (void*, Am_Get_Computed_Colors_Record_Form);

void Am_Draw_Motif_Round_Box (int left, int top, int width, int height,
			      int radius, bool depressed,
			      Computed_Colors_Record* rec, Am_Drawonable* draw)
{
  Am_Style top_fill;
  Am_Style bot_fill;
  Am_Style inside_fill;
  if (depressed) {
    top_fill = rec->shadow_style;
    bot_fill = rec->highlight_style;
    inside_fill = rec->background_style;
  }
  else {
    top_fill = rec->highlight_style;
    bot_fill = rec->shadow_style;
    inside_fill = rec->foreground_style;
  }
  //top edges
  draw->Draw_Roundtangle(Am_No_Style, top_fill, left, top, width, height,
			 radius, radius);
  //bottom edges
  draw->Draw_Roundtangle(Am_No_Style, bot_fill, left + 1, top + 1, width-1,
			 height-1, radius, radius);
  //inside of box
  draw->Draw_Roundtangle(Am_No_Style, inside_fill, left+1, top+1, width-2,
			 height-2, radius, radius);
}

// draw routine for a plain border'ed round rectangle. added by toyoda
void border_roundtangle_draw (Am_Object self, Am_Drawonable* drawonable,
			    int x_offset, int y_offset)
{
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  int radius = self.Get(Am_DRAW_RADIUS);
  bool selected = self.Get (Am_SELECTED);
  Computed_Colors_Record* rec =
      Computed_Colors_Record::Narrow(self.Get(Am_STYLE_RECORD));
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);
  if (look == Am_MOTIF_LOOK)
    Am_Draw_Motif_Round_Box(left, top, width, height, radius, selected, rec, drawonable);
  else
    Am_Error("Sorry, only the Motif Style implemented for now");
}

// exported objects

Am_Object Am_Border_Roundtangle = 0;

void My_Widgets_Initialize () {

  Am_Object_Advanced obj_adv; // to get at advanced features like
			       // local-only and demons.

  //////////// border round rectangle ////////////
  Am_Border_Roundtangle = Am_Roundtangle.Create("Border_Roundtangle")
    .Set (Am_SELECTED, false)
    .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
    .Set (Am_STYLE_RECORD,
	   Am_Formula::Create (Am_Get_Computed_Colors_Record_Form))
    .Set (Am_FILL_STYLE, Am_Motif_Light_Blue)
    .Set (Am_DRAW_METHOD, (Am_Object_Proc*)border_roundtangle_draw)
    ;
  
  obj_adv = (Am_Object_Advanced&)Am_Border_Roundtangle;
  obj_adv.Get_Slot (Am_RADIUS)
    ->Set_Demon_Bits (Am_MOVING_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_DRAW_RADIUS)
    ->Set_Demon_Bits (Am_MOVING_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_SELECTED)
    ->Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_WIDGET_LOOK)
    ->Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FILL_STYLE)
    ->Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
    
}

// Popup Menu

// export objects
Am_Object Popup_Menu;
Am_Object Popup_Button_Command;

// slot keys
Am_Slot_Key TARGET_OBJECT = Am_Register_Slot_Name("TARGET_OBJECT");

Am_Object Popup_Menu_and_Return_First_Item(Am_Object inter, Am_Object dest_obj, int x, int y)
{
  inter.Get_Owner().Set(TARGET_OBJECT, 0);
  Am_Do_Events(); // $BF1$8%?!<%2%C%H$K9g$o$;$k$H#2EYL\$O:F7W;;$5$l$J$$$N$G!%(B
  inter.Get_Owner()
    .Set(Am_VISIBLE, true)
    .Set(Am_LEFT, x - 20)
    .Set(Am_TOP, y - 10)
    .Set(TARGET_OBJECT, dest_obj);
  Am_Value_List items = (Am_Value_List)inter.Get_Owner().Get(Am_GRAPHICAL_PARTS);
  items.Start();
  return (Am_Object)items.Get();
}  

// ABORT_ACTION for Menu and Button_Command
void abort_menu(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Am_Menu.Get_Part(Am_INTERACTOR).Get_Part(Am_COMMAND),
	  Am_ABORT_ACTION, (cmd));
  cmd.Get_Owner().Get_Owner().Set(Am_VISIBLE, false);
}

void popup_do_action(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Am_Button_Command, Am_DO_ACTION, (cmd));
  cmd.Get_Owner().Get_Owner().Set(Am_VISIBLE, false);
}  

void My_Popup_Menu_Initialize(void)
{
  // Popup Menu prototype
  Popup_Menu = Am_Menu.Create("Popup_Menu")
    .Set(Am_VISIBLE, false)
    .Set(Am_FONT, Am_Font("-*-helvetica-*-r-*--12-*"))
    .Set(Am_FILL_STYLE, Am_Amulet_Purple)
    .Set(Am_V_SPACING, -3)
    .Set(TARGET_OBJECT, 0)
    .Get_Part(Am_INTERACTOR)
      .Set(Am_START_WHEN, "MIDDLE_DOWN")
      .Set(Am_RUNNING_WHERE_TEST, (Am_Object_Proc*)&Am_In_Active_Widget_Part)
      .Get_Part(Am_COMMAND)
        .Set(Am_ABORT_ACTION, &abort_menu)
        .Get_Owner()
    .Get_Owner()
    ;

  Popup_Button_Command = Am_Button_Command.Create()
    .Set(Am_DO_ACTION, &popup_do_action);
}  




/* ---------------- Initialize ---------------- */

void My_Amulet_Initialize(void)
{
  My_Opal_Initialize();
  My_Inter_Initialize();
  My_Widgets_Initialize();
  My_Popup_Menu_Initialize();
}
