// Copyright (C) 1996  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet/amulet.h>
#include <amulet/formula_advanced.h>
#include <amulet/inter_advanced.h>
#include <amulet/widgets_advanced.h>
#include <amulet/debugger.h>
#include <strstream.h>
#include <fstream.h>
#include <unistd.h>
#include "my_amulet.h"
#include "kl_widgets.h"
#include "kl_pattern.h"
#include "translator.h"
#include "save_load.h"
#include "file_box.h"

/* --- */

const char *tracer_name = TRACER;
const char *tracer_program = KLIEG_BINDIR "/" TRACER;

/* ---------------- Initial size and position of windows ---------------- */

int top_win_left = 0;
int top_win_top = 0;
int top_win_width = 600;
int top_win_height = 600;

int canvas_margin = 2;
int canvas_tools_gap = 10;

char* default_font = "-*-helvetica-*-r-*--12-*";
Am_Style menu_agg_style = Am_Style("gray80", 4, Am_CAP_BUTT, Am_JOIN_ROUND,
				   Am_LINE_SOLID, Am_DEFAULT_DASH_LIST,
				   Am_DEFAULT_DASH_LIST_LENGTH,
				   Am_FILL_SOLID, Am_FILL_POLY_EVEN_ODD, Am_No_Image);


/* ------------------------ Klieg Windows ------------------------ */

// ---- global objects ----
Am_Object KL_Top_Win,
            KL_Tools_Agg,
            KL_Menu_Agg,
            KL_Canvas_Win,
              KL_Scroll_Agg,
                KL_Process_Agg,
                KL_Rules_Agg;
//                KL_Pattern_Agg;

// ---- accelerator ----
Am_Object get_process_part(Am_Object window) {
  return window.Get_Part(KL_SCROLL_PART).Get_Part(KL_PROCESS_PART);
}

Am_Object get_rules_part(Am_Object window) {
  return window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
}

// ---- formulas ----

// Canvas Left, Top, Width, Height
Am_Define_Formula(int, kl_canvas_left) {
  return (int)self.GV_Owner().GV_Part(KL_TOOLS_PART).GV(Am_WIDTH)
    + canvas_margin;
}
Am_Define_Formula(int, kl_canvas_top) {
  return (int)self.GV_Owner().GV_Part(KL_MENU_PART).GV(Am_HEIGHT)
    + canvas_margin;
}
Am_Define_Formula(int, kl_canvas_width) {
  return (int)self.GV_Owner().GV(Am_WIDTH)
    - (int)self.GV_Owner().GV_Part(KL_TOOLS_PART).GV(Am_WIDTH)
    - canvas_margin;
}
Am_Define_Formula(int, kl_canvas_height) {
  return (int)self.GV_Owner().GV(Am_HEIGHT)
    - (int)self.GV_Owner().GV_Part(KL_MENU_PART).GV(Am_HEIGHT)
    - canvas_margin * 2;
}

// Tools Top
Am_Define_Formula(int, kl_tools_top) {
  return (int)self.GV_Owner().GV_Part(KL_MENU_PART).GV(Am_HEIGHT);
}

// Scroll Inner Width, Height
Am_Define_Formula(int, scroll_inner_width_form) {
  int max_x = 0, comp_right;
  Am_Value_List components;
  Am_Object comp;
  components = self.GV(Am_GRAPHICAL_PARTS);
  for (components.Start(); !components.Last(); components.Next()) {
    comp = components.Get ();
    // compute how much of the component extends right of the origin
    comp_right = ((int)(comp.GV(Am_LEFT)) + (int)(comp.GV(Am_WIDTH)));
    max_x = imax (max_x, comp_right);
  }
  if (self.GV_Owner().Valid())
    return imax(max_x, self.GV_Owner().GV(Am_WIDTH));
  else
    return max_x;
}
Am_Define_Formula (int, scroll_inner_height_form) {
  int max_y = 0, comp_bottom;
  Am_Value_List components;
  Am_Object comp;
  components = self.GV(Am_GRAPHICAL_PARTS);
  for (components.Start(); !components.Last(); components.Next()) {
    comp = components.Get ();
    // compute how much of the component extends below the origin
    comp_bottom = ((int)(comp.GV(Am_TOP)) + (int)(comp.GV(Am_HEIGHT)));
    max_y = imax (max_y, comp_bottom);
  }
  if (self.GV_Owner().Valid()) 
    return imax(max_y, self.GV_Owner().GV(Am_HEIGHT));
  else
    return max_y;
}

void Initialize_Windows(void)
{
  KL_Top_Win = Module_Win.Create ("KL_Top_Win")
    .Set(Am_TITLE, "Klieg")
    .Set(Am_ICON_TITLE, "Klieg")
    .Set(Am_LEFT, top_win_left)
    .Set(Am_TOP, top_win_top)
    .Set(Am_WIDTH, top_win_width)
    .Set(Am_HEIGHT, top_win_height)
    .Set(Am_FILL_STYLE, menu_agg_style)
    ;
  KL_Menu_Agg = Am_Group.Create("KL_Menu_Agg")
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    ;
  KL_Tools_Agg = Am_Group.Create("KL_Tools_Agg")
    .Set(Am_TOP, Am_Formula::Create(kl_tools_top))
    .Set(Am_WIDTH, Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    ;
  KL_Canvas_Win = Am_Window.Create("KL_Canvas_Win")
    .Set(Am_LEFT, Am_Formula::Create(kl_canvas_left))
    .Set(Am_TOP, Am_Formula::Create(kl_canvas_top))
    .Set(Am_WIDTH, Am_Formula::Create(kl_canvas_width))
    .Set(Am_HEIGHT, Am_Formula::Create(kl_canvas_height))
    .Set(Am_FILL_STYLE, Am_Motif_Gray)
    ;
  KL_Scroll_Agg = Am_Scrolling_Group.Create("KL_Scroll_Agg")
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, Am_From_Owner(Am_HEIGHT))
    .Set(Am_FILL_STYLE, Am_Motif_Gray)
    .Set(Am_INNER_FILL_STYLE, Am_Motif_Light_Gray)
    .Set(Am_INNER_WIDTH, Am_Formula::Create(scroll_inner_width_form))
    .Set(Am_INNER_HEIGHT, Am_Formula::Create(scroll_inner_height_form))
    ;
  KL_Process_Agg = Process_Agg.Create("KL_Process_Agg")
    .Set(Am_WIDTH, Am_Formula::Create(scroll_inner_width_form))
    .Set(Am_HEIGHT, Am_Formula::Create(scroll_inner_height_form))
    ;
  KL_Rules_Agg = Am_Group.Create("KL_Rules_Agg")
    .Set(Am_WIDTH, Am_Formula::Create(scroll_inner_width_form))
    .Set(Am_HEIGHT, Am_Formula::Create(scroll_inner_height_form))
    ;
/*
  KL_Pattern_Agg = Am_Group.Create("KL_Pattern_Agg")
    .Set(Am_WIDTH, Am_Formula::Create(scroll_inner_width_form))
    .Set(Am_HEIGHT, Am_Formula::Create(scroll_inner_height_form))
    ;
*/
  KL_Top_Win
    .Add_Part(KL_MENU_PART,   KL_Menu_Agg)
    .Add_Part(KL_TOOLS_PART,  KL_Tools_Agg)
    .Add_Part(KL_CANVAS_PART, KL_Canvas_Win
	      .Add_Part(KL_SCROLL_PART, KL_Scroll_Agg
			.Add_Part(KL_PROCESS_PART, KL_Process_Agg)
			.Add_Part(KL_RULES_PART,   KL_Rules_Agg)
//			.Add_Part(KL_PATTERN_PART, KL_Pattern_Agg)
		));
}

/* -------------------------- Tools & Menus -------------------------- */

/* ---- Mover, Text inter ---- */

// ---- global objects ----
Am_Object process_feedback;

// ---- slot keys ----
Am_Slot_Key KL_FRAME_FEEDBACK   = Am_Register_Slot_Name("KL_FRAME_FEEDBACK");
Am_Slot_Key KL_BINDING_FEEDBACK = Am_Register_Slot_Name("KL_BINDING_FEEDBACK");
Am_Slot_Key KL_TEXT_INTER       = Am_Register_Slot_Name("KL_TEXT_INTER");
Am_Slot_Key KL_PROCESS_MOVER    = Am_Register_Slot_Name("KL_PROCESS_MOVER");
Am_Slot_Key KL_RULE_MOVER       = Am_Register_Slot_Name("KL_RULE_MOVER");
Am_Slot_Key KL_TOP_PORT_MOVER   = Am_Register_Slot_Name("KL_TOP_PORT_MOVER");
Am_Slot_Key KL_PORT_MOVER       = Am_Register_Slot_Name("KL_PORT_MOVER");
Am_Slot_Key KL_PORT_BINDER      = Am_Register_Slot_Name("KL_PORT_BINDER");

Am_Slot_Key OLD_WINDOW = Am_Register_Slot_Name ("OLD_WINDOW");

// --- Process Mover ----

void move_to_original_place(Am_Object obj, Am_Object cmd)
{
  Am_Object inter = cmd.Get_Owner();
  Am_Four_Ints_Data* data = Am_Four_Ints_Data::Narrow(cmd.Get(Am_OLD_VALUE));
  int x = data->data.rect.left;
  int y = data->data.rect.top;
  obj.Set(Am_LEFT, x).Set(Am_TOP, y);
}

Am_Object switch_windows(Am_Object obj, Am_Object new_win) {
  Am_Object old_win = Am_No_Object;
  if (obj.Valid ()) {
    old_win = obj.Get(Am_WINDOW);
    if (old_win != new_win) {
//      cout << "()*()* Moving " << obj << " to new win "
//	   << new_win << endl << flush;
      old_win.Remove_Part(obj);
      new_win.Add_Part(obj);
    }
  }
  return old_win;
}

void interim_move_obj_to_window(Am_Object cmd) {
  Am_Object obj, inter, cur_win;
  inter = cmd.Get_Owner();
  cur_win = inter.Get(Am_WINDOW);
  obj = cmd.Get(Am_FEEDBACK_OBJECT);
  if (!obj.Valid()) obj = cmd.Get(Am_OBJECT_MODIFIED); //if no feedback
  switch_windows(obj, cur_win);
  Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_INTERIM_DO_ACTION, (cmd));
}

Am_Object in_hole(Am_Object agg, int x, int y, Am_Object event_window);

void process_move_command_do(Am_Object cmd) {
  Am_Object window = (Am_Object)cmd.Get_Owner().Get(Am_WINDOW);
  Am_Object proc_agg = get_process_part(window);
  Am_Object rules_agg = get_rules_part(window);
  Am_Object proc = (Am_Object)cmd.Get(Am_OBJECT_MODIFIED);

  Am_Four_Ints_Data* data = Am_Four_Ints_Data::Narrow(cmd.Get(Am_INTERIM_VALUE));
  Am_Object ref_obj = data->ref_obj;
  int x = data->data.rect.left;
  int y = data->data.rect.top;
  int x1 = x;
  int y1 = y;
  bool window_changed = false;
  if (ref_obj != proc_agg) {
    Am_Translate_Coordinates (ref_obj, x, y, proc_agg, x1, y1);
    window_changed = true;
  }
  Am_Object scroll = window.Get_Part(KL_SCROLL_PART);
  x1 = x1 - (int)scroll.Get(Am_X_OFFSET);
  y1 = y1 - (int)scroll.Get(Am_Y_OFFSET);

  Am_Object hole = in_hole(rules_agg, x1, y1, window);
  if (hole.Valid()) {
    Hole_Replace(hole, proc);
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
    return;
  }
  Am_Object rule = Am_Point_In_Part(rules_agg, x1, y1, window);
  if (KL_Type(rule) == KL_Rule && rule.Is_Instance_Of(Vanishing_Rule_Proto)) {
    Am_Object goal = Goal_Create(proc);
    Rule_Add_Goal_At(rule, goal, x1, y1, rules_agg);
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
  } else {
    if (window_changed) {
      Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
    } else {
      data->data.rect.left = imax(0, x);
      data->data.rect.top = imax(0, y);
      cmd.Note_Changed(Am_INTERIM_VALUE);
      Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_DO_ACTION, (cmd));
    }
  }
}

void rule_move_command_do(Am_Object cmd) {
  Am_Object window = (Am_Object)cmd.Get_Owner().Get(Am_WINDOW);
  Am_Object proc_agg = get_process_part(window);
  Am_Object rules_agg = get_rules_part(window);
  Am_Object rule = (Am_Object)cmd.Get(Am_OBJECT_MODIFIED);

  Am_Four_Ints_Data* data = Am_Four_Ints_Data::Narrow(cmd.Get(Am_INTERIM_VALUE));
  Am_Object ref_obj = data->ref_obj;
  int x = data->data.rect.left;
  int y = data->data.rect.top;
  int x1 = x;
  int y1 = y;

  if (KL_Type(rule) != KL_Pattern) {
    data->data.rect.left = imax(0, x);
    data->data.rect.top = imax(0, y);
    cmd.Note_Changed(Am_INTERIM_VALUE);
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_DO_ACTION, (cmd));
    return;
  }

  bool window_changed = false;
  if (ref_obj != rules_agg) {
    Am_Translate_Coordinates (ref_obj, x, y, rules_agg, x1, y1);
    window_changed = true;
  }

  Am_Object scroll = window.Get_Part(KL_SCROLL_PART);
  x1 = x1 - (int)scroll.Get(Am_X_OFFSET);
  y1 = y1 - (int)scroll.Get(Am_Y_OFFSET);

  Am_Object hole = in_hole(rules_agg, x1, y1, window);
  if (hole.Valid()) {
    if (Hole_Replace(hole, rule)) {
      Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
      return;
    }
  }
  Am_Object dest_rule = Am_Point_In_Part(rules_agg, x1, y1, window);
  if (dest_rule.Valid() && KL_Type(dest_rule) == KL_Rule &&
      (int)dest_rule.Get(RULE_TYPE) == KL_Network_Rule) {
    Am_Object pgoal;
    Am_Function_Call(Copy_Method, rule, UNLINK_COPY, pgoal, (rule));
    Rule_Add_Goal_At(dest_rule, pgoal, x1, y1, rules_agg);
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
    return;
  }
  if (window_changed) {
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
  } else {
    data->data.rect.left = imax(0, x);
    data->data.rect.top = imax(0, y);
    cmd.Note_Changed(Am_INTERIM_VALUE);
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_DO_ACTION, (cmd));
  }
}

Am_Define_Object_Formula(process_feedback_form)
{
  Am_Object window = (Am_Object)self.GV(Am_WINDOW);
  Am_Object feedwin = (Am_Object)process_feedback.Get(Am_WINDOW);
  if (window != feedwin && window.Valid() && feedwin.Valid()) {
    feedwin.Remove_Part(process_feedback);
    window.Add_Part(process_feedback);
  }
  return process_feedback;  
}

// ---- Port Mover ----

Am_Object in_pattern_iter(Am_Object agg, Am_Object pre_found,
			  int x, int y, Am_Object event_window)
{
  Am_Object rule = Am_Point_In_Part(agg, x, y, event_window);
  if (KL_Type(rule) == KL_Pattern) {
    pre_found = rule;
    Am_Object goals_agg = Rule_Goals_Part(rule);
    return in_pattern_iter(goals_agg, pre_found, x, y, event_window);
  } else if (KL_Type(rule) == KL_Rule) {
    Am_Object goals_agg = Rule_Goals_Part(rule);
    return in_pattern_iter(goals_agg, pre_found, x, y, event_window);
  }
  return pre_found;
}  

Am_Object in_hole(Am_Object agg, int x, int y, Am_Object event_window)
{
  Am_Object pattern = in_pattern_iter(agg, NULL, x, y, event_window);
  if (pattern.Valid()) { // hole $B$O%Q%?%s$NCf$K$7$+$J$$$3$H$r2>Dj!%(B
    Am_Object goals_agg = Rule_Holes_Part(pattern);
    Am_Object hole = Am_Point_In_Part(goals_agg, x, y, event_window);
    if (hole.Is_Instance_Of(Hole_Proto)) {
      return hole;
    }
  }
  return NULL;
}

Am_Object search_goal_or_rule_port(Am_Object event_window, int x, int y) {
  Am_Object rules_agg = get_rules_part(event_window);
  if (!rules_agg.Valid()) return 0;

  // $B%H%C%W$K$"$k%k!<%k(B/$B%Q%?!<%s(B
  Am_Object rule = in_pattern_iter(rules_agg, NULL, x, y, event_window);
  if (!rule.Valid()) {
    rule = Am_Point_In_Part(rules_agg, x, y, event_window);
    if (!rule.Valid() || !rule.Is_Instance_Of(Rule_Proto)) return 0;
  }
  // $B%H%C%W%]!<%H$N%A%'%C%/!%(B
  Am_Object port = Am_Point_In_Part(Proc_Ports_Part(rule), x, y, event_window);
  // $B%G!<%?%]!<%H!)(B
  if (!port.Valid()) {
    port = Am_Point_In_Part(Rule_Data_Part(rule), x, y, event_window);
  }
  // $B%4!<%k%]!<%H!)(B
  if (!port.Valid()) {
    Am_Object goal = Am_Point_In_Part(Rule_Goals_Part(rule), x, y, event_window);
    if (goal.Valid() && goal.Is_Instance_Of(Process_Proto))
      port = Am_Point_In_Part(Proc_Ports_Part(goal), x, y, event_window);
  }
  // $B%[!<%k%]!<%H!)(B
  if (!port.Valid()) {
    Am_Object goal = Am_Point_In_Part(Rule_Holes_Part(rule), x, y, event_window);
    if (goal.Valid() && goal.Is_Instance_Of(Process_Proto))
      port = Am_Point_In_Part(Proc_Ports_Part(goal), x, y, event_window);
  }
  // $BCf?H$N%A%'%C%/!%(B
  if (!port.Valid() || !Port_Values_Part(port).Valid()) return 0;
  for ( ; ; ) {
    Am_Object value = Am_Point_In_Part(Port_Values_Part(port), x, y, event_window);
    if (value.Valid() && Port_Values_Part(value).Valid())
      port = value;
    else
      return port;
  }
}

Am_Object KL_In_Goal_Or_Rule_Slot(Am_Object /* inter */, Am_Object /* object */,
				  Am_Object event_window, Am_Input_Char /* ic */,
				  int x, int y)
{
  return search_goal_or_rule_port(event_window, x, y);
}

Am_Object in_pattern_iter(Am_Object agg, Am_Object pre_found,
			  int x, int y, Am_Object event_window);

Am_Object KL_In_Rule_Top_Port (Am_Object inter, Am_Object object, Am_Object event_window,
			       Am_Input_Char  ic , int x, int y) {
  Am_Object rules_agg = get_rules_part(event_window);
  Am_Object rule = in_pattern_iter(rules_agg, NULL, x, y, event_window);
  if (!rule.Valid())
    rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  if (!rule.Is_Instance_Of(Rule_Proto)) return 0;
  Am_Object port = Am_Point_In_Part(Proc_Ports_Part(rule), x, y, event_window);
  if (port.Valid()) return port;
  else return 0;
}

Am_Object KL_In_Value_Port(Am_Object /* inter */, Am_Object /* object */,
			   Am_Object event_window, Am_Input_Char /* ic */, int x, int y)
{
  Am_Object rules_agg = get_rules_part(event_window);
  if (!rules_agg.Valid()) return 0;

  Am_Object rule = in_pattern_iter(rules_agg, NULL, x, y, event_window);
  if (!rule.Valid())
    rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  if (!rule.Is_Instance_Of(Rule_Proto)) return 0;

  Am_Object port = Am_Point_In_Part(Proc_Ports_Part(rule), x, y, event_window);
  if (!port.Valid()) {
    Am_Object goal = Am_Point_In_Part(Rule_Goals_Part(rule), x, y, event_window);
    if (goal.Valid() && goal.Is_Instance_Of(Process_Proto))
      port = Am_Point_In_Part(Proc_Ports_Part(goal), x, y, event_window);
  }
  if (!port.Valid()) {
    Am_Object goal = Am_Point_In_Part(Rule_Holes_Part(rule), x, y, event_window);
    if (goal.Valid() && goal.Is_Instance_Of(Process_Proto))
      port = Am_Point_In_Part(Proc_Ports_Part(goal), x, y, event_window);
  }
  if (port.Valid() && Port_Values_Part(port).Valid())
    port = Am_Point_In_Part(Port_Values_Part(port), x, y, event_window);
  while (port.Valid()) {
    if (Port_Values_Part(port).Valid()) {
      Am_Object value = Am_Point_In_Part(Port_Values_Part(port), x, y, event_window);
      if (value.Valid())
	port = value;
      else {
	if ((int)port.Get(PORT_MODE) == KL_Output) {
	    return port;
	}
	else return 0; // $BF~NO$O%3%T!<6X;_(B
      }
    } else {
      return port; // atom
    }
  }
  return 0;
}

void top_port_move_command_do(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_DO_ACTION, (cmd));
}

void port_move_command_do(Am_Object cmd) {
  Am_Object src_port = (Am_Object)cmd.Get_Owner().Get(Am_START_OBJECT);
  Am_Object window = (Am_Object)cmd.Get_Owner().Get(Am_WINDOW);
  Am_Four_Ints_Data* data = Am_Four_Ints_Data::Narrow(cmd.Get(Am_INTERIM_VALUE));
  int x = data->data.rect.left;
  int y = data->data.rect.top;
  Am_Translate_Coordinates(src_port.Get_Owner(), x, y,
			   window.Get_Part(KL_SCROLL_PART), x, y);
  Am_Object scroll = window.Get_Part(KL_SCROLL_PART);
  x = x - (int)scroll.Get(Am_X_OFFSET);
  y = y - (int)scroll.Get(Am_Y_OFFSET);

  Am_Object dest_port = search_goal_or_rule_port(window, x, y);

  if (Port_Move_Allowed(src_port, dest_port)) {
    Am_Object new_port;
    Am_Function_Call(Copy_Method, src_port, PARENT_COPY, new_port, (src_port));
    Port_Add(dest_port, new_port);
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
  } else {
    Am_Call(Am_Object_Proc, Am_Move_Grow_Command, Am_ABORT_ACTION, (cmd));
  }
}

Am_Object create_binding_line(Am_Object cmd, int x1, int y1, int x2, int y2)
{
  Am_Object window = (Am_Object)cmd.Get_Owner().Get(Am_WINDOW);
  Am_Object scroll = window.Get_Part(KL_SCROLL_PART);
  x2 = x2 - (int)scroll.Get(Am_X_OFFSET);
  y2 = y2 - (int)scroll.Get(Am_Y_OFFSET);

  Am_Object src_port = (Am_Object)(cmd.Get_Owner().Get(Am_START_OBJECT));
  Am_Object dest_port = search_goal_or_rule_port(window, x2, y2);
  if (!dest_port.Valid()) return NULL;
  if (dest_port == src_port) return NULL;
  return Binding_Line_Create(src_port, dest_port);
}

void do_edit_name_in_proc(Am_Object cmd)
{
  Am_Object text_obj = (Am_Object)cmd.Get_Owner().Get(Am_START_OBJECT);
  Am_Object target = text_obj.Get_Owner().Get_Owner();
  Am_String new_name = (Am_String)text_obj.Get(Am_TEXT);
  Am_String old_name = (Am_String)cmd.Get(Am_OLD_VALUE);

  if (target.Is_Instance_Of(Process_Proto)) {
    text_obj.Set(Am_TEXT, (char*)old_name);
    if (Proc_Rename(target, new_name) && strcmp(new_name, "") != 0) {
      Am_Call(Am_Object_Proc, Am_Edit_Text_Command, Am_DO_ACTION, (cmd));
    } else {
      Am_Call(Am_Object_Proc, Am_Edit_Text_Command, Am_ABORT_ACTION, (cmd));
    }
  } else { // Is_Instance_Of(Port_Proto)
    text_obj.Set(Am_TEXT, (char*)old_name);
    if (Port_Rename(target, new_name) && strcmp(new_name, "") != 0) {
      Am_Call(Am_Object_Proc, Am_Edit_Text_Command, Am_DO_ACTION, (cmd));
    } else {
      Am_Call(Am_Object_Proc, Am_Edit_Text_Command, Am_ABORT_ACTION, (cmd));
    }
  }
}

Am_Define_Object_Formula(frame_feedback_form)
{
  Am_Object window = (Am_Object)self.GV(Am_WINDOW);
  if (window.Valid()) {
    return window.GV(KL_FRAME_FEEDBACK);
  }
  return 0;
}

Am_Define_Object_Formula(binding_feedback_form)
{
  Am_Object window = (Am_Object)self.GV(Am_WINDOW);
  if (window.Valid()) {
    return window.GV(KL_BINDING_FEEDBACK);
  }
  return 0;
}

Am_Define_Value_List_Formula(proc_mover_other_wins_form) {
  Am_Value_List other_wins = Am_Value_List();
  Am_Value_List module_wins = (Am_Value_List)Am_Screen.GV(MODULE_WINS);
  for (module_wins.Start(); !module_wins.Last(); module_wins.Next()) {
    Am_Object win = (Am_Object)module_wins.Get();
    Am_Object canvas = win.GV_Part(KL_CANVAS_PART);
    other_wins.Add(canvas);
  }
  return other_wins;
}

void Initialize_Interactors(void)
{
  process_feedback = Am_Rectangle.Create("frame_feedback")
    .Set(Am_VISIBLE, false)
    .Set(Am_FILL_STYLE, Am_No_Style)
    ;

  Am_Object frame_feedback = Am_Rectangle.Create("frame_feedback")
    .Set(Am_VISIBLE, false)
    .Set(Am_FILL_STYLE, Am_No_Style)
    ;

  Am_Object process_mover = /* Raise_ */ My_Move_Grow_Interactor.Create("process_mover")
    .Set(Am_FEEDBACK_OBJECT, Am_Formula::Create(process_feedback_form))
    .Set(Am_OTHER_WINDOWS, Am_Formula::Create(proc_mover_other_wins_form))
    .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&Am_Inter_In_Part)
//    .Set(Am_GRID_PROC, (Am_Object_Proc*)&Grid_Positive)
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_ACTION, process_move_command_do)
      .Set(Am_INTERIM_DO_ACTION, &interim_move_obj_to_window)
    .Get_Owner()
    ;

  Am_Object rule_mover = /* Raise_ */ My_Move_Grow_Interactor.Create("rule_mover")
//    .Set(Am_FEEDBACK_OBJECT, Am_Formula::Create(frame_feedback_form))
    .Set(Am_FEEDBACK_OBJECT, Am_Formula::Create(process_feedback_form))
    .Set(Am_OTHER_WINDOWS, Am_Formula::Create(proc_mover_other_wins_form))
    .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&Am_Inter_In_Part)
//    .Set(Am_GRID_PROC, (Am_Object_Proc*)&Grid_Positive)
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_ACTION, rule_move_command_do)
      .Set(Am_INTERIM_DO_ACTION, &interim_move_obj_to_window)
    .Get_Owner()
    ;

  Am_Object top_port_mover = /* Raise_ */ My_Move_Grow_Interactor.Create("port_mover")
    .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Rule_Top_Port)
    .Set(Am_GRID_PROC, (Am_Object_Proc*)&Grid_Positive)
    .Set(Am_FEEDBACK_OBJECT, Am_Formula::Create(frame_feedback_form))
    .Set(Am_PRIORITY, 2)
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_ACTION, top_port_move_command_do)
    .Get_Owner()
    ;

  Am_Object port_mover = My_Move_Grow_Interactor.Create("port_mover")
    .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Value_Port)
    .Set(Am_FEEDBACK_OBJECT, Am_Formula::Create(frame_feedback_form))
    .Set(Am_MINIMUM_WIDTH, 2)
    .Set(Am_MINIMUM_HEIGHT, 2)
    .Set(Am_PRIORITY, 3)
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_ACTION, port_move_command_do)
    .Get_Owner()
    ;

  Am_Object text_inter = Am_Text_Edit_Interactor.Create("text_inter")
    .Set(Am_PRIORITY, 1.1)
    .Set(Am_START_WHEN, "DOUBLE_LEFT_DOWN")
    .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&Am_Inter_In_Text_Leaf)
    .Get_Part(Am_COMMAND)
      .Set(Am_DO_ACTION, &do_edit_name_in_proc)
    .Get_Owner()
    ;

  Am_Object port_binder = Am_New_Points_Interactor.Create("port_binder")
    .Set(Am_PRIORITY, 11)
    .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Goal_Or_Rule_Slot)
    .Set(Am_START_WHEN, "RIGHT_DOWN")
    .Set(Am_AS_LINE, true)
    .Set(Am_GROWING, true)
    .Set(Am_FEEDBACK_OBJECT, Am_Formula::Create(binding_feedback_form))
    .Get_Part(Am_COMMAND)
    .Set(Am_CREATE_NEW_OBJECT_ACTION,
	 (Am_Object_Proc *)&create_binding_line)
    .Get_Owner()
    ;

  KL_Canvas_Win.Add_Part(KL_FRAME_FEEDBACK,   frame_feedback);
  KL_Canvas_Win.Add_Part(KL_BINDING_FEEDBACK, Append_Line);
  KL_Canvas_Win.Add_Part(process_feedback);

  KL_Process_Agg.Add_Part(KL_TEXT_INTER,    text_inter);
  KL_Process_Agg.Add_Part(KL_PROCESS_MOVER, process_mover);

  KL_Rules_Agg.Add_Part(KL_RULE_MOVER,     rule_mover);
  KL_Rules_Agg.Add_Part(KL_TOP_PORT_MOVER, top_port_mover);
  KL_Rules_Agg.Add_Part(KL_PORT_MOVER,     port_mover);
  KL_Rules_Agg.Add_Part(KL_PORT_BINDER,    port_binder);
}

/* ---------------- Menus ---------------- */

Am_Slot_Key KLIEG_TOOLS       = Am_Register_Slot_Name("KLIEG_TOOLS");
Am_Slot_Key KL_MODULE_NAME    = Am_Register_Slot_Name("KL_MODULE_NAME");

Am_Slot_Key KL_SPLIT_BAR      = Am_Register_Slot_Name("KL_SPLIT_BAR");
Am_Slot_Key TOP_MENU          = Am_Register_Slot_Name("TOP_MENU");
Am_Slot_Key PROCESS_MENU      = Am_Register_Slot_Name("PROCESS_MENU");
Am_Slot_Key PROCESS_PORT_MENU = Am_Register_Slot_Name("PROCESS_PORT_MENU");
Am_Slot_Key RULE_MENU         = Am_Register_Slot_Name("RULE_MENU");
Am_Slot_Key RULE_PORT_MENU    = Am_Register_Slot_Name("RULE_PORT");
Am_Slot_Key GOAL_MENU         = Am_Register_Slot_Name("GOAL_MENU");
Am_Slot_Key BINDER_MENU       = Am_Register_Slot_Name("BINDER_MENU");
Am_Slot_Key GUARD_MENU        = Am_Register_Slot_Name("GUARD_MENU");

Am_Slot_Key PATTERN_MENU      = Am_Register_Slot_Name("PATTERN_MENU");
Am_Slot_Key HOLE_MENU         = Am_Register_Slot_Name("HOLE_MENU");

// ---- global objects ----
Am_Object Klieg_Tools;
Am_Object KL_Menu_Bar;
Am_Object KL_Module_Name;
Am_Object KL_Split_Bar;
Am_Object Top_Menu;
Am_Object Process_Menu;
Am_Object Process_Port_Menu;
Am_Object Rule_Menu;
Am_Object Rule_Port_Menu;
Am_Object Goal_Menu;
Am_Object Binder_Menu;
Am_Object Guard_Menu;
Am_Object Pattern_Menu;
Am_Object Hole_Menu;

// -------- Menu Bar --------

Am_Define_Formula(int, split_bar_top_form) {
  Am_Object module_name = self.GV_Sibling(KL_MODULE_NAME);
  if (module_name.Valid()) {
    return (int)module_name.GV(Am_TOP) + (int)module_name.GV(Am_HEIGHT);
  }
  return 0;
}

// Module Name formula
Am_Define_String_Formula(top_win_module_name_form) {
  Am_Object module_name = self.GV_Part(KL_MENU_PART).GV_Part(KL_MODULE_NAME);
  if (!module_name.Valid()) return (Am_String)"";
  return (Am_String)module_name.GV_Part(Am_COMMAND).GV(Am_VALUE);
}  

Am_Define_String_Formula(input_widget_module_name_form) {
  Am_Object module = self.GV_Owner().GV_Owner().GV_Owner();
  if (!module.Valid()) return (Am_String)"";
  return (Am_String)module.GV(MODULE_NAME);
}

Am_Define_String_Formula(proc_agg_module_name_form) {
  Am_Object win = (Am_Object)self.GV(Am_WINDOW);
  if (!win.Valid()) return (Am_String)"";
  Am_Object top_win = win.GV_Owner();
  if (!top_win.Valid()) return (Am_String)"";
  return (Am_String)top_win.GV(MODULE_NAME);
}

// do_command
void do_rename_module_action(Am_Object cmd)
{
  Am_Object win = (Am_Object)cmd.Get_Owner().Get(Am_WINDOW);
  Am_String new_name = (Am_String)cmd.Get(Am_VALUE);
  Am_String old_name = (Am_String)cmd.Get(Am_OLD_VALUE);
  cmd.Set(Am_VALUE, (char*)old_name); // $B>.:Y9)!%(BAm_VALUE$B$,JQ$o$k$H(B,NAME_LIST$B$bJQ$o$k!%(B
  if (Module_Rename(win, new_name) && strcmp(new_name, "") != 0) {
    Am_Call(Am_Object_Proc, Am_Text_Input_Command, Am_DO_ACTION, (cmd));
  } else {
    cmd.Set(Am_VALUE, (char*)old_name);
    Am_Call(Am_Object_Proc, Am_Text_Input_Command, Am_ABORT_ACTION, (cmd));
  }
}


void init_menu_bar(void)
{
  KL_Module_Name = Am_Text_Input_Widget.Create("Module_Name")
    .Set(Am_FONT, Am_Font(default_font))
    .Set(Am_LABEL_FONT, Am_Font(default_font));

  My_Set_Single_Constraint_Mode(KL_Module_Name.Get_Part(Am_COMMAND), Am_VALUE, false);

  KL_Module_Name
    .Get_Part(Am_COMMAND)
      .Set(Am_LABEL, "Module:")
      .Set(Am_VALUE, Am_Formula::Create(input_widget_module_name_form))
      .Set(Am_DO_ACTION, &do_rename_module_action)
    .Get_Owner()
    ;

  KL_Split_Bar = Am_Border_Rectangle.Create("Split_Bar")
    .Set(Am_TOP, Am_Formula::Create(split_bar_top_form))
    .Set(Am_WIDTH, Am_From_Owner(Am_WIDTH))
    .Set(Am_HEIGHT, 6)
    .Set(Am_SELECTED, true)
    .Set(Am_FILL_STYLE, Am_Motif_Light_Gray)
    ;
  KL_Menu_Bar = Am_Menu_Bar.Create()
    .Set(Am_FONT, Am_Font(default_font))
    .Set(Am_FILL_STYLE, Am_Motif_Gray)
    .Set(Am_LAYOUT, Am_Vertical_Layout)
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Button_Command.Create("Special_Commands")
	      .Set(Am_LABEL, "File")
	      .Set(Am_ITEMS, Am_Value_List()
		   .Add(Am_Button_Command.Create("Prop_Command")
			.Set(Am_LABEL, "Propaties..."))
//		   .Add(Am_Button_Command.Create("Quit_Command")
//			.Set(Am_LABEL, "Quit")
//			.Set(Am_DO_ACTION, do_exit))
		)
	   )
      )
    ;
}
  

// ---- Popup Menu's START_WHERE_TESTs -----

Am_Object KL_In_Proc_Agg (Am_Object inter, Am_Object object, Am_Object event_window,
			  Am_Input_Char /* ic */, int x, int y) {
  Am_Object proc_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_PROCESS_PART);
  if (Am_Point_In_Obj(proc_agg, x, y, event_window).Valid()) {
    return Popup_Menu_and_Return_First_Item(inter, proc_agg, x, y);
  }
  return NULL;
}

Am_Object KL_In_Proc (Am_Object inter, Am_Object object, Am_Object event_window,
		      Am_Input_Char /* ic */, int x, int y) {
  Am_Object proc_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_PROCESS_PART);
  Am_Object proc = Am_Point_In_Part(proc_agg, x, y, event_window);
  if (proc.Is_Instance_Of(Process_Proto)) {
    return Popup_Menu_and_Return_First_Item(inter, proc, x, y);
  }
  return NULL;
}

Am_Object KL_In_Proc_Slot (Am_Object inter, Am_Object object, Am_Object event_window,
			   Am_Input_Char /* ic */, int x, int y) {
  Am_Object proc = Am_Point_In_Part(
    event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_PROCESS_PART), x, y, event_window);
  if (proc.Is_Instance_Of(Process_Proto)) {
    Am_Object slot = Am_Point_In_Part(Proc_Ports_Part(proc), x, y, event_window);
    if (slot.Valid()) {
      return Popup_Menu_and_Return_First_Item(inter, slot, x, y);
    }
  }
  return NULL;
}

Am_Object KL_In_Rule (Am_Object inter, Am_Object object, Am_Object event_window,
		      Am_Input_Char /* ic */, int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  if (KL_Type(rule) == KL_Rule && KL_Type(rule) != KL_Pattern) {
    return Popup_Menu_and_Return_First_Item(inter, rule, x, y);
  }
  return NULL;
}

Am_Object KL_In_Menu_Goal_Or_Rule_Slot(Am_Object inter, Am_Object /* object */,
				       Am_Object event_window, Am_Input_Char /* ic */,
				       int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object rule = in_pattern_iter(rules_agg, NULL, x, y, event_window);
  if (!rule.Valid())
    rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  Am_Object slot, goal, value;
  if (rule.Is_Instance_Of(Rule_Proto)) {
    slot = Am_Point_In_Part(Proc_Ports_Part(rule), x, y, event_window);
    if (!slot.Valid()) {
      goal = Am_Point_In_Part(rule.Get_Part(CONTENTS_PART).Get_Part(GOALS_PART), x, y, event_window);
      if (goal.Valid()) {
	slot = Am_Point_In_Part(Proc_Ports_Part(goal), x, y, event_window);
      }
    }
    if (!slot.Valid()) {
      goal = Am_Point_In_Part(Rule_Holes_Part(rule), x, y, event_window);
      if (goal.Valid()) {
	slot = Am_Point_In_Part(Proc_Ports_Part(goal), x, y, event_window);
      }
    }
    while (slot.Valid()) {
      value = Am_Point_In_Part(Port_Values_Part(slot), x, y, event_window);
      if (value.Valid()) {
	if (value.Is_Instance_Of(Port_Proto) || value.Is_Instance_Of(Structure_Proto))
	  slot = value;
	else
	  return Popup_Menu_and_Return_First_Item(inter, value, x, y);
      } else {
	return Popup_Menu_and_Return_First_Item(inter, slot, x, y);
      }
    }
  }
  return NULL;
}

Am_Object KL_In_Goal(Am_Object inter, Am_Object /* object */,
		     Am_Object event_window, Am_Input_Char /* ic */, int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object rule = in_pattern_iter(rules_agg, NULL, x, y, event_window);
  if (!rule.Valid()) rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  if (rule.Valid()) {
    Am_Object goal = Am_Point_In_Part(Rule_Goals_Part(rule), x, y, event_window);
    if (goal.Valid()) {
      return Popup_Menu_and_Return_First_Item(inter, goal, x, y);
    }
  }
  return NULL;
}

Am_Object KL_In_Binder(Am_Object inter, Am_Object /* object */,
		       Am_Object event_window, Am_Input_Char /* ic */, int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object rule = in_pattern_iter(rules_agg, NULL, x, y, event_window);
  if (!rule.Valid()) rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  if (rule.Valid()) {
    Am_Object binder = Am_Point_In_Part(Rule_Binder_Part(rule), x, y, event_window);
    if (binder.Valid()) {
      return Popup_Menu_and_Return_First_Item(inter, binder, x, y);
    }
  }
  return NULL;
}

Am_Object KL_In_Guard(Am_Object inter, Am_Object /* object */,
		      Am_Object event_window, Am_Input_Char /* ic */, int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  if (rule.Valid()) {
    Am_Object guard = Am_Point_In_Part(Rule_Guard_Part(rule), x, y, event_window);
    if (guard.Valid()) {
      return Popup_Menu_and_Return_First_Item(inter, guard, x, y);
    }
  }
  return NULL;
}

Am_Object KL_Menu_In_Goal_Slot(Am_Object inter, Am_Object /* object */,
			       Am_Object event_window,
			       Am_Input_Char /* ic */, int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object rule = Am_Point_In_Part(rules_agg, x, y, event_window);
  if (rule.Valid()) {
    Am_Object goal = Am_Point_In_Part(rule.Get_Part(CONTENTS_PART).Get_Part(GOALS_PART),
				      x, y, event_window);
    if (goal.Valid()) {
      Am_Object slot = Am_Point_In_Part(goal.Get_Part(CONTENTS_PART).Get_Part(PORTS_PART),
					x, y, event_window);
      while (slot.Valid()) {
	Am_Object value = Am_Point_In_Part(slot.Get_Part(CONTENTS_PART)
					   .Get_Part(VALUE_PART), x, y, event_window);
	if (value.Valid() && value.Is_Instance_Of(Port_Proto)) {
	  slot = value;
	} else {
	  return Popup_Menu_and_Return_First_Item(inter, slot, x, y);
	}
      }
    }
  }
  return NULL;
}

Am_Object KL_In_Pattern (Am_Object inter, Am_Object object, Am_Object event_window,
			 Am_Input_Char /* ic */, int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object pattern = in_pattern_iter(rules_agg, NULL, x, y, event_window);
  if (pattern.Valid()) {
    return Popup_Menu_and_Return_First_Item(inter, pattern, x, y);
  }
  return NULL;
}

Am_Object KL_In_Hole (Am_Object inter, Am_Object object, Am_Object event_window,
		      Am_Input_Char /* ic */, int x, int y) {
  Am_Object rules_agg = event_window.Get_Part(KL_SCROLL_PART).Get_Part(KL_RULES_PART);
  Am_Object hole = in_hole(rules_agg, x, y, event_window);
  if (hole.Valid()) {
    return Popup_Menu_and_Return_First_Item(inter, hole, x, y);
  }
  return NULL;
}

// Common DO_ACTIOINs

void do_dismiss(Am_Object cmd) {
  Am_Object dest_obj = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  dest_obj.Remove_From_Owner();
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

// -------- Top Menu --------

// Top Menu's DO_ACTIONs

void do_create(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object inter = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR);
  Am_Object scroll_agg = ((Am_Object)inter.Get(Am_WINDOW)).Get_Part(KL_SCROLL_PART);

  int x = (int)inter.Get(Am_FIRST_X) + (int)scroll_agg.Get(Am_X_OFFSET);
  int y = (int)inter.Get(Am_FIRST_Y) + (int)scroll_agg.Get(Am_Y_OFFSET);

  Am_Object proto = (Am_Object)cmd.Get(Am_LABEL);
  Am_Object dest_agg;
  if (proto.Is_Instance_Of(Process_Proto)) {
    dest_agg = scroll_agg.Get_Part(KL_PROCESS_PART);
    Add_Process(dest_agg, proto.Create().Set(Am_LEFT, x).Set(Am_TOP, y));
  } else {
    dest_agg = scroll_agg.Get_Part(KL_RULES_PART);
    Am_Object new_obj;
    Am_Function_Call(Copy_Method, proto, UNLINK_COPY, new_obj, (proto));
    Add_Pattern(dest_agg, new_obj.Set(Am_LEFT, x).Set(Am_TOP, y).Set(IS_ICON, false));
  }
}    

void do_new_module(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object new_win = KL_Top_Win.Create("new_win");
  Modules_Add(Am_Screen, new_win, "unnamed");
}

void do_save(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object win = ((Am_Object)cmd.Get_Owner().Get(Am_WINDOW)).Get_Owner();

  Am_Object fb_win = Am_Window.Create()
    .Set(Am_TITLE, "Save")
    .Set(Am_WIDTH, Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    ;
  Am_Screen.Add_Part(fb_win);

  char filename[100], message_string[100];
  strcpy(message_string, "File ");
  if ( FileSelectBox(fb_win, filename) ) {
    strcat(message_string, filename);
    strcat(message_string, " was chosen.");
    MessageBox(fb_win, message_string);

    ofstream ofile;
    ofile.open(filename, ios::out);
    if (!ofile) {
      cerr << "can't open " << filename << endl;
    } else {
      save_module(win, ofile);
      ofile.close();
    }
  }
  else
    MessageBox(fb_win, "Action cancelled"); 

  Am_Screen.Remove_Part(fb_win);
//  fb_win.Destroy();
}

void do_close_module(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  do_save(cmd);
  Am_Object win = ((Am_Object)cmd.Get_Owner().Get(Am_WINDOW)).Get_Owner();
  win.Remove_From_Owner();
//  win.Destroy();
}

void do_load(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object new_win = KL_Top_Win.Create("load_win");

  Am_Object fb_win = Am_Window.Create()
    .Set(Am_TITLE, "Load")
    .Set(Am_WIDTH, Am_Width_Of_Parts)
    .Set(Am_HEIGHT, Am_Height_Of_Parts)
    ;
  Am_Screen.Add_Part(fb_win);

  char filename[100], message_string[100];
  strcpy(message_string, "File ");
  if ( FileSelectBox(fb_win, filename) ) {
    strcat(message_string, filename);
    strcat(message_string, " was chosen.");
    MessageBox(fb_win, message_string);

    ifstream ifile;
    ifile.open(filename, ios::in);
    if (!ifile) {
      cerr << "can't open test" << endl;
    } else {
      load_module(new_win, ifile);
      Modules_Add(Am_Screen, new_win, (Am_String)new_win.Get(MODULE_NAME));
      Am_Initialize_Inspector (new_win.Get_Part(KL_CANVAS_PART));
    }
  }
  else
    MessageBox(fb_win, "Action cancelled"); 

  Am_Screen.Remove_Part(fb_win);
//  fb_win.Destroy();
}

void do_translate(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object inter = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR);
  Am_Object window = ((Am_Object)inter.Get(Am_WINDOW)).Get_Owner();
  char name[1000];
  ostrstream oname(name, 1000);
  oname << (Am_String)window.Get(MODULE_NAME) << ".kl1" << '\0';
  ofstream ofile;
  ofile.open(name, ios::out);
  if (!ofile) {
    cerr << "can't open test" << endl;
  } else {
    translate_module(window, ofile);
    ofile.close();
    // exec tracer
    pid_t child_pid;

    child_pid = fork();
    if (child_pid < 0) {
      perror("executing KLIEG tracer");
      return;
    }
    if (child_pid == 0) {
      execl(tracer_program, tracer_name, name, NULL);
      exit(1);
    }
  }
}

void do_exit (Am_Object /* cmd */)
{
  Am_Exit_Main_Event_Loop ();
}

Am_Define_Formula(int, menu_rule_type_form) {
  Am_Object button, menu;
  if ((button = self.GV_Owner()).Valid()) {
    if ((menu = button.GV_Owner()).Valid()) {
      Am_Object target = (Am_Object)menu.GV(TARGET_OBJECT);
      if (target.Valid())
	return target.GV(PROC_TYPE);
    }
  }
  return KL_Normal_Proc;
}

void init_top_menu(void)
{
  Am_Object create_button_proto = Popup_Button_Command.Create()
    .Set(Am_DO_ACTION, do_create)
    ;
  Top_Menu = Popup_Menu.Create("Top_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
 	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Top Menu"))
	 .Add(Am_Menu_Line_Command.Create("line"))
//	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Create..."))
	 .Add(create_button_proto.Create().Set(Am_LABEL, Process_Proto))
	 .Add(create_button_proto.Create().Set(Am_LABEL, Creator_Proto))
//	 .Add(create_button_proto.Create().Set(Am_LABEL, Pattern_Proto))
//	 .Add(create_button_proto.Create().Set(Am_LABEL, Sequence_Pattern_Proto.Set(IS_ICON, true)))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Save")
	      .Set(Am_DO_ACTION, do_save))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Load")
	      .Set(Am_DO_ACTION, do_load))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Execute")
	      .Set(Am_DO_ACTION, do_translate))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "New Module")
	      .Set(Am_DO_ACTION, do_new_module))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Close Module")
	      .Set(Am_DO_ACTION, do_close_module))
//     	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Propaties..."))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Quit")
	      .Set(Am_DO_ACTION, do_exit))
      )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Proc_Agg)
    .Get_Owner()
    ;
}

// -------- Process Menu --------

// Process Menu's DO_ACTIONs

void do_add_port(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object dest_proc = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Object port_proto = (Am_Object)cmd.Get(Am_LABEL);

  int x = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_X);
  int y = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_Y);

  Am_Object port = port_proto.Create();
  Port_Rename(port, ((int)port.Get(PORT_MODE) == KL_Input) ? "In" : "Out");
  Proc_Add_Port_At(dest_proc, port, x, y, dest_proc.Get_Owner());
}

void do_toggle_proc_type(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object dest_proc = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  int type = dest_proc.Get(PROC_TYPE);
  dest_proc.Set(PROC_TYPE, (type == KL_Normal_Proc)? KL_Creator_Proc : KL_Normal_Proc);
}

void do_toggle_io(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object dest_proc = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Value_List slots = Port_Values(dest_proc);
  Am_Object s;
  for (slots.Start(); !slots.Last(); slots.Next()) {
    s = slots.Get();
    s.Set(PORT_MODE, !(int)s.Get(PORT_MODE));
  }
}

void do_create_rule(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object scroll_agg = cmd.Get_Owner().Get_Owner().Get_Owner()
                                                    .Get_Part(KL_SCROLL_PART);
  Am_Object rules_agg = scroll_agg.Get_Part(KL_RULES_PART);
  Am_Object inter = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR);
  Am_Object proto = (Am_Object)cmd.Get(Am_LABEL);
  Am_Object new_rule = Rule_Create(cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT), proto);

  int x = (int)inter.Get(Am_FIRST_X) + (int)scroll_agg.Get(Am_X_OFFSET) + 100;
  int y = (int)inter.Get(Am_FIRST_Y) + (int)scroll_agg.Get(Am_Y_OFFSET) + 100;

  Add_Rule(rules_agg, new_rule.Set(Am_LEFT, x).Set(Am_TOP, y));
}

void do_copy_process(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object proc = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  int left = proc.Get(Am_LEFT);
  int top = proc.Get(Am_TOP);
  Add_Process(proc.Get_Owner(),
	      proc.Copy().Set(PROC_RULES, Am_Value_List())
	      .Set(Am_LEFT, left + 50).Set(Am_TOP, top + 50));
}

void do_dismiss_process(Am_Object cmd) {
  Am_Object dest_obj = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Remove_Process(dest_obj.Get_Owner(), dest_obj);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_proc_toggle_icon(Am_Object cmd)
{
  Am_Object proc = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Proc_Toggle_Icon(proc);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void init_process_menu(void)
{
  Am_Object add_button_proto = Popup_Button_Command.Create()
    .Set(Am_DO_ACTION, &do_add_port)
    ;

  Process_Menu = Popup_Menu.Create("Process_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Process"))
	 .Add(Am_Menu_Line_Command.Create("line"))
//	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Add Slot..."))
	 .Add(add_button_proto.Create().Set(Am_LABEL, In_Single_Proto))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Out_Single_Proto))
	 .Add(add_button_proto.Create().Set(Am_LABEL, In_Stream_Proto))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Out_Stream_Proto))
	 .Add(Am_Menu_Line_Command.Create("line"))
//	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Create Rule..."))
	 .Add(Popup_Button_Command.Create()
	      .Set(Am_LABEL, Continuous_Rule_Proto.Create()
		   .Get_Part(CONTENTS_PART).Get_Part(NAME_PART)
		     .Set(Am_TEXT, "Transition")
		   .Get_Owner().Get_Owner()
//		   .Set(PROC_NAME, "Transition Rule")
		   .Set(PROC_TYPE, Am_Formula::Create(menu_rule_type_form)))
	      .Set(Am_DO_ACTION, &do_create_rule))
	 .Add(Popup_Button_Command.Create()
	      .Set(Am_LABEL, Vanishing_Rule_Proto.Create()
		   .Get_Part(CONTENTS_PART).Get_Part(NAME_PART)
		     .Set(Am_TEXT, "Network")
		   .Get_Owner().Get_Owner()
//		   .Set(PROC_NAME, "Network Rule")
		   .Set(PROC_TYPE, Am_Formula::Create(menu_rule_type_form)))
	      .Set(Am_DO_ACTION, &do_create_rule))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "De/Iconify")
	      .Set(Am_DO_ACTION, &do_proc_toggle_icon))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Normal/Creator")
	      .Set(Am_DO_ACTION, &do_toggle_proc_type))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Copy")
	      .Set(Am_DO_ACTION, &do_copy_process))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_process))
      )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 2)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Proc)
    .Get_Owner()
    ;
}

// -------- Process Port Menu --------

// Process Port Menu's DO_ACTIONs

void do_dismiss_port(Am_Object cmd) {
  Am_Object port = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Proc_Remove_Port(Port_Parent_Proc(port), port);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_stream_dir_toggle(Am_Object cmd) {
  Am_Object port = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  if (port.Valid()) 
    port.Set(PORT_DIR, 1 - (int)port.Get(PORT_DIR));
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

Am_Define_Formula(bool, dir_toggle_active_form) {
  Am_Object menu = self.GV_Owner().GV_Owner();
  if (!menu.Valid()) return false;
  Am_Object target = (Am_Object)menu.GV(TARGET_OBJECT);
  if (!target.Valid()) return false;
  if ((int)target.GV(PORT_TYPE) == KL_Stream) return true;
  return false;
}

void init_process_port_menu()
{
  Process_Port_Menu = Popup_Menu.Create("Process_Port_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Process Port"))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Left/Right")
	      .Set(Am_DO_ACTION, &do_stream_dir_toggle)
	      .Set(Am_ACTIVE, Am_Formula::Create(dir_toggle_active_form)))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_port))
      )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 2)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Proc_Slot)
    .Get_Owner()
    ;
}

// -------- Rule Menu --------

// Rule Menu's DO_ACTIONs

void do_add_guard(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object window = (Am_Object)cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_WINDOW);
  Am_Object dest_rule = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);

  int clip_x = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_X);
  int clip_y = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_Y);
  Rule_Add_Guard_At(dest_rule, Guard_Proto.Create(), clip_x, clip_y, window);
}

void do_toggle_icon(Am_Object cmd)
{
  Am_Object rule = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Rule_Toggle_Icon(rule);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_dismiss_rule(Am_Object cmd) {
  Am_Object rule = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Remove_Rule(rule.Get_Owner(), rule);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_add_data(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object rule = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Object port_proto = (Am_Object)cmd.Get(Am_LABEL);

  Am_Object port = port_proto.Create();
  char *name_base = ((int)port.Get(PORT_MODE) == KL_Input) ? "In" : "Out";
  Port_Rename(port, name_base);
  Rule_Add_Data(rule, port);
}

void init_rule_menu(void)
{
/*
  Am_Object add_data_button_proto = Popup_Button_Command.Create()
    .Set(Am_DO_ACTION, &do_add_data)
    ;
  */

  Rule_Menu = Popup_Menu.Create("Rule_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Rule"))
	 .Add(Am_Menu_Line_Command.Create("line"))
//	 .Add(add_data_button_proto.Create().Set(Am_LABEL, Out_Stream_Proto))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Add Guard")
	      .Set(Am_DO_ACTION, &do_add_guard))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "De/Iconify")
	      .Set(Am_DO_ACTION, &do_toggle_icon))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_rule))
      )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 2)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Rule)
    .Get_Owner()
    ;
}

// -------- Rule Port Menu --------

// Rule_Port_Menu do actions

void do_add_value_port(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object dest_port = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Object rule = Port_Parent_Rule(dest_port);
  Am_Object port_proto = (Am_Object)cmd.Get(Am_LABEL);

  if (KL_Type(dest_port) == KL_Port)
  if ((int)dest_port.Get(PORT_SPECIAL) == KL_Map) {
      Seq_Port_Add(dest_port, port_proto.Create());
      return;
  }
  Am_Object port = port_proto.Create();
  char *name_base = ((int)port.Get(PORT_MODE) == KL_Input) ? "In" : "Out";
  Port_Rename(port, name_base);
  Rule_Add_Inner_Port(rule, dest_port, port);
}

void do_dismiss_value_port(Am_Object cmd)
{
  Am_Object dest_port = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Object rule = Port_Parent_Rule(dest_port);
  Am_Object parent = Port_Parent(dest_port);
  if (KL_Type(parent) == KL_Pattern) Pattern_Remove_Port(parent, dest_port);
  else if (KL_Type(parent) == KL_Hole) Hole_Remove_Port(parent, dest_port);
  else Port_Remove(parent, dest_port);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_toggle_open(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object dest_slot = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  if (dest_slot.Is_Instance_Of(Port_Proto)) 
    dest_slot.Set(PORT_OPEN, 1 - (int)dest_slot.Get(PORT_OPEN));
}

Am_Define_Formula(bool, dismiss_active_form) {
  Am_Object menu = self.GV_Owner().GV_Owner();
  if (!menu.Valid()) return true;
  Am_Object target = (Am_Object)menu.GV(TARGET_OBJECT);
  if (!target.Valid()) return true;
  Am_Object parent = Port_Parent(target);
  if (KL_Type(parent) == KL_Process || KL_Type(parent) == KL_Rule) {
    return false;
  }
  return true;
}

Am_Define_Formula(bool, toggle_open_active_form) {
  Am_Object menu = self.GV_Owner().GV_Owner();
  if (!menu.Valid()) return false;
  Am_Object target = (Am_Object)menu.GV(TARGET_OBJECT);
  if (!target.Valid()) return false;
  if (target.Is_Instance_Of(Port_Proto)) {
    if ((int)target.Get(PORT_TYPE) == KL_Stream) {
/*      self.Get_Owner().Set(Am_VISIBLE, true); */
      return true;
    }
  }
/*  self.Get_Owner().Set(Am_VISIBLE, false); */
  return false;
}

Am_Define_Formula(bool, add_port_active_form) {
  // menu $B$O$"$k$+!%(B
  Am_Object menu = self.GV_Owner().GV_Owner();
  if (!menu.Valid()) return false;
  // target $B$O$"$k$+!%(B
  Am_Object target = (Am_Object)menu.GV(TARGET_OBJECT);
  if (!target.Valid()) return false;
  // src $B$O2?$+!%(B
  Am_Object src = (Am_Object)self.GV(Am_LABEL);
  if (Port_Add_Allowed(src, target)) {
    self.Get_Owner().Set(Am_VISIBLE, true);
    return true;
  }
  self.Get_Owner().Set(Am_VISIBLE, false);
  return false;
}

/* ----------------------------------------
Am_Define_Formula(bool, add_single_active_form) {
  Am_Object menu = self.GV_Owner().GV_Owner();
  if (!menu.Valid()) return false;
  Am_Object target = (Am_Object)menu.GV(TARGET_OBJECT);
  if (!target.Valid()) return false;
  if (target.Is_Instance_Of(Atom_Proto)) {
    self.Get_Owner().Set(Am_VISIBLE, false);
    return false;
  }
  int mode = ((Am_Object)self.GV(Am_LABEL)).GV(PORT_MODE);
  if ((int)target.GV(PORT_TYPE) == KL_Single
      && !(mode == KL_Output
	   && Port_Parent(target).Is_Instance_Of(Process_Proto)
	   && (int)target.GV(PORT_MODE) == KL_Output)
      && !(mode == KL_Output
	   && Port_Parent(target).Is_Instance_Of(Rule_Proto)
	   && (int)target.GV(PORT_MODE) == KL_Input)) {
    self.Get_Owner().Set(Am_VISIBLE, false);
    return false;
  }
  self.Get_Owner().Set(Am_VISIBLE, true);
  return true;
}

Am_Define_Formula(bool, add_stream_active_form) {
  Am_Object menu = self.GV_Owner().GV_Owner();
  if (!menu.Valid()) return false;
  Am_Object target = (Am_Object)menu.GV(TARGET_OBJECT);
  if (!target.Valid()) return false;
  if (target.Is_Instance_Of(Atom_Proto)) {
    self.Get_Owner().Set(Am_VISIBLE, false);
    return false;
  }
  if ((int)target.GV(PORT_TYPE) == KL_Single) {
    self.Get_Owner().Set(Am_VISIBLE, false);
    return false;
  }
  self.Get_Owner().Set(Am_VISIBLE, true);
  return true;
}
---------------------------------------- */

void init_rule_port_menu(void)
{
  Am_Object add_value_button_proto = Popup_Button_Command.Create()
    .Set(Am_DO_ACTION, &do_add_value_port)
    ;

  Rule_Port_Menu = Popup_Menu.Create("Rule_Port_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Rule Port"))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Left/Right")
	      .Set(Am_DO_ACTION, &do_stream_dir_toggle)
	      .Set(Am_ACTIVE, Am_Formula::Create(dir_toggle_active_form)))
	 .Add(Am_Menu_Line_Command.Create("line"))
//	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Add..."))
	 .Add(add_value_button_proto.Create().Set(Am_LABEL, Atom_Proto)
	      .Set(Am_ACTIVE, Am_Formula::Create(add_port_active_form)))
	 .Add(add_value_button_proto.Create().Set(Am_LABEL, Out_Single_Proto)
	      .Set(Am_ACTIVE, Am_Formula::Create(add_port_active_form)))
	 .Add(add_value_button_proto.Create().Set(Am_LABEL, Structure_Proto)
	      .Set(Am_ACTIVE, Am_Formula::Create(add_port_active_form)))
	 .Add(add_value_button_proto.Create().Set(Am_LABEL, Out_Stream_Proto)
	      .Set(Am_ACTIVE, Am_Formula::Create(add_port_active_form)))
	 .Add(add_value_button_proto.Create().Set(Am_LABEL, In_Single_Proto)
	      .Set(Am_ACTIVE, Am_Formula::Create(add_port_active_form)))
	 .Add(add_value_button_proto.Create().Set(Am_LABEL, In_Stream_Proto)
	      .Set(Am_ACTIVE, Am_Formula::Create(add_port_active_form)))
	 .Add(Am_Menu_Line_Command.Create("line"))
 	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Open/Close")
 	      .Set(Am_DO_ACTION, &do_toggle_open)
 	      .Set(Am_ACTIVE, Am_Formula::Create(toggle_open_active_form)))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_value_port)
	      .Set(Am_ACTIVE, Am_Formula::Create(dismiss_active_form)))
      )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 4)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Menu_Goal_Or_Rule_Slot)
    .Get_Owner()
    ;
}

// -------- Goal Menu --------

void do_goal_toggle_icon(Am_Object cmd)
{
  Am_Object goal = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Proc_Toggle_Icon(goal);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_dismiss_goal(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object dest_goal = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Remove_Pattern(dest_goal); // hole $B$+$i%Q%?!<%s$r>C$9$N$HF1$8!%(B
}

void init_goal_menu(void)
{
  Goal_Menu = Popup_Menu.Create("Goal_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Goal Menu"))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "De/Iconify")
	      .Set(Am_DO_ACTION, &do_goal_toggle_icon))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_goal))
	 )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 3)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Goal)
    .Get_Owner()
    ;
}

// -------- Binder Menu --------

void do_dismiss_binder(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object binder = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Binding_Line_Remove(binder);
}

void init_binder_menu(void)
{
  Binder_Menu = Popup_Menu.Create("Binder_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Binder"))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_binder))
	 )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 3)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Binder)
    .Get_Owner()
    ;
}

// -------- Guard Menu --------

void do_dismiss_guard(Am_Object cmd)
{
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object guard = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  guard.Remove_From_Owner();
}

void init_guard_menu(void)
{
  Guard_Menu = Popup_Menu.Create("Guard_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Guard"))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_guard))
	 )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 3)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Guard)
    .Get_Owner()
    ;
}

// -------- Pattern Menu --------

void do_add_port_to_pattern(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object rule = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Object win = (Am_Object)rule.Get(Am_WINDOW);
  Am_Object port_agg = Proc_Ports_Part(rule);
  Am_Object port_proto = (Am_Object)cmd.Get(Am_LABEL);
  Am_Object port = port_proto.Create();

  int x = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_X);
  int y = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_Y);

  Am_Translate_Coordinates(win, x, y, port_agg, x, y); // $B:BI8$NJQ49!%(B
  x = imax(0, x);
  y = imax(0, y);
  port.Set(Am_LEFT, x).Set(Am_TOP, y);

  Port_Rename(port, ((int)port.Get(PORT_MODE) == KL_Input) ? "In" : "Out");
  Pattern_Add_Port(rule, port);
}

void do_copy_pattern(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object pattern = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  int left = pattern.Get(Am_LEFT);
  int top = pattern.Get(Am_TOP);
  Am_Object new_pattern;
  Am_Function_Call(Copy_Method, pattern, UNLINK_COPY, new_pattern, (pattern));
  Add_Pattern(pattern.Get_Owner(),
	      new_pattern.Set(Am_LEFT, left + 50).Set(Am_TOP, top + 50));
}

void do_dismiss_pattern(Am_Object cmd) {
  Am_Object dest_obj = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Remove_Pattern(dest_obj);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_add_hole(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object pattern = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  int x = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_X);
  int y = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_Y);
  Pattern_Add_Hole_At(pattern, Hole_Proto.Create(), x, y, pattern.Get_Owner());
}

void do_merge_pattern(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void do_rotate_pattern(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object pattern = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  if (pattern.Is_Instance_Of(Sequence_Pattern_Proto))
    Seq_Pattern_Rotate(pattern);
}

void do_iconify_pattern(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object pattern = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  pattern.Set(IS_ICON, !pattern.Get(IS_ICON));
}

void init_pattern_menu(void)
{
  Am_Object add_button_proto = Popup_Button_Command.Create()
    .Set(Am_DO_ACTION, &do_add_port_to_pattern)
    ;

  Pattern_Menu = Popup_Menu.Create("Pattern_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Pattern"))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, Hole_Proto.Create())
	      .Set(Am_DO_ACTION, &do_add_hole))
	 .Add(Am_Menu_Line_Command.Create("line"))
//	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Add Slot..."))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Pat_In_Single_Port.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Pat_Out_Single_Port.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Pat_In_Stream_Port.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Pat_Out_Stream_Port.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Map_In_Port))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Map_Out_Port.Create()))
//	 .Add(add_button_proto.Create().Set(Am_LABEL, First_Item_Single_Port.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Broadcast_Single_Port.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Broadcast_Stream_Port.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Merge_Port.Create()))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Am_Menu_Line_Command.Create("line"))
//	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Merge")
//	      .Set(Am_DO_ACTION, &do_merge_pattern))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "De/Iconify")
	      .Set(Am_DO_ACTION, &do_iconify_pattern))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Rotate")
	      .Set(Am_DO_ACTION, &do_rotate_pattern))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Copy")
	      .Set(Am_DO_ACTION, &do_copy_pattern))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_pattern))
      )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 2)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Pattern)
    .Get_Owner()
    ;
}

// -------- Hole Menu --------

void do_add_port_to_hole(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object hole = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Object win = (Am_Object)hole.Get(Am_WINDOW);
  Am_Object port_agg = Proc_Ports_Part(hole);

  Am_Object port_proto = (Am_Object)cmd.Get(Am_LABEL);
  Am_Object port = port_proto.Create();

  int x = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_X);
  int y = cmd.Get_Owner().Get_Owner().Get_Part(Am_INTERACTOR).Get(Am_FIRST_Y);

  Am_Translate_Coordinates(win, x, y, port_agg, x, y); // $B:BI8$NJQ49!%(B
  x = imax(0, x);
  y = imax(0, y);
  port.Set(Am_LEFT, x).Set(Am_TOP, y);

  Port_Rename(port, ((int)port.Get(PORT_MODE) == KL_Input) ? "In" : "Out");
  Hole_Add_Port(hole, port);
}

void do_copy_hole(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object hole = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Am_Object pattern = Hole_Parent_Pattern(hole);
  int left = hole.Get(Am_LEFT);
  int top = hole.Get(Am_TOP);
  Am_Object new_hole;
  Am_Function_Call(Copy_Method, hole, PARENT_COPY, new_hole, (hole));
  Pattern_Add_Hole_At(pattern, new_hole, left + 30, top + 30, hole.Get_Owner());
}

void do_iconify_hole(Am_Object cmd) {
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
  Am_Object hole = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  hole.Set(IS_ICON, !hole.Get(IS_ICON));
}

void do_dismiss_hole(Am_Object cmd) {
  Am_Object dest_obj = (Am_Object)cmd.Get_Owner().Get_Owner().Get(TARGET_OBJECT);
  Pattern_Remove_Hole(Hole_Parent_Pattern(dest_obj), dest_obj);
  Am_Call(Am_Object_Proc, Popup_Button_Command, Am_DO_ACTION, (cmd));
}

void init_hole_menu(void)
{
  Am_Object add_button_proto = Popup_Button_Command.Create()
    .Set(Am_DO_ACTION, &do_add_port_to_hole)
    ;

  Hole_Menu = Popup_Menu.Create("Hole_Menu")
    .Set(Am_ITEMS, Am_Value_List()
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Hole"))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "De/Iconify")
	      .Set(Am_DO_ACTION, &do_iconify_hole))
//	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Add Slot..."))
	 .Add(add_button_proto.Create().Set(Am_LABEL, In_Single_Proto.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Out_Single_Proto.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, In_Stream_Proto.Create()))
	 .Add(add_button_proto.Create().Set(Am_LABEL, Out_Stream_Proto.Create()))
	 .Add(Am_Menu_Line_Command.Create("line"))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Copy")
	      .Set(Am_DO_ACTION, &do_copy_hole))
	 .Add(Popup_Button_Command.Create().Set(Am_LABEL, "Dismiss")
	      .Set(Am_DO_ACTION, &do_dismiss_hole))
      )
    .Get_Part(Am_INTERACTOR)
      .Set(Am_PRIORITY, 2)
      .Set(Am_START_WHERE_TEST, (Am_Object_Proc*)&KL_In_Hole)
    .Get_Owner()
    ;
}


void Initialize_Tools(void)
{
  init_menu_bar();
  init_top_menu();
  init_process_menu();
  init_process_port_menu();
  init_rule_menu();
  init_rule_port_menu();
  init_goal_menu();
  init_binder_menu();
  init_guard_menu();
  init_pattern_menu();
  init_hole_menu();

  KL_Menu_Agg.Add_Part(KL_MODULE_NAME, KL_Module_Name);
  KL_Menu_Agg.Add_Part(KL_SPLIT_BAR, KL_Split_Bar);

  KL_Top_Win.Set(MODULE_NAME, Am_Formula::Create(top_win_module_name_form));
  KL_Process_Agg.Set(MODULE_NAME, Am_Formula::Create(proc_agg_module_name_form));

  KL_Canvas_Win.Add_Part(TOP_MENU, Top_Menu);
  KL_Canvas_Win.Add_Part(PROCESS_MENU, Process_Menu);
  KL_Canvas_Win.Add_Part(PROCESS_PORT_MENU, Process_Port_Menu);
  KL_Canvas_Win.Add_Part(RULE_MENU, Rule_Menu);
  KL_Canvas_Win.Add_Part(RULE_PORT_MENU, Rule_Port_Menu);
  KL_Canvas_Win.Add_Part(GOAL_MENU, Goal_Menu);
  KL_Canvas_Win.Add_Part(BINDER_MENU, Binder_Menu);
  KL_Canvas_Win.Add_Part(GUARD_MENU, Guard_Menu);
  KL_Canvas_Win.Add_Part(PATTERN_MENU, Pattern_Menu);
  KL_Canvas_Win.Add_Part(HOLE_MENU, Hole_Menu);
}  


/* ---- Main Initialize routine ---- */

void Klieg_Initialize(void)
{
  Initialize_FileSelectBox();
  Initialize_KL_Widgets();
  Initialize_Pattern();
  Initialize_Windows();
  Initialize_Translator();
  Initialize_Interactors();
  Initialize_Tools();
}

/* ---------- Main Routine ---------- */

main (void)
{
  Am_Initialize();
  My_Amulet_Initialize();
  Klieg_Initialize();

  Am_Object main_win = KL_Top_Win.Create("main_win");
  Modules_Add(Am_Screen, main_win, "unnamed");

  Am_Initialize_Inspector (main_win.Get_Part(KL_CANVAS_PART));
  Am_Main_Event_Loop ();
}
