% Copyright (C) 1996   ͵

/********************************/
/*  Utility Programs in Prolog  */
/********************************/

no_gc_time.
no_ss_time.

/*  not  */

not(P) :-
   P,!,
   fail.
not(_).

/*  a subset of a list  */

subset([],_).
subset([A|X],Y) :-
   member(A,Y),
   subset(X,Y).

/*  the intersection of two lists  */

intersect([],_,[]) :- !.
intersect(_,[],[]) :- !.
intersect([A|B],L,[A|C]) :-
   member(A,L),!,
   intersect(B,L,C).
intersect([_|B],L,C) :-
   intersect(B,L,C).

intersect1([A|_],L) :-
   member(A,L),!.
intersect1([_|B],L) :-
   intersect1(B,L).

/*  the union of two lists  */

union([],X,X).
union(X,[],X).
union([X|R],Y,Z) :-
   member(X,Y),
   union(R,Y,Z).
union([X|R],Y,[X|Z]) :-
   union(R,Y,Z).

/*  the reverse of a list  */

:- public reverse/2.
:- mode reverse(+,-).

/*  equal lists  */

equal_lists([],[]) :- !.
equal_lists([A|B],[A|C]) :-
   equal_lists(B,C).

/*  create dlist  */

create_dlist([A|X],X,[A|U],U) :- !.
create_dlist([A|X],Y,[A|U],V) :-
   create_dlist(X,Y,U,V).

/*  gensym  */

gensym(X) :-
   gensym(X,$).
gensym(X,S) :-
 ( symbol(S,N),!;
   assert(symbol(S,1)),
      N=1 ),
   name(N,NUM),
   name(S,SYM),
   append(SYM,NUM,CONST),
   name(X,CONST),!,
   incr(S).

incr(S) :-
   retract(symbol(S,N)),
   N1 is N+1,
   assert(symbol(S,N1)),!.

init_symbol :-
   abolish(symbol,2).

/*  the depth of a list  */

:- mode depth(+,-).

depth([],1) :- !.
depth(L,0) :-
   atomic(L),!.
depth([L|R],N) :-
   depth(L,Ln),
   depth(R,Rn),!,
 ( Ln<Rn,!,
      N is Rn;
   N is Ln+1 ).

/*  reduce  */

reduce(X,X):- atom(X).
reduce([[]|L1],L2):- reduce(L1,L2).
reduce([L1|L2],[L3|L4]):- reduce(L1,L3),reduce(L2,L4).

/*  flatten  */
/*
flatten([],[]) :- !.
flatten([[]|T],T1) :- !,
   flatten(T,T1).
flatten([H|T],[H|T1]) :-
   atomic(H),!,
   flatten(T,T1).
flatten([[HH|HT]|T],T1) :-
   flatten([HH,HT|T],T1).
*/
flatten(B/S,S1):-!,
	flatten1(B/S,[],L),
	flatten2(L,S1).
flatten(S,S).

flatten1(B/S,L,L1):-!,
	( S=[_|_],!,
	    flatten1(B,[S|L],L1)
	; flatten1(S,L,L0),
	  flatten1(B,L0,L1) ).
flatten1(S,L,[S|L]).

flatten2([S],S):-!.
flatten2([S|L],B/S):-
	flatten2(L,B).

/*  bell  */

:- public bell/1.

bell(0) :- !.
bell(N) :- put(7),N1 is N-1,bell(N1).

/*  print list contents  */

printList([]) :- !,
   nl.
printList([','|B]) :- !,
   display(','),
   printList(B).
printList(['.'|B]) :- !,
   display('.'),
   printList(B).
printList([A|B]) :-
   display(' '),
   display(A),
   printList(B).

/* time measuring program for SICStus Prolog */

time(Type,Proc) :-
	statistics(runtime,[Rn_base,_]),
	statistics(garbage_collection,[_,_,Gc_base]),
	statistics(stack_shifts,[_,_,Ss_base]),
	assert(gc_time(Gc_base)),
	assert(rn_time(Rn_base)),
	assert(ss_time(Ss_base)),
      (	call(Proc) ;
	write(failed) ),
	statistics(runtime,[Rn_end,_]),
	statistics(garbage_collection,[_,_,Gc_end]),
	statistics(stack_shifts,[_,_,Ss_end]),
	Gc_time is Gc_end - Gc_base,
	Ss_time is Ss_end - Ss_base,
	Rn_time is Rn_end - Rn_base, nl,
      (
	  no_ex_time
      ;
	  write('time for '), write(Type), write(' = '),
	  write(Rn_time), write(' msec'), nl
      ),
      ( no_gc_time ;
	write('garbage collection time = '),
	write(Gc_time), write(' msec'), nl ),
      ( no_ss_time ;
	write('stack shifting time     = '),
	write(Ss_time), write(' msec'), nl ),
	abolish(gc_time,1),
	abolish(rn_time,1),
	abolish(ss_time,1), nl, !.

initialize_time :-
	write('Do you need the execution time?      '), ttyflush,
	read(EX),
      ( EX = y ; EX = yes ; assert(no_ex_time) ),
	write('Do you need garbage collection time? '), ttyflush,
	read(GC),
      ( GC = y ; GC = yes ; assert(no_gc_time) ),
	write('Do you need stack shifting time?     '), ttyflush,
	read(SS),
      ( SS = y ; SS = yes ; assert(no_ss_time) ).

% indent
indent(X) :- 
   indent(X,[],last).
indent(X,N,_) :- 
   atom(X),
   prline(N),
   printf(X),nlf,!.
indent([X|L],N,H) :-
   abolish(p,1),
   ( L=_/_,!,assert(p(packed))
   ; assert(p(not_packed)) ),
   atom(X),
   prline(N),
   printf(X),!,
   ( L=[Y/_],
   ( H==last,
     append(N,['   '],NN)
     ;
     append(N,[' \ '],NN)
   ),
   mapidt(L,NN)
 ; L=[Y],
   atom(Y),
   printf(' -- '),
   printf(Y),nlf 
   ;
   nlf,
   ( H==last,
     append(N,['   '],NN)
     ;
     ( p(packed),!,
      append(N,[' \ '],NN)
     ; append(N,[' | '],NN) )
   ),
   mapidt(L,NN)
 ),!.
indent(X,N,H) :-
   X=..Y,
   indent(Y,N,H).
indent(X,N,_) :- 
   mapidt(X,N),!.

% nlf
nlf :-
   ttynl,
   ( to_file,nl ; true),!.

% printf
printf(A) :-
   display(A),
   ( to_file,print(A) ; true),!.

% prline
prline([]) :-
   ( p(packed),!,
       printf(' \-')
   ; printf(' |-') ).
prline([X|L]) :- 
   printf(X),
   prline(L).


% mapidt
mapidt([],_).
mapidt([X],N) :- 
 ( X=[] ;  indent(X,N,last) ),!.
mapidt([X|L],N) :- 
 ( X=[]
   ;
   ( L=[],indent(X,N,last) 
     ;
     indent(X,N,middle)
   )
 ),!,
 mapidt(L,N).
mapidt(X/L,N):-
   mapidt(X,N),
   mapidt(L,N).

% count_stack
count_stack(N,N,[]):-!.
count_stack(N,M,[A/_|Y]) :-!,
	N1 is N+1,
	count_stack(N1,M,[A|Y]).
count_stack(N,M,[_|Y]) :-
	N1 is N+1,
	count_stack(N1,M,Y).

% disp_tree
disp_tree(N,N,[]):-!.
disp_tree(N,M,[[T|_]|Y]) :-
	show(T,n,[],last),
	ttynl,
	display('Argument Information:'),nl,
	( output_result(_),! ; true ),
	ttynl,
	N1 is N+1,
	disp_tree(N1,M,Y).

% output_result
output_result([]) :-!.
output_result(A) :-!,
  nl,
% numbervars(A,0,_),
%	write(A),nl.
  A=[(SYN,_,REL,(F0,F1),_)],
%	write(SYN),nl,
%	write(REL),nl,
%	write(F),nl.
  write_str({SYN,REL,F0-F1},0).

nest(0) :- !.
nest(N) :-
  write('|  '),
  N1 is N - 1,
  nest(N1).

write_str({SYN,REL,F},L) :-
  L2 is L+2,
  nest(L), write('+==='),nl,
  nest(L),write('| S+---'),nl,
  write_feature(SYN,L2),
  nest(L),write('|  +---'),nl,
  nest(L),write('| R+---'),nl,
  write_feature(REL,L2),
  nest(L),write('|  +---'),nl,
  nest(L),write('| F:'),
  write(F),nl,
  nest(L), write('+==='),nl.

write_feature([],_) :- !.
write_feature([{SYN,REL,F0,F1}|Z],L) :- !,
  write_str({SYN,REL,F0-F1},L),
  write_feature(Z,L).
write_feature([X|Z],L) :-
  nest(L),
  write(X),nl,
  write_feature(Z,L).


show([Nt|T],P,X,H):-	% T=[[det,the],[n,door]]/[[pron,i]]
	( X==[],!
	; displ(X) ),
	( P==p,!,
	   display(' \-'),!
	; display(' |-') ),
	display(Nt),
	( T=[T1],atom(T1),!
	; T=T1/[T2],atom(T2),!
	; ttynl ),
	( H==last,!,append(X,['   '],NN)
	; P==p,!,append(X,[' \ '],NN)
	; append(X,[' | '],NN) ),
	showt(T,n,NN).

showt([T],_,_):- atom(T),!,
	display(' -- '),display(T),ttynl.
showt(T/T1,_,X):-!,
	showt(T1,p,X),
	showt(T,p,X).
showt([T],P,X):-!,
	show(T,P,X,last).
showt([T|R],P,X):-!,
	show(T,P,X,middle),
	showt(R,P,X).

displ([X|Y]):-!,
	display(X),
	displ(Y).
displ([]).
displ(X):-
	display(X).
