/* Copyright (C) 1996   */
/*-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 *     mlr.c  ---  SLR/CLR table constructor to incorporate the connection
 *     ~~~~~       constraints into the LR table
 *       
 *       Tanaka & Tokunaga Lab.,
 *       Dept. of Computer Science,
 *       Tokyo Institute of Techmology
 *      
 *       Tel: 03-5734-2831,  E-mail: li@cs.titech.ac.jp
 *
 *       3-April-1995
 *-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 */


#include <stdio.h>
#include <time.h>
#include "config.h"
#include "LR.h"

#define FALSE		0
#define TRUE		1
#define DUMMY		0
#define MAXVN		MAXSYMBOLS  /* Must be a prime number. */
#define EOR		(-1)        /* End of rules            */
#define MAXITEMS	2*MAXSTATES
#define streq(s,t)	(*(s)==*(t) && strcmp(s,t)==0)
#define CheckOption(opt)  if(argc - 1 <= 0 || **(argv+1) == '-')\
			  fatal_error("! Missing value after %s option\n",opt)
long     cputime ( ) ;
     
typedef struct VN
{
    char	*name;	/* nonterminal */
    int	*rules;	/* indices to GrammarTable */
}   VN;

VN	*VN_Table[MAXVN];

typedef struct _Item
{
    int	ruleno;
    int	dot;
#if CLR
    int lookahead;
#endif	
    struct _Item	*next;
}   Item;

typedef struct _Gotos
{
    int	symbol;
    Item	*item;
    struct _Gotos	*next;
}   Gotos;

int	verbose = 0;

usage(progname)
char	*progname;
{
    fprintf(stderr, "Usage: %s [-g InputGrammarFile] [-c ConnectMatrixFile] [-o OutputTableFile] [-v]\n", progname);
    exit();
}

main(argc, argv)
int	argc;
char	*argv[];
{
    int	    n, i, k;
    int	    nonterminals, terminals;
    char    *progname, grammar[80], infile[80], confile[80], outfile[80];
    FILE    *infp, *outfp;

    progname = argv[0];
    *grammar = *infile = *confile = *outfile = NULL;

    if(argc < 2)
      {
	  usage(progname);
      }
	
    while(--argc)
      {
	  if(**++argv == '-')
	    {
		switch((*argv)[1])
		  {
		    case 'h':
		      usage(progname);
		    case 'g':
		      CheckOption("g");
		      strcpy(infile, *(argv+1));
		      --argc;
		      ++argv;
		      break;
		    case 'o':
		      CheckOption("o");
		      strcpy(outfile, *(argv+1));
		      --argc;
		      ++argv;
		      break;
		    case 'c':
		      CheckOption("c");
		      strcpy(confile,*(argv+1)) ;
		      --argc;
		      ++argv;
		      break;
		    case 'v':
		      ++verbose;
		      break;
		    default:
		      fatal_error("! Unknown option : %c\n", (*argv)[1]);
		  }
	    }
	  else
	    strcpy(grammar, *argv);
      }
    cputime ( ) ;
	
    if(!*grammar && !*infile)
      fatal_error("No grammar name.\n");
    if(!*infile)  sprintf(infile, "%s.gra", grammar);
    if(!*confile) sprintf(confile, "%s.con", grammar);
    if(!*outfile) sprintf(outfile, "%s.tab", grammar);
    if((infp = fopen(infile, "r")) == NULL)
      fatal_error("Can't open %s (READ)\n", infile);
    init_LR();
    InstallSymbol("$");
	
    /*
     * Read grammar.
     */
	
    NumRules = ReadRules(infp);
    fprintf(stderr, "\nRead %d rules.\n", NumRules);
    if(!streq(SymbolTable[GrammarTable[0]->lhs], "<start>"))
      fatal_error("First rule not include <start>\n");

    terminals = 0;
    nonterminals = 0;
    for( i = 0; i < MAXSYMBOLS; i++)
      if( SymbolTable[i] != NULL)
	{
	    if(SymbolTable[i][0] == '<')
	      nonterminals++;
	    else
	      terminals++;
	}
    fprintf(stderr, "(%d nonterminal symbols, %d terminal symbols)\n\n",
	    nonterminals, terminals);

    cputime2();
#if Make_Extended_Matrix
    fprintf(stderr, "Read Connection Matrix ...\n\n");
    ReadConTable(confile);
#else
    fprintf(stderr, "Read Extended Connection Matrix ...\n\n");
    ReadConTable(confile);
#endif
    fprintf(stderr,"Done. CPU time for Read Connection Table: %ld seconds\n\n", cputime2());
    /*
     * Construct first & follow tables.
     */

    cputime2();
    init_Ftable();
    fprintf(stderr, "Constructing First Table.\n");
    ConstFirst();
    fprintf(stderr,"Done. CPU time for First Table: %ld seconds\n\n",
	    cputime2());
#if test
    PrintFirst();	
#endif

#if SLR
    fprintf(stderr, "Constructing Follow Table.\n");
    ConstFollow();
    fprintf(stderr,"Done. CPU time for Follow Table: %ld seconds\n\n",
	    cputime2());
#if test
    PrintFollow();	
#endif
#endif

    fprintf(stderr, "Constructing last table.\n");
    ConstLast();
    fprintf(stderr, "Done. CPU time for Last Table: %ld seconds\n\n",
	    cputime2());
#if test
    PrintLast();
#endif
#if Make_Extended_Matrix
    fprintf(stderr, "Constructing Extended Connection Matrix...\n");
    InstallConnect2();
    fprintf(stderr, "Done. CPU time for Extended Matrix = %ld seconds\n\n",
	    cputime2());

#if test
    PrintConnect();
#endif
#endif
    
    /*
     * Construct LR parsing table.
     */
    
#if CLR
    fprintf(stderr, "Constructing canonical LR(1) collection.\n");
#endif
#if SLR	
    fprintf(stderr, "Constructing canonical LR(0) collection.\n");
#endif
    n = LRconst();
    fprintf(stderr, "Done. CPU time for LR collection = %ld seconds\n",
	    cputime2());
#if CPM	
    cpm(n);
    fprintf(stderr, "Done. CPU time for CPM = %ld seconds\n\n",
	    cputime2());
#endif

    /*
     * Write LR table.
     */

  write_lr:
    if((outfp = fopen(outfile, "w")) == NULL)
      {
	  error("*** Can't open %s (WRITE)\n", outfile);
	  printf("Another File: "); fflush(stdout);
	  scanf("%s", outfile);
	  goto write_lr;
      }
    fprintf(stderr, "Writing LR-table to %s. ", outfile);

    if (1){
	long  time(), tt;
	char  *ctime();
	time(&tt);
	fprintf(outfp, ";;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-\n");
	fprintf(outfp, ";;\n");
#if SLR
	fprintf(outfp, ";;     SLR parsing table\n");
#endif
#if CLR
	fprintf(outfp, ";;     CLR parsing table\n");
#endif
	fprintf(outfp, ";;     ~~~~~~~~~~~~~~~~~~\n");
	fprintf(outfp, ";;     Created on %s", ctime(&tt));
	fprintf(outfp, ";;\n");
	fprintf(outfp, ";;-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-\n");
#if SLR
	fprintf(outfp, "(slr-table)\n");
#endif
#if CLR
	fprintf(outfp, "(clr-table)\n");
#endif
    }
#if CPM
    WriteLR_cpm(outfp, n);
#else
    fprintf(stderr, "[%d states]\n", n);
    WriteLR(outfp, n);
#endif
    fclose(infp);
    fclose(outfp);
    fprintf(stderr, "Total CPU time = %ld seconds\n",cputime());
}

LRconst()
{
    register int	i, j, n = 0;
    int	    symbol, state, nfollows, *follows;
    int     predotsym;
    Item    *C[MAXSTATES], *C0, *top, *item, *newitem,
            *Closure(), *Goto(), *sortitem(), *copyitem(), *getitem();
    Gotos   *gotos, *gp, *InstallGotos();

    /*
     * Construct canonical LR(0)/LR(1) collection.
     */
	
    item = getitem();
    item->ruleno = 0;
    item->dot = 0;
#if CLR
    item->lookahead = HashSymbol("$");
#endif
    item->next = NULL;

    /*C[n++] = Closure(item);*/

    C[n++] = item;
    for(i = 0; i < n; i++)
      {
	  C0 = copyitem(C[i]);
	  C0 = Closure(C0);

	  if(verbose)
	    {
		printf("*** I %d ***\n", i);
		print_item(C0);
	    }

	  gotos = NULL;

	  for(top = item = C0; item != NULL; item = item->next)
	    {
		if(item->dot >= GrammarTable[item->ruleno]->length)
		
		  /* dot is rightmost position. */
		  
		  {
		      predotsym = GrammarTable[item->ruleno]->rhs[item->dot-1];
			  
		      if(item->ruleno == 0)  /* accept */
			makeaction(i, 'A', DUMMY, DUMMY);
		      else {                   /* reduce */
#if CLR
			  nfollows =
			    First(item->lookahead, &follows);
#endif			
#if SLR
			  nfollows =
			    Follow(GrammarTable[item->ruleno]->lhs, &follows);
#endif
			  for(j = 0; j < nfollows; j++)
			    makeaction(i, 'R', follows[j], item->ruleno);
		      }
		  }
		else /* shift and goto */
		  {
		      gotos = InstallGotos(gotos, item);
		  }
	    }

	  for(gp = gotos; gp != NULL; gp = gp->next)
	    {
		newitem = gp->item;
		symbol = gp->symbol;

		newitem = sortitem(newitem);
		if((state = included(newitem, C, n)) == -1)
		  C[n++] = newitem;
		state = (state == -1 ? n-1 : state);

		if(nonterminal(symbol)) /* goto */
		  makeaction(i, 'G', symbol, state);
		else /* shift */
		  makeaction(i, 'S', symbol, state);

		if(n >= MAXSTATES)
		  fatal_error("Overflow: MAXSTATES\n");
	    }
#if 0
	  check_action_table(i);
#endif
      }
    return(n);
}

makeaction(state, id, symbol, action)
int	state, id, symbol, action;
{
    register Action	*p;
    char	*input;

    switch(id)
      {
	case 'S':  /*SHIFT*/
	  input = SymbolTable[symbol];
	  break;
	case 'R':  /*REDUCE*/
	  action += 2*MAXSTATES;
	  input = SymbolTable[symbol];
	  break;
	case 'A':  /*ACCEPT*/
	  action = ACCEPT;
	  input = "$";
	  break;
	case 'G':  /*GOTO*/
	  action += MAXSTATES;
	  input = SymbolTable[symbol];
	  break;
      }
    for(p = ActionTable[state]; p != NULL; p = p->next)
      {
	  if(p->action == action
	     && strcmp(SymbolTable[p->input], input) == 0)
	    break;
      }
    if(!p)
      {
	  InstallAction(state, input, action);
      }
}

static	int     added[MAXGRAMMARS];

#if CLR
static	int     lookahead[MAXGRAMMARS];
#endif

Item	*Closure(I)
Item	*I;
{
    register int   i;
    int	           ap = 0, dotsym, nextdotsym;
    Item	   *top, *item, *newitem, *getitem(), *sortitem();
    Item           *del_duplicate();
    VN	           *vp;
    int            mothersym;
    int            nfollows, j, *follows;

#if CLR
    int            count = 0;
#endif

    if( I->dot > 0)
      mothersym = GrammarTable[I->ruleno]->rhs[I->dot-1];
    else
      mothersym = -1;
	
    for(top = item = I; item != NULL; item = item->next)
      {
	  dotsym = GrammarTable[item->ruleno]->rhs[item->dot];
#if CLR
	  if(item->dot < GrammarTable[item->ruleno]->length - 1) {
	      nextdotsym = GrammarTable[item->ruleno]->rhs[item->dot+1];
	  }else {
	      nextdotsym = item->lookahead;
	  }
#endif
	  if(!nonterminal(dotsym))
	    continue;

	  for(i = 0; i < ap; i++)
	    {
		if(added[i] == dotsym
#if CLR
		   && cmpfirst(nextdotsym, lookahead[i]) == 1
#endif
		   )
		  break;
	    }
	  if(i == ap)  /* dotsym not found */
	    {
#if CLR
		lookahead[ap] = nextdotsym;
#endif

		added[ap++] = dotsym;

		if(ap > MAXGRAMMARS - 1)
		  printf("There is too much items(ap = %d)...\n", ap);
		  
		if((vp = VN_Table[VNhash(SymbolTable[dotsym])]) == NULL)
		  continue;
#if CLR		  
		nfollows = 1;
		if(item->dot < GrammarTable[item->ruleno]->length - 1) {
		    follows =
		      &GrammarTable[item->ruleno]->rhs[item->dot+1];
		}else {
		    follows = &item->lookahead;
		}	
#endif
		for(i = 0; vp->rules[i] != EOR; i++)
		  {
		      newitem = getitem();
		      newitem->ruleno = vp->rules[i];
		      newitem->dot = 0;
#if CLR
		      newitem->lookahead = follows[0];
#endif
		      newitem->next = item->next;
		      item->next = newitem;
#if CLR
		      ++count;
		      if (0/*verbose*/ && (count%100 == 0))
			printf("count= %d\n",count);
#endif
		  }
	    }
      }
#if CLR
    top = sortitem(top);
    top = del_duplicate(top);
#endif
    return(top);
}


included(I, C, n)
Item	*I, **C;
int	n;
{
    register int   i;
    Item	   *item1, *item2;

    for(i = 0; i < n; i++)
      {
	  /* C[i] == I ? */
	  for(item1 = I, item2 = C[i];
	      item1 != NULL && item2 != NULL;
	      item1 = item1->next, item2 = item2->next)
	    {
		if(item1->ruleno != item2->ruleno
		   || item1->dot != item2->dot
#if CLR
		   || cmpfirst(item1->lookahead, item2->lookahead) != 1
#endif
		   )
		  goto next;
	    }
	  if(item1 == NULL && item2 == NULL)
	    return(i);
	next:	;
      }
    return(-1);
}

Item    *del_duplicate(I)
Item	*I;
{
    int  n;
    Item *top, *pitem, *item, *nitem;

    if (I == NULL) return(I);
    top = pitem = I;
    for (item=I->next; item != NULL; item = nitem) {
	nitem = item->next;
	if (pitem->ruleno == item->ruleno 
	    && pitem->dot    == item->dot
#if CLR	
	    && cmpfirst(pitem->lookahead,item->lookahead) == 1
#endif	
	    ) {
	    pitem->next = item->next;
	    free(item);
	} else {
	    pitem = item;
	}
    }
    return (top);
}


#if CLR
cmpfirst(symbol1, symbol2)
int symbol1, symbol2;
{
    int size1, size2, *first1, *first2;

    size1 = First(symbol1, &first1);
    size2 = First(symbol2, &first2);
    if( size1 != size2 )
      return (0);
    if (bcmp(first1, first2, size1 * sizeof(int)) == 0)
      return (1);
    else
      return (0);
}
#endif

Item	*sortitem(I)
Item	*I;
{
    extern char *malloc();
    register int	i, j, k, gap, count;
    Item	*item, *t;

    /* count number of item */

    for(count = 0, item = I; item != NULL; item = item->next, ++count)
	       ;
    if (0 && verbose)
      printf("sorting (%d) item... \n",count);

    t = (Item *)malloc(count * sizeof(Item));
    if (t == NULL)
      fatal_error("Error malloc()\n");

    for(k = 0, item = I; item != NULL; item = item->next, ++k)
      {
	  t[k].ruleno = item->ruleno;
	  t[k].dot = item->dot;
#if CLR
	  t[k].lookahead = item->lookahead;
#endif
      }
    for(gap = k/2; gap > 0; gap /= 2)	
      {
	  for(i = gap; i < k; i++)
	    {
		for(j = i - gap; j >= 0; j -= gap)
		  {
		      if((t[j].ruleno > t[j+gap].ruleno)
			 || (t[j].ruleno == t[j+gap].ruleno
			     && t[j].dot > t[j+gap].dot)
#if CLR
			 || (t[j].ruleno == t[j+gap].ruleno
			     && t[j].dot == t[j+gap].dot
			     && t[j].lookahead > t[j+gap].lookahead)
#endif
			 )
			swapitem(&t[j], &t[j+gap]);
		  }
	    }
      }
    for(k = 0, item = I; item != NULL; item = item->next, ++k)
      {
	  item->ruleno = t[k].ruleno;
	  item->dot = t[k].dot;
#if CLR
	  item->lookahead = t[k].lookahead;
#endif
      }
    free(t);
    return(I);
}

swapitem(I1, I2)
Item	*I1, *I2;
{
    Item	t;

    t.ruleno = I1->ruleno;
    t.dot = I1->dot;
#if CLR
    t.lookahead = I1->lookahead;
#endif
    I1->ruleno = I2->ruleno;
    I1->dot = I2->dot;
#if CLR
    I1->lookahead = I2->lookahead;
#endif
    I2->ruleno = t.ruleno;
    I2->dot = t.dot;
#if CLR
    I2->lookahead = t.lookahead;
#endif
}

Item	*copyitem(I)
Item	*I;
{
    Item	*item, *item2, *newitem, *getitem();

    newitem = NULL;
    for(item = I; item != NULL; item = item->next)
      {
	  item2 = getitem();
	  item2->ruleno = item->ruleno;
	  item2->dot = item->dot;
#if CLR
	  item2->lookahead = item->lookahead;
#endif
	  item2->next = newitem;
	  newitem = item2;
      }
    return(newitem);
}

Item	*getitem()
{
    Item	*p;
    char	*malloc();

    if((p = (Item *)malloc(sizeof(Item))) == NULL)
      fatal_error("Can't alloc item.\n");
    return(p);
}

Gotos	*InstallGotos(gotos, item)
Gotos	*gotos;
Item	*item;
{
    int	symbol;
    Gotos	*p, *getgotos();
    Item	*newitem, *getitem();

    symbol = GrammarTable[item->ruleno]->rhs[item->dot];
    for(p = gotos; p != NULL; p = p->next)
      {
	  if(p->symbol == symbol)
	    break;
      }
    if(!p)
      {
	  p = getgotos();
	  p->symbol = symbol;
	  p->item = NULL;
	  p->next = gotos;
	  gotos = p;
      }
    newitem = getitem();
    newitem->ruleno = item->ruleno;
    newitem->dot = item->dot + 1;
#if CLR
    newitem->lookahead = item->lookahead;
#endif
    newitem->next = p->item;
    p->item = newitem;
    return(gotos);
}

Gotos	*getgotos()
{
    Gotos	*p;
    char	*malloc();

    if((p = (Gotos *)malloc(sizeof(Gotos))) == NULL)
      fatal_error("Can't alloc gotos.\n");
    return(p);
}

nonterminal(n)
int	n;
{
    if(n < 0)
      return(FALSE);
    if(SymbolTable[n][0] == '<')
      return(TRUE);
    return(FALSE);
}

print_c(C, n)
Item	**C;
int	n;
{
    register int	i;

    for(i = 0; i < n; i++)
      {
	  printf("*** ITEM %d ***\n", i);
	  print_item(C[i]);
      }
}

print_item(I)
Item	*I;
{
    register Item	*item;

    for(item = I; item != NULL; item = item->next)
      print_rule(item);
    printf("\n");
}

print_rule(I)
Item	*I;
{
    register int	i;
    Rule	*rule;

    rule = GrammarTable[I->ruleno];
    printf("  %s --> ", SymbolTable[rule->lhs]);

    for(i = 0; i < rule->length; i++)
      {
	  if(i == I->dot)
	    printf(".");
		
	  printf("%s ", SymbolTable[rule->rhs[i]]);
      }

    if(i == I->dot)
      printf(".");
#if CLR
    printf(", %s", SymbolTable[I->lookahead]);
#endif
    printf("\n");
}

print_rule2(ruleno, dot)
int	ruleno, dot;
{
    register int	i;
    Rule	*rule;

    rule = GrammarTable[ruleno];
    printf("  %s --> ", SymbolTable[rule->lhs]);
    for(i = 0; i < rule->length; i++)
      {
	  if(i == dot)
	    printf(".");
	  printf("%s ", SymbolTable[rule->rhs[i]]);
      }
    if(i == dot)
      printf(".");
    printf("\n");
}

/*
 * VN functions.
 */

int	*Aproductions(X)
int	X;
{
    VN	*vp;

    if((vp = VN_Table[VNhash(SymbolTable[X])]) == NULL)
      return(NULL);
    return(vp->rules);
}

InstallVN(name, ruleno)
char	*name;
int	ruleno;
{
    register int	i;
    int	index, size, *newrules;
    char	*p, *malloc();
    VN	*vp;

    if((vp = VN_Table[index = VNhash(name)]) == NULL)
      {
	  if((vp = (VN *)malloc(sizeof(VN))) == NULL)
	    goto NO_MEMORY;
	  if((p = (char *)malloc(strlen(name)+1)) == NULL)
	    goto NO_MEMORY;
	  strcpy(p, name);
	  if((newrules = (int *)malloc(2*sizeof(int))) == NULL)
	    goto NO_MEMORY;
	  newrules[0] = ruleno;
	  newrules[1] = EOR;
	  vp->name = p;
	  vp->rules = newrules;
	  VN_Table[index] = vp;
      }
    else
      {
	  for(size = 0; vp->rules[size] != EOR; size++)
	    ;
	  if((newrules = (int *)malloc((size+2)*sizeof(int))) == NULL)
	    goto NO_MEMORY;
	  for(i = 0; vp->rules[i] != EOR; i++)
	    newrules[i] = vp->rules[i];
	  newrules[i++] = ruleno;
	  newrules[i] = EOR;
	  free((char *)vp->rules);
	  vp->rules = newrules;
      }
    return(index);
  NO_MEMORY:
    fatal_error("No memory for non-terminal '%s'.\n", name);
}

VNhash(s)
char	*s;
{
    char	*p;
    int	old_index, index, n;

    for(index = 0, p = s; *p != NULL; )
      index += *p++;
  retry:
    old_index = index = index % MAXVN;
    for(n = 0; VN_Table[index] != NULL; ++n, index = (old_index+n*n) % MAXVN)
      {
	  if(n != 0 && index == old_index)
	    {
		warning("Bad hashsize for VN table.\n");
		++index;
		goto retry;
	    }
	  if(strcmp(VN_Table[index]->name, s) == 0)
	    break;
      }
    return(index);
}

