/*******************************************************/
/*               SAX User Definition File              */
/*          Sample for SAX with VisIPS and JUMAN       */
/*                                                     */
/*         Copyright (C) 1993 Yasuharu Den             */
/*                   (12 March 1993)                   */
/*  by Yasuharu Den (den@forest.kuee.kyoto-u.ac.jp)    */
/*  Dept. of Electrical Engineering, Kyoto University  */
/*******************************************************/

%   load SAX system
:- use_module(library(sax)).
:- use_module(library(sax_trans)).

%   load interface programs
:- ensure_loaded(library('sax_user/extra')).
:- ensure_loaded(library('sax_user/juman_sax')).
:- ensure_loaded(library('VisIPS/visips')).
:- ensure_loaded(library('sax_user/tree_sax')).
:- ensure_loaded(library('sax_user/visips_process')).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   user defined predicate for VisIPS
write_string(Str, String) :- write_string1(Str, String).
write_string(_, _).

write_string1(Str, String) :-
	numbervars(String, 0, _),
	write(Str, String),
	!, fail.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Flag for SAX translator
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_trans_idterm(+Type)
%   ӣإȥ󥹥졼Υ Type ꤹ롥
%
sax_trans_idterm(short).

%   sax_trans_tp_filter(+Type)
%   ӣإȥ󥹥졼ףˤ벼ͽ¬ե
%   Υ Type ꤹ롥
%
sax_trans_tp_filter(no).

%   sax_trans_block(+Use)
%   ӣإȥ󥹥졼ף block 뤫
%    Use ꤹ롥
%
sax_trans_block(no).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Flag for SAX system
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_input_mode(+Mode)
%   ӣإƥϥ⡼ Mode ꤹ롥
%
sax_input_mode(jstring).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% User defined predicate for SAX translator 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_term_expansion(+Rule, -ExpandedRule)
%   ʸˡ§ Rule  ExpandedRule ѴƤӣإȥ
%   졼Ϥ
%
%   ̤γϰ֤ɽɲä롥ٱ䶯˼̻Ҥ
%   Ͽ뤿ΰɲä롥
%
%     sax_term_expansion(
%         (ʸ --> ֻ, Ҹ & {extra}),
%         (ʸ(Pos) --> ֻ(Pos), Ҹ(_) & {dx(extra,_)})
%                        )
%
sax_term_expansion((:- sax_top_node_category(_,Func/Arity)),
	           (:- sax_top_node_category(_,Func/Arity1))) :- !,
	Arity1 is Arity + 1.
sax_term_expansion((Head --> Body & Dx), (Head1 --> Body1 & Dx1)) :- !,
	put_extra_arg(Head, Head1, ExtraArg),
	put_extra_arg_body(Body, Body1, ExtraArg),
	put_extra_arg_delayed_extra(Dx, Dx1).
sax_term_expansion((Head --> Body), (Head1 --> Body1 & Dx1)) :-
	put_extra_arg(Head, Head1, ExtraArg),
	put_extra_arg_body(Body, Body1, ExtraArg),
	put_extra_arg_delayed_extra(true, Dx1).

%   put_extra_arg(+Term, -NewTerm, +ExtraArg)
%
put_extra_arg(Term, NewTerm, ExtraArg) :-
	Term =.. [Func|Args],
	NewTerm =.. [Func,ExtraArg|Args].

%   put_extra_arg_delayed_extra(+Dx, -NewDx)
%
put_extra_arg_delayed_extra({G}, {dx(G,_)}) :- !.
put_extra_arg_delayed_extra(G,   {dx(G,_)}).

%   put_extra_arg_body(+Body, -NewBody, ?ExtraArg)
%
put_extra_arg_body(((Body1,Body2),Body), NewBody, ExtraArg) :- !,
	put_extra_arg_body((Body1,(Body2,Body)), NewBody, ExtraArg).
put_extra_arg_body(((Body1;Body2),Body), (NewBody1;NewBody2), ExtraArg) :- !,
	put_extra_arg_body((Body1,Body), NewBody1, ExtraArg),
	!, put_extra_arg_body((Body2,Body), NewBody2, ExtraArg).
put_extra_arg_body(({Extra},Body), ({Extra},NewBody), ExtraArg) :- !,
	put_extra_arg_body(Body, NewBody, ExtraArg).
put_extra_arg_body(([],Body), NewBody, ExtraArg) :- !,
	put_extra_arg_body(Body, NewBody, ExtraArg).
put_extra_arg_body(([Word|Words],Body), ([NewWord],NewBody), ExtraArg) :- !,
	put_extra_arg(Word, NewWord, ExtraArg),
	!, put_extra_arg_body((Words,Body), NewBody, _).
put_extra_arg_body((NTerm,Body), (NewNTerm,NewBody), ExtraArg) :- !,
	put_extra_arg(NTerm, NewNTerm, ExtraArg),
	!, put_extra_arg_body(Body, NewBody, _).
put_extra_arg_body((Body1;Body2), (NewBody1;NewBody2), ExtraArg) :- !,
	put_extra_arg_body(Body1, NewBody1, ExtraArg),
	!, put_extra_arg_body(Body2, NewBody2, ExtraArg).
put_extra_arg_body({Extra}, {Extra}, _) :- !.
put_extra_arg_body([], [], _) :- !.
put_extra_arg_body([Word], [NewWord], ExtraArg) :- !,
	put_extra_arg(Word, NewWord, ExtraArg).
put_extra_arg_body([Word|Words], ([NewWord],NewBody), ExtraArg) :- !,
	put_extra_arg(Word, NewWord, ExtraArg),
	!, put_extra_arg_body(Words, NewBody, _).
put_extra_arg_body(NTerm, NewNTerm, ExtraArg) :-
	put_extra_arg(NTerm, NewNTerm, ExtraArg).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% User defined predicate for SAX system
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_query_expansion(+Sentence, -Goal)
%   ʸ Sentence  Goal ѴơӣإȤ
%   뤹롥
%
%   ʸ JUMAN ˤäƷǲϤη̤ӣ
%   Ѵ롥ΤȤ̤νλ̻֡ҤΥ
%    process ΰɲä롥ٱ䶯˼̻Ҥ
%   Ͽ뤿ΰɲä롥
%
%     sax_query_expansion(
%         "ޤǤޤ",
%         (visips_start_client,
%          visips_initialize([,,,ޤ],ʸ),
%          process([ie(ư([begin(t,[])],[ư(dx(true,_ID0)),],
%                           0),true)],1,S1),
%          process([ie(̾([begin(t,[])],[̾(dx(true,_ID1)),],
%                           0),true)],2,S2),
%          process([ie(Ҹ³(S1,[Ҹ³(dx(true,_ID2)),ޤ],
%                                   1),true),
%                   ie(ʽ(S2,[ʽ(dx(true,_ID3)),],
%                             2),true)],3,S3),
%          process([ie(ư(S3,[ư(dx(true,_ID4)),ޤ],
%                           3),true),
%                   ie(̾(S3,[̾(dx(true,_ID5)),ޤ],
%                           3),true)],4,S4),
%          fin(S4),
%          visips_finish,
%          visips_kill_client)
%                        )
%
sax_query_expansion(Sentence, Goal) :-
	sax_top_node_category(STerm/_), !,
	call_juman(Sentence, MorphList),
	juman2lattice(MorphList, Lattice),
	make_word_list(Lattice, Sentence, WordList),
	Goal = (visips_start_client,
	        visips_initialize(WordList,STerm),
		Rest),
	make_sax_goal_from_lattice_extra_arg(Lattice,
	        [0-([begin(t,[])],0)], 1,
		Rest, (visips_finish,visips_kill_client)).
sax_query_expansion(_, true) :-
	format(user_error, 'Top node category not exists!!~n', []).

%   make_sax_goal_from_lattice_extra_arg(+Lattice, +Assoc, +To,
%           -Goal, +Tail)
%
make_sax_goal_from_lattice_extra_arg([_-MListm], Assoc, M,
                (process(Em,M,Sm),fin(Sm),Tail), Tail) :- !,
	make_sax_process_from_morphs_extra_arg(MListm, Assoc, _,
	        Em, Sm, M).
make_sax_goal_from_lattice_extra_arg([_-MListj|Rest], Assoc, J,
	        (process(Ej,J,Sj),Goal), Tail) :-
	make_sax_process_from_morphs_extra_arg(MListj, Assoc, Assoc1,
	        Ej, Sj, J),
	J1 is J + 1,
	!, make_sax_goal_from_lattice_extra_arg(Rest, Assoc1, J1,
	        Goal, Tail).

%   make_sax_process_from_morphs_extra_arg(+MListj, +Assoc, -NewAssoc,
%           -Ej, +Sj, +J)
%
make_sax_process_from_morphs_extra_arg([], Assoc, Assoc, [], _, _) :- !.
make_sax_process_from_morphs_extra_arg([Morphj|Rest],
	        Assoc, [ID-(Sj,J)|Assoc1],
	        [ie(InActive,true)|EjTail], Sj, J) :-
        juman:get_ID(Morphj, ID),
	juman:get_preIDL(Morphj, [PreID|_]),
	get_key_assoc(Assoc, PreID, Sj_1, PrePos),
	getMorphInf(Morphj, NTerm, Word, Args),
	Dx =.. [NTerm,dx(true,_)],
	sax_make_phrase(NTerm, [Sj_1,[Dx,Word],PrePos|Args], InActive),
	!, make_sax_process_from_morphs_extra_arg(Rest, Assoc, Assoc1,
	        EjTail, Sj, J).

get_key_assoc([K-V|_], K, Value1, Value2) :- !, V = (Value1,Value2).
get_key_assoc([_|Rest], K, Value1, Value2) :-
	get_key_assoc(Rest, K, Value1, Value2).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Final process for SAX system
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   fin(+Sm)
%   ǸñץȤäȥ꡼ Sm Фơ
%   ǽܤ
%
%   ڤϤ롥
%
fin([]) :- !.
fin([Id|SmTail]) :- Id =.. [idend,_,[SDx],_To|Args], !,
	(   tree_writer_mode(off)
	;   call_delayed_extra(SDx, Tree),
	    treeprint(Tree), nl,
	    ( Args = [] ; write(Args), nl, nl )
	),
	!, fin(SmTail).
fin([_|SmTail]) :-
	fin(SmTail).
