/*******************************************************/
/*               SAX User Definition File              */
/*       Sample for SAX with Unification Grammar       */
/*                                                     */
/*         Copyright (C) 1995 Yasuharu Den             */
/*                  (30 January 1995)                  */
/*  by Yasuharu Den (den@itl.atr.co.jp)                */
/*  ATR Interpreting Telecommunications Research Labs. */
/*******************************************************/

%   load SAX system
:- use_module(library(sax)).
:- use_module(library(sax_trans)).

%   load interface programs
:- ensure_loaded(library('sax_user/unify_sax')).
:- ensure_loaded(library('sax_user/tree_sax')).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Flag for SAX translator
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_trans_idterm(+Type)
%   ӣإȥ󥹥졼Υ Type ꤹ롥
%
sax_trans_idterm(short).

%   sax_trans_tp_filter(+Type)
%   ӣإȥ󥹥졼ףˤ벼ͽ¬ե
%   Υ Type ꤹ롥
%
sax_trans_tp_filter(no).

%   sax_trans_block(+Use)
%   ӣإȥ󥹥졼ף block 뤫
%    Use ꤹ롥
%
sax_trans_block(no).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Flag for SAX system
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_input_mode(+Mode)
%   ӣإƥϥ⡼ Mode ꤹ롥
%
sax_input_mode(string).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% User defined predicate for SAX translator 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_term_expansion(+Rule, -ExpandedRule)
%   ʸˡ§ Rule  ExpandedRule ѴƤӣإȥ
%   졼Ϥ
%
%   䶯˽񤫤줿¤ñ첽ץ
%   Ѵ롥
%
%     sax_term_expansion(
%         (v(Mother) --> v(Head),
%                        n(Comp),
%                        {Head@cat@head   = Mother@cat@head},
%                        {Head@cat@subcat = [Comp|Mother@cat@subcat]},
%                        {Head@cont = Mother@cont}),
%         (v(Mother) --> v(Head),
%                        n(Comp),
%                        {unify(f(_#[cat:_#[head:A,subcat:B],cont:C],
%                                 _#[cat:_#[head:A,subcat:[D|B]],cont:C],
%                                 D),
%                               f(_Mother,Head,Comp),
%                               f(Mother,_Head,_Comp))})
%                        )
%
sax_term_expansion((Head --> Body & Dx), (Head1 --> Body1 & Dx)) :- !,
	Head =.. [F,X],
	Head1 =.. [F,X1],
	get_fs_body(Body, Vars),
	trans_unify_body(Body, Body1, [(X,_)|Vars], [(_,X1,_)|_]).
sax_term_expansion((Head --> Body), (Head1 --> Body1)) :-
	Head =.. [F,X],
	Head1 =.. [F,X1],
	get_fs_body(Body, Vars),
	trans_unify_body(Body, Body1, [(X,_)|Vars], [(_,X1,_)|_]).

%   get_fs_body(+Body, -Vars)
%
get_fs_body(((Body1,Body2),Body), Vars) :- !,
	get_fs_body((Body1,(Body2,Body)), Vars).
get_fs_body(((Body1;Body2),Body), Vars) :- !,
	get_fs_body((Body1,(Body2,Body)), Vars).
get_fs_body(({_},Body), Vars) :- !,
	get_fs_body(Body, Vars).
get_fs_body(([],Body), Vars) :- !,
	get_fs_body(Body, Vars).
get_fs_body(([_|_],Body), Vars) :- !,
	get_fs_body(Body, Vars).
get_fs_body((NTerm,Body), [(X,_)|Vars]) :- !,
	NTerm =.. [_,X],
	get_fs_body(Body, Vars).
get_fs_body((Body1;Body2), Vars) :- !,
	get_fs_body((Body1,Body2), Vars).
get_fs_body({_}, []) :- !.
get_fs_body([], []) :- !.
get_fs_body([_|_], []) :- !.
get_fs_body(NTerm, [(X,_)]) :-
	NTerm =.. [_,X].

%   trans_unify_body(+Body, -NewBody, +Vars, -NewVars)
%
trans_unify_body(((Body1,Body2),Body), NewBody, Vars, NewVars) :- !,
	trans_unify_body((Body1,(Body2,Body)), NewBody, Vars, NewVars).
trans_unify_body(((Body1;Body2),Body), (NewBody1;NewBody2),
	        Vars, NewVars) :- !,
	trans_unify_body((Body1,Body), NewBody1, Vars, NewVars),
	!, trans_unify_body((Body2,Body), NewBody2, Vars, NewVars).
trans_unify_body(([],Body), NewBody, Vars, NewVars) :- !,
	trans_unify_body(Body, NewBody, Vars, NewVars).
trans_unify_body((Term,Body), NewBody, Vars, NewVars) :- !,
	trans_unify_extra(Body, Term, [], NewBody, Vars, NewVars).
trans_unify_body((Body1;Body2), (NewBody1;NewBody2), Vars, NewVars) :- !,
	trans_unify_body(Body1, NewBody1, Vars, NewVars),
	!, trans_unify_body(Body2, NewBody2, Vars, NewVars).
trans_unify_body([], [], Vars, Vars) :- !.
trans_unify_body(Term, Term, Vars, Vars).

%   trans_unify_extra(+Body, +Term, +Exs, -NewBody, +Vars, -NewVars)
%
trans_unify_extra(((Body1,Body2),Body), Term, Exs, NewBody,
	        Vars, NewVars) :- !,
	trans_unify_extra((Body1,(Body2,Body)), Term, Exs, NewBody,
	        Vars, NewVars).
trans_unify_extra(((Body1;Body2),Body), Term, Exs, (NewBody1;NewBody2),
	        Vars, NewVars) :- !,
	trans_unify_extra((Body1,Body), Term, Exs, NewBody1, Vars, NewVars),
	!, trans_unify_extra((Body2,Body), Term, Exs, NewBody2, Vars, NewVars).
trans_unify_extra(({Extra},Body), Term, Exs, NewBody, Vars, NewVars) :- !,
	trans_unify_extra(Body, Term, [Extra|Exs], NewBody, Vars, NewVars).
trans_unify_extra((Body1;Body2), Term, Exs, (NewBody1;NewBody2),
	        Vars, NewVars) :- !,
	trans_unify_extra(Body1, Term, Exs, NewBody1, Vars, NewVars),
	!, trans_unify_extra(Body2, Term, Exs, NewBody2, Vars, NewVars).
trans_unify_extra({Extra}, Term, Exs, NewBody, Vars, NewVars) :- !,
	trans_unify_feq(Term, [Extra|Exs], NewBody, Vars, NewVars).
trans_unify_extra([], Term, Exs, NewBody, Vars, NewVars) :- !,
	trans_unify_feq(Term, Exs, NewBody, Vars, NewVars).
trans_unify_extra(Body, Term, Exs, (NewBody1,NewBody), Vars, NewVars) :-
	trans_unify_feq(Term, Exs, NewBody1, Vars, NVars),
	!, trans_unify_body(Body, NewBody, NVars, NewVars).

%   trans_unify_feq(+Term, +Exs, -NewBody, +Vars, -NewVars)
%
trans_unify_feq(Term, Exs, NewBody, Vars, NewVars) :-
	trans_unify(Term, NewTerm, Vars),
	trans_feq(Exs, NewTerm, NewBody, Vars, NewVars).

trans_unify(Words, Words, _) :- Words = [_|_], !.
trans_unify(NTerm, NewNTerm, Vars) :-
	NTerm =.. [F,X],
	NewNTerm =.. [F,X1],
	assoc_VAR(Vars, X, X1).

trans_feq([], Term, Term, Vars, Vars) :- !.
trans_feq(Exs, Term, (Term,NewExs), Vars, NewVars) :-
	feq2unify(Exs, NewExs, Vars, NewVars).

assoc_VAR([(X1,V,_)|_], X, V) :- X == X1, !.
assoc_VAR([_|Rest], X, V) :- assoc_VAR(Rest, X, V).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% User defined predicate for SAX system
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_query_expansion(+Sentence, -Goal)
%   ʸ Sentence  Goal ѴơӣإȤ
%   뤹롥
%
%   ӣإΤ call_residue/2 򤫤롥
%
%     sax_query_expansion(
%         [john,leaves,tokyo],
%         call_residue(
%             (process([ie(np([begin(t,[])],[np(true),john]),true)],
%                      S1),
%              process([ie(vt(S1,[vt(true),leaves]),true)],
%                      S2),
%              process([ie(np(S2,[np(true),tokyo]),true)],
%                      S3),
%              fin(S3))
%             _)
%                                )
%
sax_query_expansion(Sentence, call_residue(Goal,_)) :-
	make_sax_goal(Sentence, [begin(t,[])], Goal).

make_sax_goal([], Sm, fin(Sm)) :- !.
make_sax_goal([Wordj|Rest], Sj_1, (process(Ej,Sj),Goal)) :-
        sax_make_phrase(Wordj, [Sj_1], WordGoal),
	(   phrase(WordGoal, Ej)
	;   sax_make_phrase('UNKNOWN', [Sj_1,['UNKNOWN'(true),Wordj]],
	            InActive),
	    Ej = [ie(InActive,true)]
	),
	!, make_sax_goal(Rest, Sj, Goal).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Meta process for SAX system
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   process(+Eu, -Su)
%   ñץȤäȥ꡼ Eu Фơ᥿
%   ܤΤ Su ȤơñץϤ
%
process([], []) :- !.
process([ie(InActive,Exs)|EuTail], Su) :- !,
	(   call(Exs),
	    phrase(InActive, NewEu, EuTail)
	;   NewEu = EuTail
	),
	!, process(NewEu, Su).
process([ae(Active,Exs)|EuTail], Su) :-
	(   call(Exs),
	    Su = [Active|SuTail]
	;   Su = SuTail
	),
	!, process(EuTail, SuTail).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Final process for SAX system
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   fin(+Sm)
%   ǸñץȤäȥ꡼ Sm Фơ
%   ǽܤ
%
fin([]) :- !.
fin([Id|SmTail]) :- Id =.. [idend,_,[SDx],FS], !,
	(   tree_writer_mode(off)
	;   call_delayed_extra(SDx, Tree),
	    treeprint(Tree), nl,
	    fprint(FS), nl
	),
	!, fin(SmTail).
fin([_|SmTail]) :-
	fin(SmTail).
