/*******************************************************/
/*                 Morph-SAX Interface                 */
/*                                                     */
/*         Copyright (C) 1994 Yasuharu Den             */
/*                   (25 January 1993)                 */
/*  by Yasuharu Den (den@forest.kuee.kyoto-u.ac.jp)    */
/*  Dept. of Electrical Engineering, Kyoto University  */
/*                                                     */
/*  MODIFIED         (15 April 1994)                   */
/*  by Yasuharu Den (den@itl.atr.co.jp)                */
/*  ATR Interpreting Telecommunications Research Labs. */
/*******************************************************/

%   use morph module
:- use_module(library('Utility/morph')).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Dictionary
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   dict(+Word, -Cat, -Args)
%   ñ Word ʻ̾ Cat ȥǡ Args 򼭽񤫤鸡롥
%
dict(Word, Cat, Args) :-
	downcase_word(Word, Word1),
	sax_make_phrase(Word1, [_S], WordGoal),
	phrase(WordGoal, E), !,
	word_process(E, Cat, Args).
% dict(_, 'UNKNOWN', []).

word_process([ie(InActive,Exs)|_], Cat, Args) :-
	call(Exs),
	InActive =.. [Cat,_S,_Dx|Args].
word_process([_|Tail], Cat, Args) :- word_process(Tail, Cat, Args).

downcase_word(Word, NewWord) :-
	name(Word, [C|X]),
	(   C >= 65, C =< 90 ->
	    C1 is C + 32,
	    name(NewWord, [C1|X])
	;   NewWord = Word
	).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Morph Information
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   getMorphInf(+Morph, -Cat, -Root, -Args)
%    Morph ʻ̾ Cat, ܷ Root, ӡ¾
%   Υǡ Args Ф
%
%   ǾϰʲνҸˤäƼФ롥
%
%     morph:get_ID(    Morph, ID)     :- arg(1, Morph, ID).
%     morph:get_from(  Morph, From)   :- arg(2, Morph, From).
%     morph:get_to(    Morph, To)     :- arg(3, Morph, To).
%     morph:get_word(  Morph, Word)   :- arg(4, Morph, Word).
%     morph:get_root(  Morph, Root)   :- arg(5, Morph, Root).
%     morph:get_cat(   Morph, Cat)    :- arg(6, Morph, Cat).
%     morph:get_args(  Morph, Args)   :- arg(7, Morph, Args).
%     morph:get_preIDL(Morph, PreIDL) :- arg(8, Morph, PreIDL).
%
getMorphInf(Morph, Cat, Root, Args) :-
	morph:get_cat(Morph, Cat),
	morph:get_root(Morph, Root),
	morph:get_args(Morph, Args).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Call morph
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   call_morph(+Sentence, -MorphList)
%   ʸ (ñΥꥹ) Sentence  morph ǷǲϤ
%   ǤΥꥹ MorphList 롥
%
call_morph(Sentence, MorphList) :-
	statistics(runtime, _),
	morph(Sentence, MorphList),
	statistics(runtime, [_,Time]),
	format('Morphological Analysis Time = ~d msec~n', [Time]), nl, !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Morph to Lattice Converter
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   morph2lattice(+MorphList, -Lattice)
%   morph äǤΥꥹ MorphList ФƱ
%   ֤ǽǤޤȤ Lattice Ȥ롥
%
%   <Lattice> ::= a list of <Morph Item>'s
%   <Morph Item> ::= <To>-<Morph List>
%   <Morph List> ::= a list of <Morph>'s whose end-points are <To>
%   <To> ::= an integer
%   <Morph> ::= a morpheme
%
morph2lattice(MorphList, Lattice) :-
	morph2lattice(MorphList, [], Lattice).

morph2lattice([], Lattice, Lattice) :- !.
morph2lattice([Morph|Rest], Lattice, Lattice2) :-
	morph:get_to(Morph, To),
	insert_morph(Lattice, To, Morph, Lattice1),
	!, morph2lattice(Rest, Lattice1, Lattice2).

insert_morph([], To, Morph, [To-[Morph]]) :- !.
insert_morph([T-MList|Rest], To, Morph, Lattice) :- !,
	compare(Ord, T, To),
	insert_morph(Ord, T, MList, Rest, To, Morph, Lattice).

insert_morph(=, T, MList, Rest, _, Morph, [T-[Morph|MList]|Rest]).
insert_morph(>, T, MList, Rest, To, Morph, [To-[Morph],T-MList|Rest]).
insert_morph(<, T, MList, Rest, To, Morph, [T-MList|Lattice]) :-
	insert_morph(Rest, To, Morph, Lattice).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Make Word List
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   make_word_list(+Lattice, +Sentence, -WordList)
%   Lattice ڤܤˤäơʸ Sentence ʬ䤷
%   WordList Ȥ롥
%
make_word_list(Lattice, _, WordList) :-
	make_word_list(Lattice, WordList).

make_word_list([], []) :- !.
make_word_list([_-[Morph|_]|Rest], [Word|WordList]) :-
	morph:get_word(Morph, Word),
	!, make_word_list(Rest, WordList).
