/*******************************************************/
/*                 Juman-SAX Interface                 */
/*                                                     */
/*         Copyright (C) 1993 Yasuharu Den             */
/*                   (13 January 1993)                 */
/*  by Yasuharu Den (den@forest.kuee.kyoto-u.ac.jp)    */
/*  Dept. of Electrical Engineering, Kyoto University  */
/*******************************************************/

%   use juman module
:- use_module(library(juman)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% JSAX
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   jsax
%   jsax(+InFile, +OutFile)
%   jsax(+Sentence)
%   ܸѣӣإƥ
%
jsax :- jsax(user, user).

jsax(InFile, OutFile) :-
	juman_start_client,
	sax(InFile, OutFile),
	juman_kill_client.

jsax(Sentence) :-
	juman_start_client,
	sax(Sentence),
	juman_kill_client.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Morph Information
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   getMorphInf(+Morph, -Hinsi, -Midasi, -Args)
%    Morph ʻ̾ Hinsi, Ф Midasi, ӡ
%   ¾Υǡ Args Ф
%
%   ǾϰʲνҸˤäƼФ롥
%
%     juman:get_ID(         Morph, ID)     ̻
%     juman:get_from(       Morph, From)   ϰ
%     juman:get_to(         Morph, To)     λ
%     juman:get_score(      Morph, Score)  
%     juman:get_midasi(     Morph, Md)     Ф
%     juman:get_yomi(       Morph, Ym)     ɤ
%     juman:get_kihon(      Morph, Kh)     ܷ
%     juman:get_hinsi(      Morph, Hn)     ʻ̾
%     juman:get_katuyo_type(Morph, KT)     ѷ̾
%     juman:get_katuyo_form(Morph, KF)     ѷ̾
%     juman:get_imi(        Morph, Im)     ̣
%     juman:get_preIDL(     Morph, PreIDL) ܷǤμ̻ҤΥꥹ
%
getMorphInf(Morph, Hinsi, Midasi, Args) :-
	juman:get_hinsi(Morph, Hinsi),
	juman:get_midasi(Morph, Midasi),
	Args = [].

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Call JUMAN
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   call_juman(+Sentence, -MorphList)
%   ʸ (ʸΥꥹ) Sentence  JUMAN ǷǲϤ
%   ǤΥꥹ MorphList 롥
%
call_juman(Sentence, MorphList) :-
	name(AtomOfSentence, Sentence),
	statistics(runtime, _),
	juman(AtomOfSentence, MorphList, _),
	statistics(runtime, [_,Time]), nl,
	juman:print_morphs(MorphList, -1), nl,
	format('ǲϻ = ~d msec~n', [Time]), nl, !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% JUMAN to Lattice Converter
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   juman2lattice(+MorphList, -Lattice)
%   JUMAN äǤΥꥹ MorphList ФƱ
%   ֤ǽǤޤȤ Lattice Ȥ롥
%
%   <Lattice> ::= a list of <Morph Item>'s
%   <Morph Item> ::= <To>-<Morph List>
%   <Morph List> ::= a list of <Morph>'s whose end-points are <To>
%   <To> ::= an integer
%   <Morph> ::= a morpheme
%
juman2lattice(MorphList, Lattice) :-
	juman2lattice(MorphList, [], Lattice).

juman2lattice([], Lattice, Lattice) :- !.
juman2lattice([Morph|Rest], Lattice, Lattice2) :-
	juman:get_to(Morph, To),
	insert_morph(Lattice, To, Morph, Lattice1),
	!, juman2lattice(Rest, Lattice1, Lattice2).

insert_morph([], To, Morph, [To-[Morph]]) :- !.
insert_morph([T-MList|Rest], To, Morph, Lattice) :- !,
	compare(Ord, T, To),
	insert_morph(Ord, T, MList, Rest, To, Morph, Lattice).

insert_morph(=, T, MList, Rest, _, Morph, [T-[Morph|MList]|Rest]).
insert_morph(>, T, MList, Rest, To, Morph, [To-[Morph],T-MList|Rest]).
insert_morph(<, T, MList, Rest, To, Morph, [T-MList|Lattice]) :-
	insert_morph(Rest, To, Morph, Lattice).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Make Word List
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   make_word_list(+Lattice, +Sentence, -WordList)
%   Lattice ڤܤˤäơʸ Sentence ʬ䤷
%   WordList Ȥ롥
%
make_word_list(Lattice, Sentence, WordList) :-
	make_word_list(Lattice, 0, Sentence, WordList).

make_word_list([], _, [], []) :- !.
make_word_list([To-_|Rest], From, Sentence, [Word|WordList]) :-
	N is To - From,
	nth_first(N, Sentence, Name, Sentence1),
	name(Word, Name),
	!, make_word_list(Rest, To, Sentence1, WordList).

nth_first(0, List, [], List) :- !.
nth_first(N, [Element|List], [Element|Tail], Rest) :-
	N1 is N - 1,
	nth_first(N1, List, Tail, Rest).
