/*******************************************************/
/*                SAX Translator 2.0 #0                */
/*        for DCGs with Delayed Extra Conditions       */
/*                                                     */
/*  Copyright (C) 1993 Yuji Matsumoto, Yasuharu Den    */
/*                    (09 March 1993)                  */
/*  by Yuji Matsumoto (matsu@pine.kuee.kyoto-u.ac.jp)  */
/*     Yasuharu Den   (den@forest.kuee.kyoto-u.ac.jp)  */
/*  Dept. of Electrical Engineering, Kyoto University  */
/*******************************************************/

% The syntax of a DCG rule is:
%
%     Head --> Body & Delayed_extra.
%
% where
%
%     Head --> Body makes a normal DCG rule, and
%     Delayed_extra is an extra condition, which is evaluated
%                   after the syntactic analysis.
%
:- op(1150, xfy, &).

:- module(sax_trans, [
	sax_top_node_category/1,
	sax_top_node_category/2,
	sax_compile/1,
	sax_consult/1,
	sax_trans/2
		     ]).

:- ensure_loaded(library(sax_trans_utils)).

:- use_module(library(lists), [append/3,reverse/2]).

:- prolog_flag(version,PVer),
	name('SICStus 2.1',SICStus21),
	name(PVer,CPVer),
	( append(SICStus21,_,CPVer)
	; use_module(library(system))
	).

% Version of this system
ver('SAX Translator 2.0 #0: Tue Mar 09 1993').

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% SAX Top node category
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_top_node_category(?Old)
%   sax_top_node_category(?Old, ?New)
%   ϵ Old  New Ѥ롥
%
:- dynamic '$sax_top_node_category'/1.

sax_top_node_category(Old) :- '$sax_top_node_category'(Old), !.
sax_top_node_category(Old, New) :-
	( retract('$sax_top_node_category'(Old)) ; true ),
	assert('$sax_top_node_category'(New)), !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Load
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_compile(+Files)
%   ʸˡ Files 򥳥ѥ뤹롥
%
sax_compile(Files) :-
	sax:call_unix(mktemp('/tmp/spXXXXXX', TempFile)),
	sax_trans(Files, TempFile),
	compile(user:TempFile), !.

%   sax_consult(+Files)
%   ʸˡ Files 򥳥󥵥Ȥ롥
%
sax_consult(Files) :-
	sax:call_unix(mktemp('/tmp/spXXXXXX', TempFile)),
	sax_trans(Files, TempFile),
	consult(user:TempFile), !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Translator
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   sax_trans(+InFiles, -OutFile)
%   ʸˡ InFiles ӣѴ OutFile ˽Ϥ롥
%
sax_trans(InFiles, OutFile) :- InFiles = [_|_], !,
	trans(InFiles, OutFile).
sax_trans(InFile, OutFile) :-
	trans([InFile], OutFile).

trans(InFiles, OutFile) :-
	ver(Version),
	format(user_error, '~w~n', [Version]),
	init_side_effects,
	(   OutFile = user -> OutStr = user_output
	;   open(OutFile, write, OutStr)
	),
	statistics(runtime, _), !,
	do_trans(InFiles, Table, LTable, OutStr),
	(   sax_top_node_category(Func/Arity) ->
	    register_top_node(Func, Arity, Table)
	;   true
	),
	statistics(runtime, [_,T1]),
	format(user_error, '>> Finished Path 1 (~d msec.) <<~n', [T1]),
	format(user_error, 'Writing ~w...~n', [OutFile]), !,
	path2(Table, LTable, OutStr),
	produce_link(LTable, OutStr),
	statistics(runtime, [_,T2]),
	format(user_error, '>> Finished Path 2 (~d msec.) <<~n', [T2]),
	close(OutStr),
	format(user_error, '>> End of translation <<~n', []),
	format(user_error, 'Total execution time = ~d msec.~n', [T1+T2]), !.

do_trans([], _, _, _) :- !.
do_trans([InFile|Rest], Table, LTable, OutStr) :-
	absolute_file_name(InFile, InFileAbsolute),
	(   InFileAbsolute = user -> InStr = user_input
	;   open(InFileAbsolute, read, InStr)
	),
	format(user_error, 'Loading ~w...~n', [InFileAbsolute]),
	path1(InStr, Table, LTable, OutStr),
	close(InStr),
	!, do_trans(Rest, Table, LTable, OutStr).

%   init_side_effects
%   Ѥ롥
%
init_side_effects :-
	retractall(symbol(id,_)),
	abolish(done, 2),
	abolish(id_pair, 2),
	abolish(link, 2),
	assert(link(X,X)), !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Path1
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   path1(+InStream, ?Table, ?LTable, +OutStream)
%   InStream 񤭴§ɤ߹ߡ§ Table ˥
%    LTable Ͽ롥񤭴§ʳľ 
%   OutStream ˽Ϥ롥
%
path1(InStr, Table, LTable, OutStr) :-
	read(InStr, Clause0),
	(   user:sax_term_expansion(Clause0, Clause)
	;   Clause = Clause0
	), !,
	path1(Clause, Table, LTable, InStr, OutStr).

path1(end_of_file, _, _, _, _) :- !.
path1((Head0 --> Body0), Table, LTable, InStr, OutStr) :- !,
	get_delayed_extra(Body0, Body, Head0, Head),
	get_rule(Body, [], Rules, Head),
	register_rule(Rules, Head, Table, LTable),
	!, path1(InStr, Table, LTable, OutStr).
path1((:- Command), Table, LTable, InStr, OutStr) :- !,
	\+ \+ call(Command),
	print_quoted(OutStr, (:- Command)), nl(OutStr),
	!, path1(InStr, Table, LTable, OutStr).
path1(Clause, Table, LTable, InStr, OutStr) :-
	print_clause(OutStr, Clause), nl(OutStr),
	!, path1(InStr, Table, LTable, OutStr).

%   get_delayed_extra(+Body, -NewBody, +Head, -NewHead)
%   񤭴§α Body ٱ䶯ФƻĤ
%   NewBody ȤФٱ䶯 Head Фˤ
%   NewHead Ȥ롥
%
get_delayed_extra((Body & DExtra0), Body, Head, Head+DExtra) :- !,
	( DExtra0 = {DExtra} ; DExtra0 = DExtra ).
get_delayed_extra(Body, Body, Head, Head+true).

%%% Get Rule %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   get_rule(+Body, +LVars, -Rules, +Head)
%   դ Head, դλĤ꤬ Body Ǥʸˡ§ ()
%   üФRules Ȥ롥LVars ϱդδ˽
%   ʬ˸줿ѿΥꥹȡ
%
get_rule(((Body1,Body2),Body), LVars, Rule, Head) :- !,
	get_rule((Body1,(Body2,Body)), LVars, Rule, Head).
get_rule(((Body1;Body2),Body), LVars, [Rule|Rule1], Head) :- !,
	get_rule((Body1,Body), LVars, Rule, Head),
	!, get_rule((Body2,Body), LVars, Rule1, Head).
get_rule(([],Body), LVars, Rule, Head) :- !,
	get_rule(Body, LVars, Rule, Head).
get_rule(([Word],Body), LVars, Rule, Head) :- !,
	get_extra(Body, [Word], [], LVars, Rule, Head).
get_rule(([Word|Words],Body), LVars,
	        [([NewWord],NewExs,RVars)|Rule], Head) :- !,
	move_depend_vars(Word, NewWord, LVars, [], NewExs),
	mark_vars(LVars, f(NewWord,NewExs), NewVars),
	copy_vars(LVars, CopyVars),
	append(NewVars, CopyVars, RVars),
	!, get_rule((Words,Body), RVars, Rule, Head).
get_rule((NTerm,Body), LVars, Rule, Head) :- !,
	get_extra(Body, NTerm, [], LVars, Rule, Head).
get_rule((Body1;Body2), LVars, [Rule|Rule1], Head) :- !,
	get_rule(Body1, LVars, Rule, Head),
	!, get_rule(Body2, LVars, Rule1, Head).
get_rule([], _, _, _) :- !, fail.
get_rule([Word], LVars, [([NewWord],NewExs,Head)], Head) :- !,
	move_depend_vars(Word, NewWord, LVars, [], NewExs),
	mark_vars(LVars, f(NewWord,NewExs,Head), _).
get_rule([Word|Words], LVars, [([NewWord],NewExs,RVars)|Rule], Head) :- !,
	move_depend_vars(Word, NewWord, LVars, [], NewExs),
	mark_vars(LVars, f(NewWord,NewExs), NewVars),
	copy_vars(LVars, CopyVars),
	append(NewVars, CopyVars, RVars),
	!, get_rule(Words, RVars, Rule, Head).
get_rule(NTerm, LVars, [(NewNTerm,NewExs,Head)], Head) :-
	move_depend_vars(NTerm, NewNTerm, LVars, [], NewExs),
	mark_vars(LVars, f(NewNTerm,NewExs,Head), _).

%   get_extra(+Body, +Term, +Exs, +LVars, -Rules, +Head)
%   դ Head, դλĤ꤬ Body Ǥʸˡ§ ()
%   ü Term 䶯 Exs ФΥǡ
%   Rules ˲ä롥κݡդδ˽ʬ˸줿
%    LVars ΤTerm  Exs ǻȤΤФ
%   ޡĤ롥
%
get_extra(((Body1,Body2),Body), Term, Exs, LVars, Rule, Head) :- !,
	get_extra((Body1,(Body2,Body)), Term, Exs, LVars, Rule, Head).
get_extra(((Body1;Body2),Body), Term, Exs, LVars, [Rule|Rule1], Head) :- !,
	get_extra((Body1,Body), Term, Exs, LVars, Rule, Head),
	!, get_extra((Body2,Body), Term, Exs, LVars, Rule1, Head).
get_extra(({Extra},Body), Term, Exs, LVars, Rule, Head) :- !,
	get_extra(Body, Term, [Extra|Exs], LVars, Rule, Head).
get_extra((Body1;Body2), Term, Exs, LVars, [Rule|Rule1], Head) :- !,
	get_extra(Body1, Term, Exs, LVars, Rule, Head),
	!, get_extra(Body2, Term, Exs, LVars, Rule1, Head).
get_extra(([],Body), Term, Exs, LVars, Rule, Head) :- !,
    get_extra(Body, Term, Exs, LVars, Rule, Head).
get_extra({Extra}, Term, Exs, LVars, [(NewTerm,NewExs,Head)], Head) :- !,
	reverse([Extra|Exs], RExs),
	move_depend_vars(Term, NewTerm, LVars, RExs, NewExs),
        mark_vars(LVars, f(NewTerm,NewExs,Head), _).
get_extra([], Term, Exs, LVars, [(NewTerm,NewExs,Head)], Head) :- !,
	reverse(Exs, RExs),
	move_depend_vars(Term, NewTerm, LVars, RExs, NewExs),
        mark_vars(LVars, f(NewTerm,NewExs,Head), _).
get_extra(Body, Term, Exs, LVars, [(NewTerm,NewExs,RVars)|Rule], Head) :-
	reverse(Exs, RExs),
	move_depend_vars(Term, NewTerm, LVars, RExs, NewExs),
        mark_vars(LVars, f(NewTerm,NewExs), NewVars),
        copy_vars(LVars, CopyVars),
	append(NewVars, CopyVars, RVars),
	!, get_rule(Body, RVars, Rule, Head).

%   move_depend_vars(+Term, -NewTerm, +Vars, +Exs, -NewExs)
%   Term ѿǤʤप Term ¾ѿ Vars 
%   ѿȽʣƤѿ򿷤ѿ֤Τ
%   NewTerm Ȥ롥֤줿ѿȤȤιȤ«
%    Exs ƬɲäNewExs Ȥ롥
%
move_depend_vars([Word], [NewWord], Vars, Exs, NewExs) :- !,
	functor(Word, Func, Arity),
        functor(NewWord, Func, Arity),
        move_depend_vars(Arity, Word, NewWord, Vars, Exs, NewExs).
move_depend_vars(NTerm, NewNTerm, Vars, Exs, NewExs) :-
	functor(NTerm, Func, Arity),
        functor(NewNTerm, Func, Arity),
        move_depend_vars(Arity, NTerm, NewNTerm, Vars, Exs, NewExs).

move_depend_vars(0, _, _, _, Exs, Exs) :- !.
move_depend_vars(N, Term, NewTerm, Vars, Exs, NewExs) :-
	arg(N, Term, Arg),
	arg(N, NewTerm, NewArg),
	(   ( nonvar(Arg) ; member_VAR(Vars, Arg) ) -> 
	    Exs1 = [NewArg=Arg|Exs], Vars1 = Vars
	;   Arg = NewArg, Exs1 = Exs, Vars1 = [(Arg,_)|Vars]
	),
	N1 is N - 1,
	!, move_depend_vars(N1, Term, NewTerm, Vars1, Exs1, NewExs).

member_VAR(Vars, Var) :- Vars = [(Var1,_)|_], Var == Var1, !.
member_VAR([_|Rest], Var) :- member_VAR(Rest, Var).

%   mark_vars(+LVars, +Term, -NewVars)
%   ѿΥꥹ LVars ΤTerm ǻȤΤФ
%   ޡĤ롥ޤTerm ˿˸줿ѿ
%   NewVars Ȥ롥
%
mark_vars(LVars, Term, NewVars) :- mark_vars(Term, LVars, [], NewVars).

mark_vars(Var, LVars, NewVars, NewVars1) :- var(Var), !,
	(   check_used_VAR(LVars, Var) -> NewVars1 = NewVars
	;   add_VAR(NewVars, Var, NewVars1)
	).
mark_vars(Term, LVars, NewVars, NewVars1) :-
	functor(Term, _, N),
	loop_mark_vars(0, N, Term, LVars, NewVars, NewVars1).

loop_mark_vars(N, N, _, _, NewVars, NewVars) :- !.
loop_mark_vars(M, N, Term, LVars, NewVars, NewVars2) :-
        M1 is M + 1,
        arg(M1, Term, Arg),
        mark_vars(Arg, LVars, NewVars, NewVars1),
        !, loop_mark_vars(M1, N, Term, LVars, NewVars1, NewVars2).

check_used_VAR([(Var1,YN)|_], Var) :- Var == Var1, !, YN = yes.
check_used_VAR([_|Rest], Var) :- check_used_VAR(Rest, Var).

add_VAR([], Var, [(Var,_)]) :- !.
add_VAR(Vars, Var, Vars) :- Vars = [(Var1,_)|_], Var == Var1, !.
add_VAR([Var1|Rest], Var, [Var1|Rest1]) :- add_VAR(Rest, Var, Rest1).

%   copy_vars(+Vars, -CopyVars)
%   Vars γѿλȾ֤ꥻåȤ CopyVars Ȥ롥
%
copy_vars([], []) :- !.
copy_vars([(Var,YN)|Rest], [(Var,CP)|Rest1]) :-
      ( YN == yes ; YN = CP ),
      !, copy_vars(Rest, Rest1).

%%% Register Rule %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   register_rule(+Rules, +Head, +Table, +LTable)
%   񤭴§ Rules ᤷ§ Table ˥󥯾
%    LTable Ͽ롥Head Ϻա
%
%   sgldict(Word,Head,Exs)-t                  դñȤǸ뽪ü
%   sgl(NTerm,Head,Exs)-t                     դñȤǸü
%   lcdict(Word,Exs)-(PutId,PutVars)          դƬ˸뽪ü
%   lc(NTerm,Exs)-(PutId,PutVars)             դƬ˸ü
%   lstdict(Word,GetVars,Head,Exs)-t          դ˸뽪ü
%   lst(NTerm,GetVars,Head,Exs)-t             դ˸ü
%   middict(Word,GetVars,Exs)-(PutId,PutVars) դۤɤ˸뽪ü
%   mid(NTerm,GetVars,Exs)-(PutId,PutVars)    դۤɤ˸ü
%
%       Word    ü
%       NTerm   ü
%       GetVars ¦ѿΥꥹ
%       Head     + ٱ䶯
%       Exs     䶯
%
%       PutId   ¦Ϥ̻
%       PutVars ¦ϤѿΥꥹ
%
register_rule([([Word],Exs,Head)], _, Table, _) :- !,
	functor(Word, Func, Arity),
	recdz(Table, dict(Func/Arity), type1(t),
	        sgldict(Word,Head,Exs)-t).
register_rule([(NTerm,Exs,Head)], _, Table, LTable) :- !,
	functor(NTerm, Func, Arity),
	(   user:sax_trans_tp_filter(Use), ( Use = strong ; Use = weak ) ->
	    Head = Hd+_, functor(Hd, HF, HA),
	    put_data(LTable, Func/Arity, HF/HA-t),
	    ( Use = strong -> H = HF/HA ; Use = weak -> H = t )
	;   H = t
	),
	recdz(Table, Func/Arity, type1(H),
	        sgl(NTerm,Head,Exs)-t).
register_rule([([Word],Exs,PutVars)|Rest], _, Table, _) :- !,
	functor(Word, Func, Arity),
	recdz(Table, dict(Func/Arity), type1(t),
	        lcdict(Word,Exs)-(PutId,PutVars)),
	new_id(PutId),
	!, register_rule2(Rest, PutId, PutVars, Table).
register_rule([(NTerm,Exs,PutVars)|Rest], Head, Table, LTable) :- !,
	functor(NTerm, Func, Arity),
	(   user:sax_trans_tp_filter(Use), ( Use = strong ; Use = weak ) ->
	    Head = Hd+_, functor(Hd, HF, HA),
	    put_data(LTable, Func/Arity, HF/HA-t),
	    ( Use = strong -> H = HF/HA ; Use = weak -> H = t )
	;   H = t
	),
	recdz(Table, Func/Arity, type1(H),
	        lc(NTerm,Exs)-(PutId,PutVars)),
	new_id(PutId),
	!, register_rule2(Rest, PutId, PutVars, Table).
register_rule([Rule|Rest], Head, Table, LTable) :-
	register_rule(Rule, Head, Table, LTable),
	!, register_rule(Rest, Head, Table, LTable).

register_rule2([([Word],Exs,Head)], GetId, GetVars, Table) :- !,
	functor(Word, Func, Arity),
	recdz(Table, dict(Func/Arity), type2(GetId+GetVars),
	        lstdict(Word,GetVars,Head,Exs)-t).
register_rule2([(NTerm,Exs,Head)], GetId, GetVars, Table) :- !,
	functor(NTerm, Func, Arity),
	recdz(Table, Func/Arity, type2(GetId+GetVars),
	        lst(NTerm,GetVars,Head,Exs)-t).
register_rule2([([Word],Exs,PutVars)|Rest], GetId, GetVars, Table) :- !,
	functor(Word, Func, Arity),
	recdz(Table, dict(Func/Arity), type2(GetId+GetVars),
	        middict(Word,GetVars,Exs)-(PutId,PutVars)),
	new_id(PutId),
	!, register_rule2(Rest, PutId, PutVars, Table).
register_rule2([(NTerm,Exs,PutVars)|Rest], GetId, GetVars, Table) :- !,
	functor(NTerm, Func, Arity),
	recdz(Table, Func/Arity, type2(GetId+GetVars),
	        mid(NTerm,GetVars,Exs)-(PutId,PutVars)),
	new_id(PutId),
	!, register_rule2(Rest, PutId, PutVars, Table).
register_rule2([Rule|Rest], GetId, GetVars, Table) :-
	register_rule2(Rule, GetId, GetVars, Table),
	!, register_rule2(Rest, GetId, GetVars, Table).

%   recdz(+Table, +Key1, +Key2, +Value)
%   ȥ꡼ <Key1,Key2,Value>  Table Ͽ롥
%
recdz([[Key|Data]|_], Key, Key2, Value) :- !, put_data(Data, Key2, Value).
recdz([_|Rest], Key1, Key2, Value) :- recdz(Rest, Key1, Key2, Value).

put_data([[Key|List]|_], Key, Value) :- !, insert(List, Value).
put_data([_|Rest], Key, Value) :- put_data(Rest, Key, Value).

insert(List, Value) :- var(List), !, List = [Value|_].
insert([Item1-X1|_], Item-X) :- equiv(Item, Item1), !, X = X1.
insert([_|Rest], Value) :- insert(Rest, Value).

%   new_id(?Id)
%    Id 롥
%
new_id(Id) :- nonvar(Id), !.
new_id(Id) :- gensym(id, Id).

%%% Register Top Node %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   register_top_node(+Func, +Arity, +Table)
%   ϵ Func/Arity  Table Ͽ롥
%
register_top_node(Func, Arity, Table) :-
	functor(TopNode, Func, Arity),
	TopNode =.. [_|Args],
	yes_vars(Args, Vars),
	recdz(Table, Func/Arity, type2(begin+[]),
	        mid(TopNode,[],[])-(idend,Vars)).

%   yes_vars(+Args, -Vars)
%   ѿΥꥹ Args λȾ֤򤹤٤ yes ˤƥꥹ
%   Vars 롥
%
yes_vars([], []) :- !.
yes_vars([Var|Rest], [(Var,yes)|Rest1]) :- yes_vars(Rest, Rest1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Path2
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   path2(+Table, +LTable, +OutStream)
%   Table ϿƤ񤭴§ӣѴơ
%   OutStream ˽Ϥ롥LTable ϥ󥯾
%
path2([], _, _) :- !.
path2([[Key|Data]|Rest], LTable, OutStr) :- !,
	separate(Data, Type1, Type2),
	(   Key = dict(Func/Arity) -> Sort = dict
	;   Key = Func/Arity -> Sort = nonterminal
	),
	produce_type1(Type1, Sort, Func, Arity, Type2, LTable, OutStr),
	produce_type2(Type2, Sort, Func, Arity, OutStr),
	!, path2(Rest, LTable, OutStr).

%   separate(+Data, -Type1, -Type2)
%   ǡ Data 򥿥ף˴ؤ Type1 ȥף
%   ˴ؤ Type2 ʬ롥
%
separate([], [], []) :- !.
separate([[type1(Tp1)|Data]|Rest], [[Tp1|Data]|Type1], Type2) :- !,
	separate(Rest, Type1, Type2).
separate([[type2(Tp2)|Data]|Rest], Type1, [[Tp2|Data]|Type2]) :- !,
	separate(Rest, Type1, Type2).

%%% Produce Type1 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   produce_type1(+Type1, +Sort, +Func, +Arity, +Type2, +LTable, +OutStream)
%   () ü Func/Arity Υǡ Type1 饿ף
%   ꡤOutStream ˽Ϥ롥Sort Ͻü椫ü
%   Τ餻ե饰Type2 ϥף᤬ɬפɤ
%   餻ǡLTable ϥ󥯾
%
produce_type1(Type1, Sort, Func, Arity, Type2, LTable, OutStr) :-
	make_term(Sort, Func, Arity, Args,
	        '$VAR'('InS'), '$VAR'('Dx'), LHS),
	(   Type1 \== [], Sort = nonterminal,
	    user:sax_trans_tp_filter(Use), ( Use = strong ; Use = weak ) ->
	    get_data(LTable, Func/Arity, Data),
	    type1_link(Data, Func/Arity),
	    concat_atoms(Func, Arity, FA),
	    concat_atoms('tp_filter_', FA, TpFilter),
	    concat_atoms('link_', FA, Link),
	    (   Use = weak ->
		produce_type1_tp_filter(TpFilter, Link, OutStr),
		type1_tp_filter(TpFilter, '$VAR'('InS'), NewInS, RHS, Body)
	    ;   NewInS = '$VAR'('InS'), RHS = Body
	    )
	;   NewInS = '$VAR'('InS'), RHS = Body
	),
	type1_tail(Type2, Sort, Func, Arity, Args, Tail),
	type1_body(Type1, Sort, TpFilter, Link, NewInS, Args,
	        Body, Tail, OutStr),
	print_clause(OutStr, (LHS --> RHS)), nl(OutStr).

get_data(Data, _, _) :- var(Data), !.
get_data([[Key|Value]|_], Key, Value) :- !.
get_data([_|Rest], Key, Value) :- get_data(Rest, Key, Value).

%   type1_tail(+Type2, +Sort, +Func, +Arity, +Args, -Tail)
%   () ü Func/Arity ˥ףΥǡ Type2 
%   ʤ顤ף Tail 롥Args  () ü
%   ΰ Sort Ͻü椫ü椫Τ餻ե饰
%   
type1_tail([], _, _, _, _, !) :- !.
type1_tail(_, Sort, Func, Arity, Args, Tail) :-
	concat_atoms(Func, '_tp2', Func_tp2),
	make_term(Sort, Func_tp2, Arity, Args,
	        '$VAR'('InS'), '$VAR'('Dx'), Tail).

%   type1_tp_filter(+Filter, -InS, -OutS, -RHS, +Tail)
%   ͽ¬ե륿 Filter ꡤ Tail ˤĤ
%   Τ RHS Ȥ롥InS, OutS ϥե륿Υȥ꡼ࡥ
%
type1_tp_filter(Filter, InS, OutS, ({TpFilter},Tail), Tail) :-
	TpFilter =.. [Filter,InS,OutS,[]].

%   type1_body(+Type1, +Sort, +TpFilter, +Link, +InS, +Args,
%              -RHS, +Tail, +OutStream)
%   () üΥǡ Type1 饿ף RHS 
%   롥InS ϥȥ꡼Args  () ü
%   Tail ΤSort ü椫ü椫
%   Τ餻ե饰TpFilter ϲͽ¬ե륿Ҹ̾
%   Link ϥ󥯽Ҹ̾
%
type1_body([], _, _, _, _, _, RHS, RHS, _) :- !.
type1_body([[H|Data]|Rest], Sort, TpFilter, Link, InS, Args,
	        RHS, Tail, OutStr) :-
	(   Sort = nonterminal, user:sax_trans_tp_filter(strong) ->
	    H = HF/HA,
	    concat_atoms(HF, HA, FA),
	    concat_atoms(TpFilter, '_', TpFilter1),
	    concat_atoms(TpFilter1, FA, TpFilter_FA),
	    concat_atoms(Link, '_', Link1),
	    concat_atoms(Link1, FA, Link_FA),
	    produce_type1_tp_filter(TpFilter_FA, Link_FA, OutStr),
	    type1_tp_filter(TpFilter_FA, InS, NewInS, RHS, Body)
	;   NewInS = InS, RHS = Body
	),
	type1_conjunct(Data, NewInS, Args, Body, Body1),
	!, type1_body(Rest, Sort, TpFilter, Link, InS, Args,
	        Body1, Tail, OutStr).

%   type1_conjunct(+Data, +InS, +Args, -RHS, +Tail)
%   () üΥǡ Data 饿ף RHS 
%   롥InS ϥȥ꡼Args  () ü
%   Tail Τ
%
type1_conjunct([], _, _, RHS, RHS) :- !.
type1_conjunct([Data|Rest], InS, Args, (S,RHS), Tail) :-
        copy_term(Data, CopyOfData),
	type1_single(CopyOfData, InS, Args, S),
	!, type1_conjunct(Rest, InS, Args, RHS, Tail).

%   type1_single(+Data, +InS, +Args, -S)
%   () üΥǡ Data 饿ףΤΰĤ
%    S 롥InS ϥȥ꡼Args  ()
%   üΰ
%
type1_single(sgldict(Word,Head+DExtra,Exs)-_, InS, Args, S) :- !,
	Word =.. [Func|Args],
	Head =.. [HFunc|HArgs],
	make_delayed_extra(HFunc, DExtra, Dx),
	make_inactive(HFunc, HArgs,
	        InS, [Dx,Func], Exs, S).
type1_single(sgl(NTerm,Head+DExtra,Exs)-_, InS, Args, S) :- !,
	NTerm =.. [_|Args],
	Head =.. [HFunc|HArgs],
	make_delayed_extra(HFunc, DExtra, Dx),
	make_inactive(HFunc, HArgs,
	        InS, [Dx,'$VAR'('Dx')], Exs, S).
type1_single(lcdict(Word,Exs)-(PutId,PutVars), InS, Args, S) :- !,
	(   user:sax_trans_idterm(long) -> all_vars(PutVars, IdVars)
	;   c_vars(PutVars, IdVars)
	),
	Word =.. [Func|Args],
	make_active(PutId, IdVars,
	        InS, [Func], Exs, S).
type1_single(lc(NTerm,Exs)-(PutId,PutVars), InS, Args, S) :-
	(   user:sax_trans_idterm(long) -> all_vars(PutVars, IdVars)
	;   c_vars(PutVars, IdVars)
	),
	NTerm =.. [_|Args],
	make_active(PutId, IdVars,
	        InS, ['$VAR'('Dx')], Exs, S).

%   type1_link(+RootList, +Left)
%    Left ǡ RootList γüؤ
%   ãǽϿ롥
%
type1_link([], _) :- !.
type1_link([Root-_|Rest], Left) :-
	make_link(Left, Root),
	!, type1_link(Rest, Left).

%   produce_type1_tp_filter(+Filter, +Link, +OutStream)
%   ͽ¬ե륿 Filter ꡤ OutStream ˽Ϥ롥
%   Link ϥ󥯽Ҹ̾
%
produce_type1_tp_filter(Filter, Link, OutStr) :-
	(   user:sax_trans_block(yes) ->
	    produce_type1_tp_filter_block(Filter, OutStr)
	;   true
	),
	produce_type1_tp_filter_nil(Filter, OutStr),
	produce_type1_tp_filter_hash(Filter, Link, OutStr),
	produce_type1_tp_filter_tail(Filter, OutStr),
	nl(OutStr), !.

produce_type1_tp_filter_block(Filter, OutStr) :-
	Block =.. [Filter,-,?,?],
	print_clause(OutStr, (:- block Block)),
	nl(OutStr).

produce_type1_tp_filter_nil(Filter, OutStr) :-
	LHS =.. [Filter,[]],
	print_clause(OutStr, (LHS --> [])).

produce_type1_tp_filter_hash(Filter, Link, OutStr) :-
	LHS =.. [Filter,['$VAR'('Id')|'$VAR'('Tail')]],
	Hash =.. [Link,'$VAR'('IdF')],
	Tail =.. [Filter,'$VAR'('Tail')],
	F = functor('$VAR'('Id'),'$VAR'('IdF'),'$VAR'('_')),
	S = ['$VAR'('Id')],
	print_clause(OutStr, (LHS --> {F},{Hash},S,!,Tail)).

produce_type1_tp_filter_tail(Filter, OutStr) :-
	LHS =.. [Filter,['$VAR'('_')|'$VAR'('Tail')]],
	Tail =.. [Filter,'$VAR'('Tail')],
	print_clause(OutStr, (LHS --> Tail)).

%%% Produce Type2 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   produce_type2(+Type2, +Sort, +Func, +Arity, +OutStream)
%   () ü Func/Arity Υǡ Type2 饿ף
%   ꡤOutStream ˽Ϥ롥Sort Ͻü椫ü
%   Τ餻ե饰
%
produce_type2([], _, _, _, _) :- !.
produce_type2(Type2, Sort, Func, Arity, OutStr) :-
	concat_atoms(Func, '_tp2', Func_tp2),
	concat_atoms(Func, '_h', Func_h),
	(   user:sax_trans_block(yes) ->
	    produce_type2_block(Sort, Func_tp2, Arity, OutStr)
	;   true
	),
	produce_type2_nil(Sort, Func_tp2, Arity, OutStr),
	produce_type2_hash(Sort, Func_tp2, Arity, Func_h, OutStr),
	produce_type2_tail(Sort, Func_tp2, Arity, OutStr),
	nl(OutStr),
	produce_type2_clause(Type2, Sort, Func_h, Arity, Func, OutStr),
	nl(OutStr).

produce_type2_nil(Sort, Func, Arity, OutStr) :-
	make_term(Sort, Func, Arity, _,
	        [], '$VAR'('_'), LHS),
	print_clause(OutStr, (LHS --> [])).

produce_type2_hash(Sort, Func, Arity, Func_h, OutStr) :-
	make_term(Sort, Func, Arity, Args,
	        ['$VAR'('Id')|'$VAR'('Tail')], '$VAR'('Dx'), LHS),
	make_term(Sort, Func_h, Arity, Args,
	        '$VAR'('Id'), '$VAR'('Dx'), Hash),
	make_term(Sort, Func, Arity, Args,
	        '$VAR'('Tail'), '$VAR'('Dx'), Tail),
	print_clause(OutStr, (LHS --> Hash,!,Tail)).

produce_type2_tail(Sort, Func, Arity, OutStr) :-
	make_term(Sort, Func, Arity, Args,
	        ['$VAR'('_')|'$VAR'('Tail')], '$VAR'('Dx'), LHS),
	make_term(Sort, Func, Arity, Args,
	        '$VAR'('Tail'), '$VAR'('Dx'), Tail),
	print_clause(OutStr, (LHS --> Tail)).

%   produce_type2_block(+Sort, +Func, +Arity, +OutStream)
%   () ü Func/Arity  block ꡤ
%   OutStream ˽Ϥ롥Sort Ͻü椫ü椫
%   餻ե饰
%
produce_type2_block(Sort, Func, Arity, OutStr) :-
	% +2 for 'OutS' and 'OutSTail'
	Arity1 is Arity + 2,
	make_term(Sort, Func, Arity1, Args, -, ?, LHS),
	block_args(Args),
	print_clause(OutStr, (:- block LHS)),
	nl(OutStr), !.

block_args([]) :- !.
block_args([?|Rest]) :- block_args(Rest).

%   produce_type2_clause(+Type2, +Sort, +Func, +Arity, +Func_l, +OutStream)
%   () ü Func/Arity Υǡ Type2 饿ף
%   ꡤOutStream ˽Ϥ롥Sort Ͻü椫ü
%   Τ餻ե饰Func_l  () ü̾
%
produce_type2_clause([], _, _, _, _, _) :- !.
produce_type2_clause([[GetId+GetVars|Type2]|Rest],
	        Sort, Func, Arity, Func_l, OutStr) :-
        (   user:sax_trans_tp_filter(Use), ( Use = strong ; Use = weak ) ->
	    make_id_pair(GetId, Func_l, Arity)
	;   true
	),
	(   user:sax_trans_idterm(long) -> all_vars(GetVars, IdVars)
	;   c_vars(GetVars, IdVars)
	),
	length(IdVars, IdArity),
	make_term(nonterminal, GetId, IdArity, IdArgs,
	        '$VAR'('InS'), '$VAR'('DxP'), IdTerm),
	make_term(Sort, Func, Arity, Args,
	        IdTerm, '$VAR'('Dx'), LHS),
	type2_body(Type2, IdArgs, Args, _, (_,RHS)),
        print_clause(OutStr, (LHS --> RHS)),
	!, produce_type2_clause(Rest, Sort, Func, Arity, Func_l, OutStr).

%   type2_body(+Type2, +IdArgs, +Args, +First, -RHS)
%   () üΥǡ Type2 饿ף RHS 
%   롥IdArgs  () ü椬ȤɣΰArgs
%    () üΰFirst ΤƬ
%
type2_body([], _, _, RHS, RHS) :- !.
type2_body([Data|Rest], IdArgs, Args, First, (First,RHS)) :-
	copy_term(Data, CopyOfData),
	type2_single(CopyOfData, IdArgs, Args, S),
	!, type2_body(Rest, IdArgs, Args, S, RHS).

%   type2_single(+Data, +IdArgs, +Args, -S)
%   () üΥǡ Data 饿ףΤΰĤ
%    S 롥IdArgs  () ü椬Ȥɣ
%   Args  () üΰ
%
type2_single(lstdict(Word,GetVars,Head+DExtra,Exs)-_, IdArgs, Args, S) :- !,
	(   user:sax_trans_idterm(long) -> all_vars(GetVars, IdArgs)
	;   c_vars(GetVars, IdArgs)
	),
	Word =.. [Func|Args],
	Head =.. [HFunc|HArgs],
	make_delayed_extra(HFunc, DExtra, Dx),
	make_inactive(HFunc, HArgs,
	        '$VAR'('InS'), [Dx,Func|'$VAR'('DxP')], Exs, S).
type2_single(lst(NTerm,GetVars,Head+DExtra,Exs)-_, IdArgs, Args, S) :- !,
	(   user:sax_trans_idterm(long) -> all_vars(GetVars, IdArgs)
	;   c_vars(GetVars, IdArgs)
	),
	NTerm =.. [_|Args],
	Head =.. [HFunc|HArgs],
	make_delayed_extra(HFunc, DExtra, Dx),
	make_inactive(HFunc, HArgs,
	        '$VAR'('InS'), [Dx,'$VAR'('Dx')|'$VAR'('DxP')], Exs, S).
type2_single(middict(Word,GetVars,Exs)-(PutId,PutVars), IdArgs, Args, S) :- !,
	(   user:sax_trans_idterm(long) ->
	    all_vars(GetVars, IdArgs),
	    all_vars(PutVars, IdVars)
	;   c_vars(GetVars, IdArgs),
	    c_vars(PutVars, IdVars)
	),
	Word =.. [Func|Args],
	make_active(PutId, IdVars,
	        '$VAR'('InS'), [Func|'$VAR'('DxP')], Exs, S).
type2_single(mid(NTerm,GetVars,Exs)-(PutId,PutVars), IdArgs, Args, S) :-
	(   user:sax_trans_idterm(long) ->
	    all_vars(GetVars, IdArgs),
	    all_vars(PutVars, IdVars)
	;   c_vars(GetVars, IdArgs),
	    c_vars(PutVars, IdVars)
	),
	NTerm =.. [_|Args],
	make_active(PutId, IdVars,
	        '$VAR'('InS'), ['$VAR'('Dx')|'$VAR'('DxP')], Exs, S).

%%% Produce Link %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   produce_link(+LTable, +OutStream)
%   󥯡ǡ LTable ꡤOutStream ˽
%   Ϥ롥
%
produce_link([], _) :- !.
produce_link([[Func/Arity|Data]|Rest], OutStr) :-
	concat_atoms(Func, Arity, FA),
	concat_atoms('link_', FA, Link),
	produce_link_clause(Data, Link, OutStr),
	!, produce_link(Rest, OutStr).

%   produce_link_clause(+Data, +Link, +OutStream)
%   󥯽Ҹ Link Υǡ Data ꡤ
%   OutStream ˽Ϥ롥
%
produce_link_clause([], _, _) :- !.
produce_link_clause([Goal-_|Rest], Link, OutStr) :-
	(   user:sax_trans_tp_filter(strong) ->
	    Goal = GF/GA,
	    concat_atoms(GF, GA, FA),
	    concat_atoms(Link, '_', Link1),
	    concat_atoms(Link1, FA, Link_FA)
	;   Link_FA = Link
	),
	produce_link_single(Goal, Link_FA, OutStr),
	!, produce_link_clause(Rest, Link, OutStr).

%   produce_link_single(+Goal, +Link, +OutStream)
%   󥯽Ҹ Link ΣĤΥǡ Goal ꡤ
%   OutStream ˽Ϥ롥
%
produce_link_single(Goal, Link, OutStr) :-
	link(Goal, Root),
	id_pair(Id, Root),
	produce_link_single1(Link, Id, OutStr).
produce_link_single(_, _, _).

produce_link_single1(Link, Id, OutStr) :-
	(   done(Link, Id)
	;   assert(done(Link,Id)),
	    LHS =.. [Link,Id],
	    print_clause(OutStr, LHS)
	),
	!, fail.

%%% Produce Utils %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   make_term(+Sort, +Func, +Arity, ?Args, +InS, +Dx, -Term)
%    Func/Arity 饹ȥ꡼ InS, ٱ䶯 Dx 
%   Ĺ Term 롥Sort Ͻü椫ü椫
%   餻ե饰Args Ϲΰ
%
make_term(dict, Func, Arity, Args, InS, _, Term) :-
	length(Args, Arity),
	make_phrase(Func, [InS|Args], Term).
make_term(nonterminal, Func, Arity, Args, InS, Dx, Term) :-
	length(Args, Arity),
	make_phrase(Func, [InS,Dx|Args], Term).

%   make_delayed_extra(+Func, +DExtra, -Dx)
%   ü Func ٱ䶯 DExtra Υǡ Dx 롥
%
make_delayed_extra(Func, DExtra, Dx) :-
	Dx =.. [Func,DExtra].

%   make_inactive(+Func, +Args, +InS, +Dx, +Exs, -S)
%   ü Func/Args 饹ȥ꡼ InS, ٱ䶯
%   Dx 䶯 Exs Գ̥ǡ S 롥
%
make_inactive(Func, Args, InS, Dx, Exs, [ie(InActive,Exs1)]) :-
	InActive =.. [Func,InS,Dx|Args],
	put_extras(Exs, Exs1).

%   make_active(+Id, +IdArgs, +InS, +Dx, +Exs, -S)
%   ɣ Id/IdArgs 饹ȥ꡼ InS, ٱ䶯 Dx,
%   䶯 Exs ĳ̥ǡ S 롥
%
make_active(Id, IdArgs, InS, Dx, Exs, [ae(Active,Exs1)]) :-
	Active =.. [Id,InS,Dx|IdArgs],
	put_extras(Exs, Exs1).

%   put_extras(+Exs, -S)
%   䶯 Exs Υ S 롥
%
put_extras([], true) :- !.
put_extras(Exs, S) :- put_extras1(Exs, S).

put_extras1([Extra], Extra) :- !.
put_extras1([Extra|Rest], (Extra,S)) :- put_extras1(Rest, S).

%   make_id_pair(+Id, +Func, +Arity)
%   ̻ Id ü Func/Arity Ͽ롥
%
make_id_pair(Id, Func, Arity) :- assert(id_pair(Id, Func/Arity)).

%   make_link(+Goal, +Root)
%   Goal  Root ؤãǽϿ롥
%
make_link(Goal, Root) :-
	(   link(Goal, Root)
	;   make_link1(Goal, Root)
	).

make_link1(Goal, Root) :-
	(   link(Leaf, Goal),
	    \+ make_link2(Root, Leaf)
	;   true
	).

make_link2(Root, Leaf) :-
	(   link(Leaf, Root)
	;   link(Root, Ground),
	    \+ make_link3(Leaf, Ground)
	;   true
	).

make_link3(Leaf, Ground) :-
	(   link(Leaf, Ground)
	;   assert(link(Leaf,Ground))
	).

%   all_vars(+Vars, -CVars)
%   ѿΥꥹ Vars 椫ѿ򽸤 CVars Ȥ롥
%
all_vars([], []) :- !.
all_vars([(Var,_)|Rest], [Var|Rest1]) :-
	all_vars(Rest, Rest1).

%   c_vars(+Vars, -CVars)
%   ѿΥꥹ Vars ΤȤ줿Τ򽸤 CVars
%   Ȥ롥
%
c_vars([], []) :- !.
c_vars([(Var,YN)|Rest], CVars) :- YN == yes, !,
	CVars = [Var|Rest1],
	c_vars(Rest, Rest1).
c_vars([_|Rest], CVars) :-
	c_vars(Rest, CVars).

