% tfunify.pl (17 Nov 1995) by Takuya NAKAYAMA (takuya-n@is.aist-nara.ac.jp)
%     Copyright (C) 1995 Takuya NAKAYAMA, Yasuharu DEN
%
% This program is an extension of ...
%
% funify.pl (26 January 1993) by Yasuharu DEN (den@forest.kuee.kyoto-u.ac.jp)
% BUG fix   (13 June 1994) by Yasuharu DEN (den@itl.atr.co.jp)
%
:- use_module(library(lists)).
:- module(tfunify, [
	funify/3,
	funify1/3,
	funify1org/3,
	funify/2,
	fcopy_term/2,
	freduce_term/2,
	tfs_to_fs/2,
	fs_to_tfs/2
		  ]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Feature Structure Unification
%
% This is based on Wroblewski's algorithm.
% The data structure of user terms is as follows:
%
%	  <Term>              ::= <Feature Structure> | <Type>
%	  <Feature Structure> ::= <ID> | <ID>#<Type>/<List>
%	  <List> ::= a list of <Feature Value Pair> |
%		     a list of <Feature Structure>  | 
%		     a set of <Feature Structure> 
%	  <Feature Value Pair> ::= <Attribute>:<Term>
%	  <ID>        ::= a variable
%	  <Attribute> ::= an atomic
%	  <Type>      ::= an atomic
%
% Restrictions:
%
% 1. A feature structure must be well-founded, i.e. it contains no cycle.
% 2. Feature value pairs must be sorted in alphabetical order of slots.
% 3. IDs are identical if and only if arc lists are identical.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% TERMs:
%    FS  ... A feature structure written in the system format.
%            (ID#ArcList)
%    SFS ... A sorted feature structure written in the system format.
%            (ID#Sort/ArcList)
%      * {SFS} are included in {FS}. i.e. is_tfs(X) => is_fs(X)
%    TFS ... A feature structure written in the user format.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   funify(+FS1, +FS2, ?FS3)
%
funify(X, Y, Z) :-
	funify2(X, Y, Z).

%   funify(+FS1, +FS2)
%
funify(X, Y) :- funify0(X, Y).

%   fcopy_term(+FS, ?FS1)
%
fcopy_term(X, Y) :-
	funify2(X, _, Y).

%   freduce_term(+FS, ?FS1)
%
freduce_term(X, Y) :-
	freduce_term(X, F, [], _),
	!, Y = F.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   is_sfs(+SFS)
%
is_sfs(X) :-
	nonvar(X),
	X = _#_/_, !.

%   is_fs(+FS)
%
is_fs(X) :-
	nonvar(X),
	X = _#_, !.

%   create_sfs(-SFS)
%
create_sfs(X) :- X = _#_/_.

%   create_fs(-FS)
%
create_fs(X) :- X = _#_.

%   fs_to_sfs(+FS, -SFS)
%
fs_to_sfs(X, X2) :-
	is_fs(X),
	create_sfs(X2),
	id_of(X,  ID),
	id_of(X2, ID),
	arclist_of(X,  Type, ArcList),
	arclist_of(X2, Type, ArcList).

%   id_of(+FS, -ID)
%
%id_of(X, ID) :- arg(1, X, ID).
id_of(X, ID) :- X = ID#_.

%   arclist_of(+SFS, -Type, -ArcList)
%
arclist_of(X, Type, ArcList) :-
	is_sfs(X), !, X = _#Type/ArcList.
arclist_of(X, _, ArcList) :-
	is_fs(X), !, X = _#ArcList.

%   dereference(+Term, -DeReference)
%       _#a/[....]#b/[....]#... =>  _#a/[....]
%
dereference(X, Y) :-
        is_fs(X),
        id_of(X, ID),
        nonvar(ID), !,
        dereference(ID, Y).
dereference(X, X).

%   forward(+Term, +Forward)
%
forward(X, Y) :- id_of(X, Y).

%   decopy(+Term, -DeCopy, +Copy)
%
decopy(X, Y, Copy) :- (var(X); is_fs(X)), !, assoc(Copy, X, Y).

%   copy(+Term, +DeCopy, +Copy, -NewCopy)
%
copy(X, Y, Copy, NewCopy) :-
	NewCopy = [X-Y|Copy].
	
%   assoc(+AList, +Key, ?Value)
%
assoc([K-V|_], X, X1) :- X == K, !, X1 = V.
assoc([_|R], X, X1) :- assoc(R, X, X1).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   coexist(+Type1, +Type2, -NewType)
%      checks whether Type1 coexists with Type2 or not.
%
coexist(Type1, Type2, Type) :-
	user:coexist(Type1, Type2, Type), !.
coexist(Type, Type, Type) :- !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   funify2(+FS1, +FS2, -FS3)
%   Nondestructively unify two terms with incremental copying
%
:- dynamic f2tmp/1.

funify2(X, Y, Z) :-
	stackall_result(X, Y),
	retract(f2tmp(Z)), !.

stackall_result(X, Y) :-
	funify0(X, Y), assertz(f2tmp(X)), fail; true.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   funify1(+FS1, +FS2, -FS3)
%   Destructively unify two terms preserving the 2nd argument
funify1(X, Y, Z) :-
	stackall_result11(X, Y),!,
	X = Z.

stackall_result11(X, Y) :-
	funify10(X, Y), assertz(f2tmp(X)), fail; retract(f2tmp(X)).

funify10(X, Y) :-
	dereference(X, X1),
	dereference(Y, Y1),
	!, xfunify10(X1, Y1).

xfunify10(X, Y) :- X == Y, !.
xfunify10(X, Y) :- var(X), !, funify2(Y,_,X).
xfunify10(X, Y) :- var(Y), !, X = Y.
xfunify10([X|XR], [Y|YR]) :- !,
	funify10(X, Y),
	!, xfunify10(XR, YR).
xfunify10(X, Y) :- is_fs(X), is_fs(Y), !,
	arclist_of(X, TypeX, XC),
	arclist_of(Y, TypeY, YC),
	coexist(TypeX, TypeY, Type),
	funify_fs0(XC, YC, ZC),
	(   XC == ZC, Type == TypeX ->
	    true
	;   create_sfs(Z),
	    arclist_of(Z, Type, ZC),
	    forward(X, Z)
	).
xfunify10(X, Y) :-
	X =.. [F|Xarg],
	Y =.. [F|Yarg],
	xfunify10(Xarg, Yarg).



%   funify_fs0(+ArcList1, +ArcList2, -ArcList3).
%
funify_fs10([], Y, Y) :- !.
funify_fs10(X, [], X) :- !.
funify_fs10([SX:VX|RestX], [SY:VY|RestY], Z) :-
	compare(Ord, SX, SY),
	funify_fs10(Ord, SX, VX, RestX, SY, VY, RestY, Z).

funify_fs10(=, S, VX, RestX, _, VY, RestY, [S:VX|RestZ]) :-
	funify10(VX, VY),
	!, funify_fs10(RestX, RestY, RestZ).
funify_fs10(<, SX, VX, RestX, SY, VY, RestY, [SX:VX|RestZ]) :-
	funify_fs10(RestX, [SY:VY|RestY], RestZ).
funify_fs10(>, SX, VX, RestX, SY, VY, RestY, [SY:VY|RestZ]) :-
	funify_fs10([SX:VX|RestX], RestY, RestZ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   funify0(+FS1, +FS2)
%   Destructively unify two terms
%
funify0(X, Y) :-
	dereference(X, X1),
	dereference(Y, Y1),
	!, xfunify0(X1, Y1).

xfunify0(X, Y) :- X == Y, !.
xfunify0(X, Y) :- ( var(X) ; var(Y) ), !, X = Y.
xfunify0([X|XR], [Y|YR]) :- !,
	funify0(X, Y),
	!, xfunify0(XR, YR).
xfunify0(X, Y) :- is_fs(X), is_fs(Y), !,
	arclist_of(X, TypeX, XC),
	arclist_of(Y, TypeY, YC),
	coexist(TypeX, TypeY, Type),
	funify_fs0(XC, YC, ZC),
	(   XC == ZC, Type == TypeX ->
	    true
	;   create_sfs(Z),
	    arclist_of(Z, Type, ZC),
	    forward(X, Z)
	),
	forward(Y, X).
xfunify0(X, Y) :-
	X =.. [F|Xarg],
	Y =.. [F|Yarg],
	xfunify0(Xarg, Yarg).

%   funify_fs0(+ArcList1, +ArcList2, -ArcList3).
%
funify_fs0([], Y, Y) :- !.
funify_fs0(X, [], X) :- !.
funify_fs0([SX:VX|RestX], [SY:VY|RestY], Z) :-
	compare(Ord, SX, SY),
	funify_fs0(Ord, SX, VX, RestX, SY, VY, RestY, Z).

funify_fs0(=, S, VX, RestX, _, VY, RestY, [S:VX|RestZ]) :-
	funify0(VX, VY),
	!, funify_fs0(RestX, RestY, RestZ).
funify_fs0(<, SX, VX, RestX, SY, VY, RestY, [SX:VX|RestZ]) :-
	funify_fs0(RestX, [SY:VY|RestY], RestZ).
funify_fs0(>, SX, VX, RestX, SY, VY, RestY, [SY:VY|RestZ]) :-
	funify_fs0([SX:VX|RestX], RestY, RestZ).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   freduce_term(+Term, -Term1, +Copy, -NewCopy)
%
freduce_term(X, Y, Copy, NewCopy) :-
        dereference(X, X1),
        !, xfreduce_term(X1, Y, Copy, NewCopy).

% xfreduce_term(X, Y, Copy, Copy) :- decopy(X, Y, Copy), !.
xfreduce_term(X, Y, Copy, Copy) :- is_fs(X), assoc(Copy, X, Y), !.
xfreduce_term(X, X, Copy, Copy) :- var(X), !.
xfreduce_term(X, Y, Copy, NewCopy) :- is_fs(X), !,
        create_sfs(Y),
	id_of(X, ID),
	id_of(Y, ID),
        arclist_of(X, Type, XC),
        arclist_of(Y, Type, YC),
        freduce_term_fs(XC, YC, Copy, NCopy),
        copy(X, Y, NCopy, NewCopy).
xfreduce_term(X, Y, Copy, NewCopy) :-
        functor(X, F, N),
        functor(Y, F, N),
        loop_freduce_term(0, N, X, Y, Copy, NewCopy).

loop_freduce_term(N, N, _, _, Copy, Copy) :- !.
loop_freduce_term(M, N, X, Y, Copy, NewCopy) :-
        M1 is M + 1,
        arg(M1, X, A),
        arg(M1, Y, B),
        freduce_term(A, B, Copy, NCopy),
        loop_freduce_term(M1, N, X, Y, NCopy, NewCopy).

%   freduce_term_fs(+ArcList, -ArcList1, +Copy, -NewCopy)
%
freduce_term_fs([], [], Copy, Copy) :- !.
freduce_term_fs([S:V|Rest], [S:V1|Rest1], Copy, NewCopy) :-
	freduce_term(V, V1, Copy, NCopy),
	!, freduce_term_fs(Rest, Rest1, NCopy, NewCopy).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Feature Structure Utilities
%
%   User terms <==> System terms
%
% The data structure of user terms is as follows:
%
%   <Term> ::= <Variable> | <Feature Structure> | <Compound Term>
%   <Variable> ::= a variable
%   <Feature Structure> ::= <Tag>#<Feature List> | <Feature List> | <Tag>
%   <Tag> ::= a variable
%   <Feature List> ::= a list of <Feature Value Pair>'s
%   <Feature Value Pair> ::= <Slot>:<Term>
%   <Slot> ::= an atomic
%   <Compound Term> ::= <Functor> followed by a sequence of <Arguments>'s
%   <Functor> ::= atom
%   <Arguments> ::= <Term>
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   tfs_to_fs(+UserTerm, ?SystemTerm)
%
tfs_to_fs(X, Y) :-
	tfs_to_fs(X, Z, [], Copy),
	copy_blocked_goals(Copy),
	!, Y = Z.

%   fs_to_tfs(+SystemTerm, ?UserTerm)
%
fs_to_tfs(X, Y) :-
	fs_to_tfs1(X, [], Copy0),
	fs_to_tfs2(X, Z, Copy0, [], Copy),
	copy_blocked_goals(Copy),
	!, Y = Z.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   is_tfs(+TFS, -Tag, -ArcList)
%
is_tfs(X, T, C) :-
	nonvar(X),
	(   functor(X, #, 2) -> tag_of(X, T), featurelist_of(X, C)
	;   X = [_:_|_] -> T = free, C = X
	).

%   create_tfs(-TFS, +Tag, +ArcList)
%
create_tfs(X, T, C) :-
	(   T == free -> X = C
	;   functor(X, #, 2),
	    tag_of(X, T),
	    featurelist_of(X, C)
	).

%   tag_of(+TFS, -Tag)
%
tag_of(X, Tag) :- arg(1, X, Tag).

%   featurelist_of(+FS, -FeatureList)
%
featurelist_of(X, FeatureList) :- arg(2, X, FeatureList).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   tfs_to_fs(+UserTerm, ?SystemTerm, +Copy, -NewCopy)
%
% tfs_to_fs(X, Y, Copy, Copy) :- decopy(X, Y, Copy), !.
tfs_to_fs(X, Y, Copy, Copy) :- var(X), assoc(Copy, X, Y), !.
tfs_to_fs(X, Y, Copy, NewCopy) :- var(X), !,
	copy(X, Y, Copy, NewCopy).
tfs_to_fs(X, Y, Copy, NewCopy) :- is_tfs(X, T, XC), !,
	create_sfs(Y),
	arclist_of(Y, _, YC),
	tfs_to_fs_arclist(XC, YC, Copy, NCopy),
	(   T == free -> NewCopy = NCopy
	;   decopy(T, Z, Copy) -> funify(Y, Z), NewCopy = NCopy
	;   copy(T, Y, NCopy, NewCopy)
	).
tfs_to_fs(X, Y, Copy, NewCopy) :-
	functor(X, F, N),
	functor(Y, F, N),
	loop_tfs_to_fs(0, N, X, Y, Copy, NewCopy).

loop_tfs_to_fs(N, N, _, _, Copy, Copy) :- !.
loop_tfs_to_fs(M, N, X, Y, Copy, NewCopy) :-
	M1 is M + 1,
	arg(M1, X, A),
	arg(M1, Y, B),
	tfs_to_fs(A, B, Copy, NCopy),
	!, loop_tfs_to_fs(M1, N, X, Y, NCopy, NewCopy).

%   tfs_to_fs_arclist(+UserArcList, -SystemArcList, +Copy, -NewCopy)
%
tfs_to_fs_arclist(XC, YC, Copy, NewCopy) :-
	sort(XC, C),
	tfs_to_fs_arclist1(C, YC, Copy, NewCopy).

tfs_to_fs_arclist1([], [], Copy, Copy) :- !.
tfs_to_fs_arclist1([S:V|Rest], [S:V1|Rest1], Copy, NewCopy) :-
	tfs_to_fs(V, V1, Copy, NCopy),
	!, tfs_to_fs_arclist1(Rest, Rest1, NCopy, NewCopy).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   fs_to_tfs1(+SystemTerm, +Copy, -NewCopy)
%
fs_to_tfs1(X, Copy, NewCopy) :-
	dereference(X, X1),
	!, xfs_to_tfs1(X1, Copy, NewCopy).

xfs_to_tfs1(X, Copy, Copy) :- decopy(X, (yes,_), Copy), !.
xfs_to_tfs1(X, Copy, NewCopy) :- var(X), !,
	copy(X, _, Copy, NewCopy).
xfs_to_tfs1(X, Copy, NewCopy) :- is_fs(X), !,
	arclist_of(X, _, XC),
	fs_to_tfs1_arclist(XC, Copy, NCopy),
	copy(X, _, NCopy, NewCopy).
xfs_to_tfs1(X, Copy, NewCopy) :-
	functor(X, _, N),
	loop_fs_to_tfs1(0, N, X, Copy, NewCopy).

loop_fs_to_tfs1(N, N, _, Copy, Copy) :- !.
loop_fs_to_tfs1(M, N, X, Copy, NewCopy) :-
	M1 is M + 1,
	arg(M1, X, A),
	fs_to_tfs1(A, Copy, NCopy),
	!, loop_fs_to_tfs1(M1, N, X, NCopy, NewCopy).

%   fs_to_tfs1_arclist(+SystemArcList, +Copy, -NewCopy)
%
fs_to_tfs1_arclist([], Copy, Copy) :- !.
fs_to_tfs1_arclist([_:V|Rest], Copy, NewCopy) :-
	fs_to_tfs1(V, Copy, NCopy),
	!, fs_to_tfs1_arclist(Rest, NCopy, NewCopy).

%   fs_to_tfs2(+SystemTerm, -UserTerm, +Copy0, +Copy, -NewCopy)
%
fs_to_tfs2(X, Y, Copy0, Copy, NewCopy) :-
	dereference(X, X1),
	!, xfs_to_tfs2(X1, Y, Copy0, Copy, NewCopy).

xfs_to_tfs2(X, Y, Copy0, Copy, NewCopy) :- var(X), !,
	decopy(X, (yes,Y), Copy0),
	copy(X, Y, Copy, NewCopy).
xfs_to_tfs2(X, Y, Copy0, Copy, NewCopy) :- is_fs(X), !,
	decopy(X, (YorN1,YorN2,T), Copy0),
	(   YorN1 == yes, YorN2 == yes -> Y = T, NewCopy = Copy
	;   ( YorN1 == yes ; T = free ),
	    YorN2 = yes,
	    create_tfs(Y, T, YC),
	    arclist_of(X, _, XC),
	    fs_to_tfs2_arclist(XC, YC, Copy0, Copy, NewCopy)
	).
xfs_to_tfs2(X, Y, Copy0, Copy, NewCopy) :-
	functor(X, F, N),
	functor(Y, F, N),
	loop_xfs_to_tfs2(0, N, X, Y, Copy0, Copy, NewCopy).

loop_xfs_to_tfs2(N, N, _, _, _, Copy, Copy) :- !.
loop_xfs_to_tfs2(M, N, X, Y, Copy0, Copy, NewCopy) :-
	M1 is M + 1,
	arg(M1, X, A),
	arg(M1, Y, B),
	xfs_to_tfs2(A, B, Copy0, Copy, NCopy),
	!, loop_xfs_to_tfs2(M1, N, X, Y, Copy0, NCopy, NewCopy).

%   fs_to_tfs2_arclist(+SystemArcList, -UserArcList, +Copy0, +Copy, -NewCopy)
%
fs_to_tfs2_arclist([], [], _, Copy, Copy) :- !.
fs_to_tfs2_arclist([S:V|Rest], [S:V1|Rest1], Copy0, Copy, NewCopy) :-
	fs_to_tfs2(V, V1, Copy0, Copy, NCopy),
	!, fs_to_tfs2_arclist(Rest, Rest1, Copy0, NCopy, NewCopy).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Blocked Goal Handling
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   copy_blocked_goals(+Copy)
%
copy_blocked_goals(Copy) :-
	collect_vars(Copy, Vars, NewVars),
	copy_term(Vars, NewVars), !.

%   collect_vars(+Copy, -Vars, -NewVars)
%
collect_vars([], [], []) :- !.
collect_vars([X-Y|Rest], Vars, NewVars) :- var(X), !,
	Vars = [X|Vars1], NewVars = [Y|NewVars1],
	collect_vars(Rest, Vars1, NewVars1).
collect_vars([_|Rest], Vars, NewVars) :- collect_vars(Rest, Vars, NewVars).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   fcopy_term(+Term, -Term1, +Copy, -NewCopy)
%
fcopy_term(X, Y, Copy, NewCopy) :-
        dereference(X, X1),
        !, xfcopy_term(X1, Y, Copy, NewCopy).

xfcopy_term(X, Y, Copy, Copy) :- decopy(X, Y, Copy), !.
xfcopy_term(X, Y, Copy, NewCopy) :- var(X), !,
        copy(X, Y, Copy, NewCopy).
xfcopy_term(X, Y, Copy, NewCopy) :- is_fs(X), !,
        create_sfs(Y),
        arclist_of(X, Type, XC),
        arclist_of(Y, Type, YC),
        fcopy_term_fs(XC, YC, Copy, NCopy),
        copy(X, Y, NCopy, NewCopy).
xfcopy_term(X, Y, Copy, NewCopy) :-
        functor(X, F, N),
        functor(Y, F, N),
        loop_fcopy_term(0, N, X, Y, Copy, NewCopy).

loop_fcopy_term(N, N, _, _, Copy, Copy) :- !.
loop_fcopy_term(M, N, X, Y, Copy, NewCopy) :-
        M1 is M + 1,
        arg(M1, X, A),
        arg(M1, Y, B),
        fcopy_term(A, B, Copy, NCopy),
        loop_fcopy_term(M1, N, X, Y, NCopy, NewCopy).

%   fcopy_term_fs(+ArcList, -ArcList1, +Copy, -NewCopy)
%
fcopy_term_fs([], [], Copy, Copy) :- !.
fcopy_term_fs([S:V|Rest], [S:V1|Rest1], Copy, NewCopy) :-
	fcopy_term(V, V1, Copy, NCopy),
	!, fcopy_term_fs(Rest, Rest1, NCopy, NewCopy).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   funify1org(+FS1, +FS2, +Copy, -NewCopy)
%   Destructively unify two terms preserving the 2nd argument
%
funify1org(X, Y, Z) :-
	funify1(X, Y, [], Copy),
	copy_blocked_goals(Copy),
	!, Z = X.

funify1(X, Y, Copy, NewCopy) :-
	dereference(X, X1),
	dereference(Y, Y1),
	!, xfunify1(X1, Y1, Copy, NewCopy).

xfunify1(X, Y, Copy, NewCopy) :- X == Y, !,
	NewCopy = Copy.
xfunify1(X, Y, Copy, NewCopy) :- decopy(Y, Y1, Copy), !,
	funify0(X, Y1),
	NewCopy = Copy.
xfunify1(X, Y, Copy, NewCopy) :- var(X), !,
	fcopy_term(Y, Y1, Copy, NewCopy),
	X = Y1.
xfunify1(X, Y, Copy, NewCopy) :- var(Y), !,
	copy(Y, X, Copy, NewCopy).
xfunify1(X, Y, Copy, NewCopy) :- is_fs(X), is_fs(Y), !,
	arclist_of(X, Type1, XC),
	arclist_of(Y, Type2, YC),
	coexist(Type1, Type2, Type),
	funify_fs1(XC, YC, ZC, Copy, NCopy),
	(   XC == ZC
	;   create_fs(Z),
	    arclist_of(Z, Type, ZC),
	    forward(X, Z)
	),
	copy(Y, X, NCopy, NewCopy).
xfunify1(X, Y, Copy, NewCopy) :-
	functor(X, F, N),
	functor(Y, F, N),
	loop_funify1(0, N, X, Y, Copy, NewCopy).

loop_funify1(N, N, _, _, Copy, Copy) :- !.
loop_funify1(M, N, X, Y, Copy, NewCopy) :-
	M1 is M + 1,
	arg(M1, X, A),
	arg(M1, Y, B),
	funify1(A, B, Copy, NCopy),
	!, loop_funify1(M1, N, X, Y, NCopy, NewCopy).

%   funify_fs1(+ArcList1, +ArcList2, -ArcList3, +Copy, -NewCopy)
%
funify_fs1([], [], [], Copy, Copy) :- !.
funify_fs1([], Y, Z, Copy, NewCopy) :- !,
	fcopy_term_fs(Y, Z, Copy, NewCopy).
funify_fs1(X, [], X, Copy, Copy) :- !.
funify_fs1([SX:VX|RestX], [SY:VY|RestY], Z, Copy, NewCopy) :-
	compare(Ord, SX, SY),
	funify_fs1(Ord, SX, VX, RestX, SY, VY, RestY, Z, Copy, NewCopy).

funify_fs1(=, S, VX, RestX, _, VY, RestY, [S:VX|RestZ], Copy, NewCopy) :-
	funify1(VX, VY, Copy, NCopy),
	!, funify_fs1(RestX, RestY, RestZ, NCopy, NewCopy).
funify_fs1(<, SX, VX, RestX, SY, VY, RestY, [SX:VX|RestZ], Copy, NewCopy) :-
	funify_fs1(RestX, [SY:VY|RestY], RestZ, Copy, NewCopy).
funify_fs1(>, SX, VX, RestX, SY, VY, RestY, [SY:VY1|RestZ], Copy, NewCopy) :-
	fcopy_term(VY, VY1, Copy, NCopy),
	!, funify_fs1([SX:VX|RestX], RestY, RestZ, NCopy, NewCopy).

