% Morphological Analyzer for English
%
%     Copyright (C) 1993 Yuji Matsumoto
%     written  by Yuji Matsumoto   (matsu@pine.kuee.kyoto-u.ac.jp)
%                 Takehito Utsuro (utsuro@pine.kuee.kyoto-u.ac.jp)
%                 Yasuharu Den     (den@forest.kuee.kyoto-u.ac.jp)

:- module(morph, [
	morph/2
		 ]).

:- use_module(library(lists), [reverse/2,append/3]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Morph
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   morph(+Sentence, -MorphList)
%
morph(Sentence, MorphList) :-
	morph(Sentence, MorphList, [], [0], 0, 0).

morph([], MorphList, MorphList, _, _, _) :- !.
morph([Word|Rest], MorphList, Tail, PreIDL, From, ID) :-
	findall(Connect-Morph, morph_word(Word,Morph,Connect), Morphs),
	( Morphs \== []
	; format(user_error, '~w: undefined ... ', [Word]), !, fail
	), !,
	classify(Morphs, [], Class),
	Pos is From + 1,
	make_morph(Class, MorphList, MorphList1, PreIDL, From,
	        NewPreIDL, [], Pos, NewPos, ID, NewID),
	!, morph(Rest, MorphList1, Tail, NewPreIDL, NewPos, NewID).

%   classify(+Morphs, +Class, -NewClass)
%
classify([], Class, Class) :- !.
classify([Connect-Morph|Rest], Class, NewClass) :-
	add_class(Class, Connect, Morph, NClass),
	!, classify(Rest, NClass, NewClass).

add_class([], Key, Value, [Key-[Value]]) :- !.
add_class([Key-Class|Rest], Key, Value, [Key-[Value|Class]|Rest]) :- !.
add_class([KC|Rest], Key, Value, [KC|Rest1]) :-
	add_class(Rest, Key, Value, Rest1).

%   make_morph(+Class, -MorphList, ?MorphListTail, +PreIDL, +From,
%              +IDL, ?IDLTail, +Pos, -NewPos, +ID, -NewID)
%
make_morph([], MorphList, MorphList, _, _, IDL, IDL, Pos, Pos, ID, ID) :- !.
make_morph([(Prefix+Suffix)-Class|Rest], MorphList, Tail, PreIDL, From,
	        IDL, IDLTail, Pos, NewPos, ID, NewID) :-
	findall(SMorph, morph_suffix(Suffix,SMorph), SClass),
        make_morph1(Class, MorphList, MorphList1, Prefix, PreIDL, From, Pos,
	        IDL0, [], ID, ID1),
	make_morph1(SClass, MorphList1, MorphList2, Suffix, IDL0, Pos, NewPos,
	        IDL, IDL1, ID1, ID2),
        Pos1 is Pos + 1,
	!, make_morph(Rest, MorphList2, Tail, PreIDL, From,
	        IDL1, IDLTail, Pos1, NewPos, ID2, NewID).
make_morph([Word-Class|Rest], MorphList, Tail, PreIDL, From,
	        IDL, IDLTail, Pos, NewPos, ID, NewID) :-
        make_morph1(Class, MorphList, MorphList1, Word, PreIDL, From, NewPos,
	        IDL, IDL1, ID, ID1),
	!, make_morph(Rest, MorphList1, Tail, PreIDL, From,
	        IDL1, IDLTail, Pos, NewPos, ID1, NewID).

make_morph1([], MorphList, MorphList, _, _, _, _, IDL, IDL, ID, ID) :- !.
make_morph1([morph(Root,Cat,Args)|Rest],
	        [morph(ID,From,To,Word,Root,Cat,Args,PreIDL)|MorphList], Tail,
		Word, PreIDL, From, To, [ID|IDL], IDLTail, ID, NewID) :-
        ID1 is ID + 1,
	make_morph1(Rest, MorphList, Tail, Word, PreIDL, From, To,
	        IDL, IDLTail, ID1, NewID).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Morph Word
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   桼: dict(Word, Cat, Args)
%
%       Word ñ
%       Cat  ʻ
%       Args ¾ξ
%

%   morph_word(+Word, -Morph, -Connect)
%
morph_word(Word, Morph, Connect) :-
	user:dict(Word, Cat, Args),
	Morph = morph(Word,Cat,Args),
        Connect = Word.
morph_word(Word, Morph, Connect) :-
	root(Word, Root, Prefix, Suffix),
	user:dict(Root, Cat, Args),
	Morph = morph(Root,Cat,Args),
	Connect = Prefix+Suffix.

%   morph_suffix(+Suffix, -Morph)
%
morph_suffix(Suffix, Morph) :-
	user:dict(Suffix, Cat, Args),
	Morph = morph(Suffix,Cat,Args).

%   root(+Word, -Root, -Prefix, -Suffix)
%
root(Word, Root, Prefix, Suffix) :-
	name(Word, WordL),
	reverse(WordL, RevWordL),
	tail(RevWordL, RevPrefixL, Suffix),
	reverse(RevPrefixL, PrefixL),
	name(Prefix, PrefixL), !,
	infl(Suffix, RevPrefixL, RevRootL),
	reverse(RevRootL, RootL),
	name(Root, RootL).

%   tail(+RevWord, -RevPrefix, -Suffix)
%
tail([115,39|R], R, '''s') :- !.              /*  +'s  */
tail([109,39|R], R, am) :- !.                 /*  +'m  */
tail([101,114,39|R], R, are) :- !.            /*  +'re  */
tail([101,118,39|R], R, have) :- !.           /*  +'ve  */
tail([116,39,110|R], R, not) :- !.            /*  +n't  */
tail([116,111,110|R], R, not) :- !.           /*  +not  */
tail([115|R], R, s) :- !.                     /*  +s  */
tail([100,101|R], R, ed) :- !.                /*  +ed  */
tail([114,101|R], R, er) :- !.                /*  +er  */
tail([116,115,101|R], R, est) :- !.           /*  +est  */
tail([103,110,105|R], R, ing) :- !.           /*  +ing  */

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Inflection
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   infl(+Suffix, -RevPrefix, -RevRoot)
%

infl('''s', R, R) :- !.
infl(am, R, R) :- !.
infl(are, R, R) :- !.
infl(have, R, R) :- !.
infl(not, R, RevRootL) :- !, infl_not(R, RevRootL).
infl(s, R, RevRootL) :- !, infl_s(R, RevRootL).
infl(ed, R, RevRootL) :- !, infl_ed(R, RevRootL).
infl(er, R, RevRootL) :- !, infl_er(R, RevRootL).
infl(est, R, RevRootL) :- !, infl_est(R, RevRootL).
infl(ing, R, RevRootL) :- !, infl_ing(R, RevRootL).     % GW 96/05

infl_not([97,99], [110,97,99]).                  /*  can't  */
infl_not([111,119], [108,108,105,119]).          /*  won't  */
infl_not(R, R).                                  /*  (others)  */

infl_s([101,105|R], [121|R]).                    /*  armies  */
infl_s([101,104,116|R], [101,104,116|R]).        /*  clothes  */
infl_s([101,104|R], [104|R]).                    /*  benches  */
infl_s([101,120|R], [120|R]).                    /*  foxes  */
infl_s([101,115,115|R], [115|R]).                /*  grasses  */
infl_s([101,122|R], [122|R]).                    /*  adzes  */
infl_s([101,118|R], [102|R]).                    /*  halves  */
infl_s([101,118|R], [101,102|R]).                /*  wives  */
infl_s([101,111|R], [111|R]).                    /*  cargoes  */
infl_s(R, R).                                    /*  (others)  */

infl_ed([105|R], [121|R]).                       /*  studied  */
infl_ed([X,X,Y|R], [X,Y|R]) :-                   /*  stopped  */
	consonant(X),
	vowel(Y).
infl_ed(R, [101|R]).                             /*  loved  */
infl_ed(R, R).                                   /*  (others)  */

infl_er([X,X,Y|R], [X,Y|R]) :-                   /*  bigger  */
	consonant(X),
	vowel(Y).
infl_er([105|R], [121|R]).                       /*  happier  */
infl_er(R, [101|R]).                             /*  larger  */
infl_er(R, R).                                   /*  (others)  */

infl_est([X,X,Y|R], [X,Y|R]) :-                  /*  biggest  */
	consonant(X),
	vowel(Y).
infl_est([105|R], [121|R]).                      /*  happiest  */
infl_est(R, [101|R]).                            /*  largest  */
infl_est(R, R).                                  /*  (others)  */

infl_ing([X,X,Y|R], [X,Y|R]) :-                  /*  getting  */
	vowel(Y).
infl_ing([121|R], [101,105|R]).                  /*  dying  */
infl_ing(R, [101|R]).                            /*  taking  */
infl_ing(R, R).                                  /*  (others)  */

%   Charactor Types
%
vowel(97). 					 /*  a  */
vowel(101).					 /*  e  */
vowel(105).					 /*  i  */
vowel(111).					 /*  o  */
vowel(117).					 /*  u  */

consonant(98).					 /*  b  */
consonant(99).					 /*  c  */
consonant(100).					 /*  d  */
consonant(103).					 /*  g  */
consonant(107).					 /*  k  */
consonant(108).					 /*  l  */
consonant(109).					 /*  m  */
consonant(110).					 /*  n  */
consonant(112).					 /*  p  */
consonant(114).					 /*  r  */
consonant(115).					 /*  s  */
consonant(116).					 /*  t  */
consonant(118).					 /*  v  */
consonant(119).					 /*  w  */
consonant(121).					 /*  y  */
consonant(122).					 /*  z  */

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Morph Data Handling
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   Ǿ: morph(ID,From,To,Word,Root,Cat,Args,PreIDL)
%
%       ID     ̻
%       From   ϰ
%       To     λ
%       Word   Ф
%       Root   ܷ
%       Cat    ʻ̾
%       Args   ¾ξ
%       PreIDL ܷǤμ̻ҤΥꥹ
%
get_ID(    Morph, ID)     :- arg(1, Morph, ID).
get_from(  Morph, From)   :- arg(2, Morph, From).
get_to(    Morph, To)     :- arg(3, Morph, To).
get_word(  Morph, Word)   :- arg(4, Morph, Word).
get_root(  Morph, Root)   :- arg(5, Morph, Root).
get_cat(   Morph, Cat)    :- arg(6, Morph, Cat).
get_args(  Morph, Args)   :- arg(7, Morph, Args).
get_preIDL(Morph, PreIDL) :- arg(8, Morph, PreIDL).
