% fprint.pl (26 January 1993) by Yasuharu DEN (den@forest.kuee.kyoto-u.ac.jp)
/*  Copyright (C) 1993 Yasuharu Den    */

:- module(fprint, [
	fprint/1,
	fprint/2
		  ]).

%  Needs append/3 from library lists.
:- use_module(library(lists), [append/3,remove_duplicates/2]).

% Pretty Printer for Feature Structures
%
% The data structure of terms is as follows:
%
%   <Term> ::= <Variable> | <Feature Structure> | <Compound Term>
%   <Variable> ::= a variable
%   <Feature Structure> ::= <ID>#<Arc List>
%   <ID> ::= a variable
%   <Arc List> ::= a list of <Feature Value Pair>'s
%   <Feature Value Pair> ::= <Slot>:<Term>
%   <Slot> ::= an atomic
%   <Compound Term> ::= <Functor> followed by a sequence of <Arguments>'s
%   <Functor> ::= atom
%   <Arguments> ::= <Term>
%

%   fprint(+Term).
%   fprint(+Stream, +Term)
%
fprint(X) :-
	current_output(Str),
	fprint(Str, X), !.

fprint(Str, X) :-
	retractall(symbol('F',_)),
	fconvert(X, Y, Copy),
	term_width(Y, _, [], Widths),
	fprint_term(Y, 0, [], [], Widths, Str),
	collect_blocked_goals(Copy, Goals),
	fprint_blocked_goals(Goals, Str), !.

% fconvert(X, Y, Copy) :- fconv_term(X, Y, [], Copy).
fconvert(X, Y, Copy) :- fconvert([X], [Y], XL, XL, YL, YL, [], Copy).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   is_fs(+FS)
%
is_fs(X) :- nonvar(X), functor(X, #, 2), !.

%   create_fs(-FS)
%
create_fs(X) :- functor(X, #, 2), !.

%   id_of(+FS, -ID)
%
id_of(X, ID) :- arg(1, X, ID).

%   arclist_of(+FS, -ArcList)
%
arclist_of(X, ArcList) :- arg(2, X, ArcList).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   dereference(+Term, -DeReference)
%
dereference(X, Y) :-
        is_fs(X),
        id_of(X, ID),
        nonvar(ID), !,
        dereference(ID, Y).
dereference(X, X).

%   decopy(+Term, -DeCopy, +Copy)
%
decopy(X, Y, Copy) :- ( var(X) ; is_fs(X) ), !,
	assoc(Copy, X, Y).

%   copy(+Term, +DeCopy, +Copy, -NewCopy)
%
copy(X, Y, Copy, NewCopy) :- % ( var(X) ; is_fs(X) ), !,
	NewCopy = [X-Y|Copy].
	
%   assoc(+AList, +Key, ?Value)
%
assoc([K-V|_], X, X1) :- X == K, !, X1 = V.
assoc([_|R], X, X1) :- assoc(R, X, X1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   fconv_term(+Term, -NewTerm, +Copy, -NewCopy)
%
fconv_term(X, Y, Copy, NewCopy) :-
	dereference(X, X1),
	!, xfconv_term(X1, Y, Copy, NewCopy).

xfconv_term(X, Y, Copy, Copy) :- decopy(X, Y, Copy), !.
xfconv_term(X, Y, Copy, NewCopy) :- var(X), !,
	gensym('F', Y),
	copy(X, Y, Copy, NewCopy).
xfconv_term(X, Y, Copy, NewCopy) :- is_fs(X), !,
	gensym('F', T),
	copy(X, T, Copy, NCopy),
	create_fs(Y),
	id_of(Y, T),
	arclist_of(X, XC),
	arclist_of(Y, YC),
	fconv_term_fs(XC, YC, NCopy, NewCopy).
xfconv_term(X, Y, Copy, NewCopy) :-
	functor(X, F, N),
	functor(Y, F, N),
	loop_fconv_term(0, N, X, Y, Copy, NewCopy).

loop_fconv_term(N, N, _, _, Copy, Copy) :- !.
loop_fconv_term(M, N, X, Y, Copy, NewCopy) :-
	M1 is M + 1,
	arg(M1, X, A),
	arg(M1, Y, B),
	fconv_term(A, B, Copy, NCopy),
	!, loop_fconv_term(M1, N, X, Y, NCopy, NewCopy).

%   fconv_term_fs(+ArcList, -NewArcList, +Copy, -NewCopy)
%
fconv_term_fs([], [], Copy, Copy) :- !.
fconv_term_fs([S:V|R], [S:V1|R1], Copy, NewCopy) :-
	fconv_term(V, V1, Copy, NCopy),
	!, fconv_term_fs(R, R1, NCopy, NewCopy).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   fconvert(+Term, -NewTerm, +SubTermL, -NewSubTermL,
%            +SubNewTermL, -NewSubNewTermL, +Copy, -NewCopy)
%
fconvert([], [], [], [], [], [], Copy, Copy) :- !.
fconvert([], [], [], XL, [], YL, Copy, NewCopy) :- !,
	fconvert(XL, YL, XXL, XXL, YYL, YYL, Copy, NewCopy).
fconvert([X|R], [Y|S], XL, XL2, YL, YL2, Copy, NewCopy) :-
	fconv_term(X, Y, XL, XL1, YL, YL1, Copy, NCopy),
	!, fconvert(R, S, XL1, XL2, YL1, YL2, NCopy, NewCopy).

%   fconv_term(+Term, -NewTerm, +SubTermL, -NewSubTermL,
%              +SubNewTermL, -NewSubNewTermL, +Copy, -NewCopy)
%
fconv_term(X, Y, XL, XL1, YL, YL1, Copy, NewCopy) :-
	dereference(X, X1),
	!, xfconv_term(X1, Y, XL, XL1, YL, YL1, Copy, NewCopy).

xfconv_term(X, Y, XL, XL, YL, YL, Copy, Copy) :- decopy(X, Y, Copy), !.
xfconv_term(X, Y, XL, XL, YL, YL, Copy, NewCopy) :- var(X), !,
	gensym('F', Y),
	copy(X, Y, Copy, NewCopy).
xfconv_term(X, Y, XL, XL1, YL, YL1, Copy, NewCopy) :- is_fs(X), !,
	gensym('F', T),
	copy(X, T, Copy, NewCopy),
	create_fs(Y),
	id_of(Y, T),
	arclist_of(X, XC),
	arclist_of(Y, YC),
	fconv_term_fs(XC, YC, XL, XL1, YL, YL1).
xfconv_term(X, Y, XL, XL1, YL, YL1, Copy, NewCopy) :-
	functor(X, F, N),
	functor(Y, F, N),
	loop_fconv_term(0, N, X, Y, XL, XL1, YL, YL1, Copy, NewCopy).

loop_fconv_term(N, N, _, _, XL, XL, YL, YL, Copy, Copy) :- !.
loop_fconv_term(M, N, X, Y, XL, XL2, YL, YL2, Copy, NewCopy) :-
	M1 is M + 1,
	arg(M1, X, A),
	arg(M1, Y, B),
	fconv_term(A, B, XL, XL1, YL, YL1, Copy, NCopy),
	!, loop_fconv_term(M1, N, X, Y, XL1, XL2, YL1, YL2, NCopy, NewCopy).

%   fconv_term_fs(+ArcList, -NewArcList, +SubTermL, -NewSubTermL,
%                 +SubNewTermL, -NewSubNewTermL)
%
fconv_term_fs([], [], XL, XL, YL, YL) :- !.
fconv_term_fs([S:V|R], [S:V1|R1], [V|XL], XL1, [V1|YL], YL1) :-
	fconv_term_fs(R, R1, XL, XL1, YL, YL1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   term_width(+Term, -Width, +Copy, -NewCopy)
%
term_width(X, Width, Copy, NewCopy) :- is_fs(X), !,
	id_of(X, T),
	arclist_of(X, XC),
	width(T, TWidth),
	term_width_fs(XC, 0, CWidth, Copy, NCopy),
	% +4 for '| ' and ' |'
	Width is TWidth + CWidth + 4,
	copy(X, Width, NCopy, NewCopy).
term_width(X, Width, Copy, Copy) :- atomic(X), !,
	width(X, Width).
term_width([A|X], Width, Copy, NewCopy) :- !,
	term_width(A, AWidth, Copy, NCopy),
	term_width_list(X, AWidth, XWidth, NCopy, NewCopy),
	% +2 for indentation
	Width is XWidth + 2. 
xterm_width(X, Width, Copy, NewCopy) :-
	functor(X, F, N),
	width(F, FWidth),
	% + 1 for '('
	FWidth1 is FWidth + 1,
	loop_term_width(0, N, X, FWidth1, Width, Copy, NewCopy).

loop_term_width(N, N, _, Width, Width, Copy, Copy) :- !.
loop_term_width(M, N, X, Width, NewWidth, Copy, NewCopy) :-
	M1 is M + 1,
	arg(M1, X, A),
	term_width(A, AWidth, Copy, NCopy),
	% + 1 for ',' or ')'
	NWidth is Width + AWidth + 1,
	!, loop_term_width(M1, N, X, NWidth, NewWidth, NCopy, NewCopy).

%   term_width_list(+List, +Max, -NewMax, +Copy, -NewCopy)
%
term_width_list([], Max, Max, Copy, Copy) :- !.
term_width_list([A|X], Max, NewMax, Copy, NewCopy) :- !,
	term_width(A, Width, Copy, NCopy),
	max(Max, Width, NMax),
	!, term_width_list(X, NMax, NewMax, NCopy, NewCopy).
term_width_list(X, Max, NewMax, Copy, NewCopy) :-
	term_width(X, Width, Copy, NewCopy),
	max(Max, Width, NewMax).

%   term_width_fs(+ArcList, +Width, -NewWidth, +Copy, -NewCopy)
%
term_width_fs([], Max, Max, Copy, Copy) :- !.
term_width_fs([S:V|Rest], Max, NewMax, Copy, NewCopy) :-
	term_width_node(S, V, Width, Copy, NCopy),
	max(Max, Width, NMax),
	!, term_width_fs(Rest, NMax, NewMax, NCopy, NewCopy).

term_width_node(S, V, Width, Copy, NewCopy) :-
	width(S, SWidth),
	term_width(V, VWidth, Copy, NewCopy),
	% +2 for ': '
	Width is SWidth + VWidth + 2.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   fprint_term(+Term, +CurrPos, +Left, +Right, +Copy, +Stream)
%
fprint_term(X, C, L, R, Copy, Str) :- is_fs(X), !,
	id_of(X, T),
	arclist_of(X, XC),
	decopy(X, Width, Copy),
	fprint_object(T, C, C1, Str),
	To is C + Width,
	fprint_fs(XC, C1, [C1|L], [To|R], Copy, Str).
fprint_term([A|X], C, L, R, Copy, Str) :- !,
	fprint_object('[ ', C, C1, Str),
	fprint_right(R, C1, Str),
	fprint_list([A|X], 0, C, L, R, Copy, Str),
	fprint_left(L, 0, C2, Str),
	fprint_object_from('] ', C, C2, C3, Str),
	fprint_right(R, C3, Str).
fprint_term(X, C, _, R, _, Str) :-
	fprint_object(X, C, C1, Str),
	fprint_right(R, C1, Str).

%   fprint_object(+Term, +CurrPos, -NewCurrPos, +Stream)
%
fprint_object(X, C, C1, Str) :- atomic(X), !,
	width(X, Width),
	write(Str, X),
	C1 is C + Width.
fprint_object(X, C, C2, Str) :-
	functor(X, F, N),
	write(Str, F),
	write(Str, '('),
	width(F, FWidth),
	% +1 for '('
	C0 is C + FWidth + 1,
	loop_fprint_object(1, N, X, C0, C1, Str),
	write(Str, ')'),
	% +1 for ')'
	C2 is C1 + 1.

loop_fprint_object(N, N, X, C, C1, Str) :- !,
	arg(N, X, A),
	fprint_object(A, C, C1, Str).
loop_fprint_object(M, N, X, C, C2, Str) :-
	arg(M, X, A),
	fprint_object(A, C, C0, Str),
	write(','),
	% +1 for ',' or ')'
	C1 is C0 + 1,
	M1 is M + 1,
	!, loop_fprint_object(M1, N, X, C1, C2, Str).

%   fprint_list(+List, +CurrPos, +RefPos, +Left, +Right, +Copy, +Stream)
%
fprint_list([], _, _, _, _, _, _) :- !.
fprint_list([A|X], C, Ref, L, R, Copy, Str) :- !,
	fprint_left(L, C, C1, Str),
	fprint_object_from('  ', Ref, C1, C2, Str),
	fprint_term(A, C2, L, R, Copy, Str),
	!, fprint_list(X, C, Ref, L, R, Copy, Str).
fprint_list(X, C, Ref, L, R, Copy, Str) :-
	fprint_left(L, C, C1, Str),
	fprint_object_from('| ', Ref, C1, C2, Str),
	fprint_right(R, C2, Str),
	fprint_left(L, C, C3, Str),
	fprint_object_from('  ', Ref, C3, C4, Str),
	fprint_term(X, C4, L, R, Copy, Str).

%   fprint_fs(+ArcList, +CurrPos, +Left, +Right, +Copy, +Stream)
fprint_fs([], _, _, _, _, _) :- !.
fprint_fs([S:V|Rest], C, L, R, Copy, Str) :-
	fprint_node(S, V, C, L, R, Copy, Str),
	!, fprint_fs(Rest, 0, L, R, Copy, Str).

fprint_node(S, V, C, L, R, Copy, Str) :-
	fprint_left(L, C, C1, Str),
	fprint_object(S, C1, C2, Str),
	fprint_object(': ', C2, C3, Str),
	fprint_term(V, C3, L, R, Copy, Str).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   fprint_left(+Left, +CurrPos, -NewCurrPos, +Stream)
%
fprint_left([], C, C, _) :- !.
fprint_left([From|_], C, C, _) :- From < C, !.
fprint_left([From|Rest], C, C2, Str) :-
	fprint_left(Rest, C, C1, Str),
	fprint_object_from('| ', From, C1, C2, Str).

%   fprint_right(+Right, +CurrPos, +Stream)
%
fprint_right([], _, Str) :- !, nl(Str).
fprint_right([To|Rest], C, Str):-
	fprint_object_to(' |', To, C, C1, Str),
	!, fprint_right(Rest, C1, Str).

%   fprint_object_from(+Obj, +FromPos, +CurrPos, -NewCurrPos, +Stream)
%
fprint_object_from(X, From, C, C1, Str) :-
	width(X, Width),
	Skip is From - C,
	tab(Str, Skip),
	write(Str, X),
	C1 is From + Width.

%   fprint_object_to(+Obj, +To, +CurrPos, -NewCurrPos, +Stream)
%
fprint_object_to(X, To, C, To, Str) :-
	width(X, Width),
	Skip is To - Width - C,
	tab(Str, Skip),
	write(Str, X).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   collect_blocked_goals(+Copy, -Goals)
%
collect_blocked_goals(Copy, Goals) :-
	collect_vars(Copy, Vars, NewVars),
	call_residue(copy_term(Vars,CopyOfVars), Residue),
	CopyOfVars = NewVars,
	collect_goals(Residue, Goals0),
	remove_duplicates(Goals0, Goals), !.

%   collect_vars(+Copy, -Vars, -NewVars)
%
collect_vars([], [], []) :- !.
collect_vars([X-Y|Rest], Vars, NewVars) :- var(X), !,
	Vars = [X|Vars1], NewVars = [Y|NewVars1],
	collect_vars(Rest, Vars1, NewVars1).
collect_vars([_|Rest], Vars, NewVars) :- collect_vars(Rest, Vars, NewVars).

%   collect_goals(+Residue, -Goals)
%
collect_goals([], []) :- !.
collect_goals([V-(prolog:when(_,C))|Rest], Goals) :- !,
	collect_goals([V-C|Rest], Goals).
collect_goals([_-(prolog:freeze(V,C))|Rest], Goals) :- !,
	collect_goals([V-C|Rest], Goals).
collect_goals([_-(_:call(C))|Rest], [C|Goals]) :- !,
	collect_goals(Rest, Goals).
collect_goals([_-(_:C)|Rest], [C|Goals]) :-
	collect_goals(Rest, Goals).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%   fprint_blocked_goals(+Goals, +Stream)
%
fprint_blocked_goals([], _) :- !.
fprint_blocked_goals(Goals, Str) :-
	nl(Str),
	fprint_blocked_goals1(Goals, Str).

fprint_blocked_goals1([], _) :- !.
fprint_blocked_goals1([C|Rest], Str) :-
	write(Str, C), nl(Str),
	fprint_blocked_goals1(Rest, Str).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   max(+X, +Y, -Max)
%
max(X, Y, X) :- X >= Y, !.
max(_, Y, Y).

%   width(+Atom, -Width)
%
width(X, Width) :-
	name(X, Name),
	length(Name, Width).

%   concat_atoms(+Atom1, +Atom2, ?NewAtom)
%
concat_atoms(Atom1, Atom2, NewAtom) :-
        name(Atom1, Name1),
        name(Atom2, Name2),
        append(Name1, Name2, Name3),
        name(Atom3, Name3),
        !, NewAtom = Atom3.

%   gensym(+Name, -Symbol)
%
:- dynamic symbol/2.

gensym(Name, Symbol) :-
        (   retract(symbol(Name,N))
        ;   N = 1
        ),
        N1 is N + 1,
        asserta(symbol(Name,N1)),
        concat_atoms(Name, N, Symbol), !.
