/*
 * xwrdplay.cc -- WRD file player
 *
 * Copyright (C) 1996-1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: xwrdplay.cc,v 1.12 1998-03-25 07:56:38+09 satoshi Exp $

#define getopt getopt__
#include <stdlib.h>
#undef getopt
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/Intrinsic.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xresource.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#include "getopt.h"
#endif

#include "magfile.h"
#include "kconv.h"
#include "shmem.h"
#include "server.h"

static const char* version = "(" VERSION ")";

#define SERVER "eplaymidi"	// server name

enum verboseLevel {verbose_quiet = 0, verbose_normal = 1, verbose_debug = 2};
static verboseLevel verbose;

#define CLASS_NAME	"XWrdplay"
#define RESOURCE_NAME	"xwrdplay"
#define NAME		"xwrdplay"	// WM_NAME, WM_ICON_NAME

struct Resource {
  const char* specifier;
  const char* value;
};

static Resource resources[] = {
  {"xwrdplay*font",
   "-adobe-courier-medium-r-normal--16-110-100-0-m-*-iso8859-*"},
#ifdef KANJI
  {"xwrdplay*kfont",
   "-jis-fixed-medium-r-normal--16-110-100-100-*-*-jisx0208.1983-*"},
  {"xwrdplay*rfont",
   "-misc-fixed-medium-r-normal--16-110-100-*-*-*-jisx0201.1976-*"},
#endif
  {"xwrdplay.textColor0", "black"},
  {"xwrdplay.textColor1", "red"},
  {"xwrdplay.textColor2", "green"},
  {"xwrdplay.textColor3", "yellow"},
  {"xwrdplay.textColor4", "blue"},
  {"xwrdplay.textColor5", "magenta"},
  {"xwrdplay.textColor6", "cyan"},
  {"xwrdplay.textColor7", "white"},
};

#undef MAX_PATH		// ???
static const int MAX_PATH = 80;		// maximum length of path
static const int MAX_FILENAME = 20;	// maximum length of file name

static const unsigned GX_SIZE = 640;	// graphic screen size
static const unsigned GY_SIZE = 400;
static const unsigned TX_SIZE = 80;	// text screen size
static const unsigned TY_SIZE = 25;

static const char CR = '\x0d', LF = '\x0a';

static const unsigned char SSO = 0x8e;	// EUC SS0 X0201
inline bool iskanji(unsigned char c) { return (c > 0x80 && c != SSO); }
inline bool iskana(unsigned char c) { return (c == SSO); }
inline bool isiso(unsigned char c) { return (0x1f < c && c < 0x80); }

// SJIS (MS KANJI)
inline bool issjkana(unsigned char c) { return (0xa0 <= c && c <= 0xdf); }
					// JIS X0201 kana
inline bool issjkanji(unsigned char c) { return (c > 0x80 && !issjkana(c)); }

static const int SHORTEST_CMD = 3;  // the length of the shortest macro command
static const int LONGEST_CMD = 7;   // the length of the longest macro command
enum token_type {	// token type
  // for internal use
  tINVALID = 0,	// invalid macro name
  tATMARK = 1,	// `@' alone
  tPRINT,	// print word
  // for MIMPI
  tCOLOR,
  tEND, 
  tESC, 
  tEXEC, 
  tFADE, 
  tGCIRCLE, 
  tGCLS, 
  tGINIT, 
  tGLINE, 
  tGMODE,
  tGMOVE, 
  tGON,
  tGSCREEN, 
  tINKEY, 
  tLOCATE, 
  tLOOP, 
  tMAG, 
  tMIDI,
  tOFFSET, 
  tPAL, 
  tPALCHG, 
  tPALREV, 
  tPATH, 
  tPLOAD, 
  tREM, 
  tREMARK, 
  tREST, 
  tSCREEN, 
  tSCROLL, 
  tSTARTUP, 
  tSTOP, 
  tTCLS, 
  tTON, 
  tWAIT, 
  tWMODE,
  // for ensyutsukun mechanism
  tFONTM,
  tFONTP,
  tFONTR,
  tGSC,
  tLINE,
  tPAL_,	// ^PAL
  tREGSAVE,
  tSCROLL_, 	// ^SCROLL
  tTEXTDOT,
  tTMODE,
  tTSCRL,
  tVCOPY,
  tVSGET,
  tVSRES,
  tXCOPY
};

#include "wrd.h"

static Server* server = NULL;
static volatile Register* reg = NULL;

static struct {
  volatile unsigned count;	// counter
  unsigned p1, p2;	// source palette, destination palette
  unsigned maxdif;	// maximum difference
  long paldif[16*3];	// differences between p1 and p2
  unsigned tempo;
} gradate = { 0 };

inline void print_version(void);
inline void print_usage(void);
static int compar(const char** e1, const char** e2);
static void strdump(const unsigned char *str);
static void gradatePalette(int arg);
static void sigcld_handler(int sig);

extern "C" int strncasecmp(const char*, const char*, size_t);


/*
 * directory
 */
class Directory {
private:
  char* dir;
  DIR* dirp;
  struct dirent* dir_ent;

  struct entry {
    char* name;
    entry* next;
  };

public:
  Directory(const char* name);
  const char** read(void);
  bool ismidifile(const char* filename);
  bool isdirectory(const char* filename);
};


Directory::Directory(const char* name)
{
  if((dirp = opendir(name)) == NULL) {
    fprintf(stderr, "cannot open directory `%s'.\n", name);
  }
  dir = new char[strlen(name)+1];
  strcpy(dir, name);
}


const char** Directory::read(void)
{
  entry* first = NULL;
  entry* e = NULL;
  int n = 0;
  while((dir_ent = readdir(dirp)) != NULL) {
    char filename[MAX_PATH+MAX_NAME+1];
    sprintf(filename, "%s/%s", dir, dir_ent->d_name);
//    if(isdirectory(filename) || ismidifile(filename)) {
      if(first == NULL)
	first = e = new entry;
      else
	e = e->next = new entry;
      n++;
      e->name = new char[strlen(dir_ent->d_name)+1];
      strcpy(e->name, dir_ent->d_name);
      e->next = NULL;
//    }
  }
  closedir(dirp);
  const char** names = new const char*[n+1];
  int i;
  e = first;
  for(i = 0; i < n; i++) {
    names[i] = e->name;
    e = e->next;
  }
  qsort(names, n, sizeof(const char*),
	(int(*)(const void*, const void*))compar);
  names[i] = NULL;
  return names;
}


static int compar(const char** e1, const char** e2)
{
  if(*e1 == *e2)
    return 0;
  if(*e1 == NULL)
    return -1;
  if(*e2 == NULL)
    return 1;
  int l1 = strlen(*e1);
  int l2 = strlen(*e2);
  int i;
  for(i = 0; i < l1 && i < l2; i++) {
    if((*e1)[i] != (*e2)[i])
      return ((*e1)[i] < (*e2)[i]) ? -1 : 1;
  }
  return (l1 == l2) ? 0 : (l1 < l2) ? -1 : 1;
}


bool Directory::ismidifile(const char* filename)
{
  char ext[1+3+1];
  int len = strlen(filename);
  if(len < 4)
    return false;
  int i;
  for(i = 0; i < 1+3+1; i++)
    ext[i] = tolower(filename[len-4+i]);
  if(!strcmp(ext, ".rcp") || !strcmp(ext, ".r36") ||
     !strcmp(ext, ".g18") || !strcmp(ext, ".g36") ||
     !strcmp(ext, ".mdf") ||
     !strcmp(ext, ".mid"))
    return true;
  return false;
}


bool Directory::isdirectory(const char* filename)
{
  struct stat buf;
  if(stat(filename, &buf) == -1) {
    perror("stat");
    return false;
  }
  return S_ISDIR(buf.st_mode);
}


/*
 * screen
 */
class ScreenBase {
public:
  ScreenBase(const char* xhost);
  void changeTitle(const char* filename);
  const char* getResource(const char* name);

protected:
  const char* XHost;
  Display* XDisplay;
  Window RootWindow;
  Window XWindow;
  int XScreen;
  int DDepth;
  unsigned long Black, White;
  XVisualInfo vis_tmp, *visualList;
  int vis_match;
  Visual* visual;
  GC XGC;
  int WinWidth, WinHeight;
  Colormap cmap;
  unsigned long pixel[16+8];	// 0-15:graphic, 16-23:text
  Font font;
  XFontStruct* xfs;
#ifdef KANJI
  Font kfont, rfont;
  XFontStruct *kfs, *rfs;
#endif
  XEvent event;
  XrmDatabase rdb, rdb2;
  const char* name;
  int vis;
  const char* wmname;
};


/*
 * graphic screen
 */
class GraphicScreen : public virtual ScreenBase {
public:
  GraphicScreen(const char* xhost);
  ~GraphicScreen();
  void drawLine(unsigned x1, unsigned y1, unsigned x2, unsigned y2,
		unsigned char c);
  void drawRectangle(unsigned x1, unsigned y1, unsigned x2, unsigned y2,
		     unsigned char c);
  void fillRectangle(unsigned x1, unsigned y1, unsigned x2, unsigned y2,
		     unsigned char c);
  void drawCircle(unsigned x, unsigned y, unsigned r, unsigned char c);
  void fillCircle(unsigned x, unsigned y, unsigned r, unsigned char c);
  void move(unsigned x1, unsigned y1, unsigned x2, unsigned y2,
	    unsigned xd, unsigned yd, unsigned vs, unsigned vd,
	    unsigned char sw);
  void gclear(unsigned char sw);
  void screen(unsigned char new_act, char unsigned new_dsp);
  void gon(unsigned char sw);
  void gmode(unsigned char sw);
  void palette(unsigned char no, unsigned short* p);
  void palrev(unsigned char no);
  void fade(unsigned char p1, unsigned char p2, int speed);
  void loadMag(const char* filename, int x, int y, unsigned char s,
	       unsigned char pixflag, unsigned char palflag);
  void loadPho(FILE* fp);
  void getVS(unsigned n);
  void releaseVS(void);
  void vcopy(int sx1, int sy1, int sx2, int sy2, int tx, int ty, int ss,
	     int ts, int mode);

protected:
  Pixmap gvram[2];		// graphic VRAM
  XColor gcolor[16];
  unsigned short palet[20*16];	// color palette
  unsigned char graph;		// graphic screen mode, 0:off, 1:on
  unsigned char act, dsp;	// active, display screen, 0:$BI=(B, 1:$BN"(B
  // for ensyutsukun mechanism
  Pixmap* vsc;			// virtual screen
  unsigned n_vsc;		// number of virtual screen
};

#undef putchar


enum charset {charset_iso, charset_kanji1, charset_kanji2, charset_kana};

/*
 * text screen
 */
class TextScreen : public virtual ScreenBase {
public:
  TextScreen(const char* xhost);
  void locate(unsigned char x, unsigned char y);
  void tcolor(unsigned char c) {color = c;}
  void right(int x = 1) {
    if((loc_x += x) > TX_SIZE) loc_x = TX_SIZE;
    if(loc_x < 1) loc_x = 1;
  }
  void down(int x = 1) {
    if((loc_y += x) > TY_SIZE) loc_y = TY_SIZE;
    if(loc_y < 1) loc_y = 1;
  }
  void left(int x = 1) {right(-x);}
  void up(int x = 1) {down(-x);}
  void ton(unsigned char sw);
  void x(unsigned char new_x) {loc_x = new_x;}

protected:
  unsigned long text_palet[48];	// text palette
  unsigned char loc_x, loc_y;	// cursor location
  unsigned char color;		// text color
  unsigned char text;		// text screen mode, 0:off, 1:on, 2:double
  struct {
    unsigned char chr;		// character
    unsigned char col;		// color
    unsigned char redraw;	// redraw flag
    enum charset charset;	// character set
  } tram[TY_SIZE][TX_SIZE];	// text RAM
  unsigned char sx, sy;		// saved cursor location
  unsigned char scolor;		// saved color
};


/*
 * screen for WRD player
 */
class WrdScreen : public GraphicScreen, public TextScreen {
public:
  WrdScreen(const char* xhost) :
  ScreenBase(xhost), GraphicScreen(xhost), TextScreen(xhost), fontmode(0) {}
  ~WrdScreen();
  void doExpose(void);
  void doEnterWindow(void);
  void doLeaveWindow(void);
  void doKeyRelease(void);
  void print(const char* word);
  void doEscSeq(const char* word, int* j);
  void putchar(char c);
  void putchar_(unsigned char loc_x, unsigned char loc_y, unsigned char color,
		char c, bool restore_bg);
#ifdef KANJI
  void putchar2b(unsigned char c1, unsigned char c2);
  void putchar2b_(unsigned char loc_x, unsigned char loc_y,
		  unsigned char color, unsigned char c1, unsigned char c2,
		  bool restore_bg);
#endif
  void redrawText(int x1, int y1, int x2, int y2, bool restore_bg);
  void tclear(unsigned char x1 = 1, unsigned char y1 = 1,
	      unsigned char x2 = TX_SIZE, unsigned char y2 = TY_SIZE,
	      unsigned char color = 0, char char_ = ' ');
  void tscroll(unsigned char x1 = 1, unsigned char y1 = 1,
	       unsigned char x2 = TX_SIZE, unsigned char y2 = TY_SIZE,
	       unsigned char mode = 0, unsigned char color = 0,
	       char char_ = ' ');
  void inkey(unsigned bar);
  void loop(unsigned count);
  void gradatePalette(unsigned step);
  void drawFont(int num, int x, int y, int atr);

  // for ensyutsukun mechanism
  int fontmode;
  struct {
    unsigned short data[16];
  } fontdata[16];
};


/*
 * WRD file
 */
class WrdFile {
public:
  WrdFile(const char* filename_, int initial_timebase, int initial_tempo);
  ~WrdFile();
  void analyze(void);
  void play(const char* xhost);
  void setVersion(int version_) {version = version_;}

private:
  Directory* dir;
  const char** files;

  char* filename;
  FILE* fp;
  unsigned line;	// processing line's number
  unsigned error;

  char* magfile;	// startup MAG file
  char* phofile;	// startup PHO file
  char* midfile;	// MIDI data file synchronized with
  enum Midi_file_format {
    RCP_format, G36_format, SMF_format
  } midfileformat;

  int version;			// -1: omit, 3xx-4xx: version
  unsigned char wmode_n;	// 0: measure, 1-255: timing
  unsigned char wmode;		// 0: per line, 1: per character
  unsigned offset;		// offset
  char path[MAX_PATH+1+1]; // directory search path used by @EXEC, @MAG, @PLOAD

  unsigned nowbar;
  unsigned nowstep;
  unsigned timebase;
  unsigned tempo;

  enum Param_type {
    param_int,		// positive number
    param_no,		// positive number with a preceding `#'
    param_hex,		// hexadecimal number
    param_eint,		// number (positive, negative and hexadecimal)
    param_str,		// strings
    param_end		// end of parameter
  };
  enum Param_omit {param_omit, param_noomit};

  struct Param {
    Param_type type;
    void* ptr;
    Param_omit omit;
  };

  int param(const char* cmdname, Param* p, unsigned* ret);
  token_type macro(char* cmd);

public:		// ???
  WrdScreen* scr;
private:	// ???

  void wait(unsigned bar, unsigned step);
  void exec(const char* file_name);
  FILE* dos_fopen(char* path, char* mode);
  char* dos_filename(char* path);

  struct wrddata {
    unsigned bar;
    unsigned step;
    token_type cmd;
    void* param;
  };
  wrddata* data;
  const unsigned DATABLOCK;
  unsigned maxdata;
  unsigned datasize;

  void record(token_type cmd, void* param);

  bool preloadMag;
  bool preloadPho;
  struct magList {
    char* file;
    int32 width, height, size;
    int16* image;
    int16 left, top;
    char mode;
  } *magList;
  char** phoList;
};


enum debugMode {
  normalMode = 0,
  nonstopMode = 1,
  emulationMode = 2,
  stepMode = 3
};
static debugMode debugMode = normalMode;
static volatile unsigned debugStep = 0;

static const char* prgname;


/*
 * print version number
 */
inline void print_version(void)
{
  fprintf(stderr,
	  "WrdPlayer %s Copyright (C) 1996-1998 S.Kuramochi\n",
	  version);
}


/*
 * print usage
 */
inline void print_usage(void)
{
  fprintf(stderr,
	  "usage: %s [-bdehnqstvV] wrdfile\n"
	  "  -b, --timebase value  specifies timebase\n"
	  "  -d, --debug           debug mode\n"
	  "  -e, --emulate         emulation mode\n"
	  "  -h, --help            show this help\n"
	  "  -n, --nonstop         nonstop mode\n"
	  "  -q, --quiet           quiet mode\n"
	  "  -s, --step            step execution mode\n"
	  "  -t, --tempo value     specifies tempo\n"
	  "  -v, --verbose         verbose mode\n"
	  "  -V, --version         display version number\n"
	  "  -display display      X server to use\n"
//	  "  -geometry geometry    initial size and location\n"
	  ,
	  prgname);
}


/*
 * main routine
 */
int main(int argc, char* const argv[])
{
  prgname = argv[0];

  const char* xhost = NULL;
  const char* filename = NULL;
  int tempo = -1;
  int timebase = -1;

  struct option long_options[] = {
    {"debug",    no_argument,       NULL, 'd'},
    {"display",	 required_argument, NULL, 'D'},
    {"emulate",  no_argument,       NULL, 'e'},
    {"geometry", required_argument, NULL, 'G'},
    {"help",     no_argument,       NULL, 'h'},
    {"iconic",	 no_argument,       NULL, 'I'},
    {"nonstop",  no_argument,       NULL, 'n'},
    {"quit",     no_argument,       NULL, 'q'},
    {"step",     no_argument,       NULL, 's'},
    {"tempo",    required_argument, NULL, 't'},
    {"timebase", required_argument, NULL, 'b'},
    {"verbose",  no_argument,       NULL, 'v'},
    {"version",  no_argument,       NULL, 'V'},
    {"xrm",      required_argument, NULL, 'R'},
    {NULL,       no_argument,       NULL,  0 }
  };
  optind = 0;
  opterr = 1;
  int c;
  for(;;) {
    int option_index = 0;
    if((c = getopt_long_only(argc, argv, "b:dehnqst:vV",
			     long_options, &option_index)) == EOF)
      break;
    switch(c) {
    case 'b': // specifies timebase
      timebase = atoi(optarg);
      break;
    case 'D': // X server to use
      xhost = optarg;
      break;
    case 'd': // debug mode
      verbose = verbose_debug;
      break;
    case 'e': // emulation mode
      debugMode = emulationMode;
      break;
    case 'G': // geometry
      break;
    case 'h': // show help
      print_usage();
      exit(EXIT_SUCCESS);
      break;
    case 'I': // iconic
      break;
    case 'n': // nonstop mode
      debugMode = nonstopMode;
      break;
    case 'q': // quiet mode
      verbose = verbose_quiet;
      break;
    case 'R': // resource
      break;
    case 's': // step execution mode
      debugMode = stepMode;
      break;
    case 't': // specifies tempo
      tempo = atoi(optarg);
      break;
    case 'v': // verbose mode
      verbose = verbose_normal;
      break;
    case 'V': // display version number
      print_version();
      fprintf(stderr, 
	      "This is free software with ABSOLUTELY NO WARRANTY.\n"
	      "For details please see the file COPYING.\n");
      exit(EXIT_SUCCESS);
      break;
    case ':': // missing parameter
      break;
    case '?': // unknown option character
      break;
    default:
      fprintf(stderr,
	      "getopt_long_only() returned character code %02x\n", c&0xff);
      break;
    }
  }
  if(optind < argc)
    filename = argv[optind];
  else {
    print_version();
    print_usage();
    exit(EXIT_FAILURE);
  }

#ifdef NOSERVER
  if(debugMode == normalMode)
    debugMode == emulationMode;
#endif

  if(debugMode == normalMode) {
    server = new Server(SERVER);
    reg = server->creg->reg;
  } else
    reg = new Register;

//signal(SIGCHLD, sigcld_handler);
  WrdFile wrdfile(filename, timebase, tempo);
  wrdfile.analyze();
  wrdfile.play(xhost);

  if(verbose != verbose_quiet)
    fprintf(stderr, "Finished.\n");

  struct timeval tv1, tv;
  gettimeofday(&tv1, NULL);
  tv1.tv_sec += 60;	// 60 seconds
  while(gettimeofday(&tv, NULL), tv.tv_sec < tv1.tv_sec) {
    if(gradate.count) {
      wrdfile.scr->gradatePalette(gradate.count);
      if(gradate.count >= gradate.maxdif)
	gradate.count = 0;
    }
    wrdfile.scr->doExpose();
    wrdfile.scr->doKeyRelease();
    wrdfile.scr->doEnterWindow();
    wrdfile.scr->doLeaveWindow();
    usleep(50*1000);	// 50ms
  }

  exit(EXIT_SUCCESS);
}


#undef error	// macro `error' causes a problem

WrdFile::WrdFile(const char* filename_, int initial_timebase,
		 int initial_tempo)
: dir(NULL), files(NULL), filename(NULL), line(1), error(0), magfile(NULL),
  phofile(NULL), midfile(NULL), version(-1), wmode_n(0), wmode(0), offset(0),
  nowbar(0), nowstep(0), scr(NULL), data(NULL), DATABLOCK(16*1024),
  maxdata(DATABLOCK), datasize(0), preloadMag(false), preloadPho(false),
  magList(NULL), phoList(NULL)
{
  if((fp = fopen(filename_, "r")) == NULL) {
    fprintf(stderr, "Can't open file `%s'.\n", filename_);
  }

  if(filename_[0] == '/') {
    const char* p = strrchr(filename_, '/');
    strncpy(path, filename_, p-filename_+1);
    path[p-filename_+1] = '\0';
    filename = new char[strlen(filename_)-(p-filename_)];
    strncpy(filename, p+1, strlen(filename_)-(p-filename_));
  } else {
    const char* p;
    if((p = strrchr(filename_, '/')) != NULL) {
      strncpy(path, filename_, p-filename_+1);
      path[p-filename_+1] = '\0';
      filename = new char[strlen(filename_)-(p-filename_)];
      strncpy(filename, p+1, strlen(filename_)-(p-filename_));
    } else {
      getcwd(path, MAX_PATH);
      strcat(path, "/");
      filename = new char[strlen(filename_)+1];
      strcpy(filename, filename_);
    }
  }

  dir = new Directory(path);
  files = dir->read();

  // check for startup MAG file and PHO file
  const char* p;
  size_t pos =
    ((p = strrchr(filename, '.')) != NULL) ? p-filename : strlen(filename);
  bool magfound = false;
  bool phofound = false;
  bool midfound = false;
  int i;
  for(i = 0; files[i] != NULL; i++) {
    if(!strncasecmp(files[i], filename, pos)) {
      if(!magfound && !strcasecmp(files[i]+pos, ".mag")) {
	magfile = new char[pos+5];
	strcpy(magfile, files[i]);
	magfound = true;
	if(midfound)	// give precedence to MAG file ???
	  break;
      }
      if(!phofound && !strcasecmp(files[i]+pos, ".pho")) {
	phofile = new char[pos+5];
	strcpy(phofile, files[i]);
	phofound = true;
	if(magfound && midfound)
	  break;
      }
      if(!midfound) {
	if(!strcasecmp(files[i]+pos, ".rcp") ||
	   !strcasecmp(files[i]+pos, ".r36")) {
	  midfileformat = RCP_format;
	  midfound = true;
	}
	if(!strcasecmp(files[i]+pos, ".g36") ||
	   !strcasecmp(files[i]+pos, ".g18")) {
	  midfileformat = G36_format;
	  midfound = true;
	}
	if(!strcasecmp(files[i]+pos, ".mid")) {
	  midfileformat = SMF_format;
	  midfound = true;
	}
	if(midfound) {
	  midfile = new char[pos+5];
	  strcpy(midfile, files[i]);
	  if(magfound && phofound)
	    break;
	}
      }
    }
  }
  if(magfound && phofound) {	// give precedence to MAG file ???
    delete[] phofile;
    phofile = NULL;
  }

  if(initial_timebase == -1) {
    timebase = 48;	// default value
    if(midfound) {	// if corresponding MIDI data file is found,
      FILE* fp;		// read timebase from it.
      if((fp = dos_fopen(midfile, "r")) == NULL) {
	fprintf(stderr, "Can't open file `%s'.\n", midfile);
      } else {
	if(midfileformat == SMF_format) {
	  unsigned char data[14];
	  if(fread(data, sizeof(unsigned char), 14, fp) != 14)
	    fprintf(stderr, "Error reading `%s'.\n", midfile);
	  else
	    timebase = (data[12]<<8) | data[13];
	}
	if(midfileformat == RCP_format) {
	  unsigned char msb, lsb;
	  if(fseek(fp, 0x01c0, SEEK_SET) == -1 ||
	     fread(&lsb, sizeof(unsigned char), 1, fp) != 1 ||
	     fseek(fp, 0x01e7, SEEK_SET) == -1 ||
	     fread(&msb, sizeof(unsigned char), 1, fp) != 1)
	    fprintf(stderr, "Error reading `%s'.\n", midfile);
	  else
	    timebase = (msb<<8) | lsb;
	}
	if(midfileformat == G36_format) {
	  unsigned char msb, lsb;
	  if(fseek(fp, 0x020a, SEEK_SET) == -1 ||
	     fread(&lsb, sizeof(unsigned char), 1, fp) != 1 ||
	     fread(&msb, sizeof(unsigned char), 1, fp) != 1)
	    fprintf(stderr, "Error reading `%s'.\n", midfile);
	  else
	    timebase = (msb<<8) | lsb;
	}
	fclose(fp);
      }
    }
  } else
    timebase = initial_timebase;

  tempo = ((initial_tempo == -1) ? 160 : initial_tempo);
		// read tempo from midi file !!!
  gradate.tempo = tempo;
  data = new wrddata[DATABLOCK];
}


WrdFile::~WrdFile()
{
  delete[] data;
}


/*
 * parameter analyzer for macro command
 */
int WrdFile::param(const char* cmdname, Param* p, unsigned* ret)
{
  const unsigned max_param = 32;
  int success = 0;	// number of parameters successfully read
			// set to -1 if error occured
  char separator = 0;	// for LOCATE
  if(ret != NULL)
    *ret = 0;		// bitmapped parameters successfully read
  int error = 0;

  int c;
  if((c = fgetc(fp)) != '(') {
    ungetc(c, fp);
    if(strcmp(cmdname, "STARTUP")) {
      if(verbose != verbose_quiet)
	fprintf(stderr, "%d: @%s: no '('.\n", line, cmdname);
      return -1;
    }
  }

  unsigned n_param = 0;	// number of parameters (includes omittables)
  while(p[n_param++].type != param_end);
  if(--n_param > max_param) {
    if(verbose != verbose_quiet)
      fprintf(stderr, "param: too many parameters expected.\n");
    return -1;
  }

  int i;
  for(i = 0; i < (int)n_param; i++) {
    if(p[i].type != param_end && p[i].ptr == NULL) {
      if(verbose != verbose_quiet)
	fprintf(stderr, "param: ptr is NULL.\n");
    } else {
      switch(p[i].type) {
      case param_int:
	if(fscanf(fp, "%u", (unsigned*) p[i].ptr) != 1) {
	  if(p[i].omit != param_omit) {
	    if(verbose != verbose_quiet)
	      fprintf(stderr, "%d: @%s: argument %d is omitted.\n",
		      line, cmdname, i+1);
	    error++;
	  }
	} else {
	  success++;
	  if(ret != NULL)
	    *ret |= 1<<i;
	}
	break;
      case param_no:
	while((c = fgetc(fp)) == ' ' || c == '\t');
	if(c != '#') {
	  ungetc(c, fp);
	  break;
	}
	if(fscanf(fp, "%u", (unsigned*) p[i].ptr) != 1) {
	  if(p[i].omit != param_omit) {
	    if(verbose != verbose_quiet)
	      fprintf(stderr, "%d: @%s: argument %d is omitted.\n",
		      line, cmdname, i+1);
	    error++;
	  }
	} else {
	  success++;
	  if(ret != NULL)
	    *ret |= 1<<i;
	}
	break;
      case param_hex:
	if(fscanf(fp, "%x", (unsigned*) p[i].ptr) != 1) {
	  if(p[i].omit != param_omit) {
	    if(verbose != verbose_quiet)
	      fprintf(stderr, "%d: @%s: argument %d is omitted.\n",
		      line, cmdname, i+1);
	    error++;
	  }
	} else {
	  success++;
	  if(ret != NULL)
	    *ret |= 1<<i;
	}
	break;
      case param_eint:
	while((c = fgetc(fp)) == ' ' || c == '\t');
	if(c != '$') {
	  ungetc(c, fp);
	  if(fscanf(fp, "%d", (int*) p[i].ptr) != 1) {
	    if(p[i].omit != param_omit) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @%s: argument %d is omitted.\n",
			line, cmdname, i+1);
	      error++;
	      break;
	    }
	  }
	} else {
	  if(fscanf(fp, "%x", (unsigned*) p[i].ptr) != 1) {
	    if(p[i].omit != param_omit) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @%s: argument %d is omitted.\n",
			line, cmdname, i+1);
	      error++;
	      break;
	    }
	  }
	}
	success++;
	if(ret != NULL)
	  *ret |= 1<<i;
	break;
      case param_str:
	{
	  char* ptr = (char*) p[i].ptr;
	  for(;;) {
	    c = fgetc(fp);
	    if(c == ',' || c == '\n') {
	      ungetc(c, fp);
	      break;
	    }
	    if(c == ')') {
	      if(!strcmp(cmdname, "ESC")) {
		if((c = fgetc(fp)) != '0' && c != '3') {
		  ungetc(c, fp);
		  ungetc(')', fp);
		  break;
		} else
		  *ptr++ = ')';
	      } else {
		ungetc(c, fp);
		break;
	      }
	    }
	    *ptr++ = c;
	  }
	  *ptr = '\0';
	}
	if(strlen((char*)p[i].ptr) == 0) {
	  if(p[i].omit != param_omit) {
	    if(verbose != verbose_quiet)
	      fprintf(stderr, "%d: @%s: argument %d is omitted.\n",
		      line, cmdname, i+1);
	    error++;
	  }
	} else {
	  success++;
	  if(ret != NULL)
	    *ret |= 1<<i;
	}
	break;
      case param_end:
	break;
      default:
	if(verbose != verbose_quiet)
	  fprintf(stderr, "param: unexpected parameter type.\n");
	break;
      }
    }
    for(;;) {
      switch(c = fgetc(fp)) {
      case ' ':
      case '\t':
	continue;
      case ',':
      case ';':
	separator = c;
	if(c == ';' &&
	   strcmp(cmdname, "LOCATE") && strcmp(cmdname, "COLOR")) {
	  if(verbose != verbose_quiet)
	    fprintf(stderr, "%d: @%s: unexpected character ';' appeared.\n",
		    line, cmdname);
	}
	if(i+1 >= (int)n_param) {
	  if(verbose != verbose_quiet)
	    fprintf(stderr, "%d: @%s: too many argument.\n", 
		    line, cmdname);
	  // error recovery
	  while((c = fgetc(fp)) != CR && c != EOF && c != ')');
	  ungetc(c, fp);
	  continue;
	}
	break;
      case ')':
	if(i+1 < (int)n_param) {
	  int j;
	  for(j = i+1; j < (int)n_param; j++) {
	    if(p[j].omit != param_omit)
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @%s: argument %d is omitted.\n",
			line, cmdname, j+1);
	  }
	}
	if(!strcmp(cmdname, "LOCATE") && separator == ',' && n_param == 2) {
	  unsigned tmp = *(unsigned*)(p[0].ptr);	// exchange parameters
	  *(unsigned*)(p[0].ptr) = *(unsigned*)(p[1].ptr);
	  *(unsigned*)(p[1].ptr) = tmp;
	}
	return (error) ? -1 : success;
      case '\n':
      case EOF:
	if(verbose != verbose_quiet)
	  fprintf(stderr, "%d: @%s: no ')'.\n", line, cmdname);
	return (error) ? -1 : success;
      default:
	ungetc(c, fp);
	break;
      }
      break;
    }
  }
  return (error) ? -1 : success;
}


/*
 * get macro command name
 */
token_type WrdFile::macro(char* cmd)
{
  int c;
  unsigned len;
  for(len = 0; len < LONGEST_CMD; len++) {
    if(!isalpha((c = fgetc(fp)))) {
      if(c == '(' || c == ' ' || c == '@' || c == '^' || c == CR || c == LF)
	ungetc(c, fp);
      else
	error++;
      break;
    } else
      cmd[len] = tolower(c);
  }
  cmd[len] = '\0';
  if(len == LONGEST_CMD) {
    if(isalpha((c = fgetc(fp)))) {
      if(!strncasecmp(cmd, "REM", 3))
	return tREM;	// consider the command started with "REM" as a remark
      else
	return tINVALID;
    }
    if(c == '(' || c == ' ' || c == '@' || c == '^' || c == CR || c == LF)
      ungetc(c, fp);
    else
      error++;
  }
  if(len < SHORTEST_CMD)
    return (len == 0) ? tATMARK : tINVALID;
  if(!strncasecmp(cmd, "REM", 3))
    return tREM;	// consider the command started with "REM" as a remark
  const resword* res = is_reserved_word(cmd, len);
  return (res != NULL) ? res->token : tINVALID;
}


/*
 * event recorder
 */
void WrdFile::record(token_type cmd, void* param)
{
  data[datasize].bar = nowbar;
  data[datasize].step = nowstep;
  data[datasize].cmd = cmd;
  data[datasize].param = param;
  if(++datasize >= maxdata) {
    maxdata += DATABLOCK;
    if((data = (wrddata*) realloc(data, maxdata)) == NULL) {
      fprintf(stderr, "Memory exhausted, aborting...\n");
    }
  }

  if(cmd == tFADE) {
    static bool first = true;
    static unsigned fadeBar = 0, fadeStep = 0;
    if(!first &&
       !(nowbar > fadeBar || (nowbar == fadeBar && nowstep > fadeStep)))
      if(verbose != verbose_quiet)
	fprintf(stderr, 
		"%d: @fade(%d:%d) appeared again before finishing preceding "
		"@fade(%d:%d)\n",
		line, nowbar, nowstep, fadeBar, fadeStep);
    if(first)
      first = false;
    int speed = (int)((unsigned*)param)[2];
    if(speed != 0) {
      fadeBar = nowbar;
      fadeStep = nowstep;
      if(speed != -1)
	fadeStep += speed;
      else
	;	// VSYNC ???
      if(fadeStep >= 4*timebase) {
	fadeBar += fadeStep/(4*timebase);
	fadeStep = fadeStep%(4*timebase);
      }
    }
  }
}


/*
 * WRD file analyzer
 */
void WrdFile::analyze(void)
{
  if(fp == NULL)
    return;

  const unsigned MAX_WORD = 255;	// maximum length of word
  const unsigned MAX_ESC = 80;		// maximum length of escape sequence
//  const unsigned MAX_REMARK = 80;	// maximum length of remark
  bool donext = true;	// do next line at a time

  while(!feof(fp)) {
    if(donext) {
      donext = false;
    } else {
      if(wmode_n == 0) {
	nowbar++;
	nowstep = 0;
      } else {
	nowstep += (wmode_n+1)*2*timebase/48;	// ???
	if(nowstep >= 4*timebase) {
	  nowbar += nowstep/(4*timebase);
	  nowstep = nowstep%(4*timebase);
	}
      }
    }

    int c;
    switch(c = fgetc(fp)) {
    case '@':	// macro command
      {
	char cmd[LONGEST_CMD+1];
	token_type t = macro(cmd);
	if(verbose == verbose_debug)
	  fprintf(stderr, "%d:%d\n@%s\n", nowbar, nowstep, cmd);
	switch(t) {
	default:
	case tINVALID:
	  if(verbose != verbose_quiet)
	    fprintf(stderr, "%d: invalid macro command `@%s'.\n", line, cmd);
	  error++;
	  break;
	case tATMARK: // no operation
	  break;
	case tCOLOR:
	  {
	    unsigned* d = new unsigned[6];	// enough ???
	    Param p[7] = {
	      {param_int, &d[0], param_noomit},	// c
	      {param_int, &d[1], param_omit},
	      {param_int, &d[2], param_omit},
	      {param_int, &d[3], param_omit},
	      {param_int, &d[4], param_omit},
	      {param_int, &d[5], param_omit},
	      {param_end}
	    };
	    if(param("COLOR", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(47 < d[0]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @COLOR: invalid argument (%d).\n",
			line, d[0]);
	      error++;
	      delete[] d;
	      break;
	    }
	    // Here, convert color code !!!
	    record(tCOLOR, d);
	  }
	  break;
	case tEND:
	  {
	    Param p[1] = {
	      {param_end}
	    };
	    param("END", p, NULL);
	    record(tEND, NULL);
	  }
	  break;
	case tESC:
	  {
	    char str[2+MAX_ESC+1] = {'\x1b', '['};
	    Param p[2] = {
	      {param_str, &str[2], param_noomit},
	      {param_end}
	    };
	    if(param("ESC", p, NULL) < 0)
	      break;
	    char* d = new char[strlen(str)+1];
	    strcpy(d, str);
	    record(tESC, d);
	  }
	  break;
	case tEXEC:
	  {
	    char file_name[MAX_FILENAME+1];
	    Param p[2] = {
	      {param_str, file_name, param_noomit},
	      {param_end}
	    };
	    if(param("EXEC", p, NULL) < 0)
	      break;
	    char* d = new char[strlen(file_name)+1];
	    strcpy(d, file_name);
	    record(tEXEC, d);
	  }
	  break;
	case tFADE:
	  {
	    unsigned* d = new unsigned[3];
	    Param p[4] = {
	      {param_int, &d[0], param_noomit},	// p1
	      {param_int, &d[1], param_noomit},	// p2
	      {param_int, &d[2], param_omit},	// speed
	      {param_end}
	    };
	    unsigned ret;
	    if(param("FADE", p, &ret) < 0) {
	      delete[] d;
	      break;
	    }
	    if(19 < d[0] || 19 < d[1]) {	// range of speed ???
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @FADE: invalid argument (%d,%d,%d).\n", 
			line, d[0], d[1], d[2]);
	      delete[] d;
	      break;
	    }
	    if(!(ret & (1<<2)))
	      d[2] = (unsigned)-1;	// VSYNC
	    record(tFADE, d);
	  }
	  break;
	case tGCIRCLE:
	  {
	    unsigned* d = new unsigned[6];
	    Param p[7] = {
	      {param_int, &d[0], param_noomit},	// x
	      {param_int, &d[1], param_noomit},	// y
	      {param_int, &d[2], param_noomit},	// r
	      {param_int, &d[3], param_noomit},	// p1
	      {param_int, &d[4], param_noomit},	// sw
	      {param_int, &d[5], param_omit},	// p2	omit ???
	      {param_end}
	    };
	    if(param("GCIRCLE", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(GX_SIZE < d[0] || GY_SIZE < d[1] || GX_SIZE < d[2] ||
	       GY_SIZE < d[2] || 15 < d[3] || 2 < d[4] || 15 < d[5]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @GCIRCLE: invalid argument "
			"(%d,%d,%d,%d,%d,%d).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5]);
	      delete[] d;
	      break;
	    }
	    record(tGCIRCLE, d);
	  }
	  break;
	case tGCLS:
	  {
	    unsigned sw = 0;
	    Param p[2] = {
	      {param_int, &sw, param_omit},
	      {param_end}
	    };
	    param("GCLS", p, NULL);
	    if(15 < sw) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "@GCLS: invalid argument.\n");
	      break;
	    }
	    record(tGCLS, new unsigned(sw));
	  }
	  break;
	case tGINIT:
	  {
	    Param p[1] = {
	      {param_end}
	    };
	    param("GINIT", p, NULL);
	    record(tGINIT, NULL);
	  }
	  break;
	case tGLINE:
	  {
	    unsigned* d = new unsigned[7];
	    d[5] = d[6] = 0;
	    Param p[8] = {
	      {param_int, &d[0], param_noomit},	// x1
	      {param_int, &d[1], param_noomit},	// y1
	      {param_int, &d[2], param_noomit},	// x2
	      {param_int, &d[3], param_noomit},	// y2
	      {param_int, &d[4], param_noomit},	// p1
	      {param_int, &d[5], param_omit},	// sw
	      {param_int, &d[6], param_omit},	// p2
	      {param_end}
	    };
	    if(param("GLINE", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(/*GX_SIZE < d[0] || GY_SIZE < d[1] || GX_SIZE < d[2] ||
		GY_SIZE < d[3] || */
	       15 < d[4] || 2 < d[5] || (d[5] == 2 ? 15U : 255U) < d[6]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @GLINE: invalid argument "
			"(%d,%d,%d,%d,%d,%d,%d).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5], d[6]);
	      delete[] d;
	      break;
	    }
	    record(tGLINE, d);
	  }
	  break;
	case tGMODE:
	  {
	    unsigned sw;
	    Param p[2] = {
	      {param_int, &sw, param_noomit},
	      {param_end}
	    };
	    if(param("GMODE", p, NULL) < 0)
	      break;
	    if(15 < sw) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @GMODE: invalid argument (%d).\n", 
			line, sw);
	      error++;
	      break;
	    }
	    record(tGMODE, new unsigned(sw));
	  }
	  break;
	case tGMOVE:
	  {
	    unsigned* d = new unsigned[9];
	    d[6] = d[7] = d[8] = 0;
	    Param p[10] = {
	      {param_int, &d[0], param_noomit},	// x1
	      {param_int, &d[1], param_noomit},	// y1
	      {param_int, &d[2], param_noomit},	// x2
	      {param_int, &d[3], param_noomit},	// y2
	      {param_int, &d[4], param_noomit},	// xd
	      {param_int, &d[5], param_noomit},	// yd
	      {param_int, &d[6], param_omit},	// vs
	      {param_int, &d[7], param_omit},	// vd
	      {param_int, &d[8], param_omit},	// sw
	      {param_end}
	    };
	    if(param("GMOVE", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if((d[6] != 0 && d[6] != 1) || (d[7] != 0 && d[7] != 1)) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @GMOVE: invalid argument "
			"(%d,%d,%d,%d,%d,%d,%d,%d,%d).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7],
			d[8]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tGMOVE, d);
	  }
	  break;
	case tGON:
	  {
	    unsigned sw = 1;
	    Param p[2] = {
	      {param_int, &sw, param_omit},
	      {param_end}
	    };
	    param("GON", p, NULL);
	    if(sw != 0 && sw != 1) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @GON: invalid argument (%d).\n", 
			line, sw);
	      error++;
	      break;
	    }
	    record(tGON, new unsigned(sw));
	  }
	  break;
	case tGSCREEN:
	  {
	    unsigned* d = new unsigned[2];
	    Param p[3] = {
	      {param_int, &d[0], param_noomit},	// act
	      {param_int, &d[1], param_noomit},	// dsp
	      {param_end}
	    };
	    if(param("GSCREEN", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if((d[0] != 0 && d[0] != 1) || (d[1] != 0 && d[1] != 1)) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @SCREEN: invalid argument (%d, %d).\n", 
			line, d[0], d[1]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tGSCREEN, d);
	  }
	  break;
	case tINKEY:
	  {
	    unsigned bar;
	    Param p[2] = {
	      {param_int, &bar, param_noomit},
	      {param_end}
	    };
	    if(param("INKEY", p, NULL) < 0)
	      break;
	    record(tINKEY, new unsigned(bar));
	  }
	  break;
	case tLOCATE:
	  {
	    unsigned* d = new unsigned[2];
	    Param p[3] = {
	      {param_int, &d[0], param_noomit},	// y
	      {param_int, &d[1], param_noomit},	// x
	      {param_end}
	    };
	    if(param("LOCATE", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(TY_SIZE < d[0] || TX_SIZE < d[1]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @LOCATE: invalid argument (%d, %d).\n", 
			line, d[0], d[1]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tLOCATE, d);
	  }
	  break;
	case tLOOP:
	  {
	    unsigned count;
	    Param p[2] = {
	      {param_int, &count, param_noomit},
	      {param_end}
	    };
	    if(param("LOOP", p, NULL) < 0)
	      break;
	    record(tLOOP, new unsigned(count));
	  }
	  break;
	case tMAG:
	  {
	    unsigned* d = new unsigned[5];
	    char file[MAX_FILENAME+1];
	    d[3] = 1;
	    d[4] = 0;
	    Param para[6] = {
	      {param_str, file, param_noomit},
	      {param_int, &d[1], param_omit},	// x
	      {param_int, &d[2], param_omit},	// y
	      {param_int, &d[3], param_omit},	// s
	      {param_int, &d[4], param_omit},	// p
	      {param_end}
	    };
	    unsigned ret;
	    if(param("MAG", para, &ret) < 0) {
	      delete[] d;
	      break;
	    }
	    if(!(ret & (1<<1)))
	      d[1] = (unsigned)-1;
	    if(!(ret & (1<<2)))
	      d[2] = (unsigned)-1;
	    if(d[3] == 0)	// for a certain WRD file ???
	      d[3] = 1;
	    if(((ret & (1<<1)) && GX_SIZE < d[1]) ||
	       ((ret & (1<<2)) && GY_SIZE < d[2]) ||
	       d[3] < 1 || 16 < d[3] || 2 < d[4]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, 
			"%d: @MAG: invalid argument (%s,%d,%d,%d,%d).\n", 
			line, file, d[1], d[2], d[3], d[4]);
	      error++;
	      delete[] d;
	      break;
	    }
	    // cast problem between unsigned and char* !!!
	    d[0] = new char[strlen(file)+1];
	    strcpy((char*)(d[0]), file);
	    record(tMAG, d);
#if 0
	    if(preloadMag) {
	      magList* list;
	      if(magList == NULL) {
		magList = new (magList*)[2];
		list = magList[0] = new magList;
		magList[1] = NULL;
	      } else {
		magList* old = magList;
		int i, j;
		for(i = 0; old[i] != NULL; i++);
		magList = new (magList*)[i+2];
		for(j = 0; j < i; j++)
		  magList[j] = old[j];
		delete[] old;
		list = magList[i] = new magList;
		magList[i+1] = NULL;
	      }
	      list->file = new char[strlen(file)+1];
	      strcpy(list->file, file);
	      char magfile[MAX_PATH+MAX_FILENAME+1];
	      strcpy(magfile, file);
//            sprintf(magfile, "%s%s", path, file);
	      FILE* magfp;
	      if((magfp = dos_fopen(magfile, "r")) == NULL) {
		if(verbose != verbose_quiet)
		  fprintf(stderr, "MAG file `%s' not found.\n", 
			  file);
		error++;
		break;
	      }
	      if(verbose != verbose_quiet)
		fprintf(stderr, "Preloading MAG file `%s'.\n", file);
	      MagLoad();
	      list->mode = MagHeader.ScreenMode;

	    }
#endif
	  }
	  break;
	case tMIDI:
	  {
	    unsigned* d = new unsigned[10];
	    Param p[11] = {
	      {param_hex, &d[0], param_noomit},
	      {param_hex, &d[1], param_omit},
	      {param_hex, &d[2], param_omit},
	      {param_hex, &d[3], param_omit},
	      {param_hex, &d[4], param_omit},
	      {param_hex, &d[5], param_omit},
	      {param_hex, &d[6], param_omit},
	      {param_hex, &d[7], param_omit},
	      {param_hex, &d[8], param_omit},
	      {param_hex, &d[9], param_omit},
	      {param_end}
	    };
	    if(param("MIDI", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    record(tMIDI, d);
	  }
	  break;
	case tOFFSET:
	  {
	    unsigned nnn;
	    Param p[2] = {
	      {param_int, &nnn, param_noomit},
	      {param_end}
	    };
	    if(param("OFFSET", p, NULL) < 0)
	      break;
	    offset = nnn;
	    break;
	  }
	  break;
	case tPAL:
	  {
	    unsigned* d = new unsigned[17];
	    d[0] = 0;
	    Param para[18] = {
	      {param_no, &d[0], param_omit},		// p
	      {param_hex, &d[1], param_noomit},		// rgb0
	      {param_hex, &d[2], param_noomit},		// rgb1
	      {param_hex, &d[3], param_noomit},		// rgb2
	      {param_hex, &d[4], param_noomit},		// rgb3
	      {param_hex, &d[5], param_noomit},		// rgb4
	      {param_hex, &d[6], param_noomit},		// rgb5
	      {param_hex, &d[7], param_noomit},		// rgb6
	      {param_hex, &d[8], param_noomit},		// rgb7
	      {param_hex, &d[9], param_noomit},		// rgb8
	      {param_hex, &d[10], param_noomit},	// rgb9
	      {param_hex, &d[11], param_noomit},	// rgb10
	      {param_hex, &d[12], param_noomit},	// rgb11
	      {param_hex, &d[13], param_noomit},	// rgb12
	      {param_hex, &d[14], param_noomit},	// rgb13
	      {param_hex, &d[15], param_noomit},	// rgb14
	      {param_hex, &d[16], param_noomit},	// rgb15
	      {param_end}
	    };
	    if(param("PAL", para, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    bool invalid = false;
	    int i;
	    for(i = 0; i < 16; i++)
	      if(0xfff < d[i+1])
		invalid = true;
	    if(19 < d[0] || invalid) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @PAL: invalid argument #%d.\n", 
			line, d[0]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tPAL, d);
	  }
	  break;
	case tPALCHG:
	  {
	    char file[MAX_FILENAME+1];
	    Param p[2] = {
	      {param_str, file, param_noomit},
	      {param_end}
	    };
	    if(param("PALCHG", p, NULL) < 0)
	      break;
	    char* d = new char[strlen(file)+1];
	    strcpy(d, file);
	    record(tPALCHG, d);
	  }
	  break;
	case tPALREV:
	  {
	    unsigned p;
	    Param para[2] = {
	      {param_int, &p, param_noomit},
	      {param_end}
	    };
	    if(param("PALREV", para, NULL) < 0)
	      break;
	    if(19 < p) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @PALREV: invalid argument %d.\n", 
			line, p);
	      error++;
	      break;
	    }
	    record(tPALREV, new unsigned(p));
	  }
	  break;
	case tPATH:
	  {
	    char path[MAX_PATH+1];
	    Param p[2] = {
	      {param_str, path, param_omit},
	      {param_end}
	    };
	    if(param("PATH", p, NULL) < 0)
	      break;
	    if(!strcmp("*", path)) {
	    }
	    if(!strcmp("@", path)) {
	    }
	    if(!strcmp("", path)) {
	    }
	    char* d = new char[strlen(path)+1];
	    strcpy(d, path);
//	    record(tPATH, d);
	  }
	  break;
	case tPLOAD:
	  {
	    char file[MAX_FILENAME+1];
	    Param p[2] = {
	      {param_str, file, param_noomit},
	      {param_end}
	    };
	    if(param("PLOAD", p, NULL) < 0)
	      break;
	    char* d = new char[strlen(file)+1];
	    strcpy(d, file);
	    record(tPLOAD, d);
	  }
	  break;
	case tREM:
	case tREMARK:
	  {
#if 0
	    char str[MAX_REMARK+1];
	    Param p[2] = {
	      {param_str, &str, param_omit},
	      {param_end}
	    };
	    param("REMARK", p, NULL);
#endif
	    for(;;) {
	      switch((c = fgetc(fp))) {
	      case ';':
		if((c = fgetc(fp)) == CR) {
		  line++;
		  donext = true;
		  if((c = fgetc(fp)) != LF)
		    ungetc(c, fp);
		} else {
		  ungetc(c, fp);
		  if(c != EOF)
		    continue;
		}
		break;
/*	      case LF:
		line++;
		break;*/
	      case CR:
		if((c = fgetc(fp)) == LF) {
		  line++;
		} else
		  ungetc(c, fp);
		break;
	      case EOF:
		break;
	      default:
		while((c = fgetc(fp)) != CR && c != EOF && c != ';');
		ungetc(c, fp);
		continue;
	      }
	      break;
	    }
	  }
	  break;
	case tREST:
	  {
	    unsigned d[2];
	    d[1] = 0;
	    Param p[3] = {
	      {param_int, &d[0], param_noomit},	// bar
	      {param_int, &d[1], param_omit},	// step
	      {param_end}
	    };
	    if(param("REST", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    nowbar += d[0];
	    nowstep = d[1];
	  }
	  break;
	case tSCREEN:
	  
	  break;
	case tSCROLL:
	  {
	    unsigned* d = new unsigned[7];
	    d[0] = 1;
	    d[1] = 1;
	    d[2] = TX_SIZE;
	    d[3] = TY_SIZE;
	    d[4] = 0;
	    d[5] = 0;
	    d[6] = ' ';
	    Param p[8] = {
	      {param_int, &d[0], param_omit},	// x1
	      {param_int, &d[1], param_omit},	// y1
	      {param_int, &d[2], param_omit},	// x2
	      {param_int, &d[3], param_omit},	// y2
	      {param_int, &d[4], param_omit},	// mode
	      {param_int, &d[5], param_omit},	// color
	      {param_int, &d[6], param_omit},	// char
	      {param_end}
	    };
	    if(param("SCROLL", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(d[0] < 1 || TX_SIZE < d[0] || d[1] < 1 || TY_SIZE < d[1] ||
	       d[2] < 1 || TX_SIZE < d[2] || d[3] < 1 || TY_SIZE < d[3] ||
	       3 < d[4] || 16 < d[5] || 255 < d[6]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @SCROLL: invalid argument "
			"(%d,%d,%d,%d,%d,%d,%d).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5], d[6]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tSCROLL, d);
	  }	    
	  break;
	case tSTARTUP:
	  {
	    unsigned ver_;
	    Param p[2] = {
	      {param_int, &ver_, param_omit},
	      {param_end}
	    };
	    // if it apears not on the top of file ???
	    unsigned ret;
	    if(param("STARTUP", p, &ret) < 0)
	      break;
	    int ver = (ret & (1<<0)) ? (int)ver_ : -1;
	    if(ver > 999) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @STARTUP: invalid argument (%d).\n", 
			line, ver);
	      error++;
	      break;
	    }
	    setVersion(ver);
	  }
	  break;
	case tSTOP:
	  {
	    Param p[1] = {
	      {param_end}
	    };
	    param("STOP", p, NULL);
	    record(tSTOP, NULL);
	  }
	  break;
	case tTCLS:
	  {
	    unsigned* d = new unsigned[6];
	    d[0] = 1;
	    d[1] = 1;
	    d[2] = TX_SIZE;
	    d[3] = TY_SIZE;
	    d[4] = 0;
	    d[5] = ' ';
	    Param p[7] = {
	      {param_int, &d[0], param_omit},	// x1
	      {param_int, &d[1], param_omit},	// y1
	      {param_int, &d[2], param_omit},	// x2
	      {param_int, &d[3], param_omit},	// y2
	      {param_int, &d[4], param_omit},	// color
	      {param_int, &d[5], param_omit},	// char
	      {param_end}
	    };
	    if(param("TCLS", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(/*d[0] < 1 ||*/TX_SIZE < d[0] ||/*d[1] < 1 ||*/TY_SIZE < d[1] ||
	       /*d[2] < 1 ||*/TX_SIZE < d[2] ||/*d[3] < 1 ||*/TY_SIZE < d[3] ||
	       47 < d[4] || 255 < d[5]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr,
			"%d: @TCLS: invalid argument (%d,%d,%d,%d,%d,%d).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5]);
	      error++;
	      delete[] d;
	      break;
	    }
	    if(d[0] == 0) d[0] = 1;
	    if(d[1] == 0) d[1] = 1;
	    if(d[2] == 0) d[2] = 1;
	    if(d[3] == 0) d[3] = 1;
	    record(tTCLS, d);
	  }
	  break;
	case tTON:
	  {
	    unsigned sw;
	    Param p[2] = {
	      {param_int, &sw, param_noomit},
	      {param_end}
	    };
	    if(param("TON", p, NULL) < 0)
	      break;
	    if(sw != 0 && sw != 1 && sw != 2) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @TON: invalid argument (%d).\n", 
			line, sw);
	      error++;
	      break;
	    }
	    record(tTON, new unsigned(sw));
	  }
	  break;
	case tWAIT:
	  {
	    unsigned d[2];
	    d[1] = 0;
	    Param p[3] = {
	      {param_int, &d[0], param_noomit},	// bar
	      {param_int, &d[1], param_omit},	// step
	      {param_end}
	    };
	    if(param("WAIT", p, NULL) < 0)
	      break;
	    if(d[0]+offset < nowbar ||
	       (d[0]+offset == nowbar && d[1] < nowstep)) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @WAIT: backward timing (%d,%d), "
			"now (%d,%d).\n", 
			line, d[0]+offset, d[1], nowbar, nowstep);
	      break;				// warning only ???
	    }
	    nowbar = d[0]+offset;
	    nowstep = d[1];
	  }
	  break;
	case tWMODE:
	  {
	    unsigned* d = new unsigned[2];
	    d[1] = wmode;
	    Param p[3] = {
	      {param_int, &d[0], param_noomit},	// n
	      {param_int, &d[1], param_omit},	// mode
	      {param_end}
	    };
	    if(param("WMODE", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(255 < d[0] || 1 < d[1]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: @WMODE: invalid argument (%d,%d).\n", 
			line, d[0], d[1]);
	      error++;
	      delete[] d;
	      break;
	    }
	    wmode_n = d[0];
	    wmode = d[1];
	  }
	  break;
#if 0
	default:
	  if(verbose != verbose_quiet)
	    fprintf(stderr, "%d: internal error: Invalid token type `%d'.\n",
		    line, t);
	  break;
#endif
	}
	for(;;) {
	  switch((c = fgetc(fp))) {
	  case ' ':
	  case '\t':
	    continue;
	  case '@':
	  case '^':
	    ungetc(c, fp);
	    donext = true;
	    break;
	  case ';':
	    if((c = fgetc(fp)) == CR) {
	      line++;
	      donext = true;
	      if((c = fgetc(fp)) != LF)
		ungetc(c, fp);
	    } else {
	      ungetc(c, fp);
	      continue;
	    }
	    break;
/*  	  case LF:
	    line++;
	    break;*/
	  case CR:
	    if((c = fgetc(fp)) == LF) {
	      line++;
	    } else
	      ungetc(c, fp);
	    break;
	  case EOF:
	    break;
	  default:
	    while((c = fgetc(fp)) != CR && c != EOF && c != ';');
	    ungetc(c, fp);
	    continue;
	  }
	  break;
	}
      }
      break;

    case '^':	// ensyutsukun mechanism
      {
	char cmd[LONGEST_CMD+1];
	token_type t = macro(cmd);
	if(verbose == verbose_debug)
	  fprintf(stderr, "%d:%d\n^%s\n", nowbar, nowstep, cmd);
	switch(t) {
	default:
	case tINVALID:
	  if(verbose != verbose_quiet)
	    fprintf(stderr, "%d: invalid macro command `^%s'.\n", line, cmd);
	  error++;
	  break;
	case tATMARK: // `^' alone ???
			// no operation or part of lyrics
	  break;
	case tFONTM:
	  {
	    unsigned mode;
	    Param p[2] = {
	      {param_eint, &mode, param_noomit},
	      {param_end}
	    };
	    if(param("FONTM", p, NULL) < 0)
	      break;
	    if(1 < mode) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^FONTM: invalid argument (%d).\n", 
			line, mode);
	      error++;
	      break;
	    }
	    record(tFONTM, new unsigned(mode));
	  }
	  break;
	case tFONTP:
	  {
	    unsigned* d = new unsigned[4];
	    Param p[5] = {
	      {param_eint, &d[0], param_noomit},	// num
	      {param_eint, &d[1], param_noomit},	// x
	      {param_eint, &d[2], param_noomit},	// y
	      {param_eint, &d[3], param_noomit},	// atr
	      {param_end}
	    };
	    if(param("FONTP", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(15 < d[0] || TX_SIZE < d[1] || TY_SIZE < d[2] || 255 < d[3]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^FONTP: invalid argument (%u,%u,%u,%u).\n", 
			line, d[0], d[1], d[2], d[3]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tFONTP, d);
	  }
	  break;
	case tFONTR:
	  {
	    unsigned* d = new unsigned[17];
	    Param p[18] = {
	      {param_eint, &d[0], param_noomit},	// num
	      {param_eint, &d[1], param_noomit},	// d1
	      {param_eint, &d[2], param_noomit},	// d2
	      {param_eint, &d[3], param_noomit},	// d3
	      {param_eint, &d[4], param_noomit},	// d4
	      {param_eint, &d[5], param_noomit},	// d5
	      {param_eint, &d[6], param_noomit},	// d6
	      {param_eint, &d[7], param_noomit},	// d7
	      {param_eint, &d[8], param_noomit},	// d8
	      {param_eint, &d[9], param_noomit},	// d9
	      {param_eint, &d[10], param_noomit},	// d10
	      {param_eint, &d[11], param_noomit},	// d11
	      {param_eint, &d[12], param_noomit},	// d12
	      {param_eint, &d[13], param_noomit},	// d13
	      {param_eint, &d[14], param_noomit},	// d14
	      {param_eint, &d[15], param_noomit},	// d15
	      {param_eint, &d[16], param_noomit},	// d16
	      {param_end}
	    };
	    if(param("FONTR", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(15 < d[0]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^FONTR: invalid argument "
			"(%u,%04x,%04x,%04x,%04x,%04x,%04x,%04x,%04x,"
			"%04x,%04x,%04x,%04x,%04x,%04x,%04x,%04x).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7],
			d[8], d[9], d[10], d[11], d[12], d[13], d[14], d[15],
			d[16]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tFONTR, d);
	  }
	  break;
	case tGSC:
	  {
	    unsigned mode;
	    Param p[2] = {
	      {param_eint, &mode, param_noomit},
	      {param_end}
	    };
	    if(param("GSC", p, NULL) < 0)
	      break;
	    if(1 < mode) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^GSC: invalid argument (%d).\n", 
			line, mode);
	      error++;
	      break;
	    }
	    record(tGSC, new unsigned(mode));
	  }
	  break;
	case tLINE:
	  {
	    unsigned mode;
	    Param p[2] = {
	      {param_eint, &mode, param_noomit},
	      {param_end}
	    };
	    if(param("LINE", p, NULL) < 0)
	      break;
	    if(32 < mode) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^LINE: invalid argument (%d).\n",
			line, mode);
	      error++;
	      break;
	    }
	    record(tLINE, new unsigned(mode));
	  }
	  break;
	case tPAL:
	  {
	    unsigned* d = new unsigned[2];
	    Param p[3] = {
	      {param_eint, &d[0], param_noomit},	// pal
	      {param_eint, &d[1], param_noomit},	// rgb
	      {param_end}
	    };
	    if(param("PAL", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(15 < d[0] || 0xfff < d[1]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^PAL: invalid argument (%u,%03x).\n", 
			line, d[0], d[1]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tPAL_, d);
	  }
	  break;
	case tREGSAVE:
	  {
	    unsigned num, value[16];
	    Param p[18] = {
	      {param_eint, &num, param_noomit},
	      {param_eint, &value[0], param_noomit},
	      {param_eint, &value[1], param_omit},
	      {param_eint, &value[2], param_omit},
	      {param_eint, &value[3], param_omit},
	      {param_eint, &value[4], param_omit},
	      {param_eint, &value[5], param_omit},
	      {param_eint, &value[6], param_omit},
	      {param_eint, &value[7], param_omit},
	      {param_eint, &value[8], param_omit},
	      {param_eint, &value[9], param_omit},
	      {param_eint, &value[10], param_omit},
	      {param_eint, &value[11], param_omit},
	      {param_eint, &value[12], param_omit},
	      {param_eint, &value[13], param_omit},
	      {param_eint, &value[14], param_omit},
	      {param_eint, &value[15], param_omit},
	      {param_end}
	    };
	    if(param("REGSAVE", p, NULL) < 0)
	      break;
	    if(9 < num) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^REGSAVE: invalid argument "
			"(%u,%u,%u,%u,%u,%u,%u,%u,%u,"
			"%u,%u,%u,%u,%u,%u,%u,%u).\n", 
			line, num, value[0], value[1], value[2], value[3],
			value[4], value[5], value[6], value[7], value[8],
			value[9], value[10], value[11], value[12], value[13],
			value[14], value[15]);
	      error++;
	      break;
	    }
	  }
	  break;
	case tSCROLL:
	  {
	    unsigned* d = new unsigned[2];
	    Param p[3] = {
	      {param_eint, &d[0], param_noomit},	// x
	      {param_eint, &d[1], param_noomit},	// y
	      {param_end}
	    };
	    if(param("SCROLL", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(39 < d[0] || 399 < d[1]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^SCROLL: invalid argument (%d,%d).\n", 
			line, d[0], d[1]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tSCROLL_, d);
	  }
	  break;
	case tTEXTDOT:
	  {
	    unsigned sw;
	    Param p[2] = {
	      {param_eint, &sw, param_noomit},
	      {param_end}
	    };
	    if(param("TEXTDOT", p, NULL) < 0)
	      break;
	    if(1 < sw) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^TEXTDOT: invalid argument (%d).\n", 
			line, sw);
	      error++;
	      break;
	    }
	    record(tTEXTDOT, new unsigned(sw));
	  }
	  break;
	case tTMODE:
	  {
	    unsigned mode;
	    Param p[2] = {
	      {param_eint, &mode, param_noomit},
	      {param_end}
	    };
	    if(param("TMODE", p, NULL) < 0)
	      break;
	    if(2 < mode) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^TMODE: invalid argument (%d).\n", 
			line, mode);
	      error++;
	      break;
	    }
	    record(tTMODE, new unsigned(mode));
	  }
	  break;
	case tTSCRL:
	  {
	    unsigned lines;
	    Param p[2] = {
	      {param_eint, &lines, param_noomit},
	      {param_end}
	    };
	    if(param("TSCRL", p, NULL) < 0)
	      break;
	    if(399/*415*/ < lines) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^TSCRL: invalid argument (%d).\n", 
			line, lines);
	      error++;
	      break;
	    }
	    record(tTSCRL, new unsigned(lines));
	  }
	  break;
	case tVCOPY:
	  {
	    unsigned* d = new unsigned[9];
	    Param p[10] = {
	      {param_eint, &d[0], param_noomit},	// sx1
	      {param_eint, &d[1], param_noomit},	// sy1
	      {param_eint, &d[2], param_noomit},	// sx2
	      {param_eint, &d[3], param_noomit},	// sy2
	      {param_eint, &d[4], param_noomit},	// tx
	      {param_eint, &d[5], param_noomit},	// ty
	      {param_eint, &d[6], param_noomit},	// ss
	      {param_eint, &d[7], param_noomit},	// ts
	      {param_eint, &d[8], param_noomit},	// mode
	      {param_end}
	    };
	    if(param("VCOPY", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(GX_SIZE < d[0] || GY_SIZE < d[1] || GX_SIZE < d[2] ||
	       GY_SIZE < d[3] || GX_SIZE < d[4] || GY_SIZE < d[5] ||
	       1 < d[8]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^VCOPY: invalid argument "
			"(%d,%d,%d,%d,%d,%d,%d,%d,%d).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7],
			d[8]);
	      error++;
	      delete[] d;
	      break;
	    }
	    record(tVCOPY, d);
	  }
	  break;
	case tVSGET:
	  {
	    unsigned page[4];	// sufficient ???
	    Param p[5] = {
	      {param_eint, &page[0], param_noomit},
	      {param_eint, &page[1], param_omit},
	      {param_eint, &page[2], param_omit},
	      {param_eint, &page[3], param_omit},
	      {param_end}
	    };
	    if(param("VSGET", p, NULL) < 0)
	      break;
	    if(page[0] < 1) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^VSGET: invalid argument (%u).\n", 
			line, page[0]);
	      error++;
	      break;
	    }
	    record(tVSGET, new unsigned(page[0]));
	  }
	  break;
	case tVSRES:
	  {
	    Param p[1] = {
	      {param_end}
	    };
	    if(param("VSRES", p, NULL) < 0)
	      break;
	    record(tVSRES, NULL);
	  }
	  break;
	case tXCOPY:
	  {
	    unsigned* d = new unsigned[14];
	    Param p[15] = {
	      {param_eint, &d[0], param_noomit},	// sx1
	      {param_eint, &d[1], param_noomit},	// sy1
	      {param_eint, &d[2], param_noomit},	// sx2
	      {param_eint, &d[3], param_noomit},	// sy2
	      {param_eint, &d[4], param_noomit},	// tx
	      {param_eint, &d[5], param_noomit},	// ty
	      {param_eint, &d[6], param_noomit},	// ss
	      {param_eint, &d[7], param_noomit},	// ts
	      {param_eint, &d[8], param_noomit},	// method
	      {param_eint, &d[9], param_omit},		// opt1
	      {param_eint, &d[10], param_omit},		// opt2
	      {param_eint, &d[11], param_omit},		// opt3
	      {param_eint, &d[12], param_omit},		// opt4
	      {param_eint, &d[13], param_omit},		// opt5
	      {param_end}
	    };
	    if(param("XCOPY", p, NULL) < 0) {
	      delete[] d;
	      break;
	    }
	    if(1 < d[6] || 1 < d[7] || 12 < d[8]) {
	      if(verbose != verbose_quiet)
		fprintf(stderr, "%d: ^XCOPY: invalid argument "
			"(%d,%d,%d,%d,%d,%d,%d,%d,%d).\n", 
			line, d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7],
			d[8]);
	      error++;
	      break;
	    }
	    record(tXCOPY, d);
	  }
	  break;
	}
	for(;;) {
	  switch((c = fgetc(fp))) {
	  case ' ':
	  case '\t':
	    continue;
	  case '@':
	  case '^':
	    ungetc(c, fp);
	    donext = true;
	    break;
	  case ';':
	    if((c = fgetc(fp)) == CR) {
	      line++;
	      donext = true;
	      if((c = fgetc(fp)) != LF)
		ungetc(c, fp);
	    } else {
	      ungetc(c, fp);
	      continue;
	    }
	    break;
/*  	  case LF:
	    line++;
	    break;*/
	  case CR:
	    if((c = fgetc(fp)) == LF) {
	      line++;
	    } else
	      ungetc(c, fp);
	    break;
	  case EOF:
	    break;
	  default:
	    while((c = fgetc(fp)) != CR && c != EOF && c != ';');
	    ungetc(c, fp);
	    continue;
	  }
	  break;
	}
      }
      break;

    default:	// lyrics
      {
	ungetc(c, fp);

	char word[MAX_WORD*2+1];
	char tmp[MAX_WORD+1];
	int last;	// the last character
	int i;
	do {
	  bool byte2 = false;	// 2nd byte of SJIS code
	  bool escape = false;	// escape special characters
	  bool multi = false;	// consider multiple characters as a character
	  if(!wmode) {
	    for(i = 0; i < (int)MAX_WORD; i++) {
	      word[i] = c = fgetc(fp);
	      if(c == EOF || c == CR) {
		ungetc(c, fp);
		break;
	      }
	      if(byte2) {
		byte2 = false;
	      } else {
		if(issjkanji((unsigned char)c))
		  byte2 = true;
		if(c == ';') {
		  if((c = fgetc(fp)) == CR || c == EOF) {
		    ungetc(c, fp);
		    c = ';';
		    break;
		  } else
		    ungetc(c, fp);
		}
	      }
	    }
	  } else {
	    for(i = 0; i < (int)MAX_WORD; i++) {
	      word[i] = c = fgetc(fp);
	      if(c == EOF || c == CR) {
		ungetc(c, fp);
		break;
	      }
	      if(!byte2 && c == ';') {
		if((c = fgetc(fp)) == CR || c == EOF) {
		  ungetc(c, fp);
		  c = ';';
		  break;
		} else {
		  ungetc(c, fp);
		  c = ';';
		}
	      }
	      if(multi) {
		if(!byte2) {
		  if(issjkanji((unsigned char)c))
		    byte2 = true;
		  if(c == '|') {
		    multi = false;
		    break;
		  }
		} else
		  byte2 = false;
	      } else {
		if(byte2 || escape) {
		  if(!byte2 && escape && issjkanji((unsigned char)c))
		    byte2 = true;
		  else {
		    i++;
		    break;
		  }
		} else {
		  if(issjkanji((unsigned char)c))
		    byte2 = true;
		  else {
		    if(c == '\\') {
		      escape = true;
		      i--;
		    }
		    if(c == '|') {
		      multi = true;
		      i--;
		    }
		    if(c == '_')
		      i--;
		    if(c != '\\' && c != '|') {
		      i++;
		      break;
		    }
		  }
		}
	      }
	    }
	  }
	  last = c;
	  word[i] = '\0';

	  if(i > 0) {
	    strcpy(tmp, word);
	    kconv((unsigned char*)tmp, (unsigned char*)word);
	    char* d = new char[strlen(word)+1];
	    strcpy(d, word);
	    record(tPRINT, d);

	    if(verbose == verbose_debug)
	      fprintf(stderr, "%d:%d\n%s\n", nowbar, nowstep, word);
	  }
	  if(wmode && last != EOF && last != CR && last != ';') {
	    if(last != ' ' || version >= 380) {
	      if(wmode_n == 0) {
		nowbar++;
		nowstep = 0;
	      } else {
		nowstep += (wmode_n+1)*2*timebase/48;
		if(nowstep >= 4*timebase) {
		  nowbar += nowstep/(4*timebase);
		  nowstep = nowstep%(4*timebase);
		}
	      }
	    }
	  }
	} while(last != EOF && last != CR && last != ';');

	if(last == ';' && (version >= 380 || version == -1))
	  donext = true;
	if(last != ';' /*|| (last == ';' && version < 380)*/) {	// ???
	  char* d = new char[3];
	  d[0] = CR;
	  d[1] = LF;
	  d[2] = '\0';
	  record(tPRINT, d);
	}

	while((c = fgetc(fp)) != EOF && c != LF);
	line++;
      }
      break;
    }
  }
  fclose(fp);
}


/*
 * WRD file player
 */
void WrdFile::play(const char* xhost)
{
  scr = new WrdScreen(xhost);

  char* fname = new char[strlen(path)+strlen(filename)+1];
  sprintf(fname, "%s%s", path, filename);
  scr->changeTitle(fname);
  delete[] fname;

  // load startup MAG, PHO file
  if(magfile) {
    const char* magfile_;
    if((magfile_ = dos_filename(magfile)) == NULL) {
      if(verbose != verbose_quiet)
	fprintf(stderr, "MAG file `%s' not found.\n", magfile);
    } else {
      if(verbose != verbose_quiet)
	fprintf(stderr, "Loading MAG file `%s'.\n", magfile);
      scr->loadMag(magfile_, -1, -1, 1, 1, 1);
      delete[] magfile_;
    }
  }
  if(phofile) {
    FILE* phofp;
    if((phofp = dos_fopen(phofile, "r")) == NULL) {
      if(verbose != verbose_quiet)
	fprintf(stderr, "PHO file `%s' not found.\n", phofile);
    } else {
      if(verbose != verbose_quiet)
	fprintf(stderr, "Loading PHO file `%s'.\n", phofile);
      scr->loadPho(phofp);
    }
  }

  unsigned i;
  for(i = 0; i < datasize; i++) {	// main loop
    union {
      void* v;
      int* i;
      unsigned* ui;
      char* c;
    } p;
    p.v = data[i].param;

    scr->doExpose();
    scr->doKeyRelease();
    scr->doEnterWindow();
    scr->doLeaveWindow();

    if(verbose == verbose_debug)
      fprintf(stderr, "%u:%u\n", data[i].bar, data[i].step);

    if(i > 0 &&
       (data[i].bar != data[i-1].bar || data[i].step != data[i-1].step))
      wait(data[i].bar, data[i].step);

    switch(data[i].cmd) {
    case tINVALID:
    case tATMARK:
    default:
      break;
    // internal command
    case tPRINT:
      scr->print(p.c);
      break;
    // MIMPI
    case tCOLOR:
      scr->tcolor(p.ui[0]);
      break;
    case tEND:
      break;
    case tESC:
      scr->print(p.c);
      break;
    case tEXEC:
      exec(p.c);
      break;
    case tFADE:
      scr->fade(p.ui[0], p.ui[1], p.i[2]);
      break;
    case tGCIRCLE:
      switch(p.ui[4]) {
      case 0:
      case 1:
	scr->drawCircle(p.ui[0], p.ui[1], p.ui[2], p.ui[3]);
	break;
      case 2:
	scr->fillCircle(p.ui[0], p.ui[1], p.ui[2], p.ui[5]);
	break;
      }
      break;
    case tGCLS:
      scr->gclear(p.ui[0]);
      break;
    case tGINIT:
      scr->screen(1, 0);
      scr->gclear(0);
      scr->screen(0, 0);
      scr->gclear(0);
      scr->gon(1);
      break;
    case tGLINE:
      switch(p.ui[5]) {
      case 0:
	scr->drawLine(p.ui[0], p.ui[1], p.ui[2], p.ui[3], p.ui[4]);
	break;
      case 1:
	scr->drawRectangle(p.ui[0], p.ui[1], p.ui[2], p.ui[3], p.ui[4]);
	break;
      case 2:
	scr->fillRectangle(p.ui[0], p.ui[1], p.ui[2], p.ui[3], p.ui[6]);
	break;
      }
      break;
    case tGMODE:
      scr->gmode(p.ui[0]);
      break;
    case tGMOVE:
      scr->move(p.ui[0], p.ui[1], p.ui[2], p.ui[3], p.ui[4], p.ui[5], p.ui[6],
		p.ui[7], p.ui[8]);
      break;
    case tGON:
      scr->gon(p.ui[0]);
      break;
    case tGSCREEN:
      scr->screen(p.ui[0], p.ui[1]);
      break;
    case tINKEY:
      scr->inkey(p.ui[0]);
      break;
    case tLOCATE:
      scr->locate(p.ui[1], p.ui[0]);
      break;
    case tLOOP:
      scr->loop(p.ui[0]);
      break;
    case tMAG:
      {
	const char* file = (char*)(p.ui[0]);
	char magfile[MAX_PATH+MAX_FILENAME+1];
	strcpy(magfile, file);
//      sprintf(magfile, "%s%s", path, file);
	const char* magfile_;
	if((magfile_ = dos_filename(magfile)) == NULL) {
	  if(verbose != verbose_quiet)
	    fprintf(stderr, "MAG file `%s' not found.\n", file);
	  break;
	}
	if(verbose != verbose_quiet)
	  fprintf(stderr, "Loading MAG file `%s'.\n", file);
	bool palflag =
	  ((p.ui[4] == 0) || (p.ui[4] == 2)); // whether to load palette
	bool pixflag =
	  ((p.ui[4] == 0) || (p.ui[4] == 1)); // whether to load pixels
	scr->loadMag(magfile_, p.i[1], p.i[2], p.ui[3], pixflag, palflag);
	delete[] magfile_;
      }
      break;
    case tMIDI:
//    scr->midi(d);
      break;
    case tOFFSET:
      break;
    case tPAL:
      {
	unsigned short rgb[16];
	int j;
	for(j = 0; j < 16; j++)
	  rgb[j] = (unsigned short)p.ui[j+1];
	scr->palette(p.ui[0], rgb);
      }
      break;
    case tPALCHG:
//	    scr->palchg(p.c);
      break;
    case tPALREV:
      scr->palrev(p.ui[0]);
      break;
    case tPATH:
      break;
    case tPLOAD:
      {
	const char* file = p.c;
	char phofile[MAX_PATH+MAX_FILENAME+1];
	strcpy(phofile, file);
//      sprintf(phofile, "%s%s", path, file);
	FILE* phofp;
	if((phofp = dos_fopen(phofile, "r")) == NULL) {
	  if(verbose != verbose_quiet)
	    fprintf(stderr, "PHO file `%s' not found.\n", file);
	  error++;
	  break;
	}
	if(verbose != verbose_quiet)
	  fprintf(stderr, "Loading PHO file `%s'.\n", file);
	scr->loadPho(phofp);
      }
      break;
    case tREM:
      break;
    case tREMARK:
      break;
    case tREST:
      break;
    case tSCREEN:
      break;
    case tSCROLL:
      scr->tscroll(p.ui[0], p.ui[1], p.ui[2], p.ui[3], p.ui[4], p.ui[5],
		   p.ui[6]);
      break;
    case tSTARTUP:
      break;
    case tSTOP:
      break;
    case tTCLS:
      scr->tclear(p.ui[0], p.ui[1], p.ui[2], p.ui[3], p.ui[4], p.ui[5]);
      break;
    case tTON:
      scr->ton(p.ui[0]);
      break;
    case tWAIT:
      break;
    case tWMODE:
      break;
    // ensyutsukun mechanism
    case tFONTM:
      scr->fontmode = p.ui[0];
      break;
    case tFONTP:
      scr->drawFont(p.ui[0], p.ui[1], p.ui[2], p.ui[3]);
      break;
    case tFONTR:
      {
	int j;
	for(j = 0; j < 16; j++)
	  scr->fontdata[p.ui[0]].data[j] = p.ui[j+1];
      }
      break;
    case tGSC:
      break;
    case tLINE:
      break;
    case tPAL_:
      break;
    case tREGSAVE:
      break;
    case tSCROLL_:
      break;
    case tTEXTDOT:
      break;
    case tTMODE:
      break;
    case tTSCRL:
      break;
    case tVCOPY:
      scr->vcopy(p.ui[0], p.ui[1], p.ui[2], p.ui[3], p.ui[4], p.ui[5],
		 p.ui[6], p.ui[7], p.ui[8]);
      break;
    case tVSGET:
      scr->getVS(p.ui[0]);
      break;
    case tVSRES:
      scr->releaseVS();
      break;
    case tXCOPY:
      break;
    }
//    delete data[i].param;
  }
}


static void strdump(const unsigned char *str)
{
  while(*str != 0) {
    fprintf(stderr, "%2x ", (unsigned) *str++ & 0xff);
  }
  fprintf(stderr, "\n");
}


ScreenBase::ScreenBase(const char* xhost)
: XHost(xhost), wmname(NULL)
{
  XSizeHints SizeHints;

  name = "WRD Player";
  if((XDisplay = XOpenDisplay(XHost)) == NULL) {
    if(XHost != NULL)
      fprintf(stderr, "Can\'t open display: %s\n", XHost);
    else
      fprintf(stderr, "Can\'t open display\n");
    exit(EXIT_FAILURE);
  }

  XrmInitialize();
  char* res = XResourceManagerString(XDisplay);
  if(res == NULL)
    rdb = NULL;
  else {
    rdb = XrmGetStringDatabase(res);
    XFree(res);
  }

  rdb2 = XrmGetDatabase(XDisplay);
  int i;
  for(i = 0; i < (int)(sizeof(resources)/sizeof(resources[0])); i++)
    XrmPutStringResource(&rdb2, resources[i].specifier, resources[i].value);

  const char* home_dir;
  char rdbfile[MAX_PATH+MAX_NAME+1];
  if((home_dir = getenv("HOME")) != NULL)
    sprintf(rdbfile, "%s/.Xdefaults", home_dir);
  XrmCombineFileDatabase(rdbfile, &rdb2, True);

  if(rdb != NULL)
    XrmCombineDatabase(rdb2, &rdb, False);
  else
    rdb = rdb2;
  XrmSetDatabase(XDisplay, rdb);

  XScreen = DefaultScreen(XDisplay);
  RootWindow = RootWindow(XDisplay, XScreen);
  DDepth = DefaultDepth(XDisplay, XScreen);
  Black = BlackPixel(XDisplay, XScreen);
  White = WhitePixel(XDisplay, XScreen);

  visualList = (XVisualInfo *)malloc (sizeof (XVisualInfo));
  if (XMatchVisualInfo (XDisplay, XScreen, DDepth, PseudoColor, visualList))
    {
      if(verbose != verbose_quiet)
	fprintf (stderr, "I will use PseudoColor visual.\n");
      vis = visualList->c_class;
      visual = visualList->visual;
      cmap = DefaultColormap (XDisplay, XScreen);
    }
  else if (XMatchVisualInfo (XDisplay, XScreen, DDepth, DirectColor, visualList))
    {
      if(verbose != verbose_quiet)
	fprintf (stderr, "I will use DirectColor visual.\n");
      vis = visualList->c_class;
      visual = visualList->visual;
      cmap = DefaultColormap (XDisplay, XScreen);
    }
  else
    {
      fprintf(stderr, "This program supports DirectColor and PseudoColor "
		      "VisualClass.\n");
      exit(EXIT_FAILURE);
    }

  if (!XAllocColorCells(XDisplay, cmap, FALSE, NULL, 0, pixel, 16+8)) {
    fprintf (stderr,
	     "Warning : XAllocColorCells failed. Try to Private Colormap.\n");
    cmap = XCreateColormap(XDisplay, RootWindow, visual, AllocNone);
    if (!XAllocColorCells(XDisplay, cmap, FALSE, NULL, 0, pixel, 16+8))
      {
	fprintf (stderr, "Private Colormap Allocation Error.\n");
	exit (EXIT_FAILURE);
      }
  }
  WinWidth = GX_SIZE;
  WinHeight = GY_SIZE;
  // Window Attribute Settings.
  XSetWindowAttributes attribute;
  unsigned long valuemask;
//  attribute.backing_store = Always;
  attribute.colormap = cmap;
  valuemask = CWColormap; // | CWBackingStore;
  XWindow = XCreateWindow(XDisplay, RootWindow, 0, 0, WinWidth, WinHeight, 3, DDepth, InputOutput, visual, valuemask, &attribute);
 
/*#define XStoreNamedColor(d,m,c,p,f) \*/
/*{XColor sdr, edr; XAllocNamedColor(d,m,c,&sdr,&edr); p=sdr.pixel;}*/

  // text color palette
  XStoreNamedColor(XDisplay, cmap, getResource("textColor0"),
		   pixel[16+0], DoRed | DoGreen | DoBlue);
  XStoreNamedColor(XDisplay, cmap, getResource("textColor1"),
		   pixel[16+2], DoRed | DoGreen | DoBlue);
  XStoreNamedColor(XDisplay, cmap, getResource("textColor2"),
		   pixel[16+3], DoRed | DoGreen | DoBlue);
  XStoreNamedColor(XDisplay, cmap, getResource("textColor3"),
		   pixel[16+4], DoRed | DoGreen | DoBlue);
  XStoreNamedColor(XDisplay, cmap, getResource("textColor4"),
		   pixel[16+5], DoRed | DoGreen | DoBlue);
  XStoreNamedColor(XDisplay, cmap, getResource("textColor5"),
		   pixel[16+6], DoRed | DoGreen | DoBlue);
  XStoreNamedColor(XDisplay, cmap, getResource("textColor6"),
		   pixel[16+7], DoRed | DoGreen | DoBlue);
  XStoreNamedColor(XDisplay, cmap, getResource("textColor7"),
		   pixel[16+1], DoRed | DoGreen | DoBlue);

  XSelectInput(XDisplay, XWindow, ExposureMask | EnterWindowMask |
	       LeaveWindowMask | KeyReleaseMask);

  SizeHints.flags = 0;
  XSetStandardProperties(XDisplay, XWindow, name, name, None, 
			 (char **)NULL, 0, &SizeHints);

  XClassHint classhints = {RESOURCE_NAME, CLASS_NAME};
  XSetClassHint(XDisplay, XWindow, &classhints);

#if 0
  XWMHints wmhints;
  wmhints.flags = IconPixmapHint;
  int icon_width, icon_height;
  const char* iconPixmap_str = getResource("iconPixmap");
  if(strcmp(iconPixmap_str, "") &&
     readXpm(iconPixmap_str, &wmhints.icon_pixmap, &icon_width, &icon_height))
    XSetWMHints(XDisplay, XWindow, &wmhints);
#endif

  XGCValues xgcv;
  xgcv.graphics_exposures = false;
  XGC = XCreateGC(XDisplay, RootWindow, GCGraphicsExposures, &xgcv);
  XSetBackground(XDisplay, XGC, Black);
  XSetForeground(XDisplay, XGC, Black);
  xfs = XLoadQueryFont(XDisplay, getResource("font"));
  font = xfs->fid;
#ifdef KANJI
  kfs = XLoadQueryFont(XDisplay, getResource("kfont"));
  rfs = XLoadQueryFont(XDisplay, getResource("rfont"));
  kfont = kfs->fid;
  rfont = rfs->fid;
#endif

  XMapWindow(XDisplay, XWindow);
  XFlush(XDisplay);

  // wait for opening window
  XWindowEvent(XDisplay, XWindow, ExposureMask, &event);
}


/*
 * change the window name and icon name
 */
void ScreenBase::changeTitle(const char* filename)
{
  delete[] wmname;
  wmname = new char[strlen(NAME)+1+strlen(filename)+1];
  sprintf((char*)wmname, "%s:%s", NAME, filename);
  XStoreName(XDisplay, XWindow, wmname);
  XSetIconName(XDisplay, XWindow, wmname);
}


/*
 * get resource
 */
const char* ScreenBase::getResource(const char* name)
{
  char rname[80];
  char cname[80];
  sprintf(rname, "%s.%s", RESOURCE_NAME, name);
  sprintf(cname, "%s.%s", CLASS_NAME, name);
  // resource name is just lowercased name of its class name.
  int i;
  for(i = 0; rname[i] != '\0'; i++)
    rname[i] = tolower(rname[i]);
  char* type;
  XrmValue value;
  if(XrmGetResource(rdb, rname, cname, &type, &value))
    return value.addr;
  else
    return "";
}


GraphicScreen::GraphicScreen(const char* xhost)
: ScreenBase(xhost), graph(1), act(0), dsp(0), vsc(NULL), n_vsc(0)
{
  XSetForeground(XDisplay, XGC, Black);
  XFillRectangle(XDisplay, XWindow, XGC, 0, 0, GX_SIZE, GY_SIZE);
  gvram[0] = XCreatePixmap(XDisplay, XWindow, WinWidth, WinHeight, DDepth);
  gvram[1] = XCreatePixmap(XDisplay, XWindow, WinWidth, WinHeight, DDepth);
  XFillRectangle(XDisplay, gvram[0], XGC, 0, 0, GX_SIZE, GY_SIZE);
  XFillRectangle(XDisplay, gvram[1], XGC, 0, 0, GX_SIZE, GY_SIZE);
  int i;
  for(i = 0; i < 16; i++) {	// graphic color palette
    gcolor[i].pixel = pixel[i];
    gcolor[i].flags = DoRed | DoGreen | DoBlue;
    gcolor[i].red =   ((i&0x4)<<15)|((i&0x1)<<14);
    gcolor[i].green = ((i&0x2)<<15)|((i&0x1)<<14);
    gcolor[i].blue =  ((i&0x8)<<15)|((i&0x1)<<14);
    XStoreColor(XDisplay, cmap, &gcolor[i]);
  }
}


#if 0
#define XStoreColor(d,c,s) \
	/*XFreeColors(d,c,&((s)->pixel),1,0),*/\
	XAllocColor(d,c,s)
#endif


GraphicScreen::~GraphicScreen()
{
  XFreePixmap(XDisplay, gvram[0]);
  XFreePixmap(XDisplay, gvram[1]);
}


/*
 * draw a line
 */
void GraphicScreen::drawLine(unsigned x1, unsigned y1,
			     unsigned x2, unsigned y2, unsigned char c)
{
  XSetForeground(XDisplay, XGC, pixel[c]);
  if(graph && act == dsp) {
    XDrawLine(XDisplay, XWindow, XGC, x1, y1, x2, y2);
    ((WrdScreen*)this)->redrawText(x1, y1, x2, y2, false);
  }
  XSetForeground(XDisplay, XGC, pixel[c]);
  XDrawLine(XDisplay, gvram[act], XGC, x1, y1, x2, y2);
  XFlush(XDisplay);
}


/*
 * draw a rectangle
 */
void GraphicScreen::drawRectangle(unsigned x1, unsigned y1,
				  unsigned x2, unsigned y2, unsigned char c)
{
  XSetForeground(XDisplay, XGC, pixel[c]);
  if(graph && act == dsp) {
    XDrawRectangle(XDisplay, XWindow, XGC, x1, y1, x2-x1+1, y2-y1+1);
    ((WrdScreen*)this)->redrawText(x1, y1, x2, y2, false);
  }
  XSetForeground(XDisplay, XGC, pixel[c]);
  XDrawRectangle(XDisplay, gvram[act], XGC, x1, y1, x2-x1+1, y2-y1+1);
  XFlush(XDisplay);
}


/*
 * draw a filled rectangle
 */
void GraphicScreen::fillRectangle(unsigned x1, unsigned y1,
				  unsigned x2, unsigned y2, unsigned char c)
{
  XSetForeground(XDisplay, XGC, pixel[c]);
  if(graph && act == dsp) {
    XFillRectangle(XDisplay, XWindow, XGC, x1, y1, x2-x1+1, y2-y1+1);
    ((WrdScreen*)this)->redrawText(x1, y1, x2, y2, false);
  }
  XSetForeground(XDisplay, XGC, pixel[c]);
  XFillRectangle(XDisplay, gvram[act], XGC, x1, y1, x2-x1+1, y2-y1+1);
  XFlush(XDisplay);
}


/*
 * draw a circle
 */
void GraphicScreen::drawCircle(unsigned x, unsigned y, unsigned r,
			       unsigned char c)
{
  XSetForeground(XDisplay, XGC, pixel[c]);
  if(graph && act == dsp) {
    XDrawArc(XDisplay, XWindow, XGC, (int)x-(int)r, (int)y-(int)r,
	     2*r, 2*r, 0, 360*64);
    ((WrdScreen*)this)->redrawText(x-r, y-r, x+r, y+r, false);
  }
  XSetForeground(XDisplay, XGC, pixel[c]);
  XDrawArc(XDisplay, gvram[act], XGC, (int)x-(int)r, (int)y-(int)r,
	   2*r, 2*r, 0, 360*64);
  XFlush(XDisplay);
}


/*
 * draw a filled circle
 */
void GraphicScreen::fillCircle(unsigned x, unsigned y, unsigned r,
			       unsigned char c)
{
  XSetForeground(XDisplay, XGC, pixel[c]);
  if(graph && act == dsp) {
    XFillArc(XDisplay, XWindow, XGC, (int)x-(int)r, (int)y-(int)r,
	     2*r, 2*r, 0, 360*64);
    ((WrdScreen*)this)->redrawText(x-r, y-r, x+r, y+r, false);
  }
  XSetForeground(XDisplay, XGC, pixel[c]);
  XFillArc(XDisplay, gvram[act], XGC, (int)x-(int)r, (int)y-(int)r,
	   2*r, 2*r, 0, 360*64);
  XFlush(XDisplay);
}


/*
 * copy or exchange regions
 */
void GraphicScreen::move(unsigned x1, unsigned y1, unsigned x2, unsigned y2,
			 unsigned xd, unsigned yd, unsigned vs, unsigned vd,
			 unsigned char sw)
{
  if(x1 & 0x0007)
    x1 = (x1 & 0xfff8) + 8;	// x $B:BI8$r(B 8dots $BC10L$G4]$a$k(B???
  if(x2 & 0x0007)
    x2 = (x2 & 0xfff8) + 8;
  if(xd & 0x0007)
    xd = (xd & 0xfff8) + 8;
//  x1 &= 0xfff8;
//  x2 &= 0xfff8;
//  xd &= 0xfff8;
  unsigned width  = abs((int)x2-(int)x1)+1;
  unsigned height = abs((int)y2-(int)y1)+1;
  switch(sw) {
  case 0: // copy
    XCopyArea(XDisplay, gvram[vs], gvram[vd], XGC, 
	      x1, y1, width, height, xd, yd);
    if(graph && vd == dsp) {
      XCopyArea(XDisplay, gvram[vd], XWindow, XGC, 
		xd, yd, width, height, xd, yd);
      ((WrdScreen*)this)->redrawText(xd, yd, xd+width-1, yd+height-1, false);
    }
    break;
  case 1: // exchange
    {
      Pixmap temp =
	XCreatePixmap(XDisplay, XWindow, width, height, DDepth);
      XCopyArea(XDisplay, gvram[vd], temp, XGC, 
		xd, yd, width, height, 0, 0);
      XCopyArea(XDisplay, gvram[vs], gvram[vd], XGC, 
		x1, y1, width, height, xd, yd);
      XCopyArea(XDisplay, temp, gvram[vs], XGC, 
		0, 0, width, height, x1, y1);
      if(graph && vs == dsp)
	XCopyArea(XDisplay, gvram[vs], XWindow, XGC, 
		  x1, y1, width, height, x1, y1);
      if(graph && vd == dsp) {
	XCopyArea(XDisplay, gvram[vd], XWindow, XGC, 
		  xd, yd, width, height, xd, yd);
	((WrdScreen*)this)->redrawText(xd, yd, xd+width-1, yd+height-1, false);
      }
      if(graph && vs == dsp)
	((WrdScreen*)this)->redrawText(x1, y1, x2, y2, false);
      XFreePixmap(XDisplay, temp);
    }
    break;
  }
  XFlush(XDisplay);
}


/*
 * clear graphic screen
 */
void GraphicScreen::gclear(unsigned char sw)
{
  XSetForeground(XDisplay, XGC, Black);
  if(graph && act == dsp) {
    XFillRectangle(XDisplay, XWindow, XGC, 0, 0, GX_SIZE, GY_SIZE);
    ((WrdScreen*)this)->redrawText(0, 0, GX_SIZE-1, GY_SIZE-1, false);
  }
  XSetForeground(XDisplay, XGC, Black);
  XFillRectangle(XDisplay, gvram[act], XGC, 0, 0, GX_SIZE, GY_SIZE);
  XFlush(XDisplay);
}


/*
 * change both active screen and displayed screen
 */
void GraphicScreen::screen(unsigned char new_act, unsigned char new_dsp)
{
  act = new_act;
  if(graph && dsp != new_dsp) {
    dsp = new_dsp;
    XCopyArea(XDisplay, gvram[new_dsp], XWindow, XGC, 
	      0, 0, GX_SIZE, GY_SIZE, 0, 0);
    ((WrdScreen*)this)->redrawText(0, 0, GX_SIZE-1, GY_SIZE-1, false);
    XFlush(XDisplay);
  }
}


/*
 * switch on/off graphic screen
 */
void GraphicScreen::gon(unsigned char sw)
{
  graph = sw;
  if(graph) {
    XCopyArea(XDisplay, gvram[dsp], XWindow, XGC, 0, 0,
	      GX_SIZE, GY_SIZE, 0, 0);
    ((WrdScreen*)this)->redrawText(0, 0, GX_SIZE-1, GY_SIZE-1, false);
  } else {
    XSetForeground(XDisplay, XGC, Black);
    XFillRectangle(XDisplay, XWindow, XGC, 0, 0, GX_SIZE, GY_SIZE);
    ((WrdScreen*)this)->redrawText(0, 0, GX_SIZE-1, GY_SIZE-1, false);
  }
  XFlush(XDisplay);
}


/*
 * change palette
 */
void GraphicScreen::palette(unsigned char no, unsigned short* p)
{
  int i;
  for(i = 0; i < 16; i++) {
    unsigned short c;
    c = palet[no*16+i] = p[i];
    if(no == 0) {
      gcolor[i].red   = ((c&0x0f00)<< 4) | ((c&0x0f00) ? 0x0fff : 0);
      gcolor[i].green = ((c&0x00f0)<< 8) | ((c&0x00f0) ? 0x0fff : 0);
      gcolor[i].blue  = ((c&0x000f)<<12) | ((c&0x000f) ? 0x0fff : 0);
      // $B2<0L%S%C%H$NId9g3HD%(B
//      XStoreColor(XDisplay, cmap, &gcolor[i]);
//      pixel[i] = gcolor[i].pixel;
    }
  }
  if(no == 0)
    XStoreColors(XDisplay, cmap, gcolor, 16);
}


/*
 * reverse palette's colors
 */
void GraphicScreen::palrev(unsigned char no)
{
  int i;
  for(i = 0; i < 16; i++) {
    palet[no*16+i] = ~palet[no*16+i] & 0x0fff;
    if(no == 0) {
      gcolor[i].red   = ~gcolor[i].red;
      gcolor[i].green = ~gcolor[i].green;
      gcolor[i].blue  = ~gcolor[i].blue;
//      XStoreColor(XDisplay, cmap, &gcolor[i]);
//      pixel[i] = gcolor[i].pixel;
    }
  }
  if(no == 0)
    XStoreColors(XDisplay, cmap, gcolor, 16);
}


/*
 * do fade effect
 */
void GraphicScreen::fade(unsigned char p1, unsigned char p2, int speed)
{
  if(gradate.count) {
//    ((WrdScreen*)this)->gradatePalette(gradate.maxdif);
    palette(0, &palet[gradate.p2*16]);
    gradate.count = 0;
    signal(SIGALRM, SIG_IGN);
#ifdef DEBUG_FADE
    fprintf(stderr, "reset preceding timer\n");
#endif
  }
  switch(speed) {
  default:
    if(debugMode != nonstopMode)
      palette(0, &palet[p1*16]);
    else
      palette(0, &palet[p2*16]);
    break;
  case 0: // immediate
    palette(0, &palet[p2*16]);
    break;
  case -1: // VSYNC ??? (wait)
    if(debugMode != nonstopMode)
      palette(0, &palet[p1*16]);
    else
      palette(0, &palet[p2*16]);
    break;
  }
  if(debugMode != nonstopMode && speed != 0) {
    gradate.maxdif = 0;
    gradate.p1 = p1;
    gradate.p2 = p2;
    int i;
    for(i = 0; i < 16; i++) {
      unsigned short c1, c2, r1, g1, b1, r2, g2, b2;
      c1 = palet[p1*16+i];
      c2 = palet[p2*16+i];
      r1 = (c1&0x0f00) >> 8;
      g1 = (c1&0x00f0) >> 4;
      b1 = (c1&0x000f);
      r2 = (c2&0x0f00) >> 8;
      g2 = (c2&0x00f0) >> 4;
      b2 = (c2&0x000f);
      gradate.paldif[i*3+0] = r2-r1;
      gradate.paldif[i*3+1] = g2-g1;
      gradate.paldif[i*3+2] = b2-b1;
      if((unsigned)abs(r2-r1) > gradate.maxdif)
	gradate.maxdif = abs(r2-r1);
      if((unsigned)abs(g2-g1) > gradate.maxdif)
	gradate.maxdif = abs(g2-g1);
      if((unsigned)abs(b2-b1) > gradate.maxdif)
	gradate.maxdif = abs(b2-b1);
    }
    if(gradate.maxdif > 0) {
      gradate.count = 0;

      struct sigaction sigact;
#if defined(linux)
      sigact.sa_handler = gradatePalette;
      sigact.sa_flags = SA_NOMASK;
      sigact.sa_restorer = NULL;
#elif defined(__FreeBSD__)
      sigact.sa_handler = (void(*)(int))gradatePalette;
      sigact.sa_flags = 0;	// ???
#elif defined(SVR4)
      sigact.sa_handler = gradatePalette;
      sigact.sa_flags = 0;	// ???
#endif
      sigemptyset(&sigact.sa_mask);
      sigaddset(&sigact.sa_mask, SIGALRM);
      sigaction(SIGALRM, &sigact, NULL);

      // start timer
      struct itimerval itv;
      itv.it_interval.tv_sec = 0;
      itv.it_interval.tv_usec =
	((speed > 0)
	 ? 60*1000*1000/gradate.tempo/48 * speed/gradate.maxdif
	 : 1000*1000/60);
      itv.it_value = itv.it_interval;
      if(setitimer(ITIMER_REAL, &itv, NULL) == -1)
	perror("setitimer");

#ifdef DEBUG_FADE
      fprintf(stderr, "start timer: %dus, maxdif: %d\n",
	      itv.it_interval.tv_usec, gradate.maxdif);
#endif
    }
  }
}


/*
 * SIGALRM signal handler for gradating palette
 */
static void gradatePalette(int arg)
{
  (void) arg;
#ifdef DEBUG_FADE
  fprintf(stderr, "SIGALRM: %d/%d\n", gradate.count, gradate.maxdif);
#endif
  if(++gradate.count >= gradate.maxdif) {
    struct itimerval itv;
    itv.it_interval.tv_sec = 0;
    itv.it_interval.tv_usec = 0;
    itv.it_value = itv.it_interval;
    if(setitimer(ITIMER_REAL, &itv, NULL) == -1)
      perror("setitimer");
    signal(SIGALRM, SIG_IGN);
#ifdef DEBUG_FADE
    fprintf(stderr, "reset timer\n");
#endif
  }
}


/*
 * gradate palette
 */
void WrdScreen::gradatePalette(unsigned step)
{
#ifdef DEBUG_FADE
  fprintf(stderr, "gradate: %d, step: %d\n", gradate.count, step);
#endif
  int i;
  for(i = 0; i < 16; i++) {
    unsigned short c, r, g, b;
    c = palet[gradate.p1*16+i];
    r = (((c&0x0f00)>>8) +
	 (int)((double)gradate.paldif[i*3+0]*(double)step
	       /(double)gradate.maxdif))<<12;
    g = (((c&0x00f0)>>4) +
	 (int)((double)gradate.paldif[i*3+1]*(double)step
	       /(double)gradate.maxdif))<<12;
    b = (((c&0x000f)>>0) +
	 (int)((double)gradate.paldif[i*3+2]*(double)step
	       /(double)gradate.maxdif))<<12;
    gcolor[i].red   = r | ((r&0xf000) ? 0x0fff : 0);
    gcolor[i].green = g | ((g&0xf000) ? 0x0fff : 0);
    gcolor[i].blue  = b | ((b&0xf000) ? 0x0fff : 0);
//    fprintf(stderr, "%04x%04x%04x ",
//	    gcolor[i].red, gcolor[i].green, gcolor[i].blue);
  }
//  fprintf(stderr, "\n");
  XStoreColors(XDisplay, cmap, gcolor, 16);
  XFlush(XDisplay);
}


/*
 * load MAG image
 */
void GraphicScreen::loadMag(const char* filename, int x, int y,
			    unsigned char s, unsigned char pixflag,
			    unsigned char palflag)
{
  MagFile mag(filename);
  if(pixflag || palflag) {
    mag.load();
    if(x == -1)
      x = mag.header.Left;
    if(y == -1)
      y = mag.header.Top;
    x &= 0xfff8;
  }
  if(mag.header.ScreenMode & 0x80 == 256) {
    if(verbose != verbose_quiet)
      fprintf(stderr, "256 color MAG file not supported.\n");
    return;
  }
  int i, j;
  for(i = 0; i < 16; i++) {
    palet[17*16+i] =
      palet[(act ? 19 : 18)*16+i] =
	((mag.Red[i]>>4)<<8) | ((mag.Green[i]>>4)<<4) | (mag.Blue[i]>>4);
	// palet[] = 0x00rrggbb;
  }
  if(palflag) {
    for(i = 0; i < 16; i++) {
      palet[i] = palet[17*16+i];
      gcolor[i].red =
	((mag.Red[i]  <<8) | ((mag.Red[i]   > 0) ? 0x00ff : 0));
      gcolor[i].green =
	((mag.Green[i]<<8) | ((mag.Green[i] > 0) ? 0x00ff : 0));
      gcolor[i].blue =
	((mag.Blue[i] <<8) | ((mag.Blue[i]  > 0) ? 0x00ff : 0));
      // $B2<0L%S%C%H$NId9g3HD%(B
//      XStoreColor(XDisplay, cmap, &gcolor[i]);
//      pixel[i] = gcolor[i].pixel;
    }
    XStoreColors(XDisplay, cmap, gcolor, 16);
  }
  if(pixflag) {
    XImage* image;
    int bitmap_pad;
//  bitmap_pad = (DDepth>16) ? 32 : ((DDepth>8) ? 16 : 8);
    image = XCreateImage(XDisplay, DefaultVisual(XDisplay, XScreen),
			 8, ZPixmap, 0, (char*)mag.Screen,
			 mag.width, mag.height, 8, 0);
//  image = XCreateImage(XDisplay, DefaultVisual(XDisplay, XScreen),
//			 DDepth, ZPixmap, 0, (char*)mag.Screen,
//			 mag.width, mag.height, bitmap_pad, 0);
    if(s == 1) {
      for(j = 0; j < mag.height; j++)
	for(i = 0; i < mag.width; i++)
	  XPutPixel(image, i, j, pixel[XGetPixel(image, i, j)]);
      XPutImage(XDisplay, gvram[act], XGC, image, 0, 0, x, y,
		mag.width, mag.height);
    } else {
      XImage* new_image;
      int m = (DDepth>16) ? 4 : ((DDepth>8) ? 2 : 1);
      bitmap_pad = (DDepth>16) ? 32 : ((DDepth>8) ? 16 : 8);
      char* new_screen =
	new char[(mag.width/s)*(mag.height/s)*m];
      new_image = XCreateImage(XDisplay, DefaultVisual(XDisplay, XScreen),
			       DDepth, ZPixmap, 0, new_screen,
			       mag.width/s, mag.height/s, bitmap_pad, 0);
      for(j = 0; j < mag.height/s; j++)
	for(i = 0; i < mag.width/s; i++)
	  XPutPixel(new_image, i, j, pixel[XGetPixel(image, i*s, j*s)]);
      XPutImage(XDisplay, gvram[act], XGC, new_image, 0, 0, x, y,
		mag.width/s, mag.height/s);
      XDestroyImage(new_image);
    }
    XDestroyImage(image);
    if(graph && act == dsp)
      XCopyArea(XDisplay, gvram[act], XWindow, XGC, 
		x, y, mag.width/s, mag.height/s, x, y);
  }
  XFlush(XDisplay);
}


/*
 * load PHO image
 */
void GraphicScreen::loadPho(FILE* fp)
{
  fseek(fp, 0L, SEEK_END);
  long size;
  if((size = ftell(fp)) != 96000 && size != 128000) {
    if(verbose != verbose_quiet)
      fprintf(stderr, "Invalid PHO file.\n");
    fclose(fp);
    return;
  }
  rewind(fp);
  unsigned char* b = new unsigned char[640*400/8];
  unsigned char* r = new unsigned char[640*400/8];
  unsigned char* g = new unsigned char[640*400/8];
  unsigned char* i =
    (size == 128000) ? new unsigned char[640*400/8] : (unsigned char*)NULL;
  fread(b, sizeof(unsigned char), 640*400/8, fp);
  fread(r, sizeof(unsigned char), 640*400/8, fp);
  fread(g, sizeof(unsigned char), 640*400/8, fp);
  if(i != NULL)
    fread(i, sizeof(unsigned char), 640*400/8, fp);
  fclose(fp);
  int x, y;
  for(y = 0; y < 400; y++) {
    for(x = 0; x < 640; x++) {
      unsigned short c;
      c = ((((b[(y*640+x)/8] & (1<<(7-x%8))) != 0) << 3) |
	   (((r[(y*640+x)/8] & (1<<(7-x%8))) != 0) << 2) |
	   (((g[(y*640+x)/8] & (1<<(7-x%8))) != 0) << 1) |
	   ((i == NULL) ? 0 :
	    ((i[(y*640+x)/8] & (1<<(7-x%8))) != 0)));
      XSetForeground(XDisplay, XGC, pixel[c]);
      XDrawPoint(XDisplay, gvram[act], XGC, x, y);
    }
  }
  delete[] b;
  delete[] r;
  delete[] g;
  delete[] i;
  if(graph && act == dsp)
    XCopyArea(XDisplay, gvram[act], XWindow, XGC,
	      0, 0, GX_SIZE, GY_SIZE, 0, 0);
  XFlush(XDisplay);
}


TextScreen::TextScreen(const char* xhost)
: ScreenBase(xhost), loc_x(1), loc_y(1), color(0), text(1), sx(1), sy(1),
  scolor(0)
{
  unsigned i, j;
  for(j = 0; j < TY_SIZE; j++) {
    for(i = 0; i < TX_SIZE; i++) {
      tram[j][i].chr = ' ';
      tram[j][i].col = color;
      tram[j][i].redraw = false;
      tram[j][i].charset = charset_iso;
    }
  }
  text_palet[0] = pixel[16+1];
  text_palet[1] = pixel[16+1];
  text_palet[2] = pixel[16+1];
  text_palet[3] = pixel[16+1];	// ???
  text_palet[4] = pixel[16+1];
  text_palet[5] = pixel[16+1];
  text_palet[6] = pixel[16+0];	// ???
  text_palet[7] = pixel[16+1];
  text_palet[8] = pixel[16+0];

  text_palet[16] = pixel[16+0];
  text_palet[17] = pixel[16+2];
  text_palet[18] = pixel[16+5];
  text_palet[19] = pixel[16+6];
  text_palet[20] = pixel[16+3];
  text_palet[21] = pixel[16+4];
  text_palet[22] = pixel[16+7];
  text_palet[23] = pixel[16+1];

  text_palet[30] = pixel[16+0];
  text_palet[31] = pixel[16+2];
  text_palet[32] = pixel[16+3];
  text_palet[33] = pixel[16+4];
  text_palet[34] = pixel[16+5];
  text_palet[35] = pixel[16+6];
  text_palet[36] = pixel[16+7];
  text_palet[37] = pixel[16+1];

  text_palet[40] = pixel[16+0];
  text_palet[41] = pixel[16+2];
  text_palet[42] = pixel[16+3];
  text_palet[43] = pixel[16+4];
  text_palet[44] = pixel[16+5];
  text_palet[45] = pixel[16+6];
  text_palet[46] = pixel[16+7];
  text_palet[47] = pixel[16+1];
}


/*
 * switch on/off text screen
 */
void TextScreen::ton(unsigned char sw)
{
  text = sw;
  if(text)
    ((WrdScreen*)this)->redrawText(0, 0, GX_SIZE-1, GY_SIZE-1, false);
}


/*
 * locate the cursor
 */
void TextScreen::locate(unsigned char x, unsigned char y)
{
  loc_x = (x > 80) ? 80 : (x == 0) ? 1 : x;	// ???
  loc_y = (y > 24) ? 24 : (y == 0) ? 1 : y;	// ???
}


WrdScreen::~WrdScreen()
{
  XDestroyWindow(XDisplay, XWindow);
  exit(EXIT_SUCCESS);
}


/*
 * escape sequence handling
 */
void WrdScreen::doEscSeq(const char* word, int* j)
{
  switch(word[(*j)++]) {
  case 'D': // (ESC D) get down the cursor
    down();
    tscroll();
    break;
  case 'E': // (ESC E) carriage return and line feed
    down();
    x(1);
    break;
  case 'M': // (ESC M) get up the cursor
    up();
    tscroll(1, 1, TX_SIZE, TY_SIZE, 1);
    break;
  case 'R': // (ESC R) remove the line and scroll up
    tclear(1, loc_y, TX_SIZE, loc_y);
    tscroll(1, loc_y+1, TX_SIZE, TY_SIZE, 1);
    break;
  case '2': // erase the lines
    tclear(1, loc_y, TX_SIZE, TY_SIZE);
    break;
  case '3': // erase the lines
    tclear(1, 1, TX_SIZE, loc_y);
    break;
  case 'T': // erase the line
    tclear(1, loc_y, TX_SIZE, loc_y);
    break;
  case 'Y': 
  case '*': // clear screen
    tclear();
    break;
  case '=': // (ESC=lc) move the cursor
    locate(word[(*j)++]-0x20, word[(*j)++]-0x20);
    break;
  case '[':
    switch(word[(*j)++]) {
    case '>':
      switch(word[(*j)++]) {
      case '1':
	switch(word[(*j)++]) {
	case 'h': // (ESC[>1h) hide function key
	  break;
	case 'l': // (ESC[>1l) show function key
	  break;
	}
	break;
      case '3':
	switch(word[(*j)++]) {
	case 'h': // (ESC[>3h) set to 20 line mode
	  break;
	case 'l': // (ESC[>3l) set to 25 line mode
	  break;
	case 'n': // (ESC[>3n) set to 31 line mode
	  break;
	}
	break;
      case '5':
	switch(word[(*j)++]) {
	case 'h': // (ESC[>5h) hide cursor
	  break;
	case 'l': // (ESC[>5l) show cursor
	  break;
	}
	break;
      }
      break;
    case 's': // (ESC[s) save both cursor position and attribute
      sx = loc_x;
      sy = loc_y;
      scolor = color;
      break;
    case 'u': // (ESC[u) recover both cursor position and attribute
      loc_x = sx;
      loc_y = sy;
      color = scolor;
      break;
    case '0': case '1': case '2': case '3': case '4':
    case '5': case '6': case '7': case '8': case '9':
      {
	int n;
	for(n = word[*j-1]-'0'; '0' <= word[*j] && word[*j] <= '9' ; (*j)++)
	  n = n*10 + word[*j]-'0';
	switch(word[(*j)++]) {
	case 'A': // 
	  up(n);
	  break;
	case 'B':
	  down(n);
	  break;
	case 'C':
	  right(n);
	  break;
	case 'D':
	  left(n);
	  break;
	case 'J':
	case 'j': // small capital for some WRDs
	  switch(n) {
	  case 0:
	    tclear(1, loc_y, TX_SIZE, TY_SIZE);	// $B%+!<%=%k9T$N=hM}(B???
	    break;
	  case 1:
	    tclear(1, 1, TX_SIZE, loc_y);	// $B%+!<%=%k9T$N=hM}(B???
	    break;
	  case 2: // clear whole screen and move cursor to home position
	    tclear();
	    locate(1, 1);
	    break;
	  }
	  break;
	case 'K':
	  switch(n) {
	  case 0:
	    tclear(loc_x, loc_y, loc_x, loc_y);
	    break;
	  case 1:
	    tclear(1, loc_y, loc_x, loc_y);
	    break;
	  case 2:
	    tclear(1, loc_y, TX_SIZE, loc_y);
	    break;
	  }
	  break;
	case 'L':
	  {
	    int i;
	    for(i = 0; i < n; i++)
	      tscroll(1, loc_y+i, TX_SIZE, TY_SIZE, 0);
	  }
	  break;
	case 'M':
	  {
	    int i;
	    for(i = 0; i < n; i++)
	      tscroll(1, loc_y, TX_SIZE, TY_SIZE-i, 1);
	  }
	  break;
	case 'm': // 
	  tcolor(n);
	  break;
	case ';':
	  {
	    unsigned m;
	    for(m = 0; '0' <= word[*j] && word[*j] <= '9' ; (*j)++)
	      m = m*10 + word[*j]-'0';
	    switch(word[(*j)++]) {
	    case 'f':
	    case 'H':
	      locate(m, n);
	      break;
	    case 'm':
	      tcolor(n);
	      break;
	    }
	  }
	  break;
	}
      }
      break;
    }
    break;
  case ')':
    switch(word[(*j)++]) {
    case '3': // (ESC)3) set to graphic character mode
      break;
    case '0': // (ESC)0) set to kanji mode
      break;
    }
    break;
  }
}


/*
 * print
 */
void WrdScreen::print(const char* word)
{
  int i, j;
  for(i = 0; word[i] != '\0'; i++) {
    if(word[i] == '\x1b') {	// escape sequence
      j = i+1;
      doEscSeq(word, &j);
    } else {
      if(isiso(word[i])) {	// ISO characters
	for(j = i; word[j] != '\0' && isiso(word[j]); j++)
	  putchar(word[j]);
      } else {
	if((unsigned char)word[i] < 0x20) {	// control code
	  switch(word[i]) {
	  case '\t':
	    {
	      int k;
	      for(k = 0; k < ~loc_x&0x07; k++)
		putchar(' ');
	    }
	    break;
	  case CR:
	    x(1);
	    break;
	  case LF:
	    down();
	    break;
	  default:
	    break;
	  }
	  j = i+1;
	} else {
	  if(iskana(word[i])) {	// JIS X0201 kana
	    for(j = i; word[j] != '\0' && iskana(word[j]); j += 2) {
#ifdef KANJI
	      putchar2b((unsigned char)word[j], (unsigned char)word[j+1]);
//	      putchar(word[j+1]);
#endif
	    }
	  } else {	// Shift JIS
	    for(j = i; word[j] != '\0' && iskanji(word[j]); j += 2) {
#ifdef KANJI
	      putchar2b((unsigned char)word[j], (unsigned char)word[j+1]);
#endif
	    }
	  }
	}
      }
    }
    i = j-1;
  }
}


/*
 * put a character
 */
void WrdScreen::putchar(char c)
{
  putchar_(loc_x, loc_y, color, c, true);
  right();
}


/*
 * put a character
 */
void WrdScreen::putchar_(unsigned char loc_x, unsigned char loc_y, 
			 unsigned char color, char c, bool restore_bg)
{
  // restore_bg: if true redraw background
  char str[1];
  str[0] = c;
  if(text && restore_bg) {	// restore background
    if(tram[loc_y-1][loc_x-1].chr != ' ' &&
       tram[loc_y-1][loc_x-1].chr != c) {
      if(graph) {
	XCopyArea(XDisplay, gvram[dsp], XWindow, XGC,
		  (loc_x-1)*8, (loc_y-1)*16, 8, 16, (loc_x-1)*8, (loc_y-1)*16);
      } else {
	XSetForeground(XDisplay, XGC, Black);
	XFillRectangle(XDisplay, XWindow, XGC,
		       (loc_x-1)*8, (loc_y-1)*16, 8, 16);
      }
    }
  }
  tram[loc_y-1][loc_x-1].chr = c;
  tram[loc_y-1][loc_x-1].col = color;
  tram[loc_y-1][loc_x-1].redraw = false;
  tram[loc_y-1][loc_x-1].charset = charset_iso;
  if(text) {
    if(color == 7 || 40 <= color && color <= 47) {	// reverse
      Pixmap p = XCreatePixmap(XDisplay, XWindow, 8, 16, 1);
      GC gc = XCreateGC(XDisplay, p, 0, NULL);
      XSetForeground(XDisplay, gc, 1);
      XFillRectangle(XDisplay, p, gc, 0, 0, 8, 16);
      XSetForeground(XDisplay, gc, 0);
      if(isiso(c)) {
	XSetFont(XDisplay, gc, font);
      }/* else {
	if(iskana(c)) {
	  XSetFont(XDisplay, gc, rfont);
	} else {
	}
      }*/
      XDrawString(XDisplay, p, gc, 0, xfs->ascent, (char*) str, 1);
      XSetClipOrigin(XDisplay, XGC, (loc_x-1)*8, (loc_y-1)*16);
      XSetClipMask(XDisplay, XGC, p);
      XSetForeground(XDisplay, XGC, text_palet[color]);
      XFillRectangle(XDisplay, XWindow, XGC, (loc_x-1)*8, (loc_y-1)*16, 8, 16);
      XSetClipMask(XDisplay, XGC, None);
      XFreePixmap(XDisplay, p);
      XFreeGC(XDisplay, gc);
    } else {
      if(isiso(c)) {
	XSetFont(XDisplay, XGC, font);
      }/* else {
	if(iskana(c)) {
	  XSetFont(XDisplay, XGC, rfont);
	} else {
	}
      }*/
      XSetForeground(XDisplay, XGC, text_palet[color]);
      XDrawString(XDisplay, XWindow, XGC,
		  (loc_x-1)*8, (loc_y-1)*16+xfs->ascent, str, 1);
    }
    XFlush(XDisplay);
  }
}


#ifdef KANJI
/*
 * put a kanji character
 */
void WrdScreen::putchar2b(unsigned char c1, unsigned char c2)
{
  putchar2b_(loc_x, loc_y, color, c1, c2, true);
  right((!iskana(c1)) ? 2 : 1);
}


/*
 * put a kanji character
 */
void WrdScreen::putchar2b_(unsigned char loc_x, unsigned char loc_y,
			   unsigned char color,
			   unsigned char c1, unsigned char c2, bool restore_bg)
{
  char str[1];
  char c = str[0] = (!iskana(c1)) ? c1 : c2;
  if(text && restore_bg) {
    if(!iskana(c1)) {
      if(tram[loc_y-1][loc_x-1].chr != c1 ||
	 tram[loc_y-1][loc_x-1+1].chr != c2) {
	if(graph) {
	  XCopyArea(XDisplay, gvram[dsp], XWindow, XGC,
		    (loc_x-1)*8, (loc_y-1)*16, 16, 16,
		    (loc_x-1)*8, (loc_y-1)*16);
	} else {
	  XSetForeground(XDisplay, XGC, Black);
	  XFillRectangle(XDisplay, XWindow, XGC,
			 (loc_x-1)*8, (loc_y-1)*16, 16, 16);
	}
      }
    } else {
      if(tram[loc_y-1][loc_x-1].chr != ' ' &&
	 tram[loc_y-1][loc_x-1].chr != c) {
	if(graph) {
	  XCopyArea(XDisplay, gvram[dsp], XWindow, XGC,(loc_x-1)*8,
		    (loc_y-1)*16, 8, 16, (loc_x-1)*8, (loc_y-1)*16);
	} else {
	  XSetForeground(XDisplay, XGC, Black);
	  XFillRectangle(XDisplay, XWindow, XGC,
			 (loc_x-1)*8, (loc_y-1)*16, 8, 16);
	}
      }
    }
  }
  if(!iskana(c1)) {
    tram[loc_y-1][loc_x-1].chr   = c1;
    tram[loc_y-1][loc_x-1+1].chr = c2;
    tram[loc_y-1][loc_x-1].col   = color;
    tram[loc_y-1][loc_x-1+1].col = color;
    tram[loc_y-1][loc_x-1].redraw   = false;
    tram[loc_y-1][loc_x-1+1].redraw = false;
    tram[loc_y-1][loc_x-1].charset   = charset_kanji1;
    tram[loc_y-1][loc_x-1+1].charset = charset_kanji2;
  } else {
    tram[loc_y-1][loc_x-1].chr = c;
    tram[loc_y-1][loc_x-1].col = color;
    tram[loc_y-1][loc_x-1].redraw = false;
    tram[loc_y-1][loc_x-1].charset =
      (!iskana(c1)) ? charset_iso : charset_kana;
  }
  if(text) {
    if(color == 7 || 40 <= color && color <= 47) {	// reverse
      if(!iskana(c1)) {
	XChar2b c;
	c.byte1 = c1 & 0x7f;
	c.byte2 = c2 & 0x7f;
	Pixmap p = XCreatePixmap(XDisplay, XWindow, 16, 16, 1);
	GC gc = XCreateGC(XDisplay, p, 0, NULL);
	XSetForeground(XDisplay, gc, 1);
	XFillRectangle(XDisplay, p, gc, 0, 0, 16, 16);
	XSetForeground(XDisplay, gc, 0);
	XSetFont(XDisplay, gc, kfont);
	XDrawString16(XDisplay, p, gc, 0, kfs->ascent, &c, 1);
	XSetClipOrigin(XDisplay, XGC, (loc_x-1)*8, (loc_y-1)*16);
	XSetClipMask(XDisplay, XGC, p);
	XSetForeground(XDisplay, XGC, text_palet[color]);
	XFillRectangle(XDisplay, XWindow, XGC,
		       (loc_x-1)*8, (loc_y-1)*16, 16, 16);
	XSetClipMask(XDisplay, XGC, None);
	XFreePixmap(XDisplay, p);
	XFreeGC(XDisplay, gc);
      } else {
	Pixmap p = XCreatePixmap(XDisplay, XWindow, 8, 16, 1);
	GC gc = XCreateGC(XDisplay, p, 0, NULL);
	XSetForeground(XDisplay, gc, 1);
	XFillRectangle(XDisplay, p, gc, 0, 0, 8, 16);
	XSetForeground(XDisplay, gc, 0);
	short ascent = 0;
	if(isiso(c1)) {
	  XSetFont(XDisplay, gc, font);
	  ascent = xfs->ascent;
	} else {
	  if(iskana(c1)) {
	    XSetFont(XDisplay, gc, rfont);
	    ascent = rfs->ascent;
	  } else {
	  }
	}
	XDrawString(XDisplay, p, gc, 0, ascent, (char*) str, 1);
	XSetClipOrigin(XDisplay, XGC, (loc_x-1)*8, (loc_y-1)*16);
	XSetClipMask(XDisplay, XGC, p);
	XSetForeground(XDisplay, XGC, text_palet[color]);
	XFillRectangle(XDisplay, XWindow, XGC,
		       (loc_x-1)*8, (loc_y-1)*16, 8, 16);
	XSetClipMask(XDisplay, XGC, None);
	XFreePixmap(XDisplay, p);
	XFreeGC(XDisplay, gc);
      }
    } else {
      if(!iskana(c1)) {
	XChar2b c;
	c.byte1 = c1 & 0x7f;
	c.byte2 = c2 & 0x7f;
	XSetForeground(XDisplay, XGC, text_palet[color]);
	XSetFont(XDisplay, XGC, kfont);
	XDrawString16(XDisplay, XWindow, XGC,
		      (loc_x-1)*8, (loc_y-1)*16+kfs->ascent, &c, 1);
      } else {
	short ascent = 0;
	if(isiso(c1)) {
	  XSetFont(XDisplay, XGC, font);
	  ascent = xfs->ascent;
	} else {
	  if(iskana(c1)) {
	    XSetFont(XDisplay, XGC, rfont);
	    ascent = rfs->ascent;
	  } else {
	  }
	}
	XSetForeground(XDisplay, XGC, text_palet[color]);
	XDrawString(XDisplay, XWindow, XGC,
		    (loc_x-1)*8, (loc_y-1)*16+ascent, str, 1);
      }
    }
    XFlush(XDisplay);
  }
}
#endif /* KANJI */


/*
 * Expose event handler
 */
void WrdScreen::doExpose(void)
{
  bool redraw = false;
  const XExposeEvent& e = event.xexpose;
  while(XCheckWindowEvent(XDisplay, XWindow, ExposureMask, &event)) {
//    fprintf(stderr, "%lu, %lu\n", e.type, e.count);
    if(e.type != NoExpose && graph) {		// GraphicsExpose $B$O(B ???
//      fprintf(stderr, "redraw(%3d,%3d,%3d,%3d)\n",
//	      e.x, e.y, e.x+e.width-1, e.y+e.height-1);
      XCopyArea(XDisplay, gvram[dsp], XWindow, XGC, 
		e.x, e.y, e.width, e.height, e.x, e.y);
      redraw = true;
    }
    if(e.type != NoExpose && text) {
      redrawText(e.x, e.y, e.x+e.width-1, e.y+e.height-1, false);
      redraw = true;
    }
  }
  if(redraw)
    XFlush(XDisplay);
}


/*
 * EnterNotify event handler
 */
void WrdScreen::doEnterWindow(void)
{
  if(XCheckWindowEvent(XDisplay, XWindow, EnterWindowMask, &event)) {
    XInstallColormap(XDisplay, cmap);
    XFlush(XDisplay);
  }
}


/*
 * LeaveNotify event handler
 */
void WrdScreen::doLeaveWindow(void)
{
  if(XCheckWindowEvent(XDisplay, XWindow, LeaveWindowMask, &event)) {
    XUninstallColormap(XDisplay, cmap);
    XFlush(XDisplay);
  }
}


/*
 * KeyRelease event handler
 */
void WrdScreen::doKeyRelease(void)
{
  if(XCheckWindowEvent(XDisplay, XWindow, KeyReleaseMask, &event)) {
    int len;
    char buf[256];
    KeySym keycode;
    len = XLookupString(&event.xkey, buf, sizeof(buf), &keycode, NULL);
    switch(keycode) {
    case 'Q':
    case 'q': // quit
      delete this;
      break;
    case ' ': // step execution
      if(debugMode == stepMode)
	debugStep++;
      break;
    }
  }
}


/*
 * clear text screen
 */
void WrdScreen::tclear(unsigned char x1, unsigned char y1,
		       unsigned char x2, unsigned char y2,
		       unsigned char color, char char_)
{
  color += (color < 8) ? 30 : 40;
  int x, y;
  if(color == 30 && char_ == ' ') {
    for(y = y1-1; y < y2; y++) {
      for(x = x1-1; x < x2; x++) {
	tram[y][x].chr = char_;
	tram[y][x].col = color;
	tram[y][x].redraw = false;
	tram[y][x].charset = charset_iso;
      }
    }
    if(text) {
      if(graph)
	XCopyArea(XDisplay, gvram[dsp], XWindow, XGC,
		  (x1-1)*8, (y1-1)*16, x2*8, y2*16, (x1-1)*8, (y1-1)*16);
      else
	XFillRectangle(XDisplay, XWindow, XGC,
		       (x1-1)*8, (y1-1)*16, x2*8, y2*16);
      XFlush(XDisplay);
    }
  } else {
    for(y = y1; y <= y2; y++)
      for(x = x1; x <= x2; x++)
	putchar_(x, y, color, char_, true);
  }
}


/*
 * scroll text screen
 */
void WrdScreen::tscroll(unsigned char x1, unsigned char y1,
			unsigned char x2, unsigned char y2,
			unsigned char mode, unsigned char color, char char_)
{
  unsigned char cx1 = x1, cy1 = y1, cx2 = x2, cy2 = y2;
  switch(mode) {
  case 0: // up
    cy1 = cy2 = y2;
    break;
  case 1: // down
    cy1 = cy2 = y1;
    break;
  case 2: // right
    cx1 = cx2 = x1;
    break;
  case 3: // left
    cx1 = cx2 = x2;
    break;
  }


  tclear(cx1, cy1, cx2, cy2, color, char_);
}


/*
 * wait the specified time
 */
void WrdFile::wait(unsigned bar, unsigned step)
{
  if(bar > 0)	// Is this OK ???
    bar--;

  if(verbose == verbose_debug)
    fprintf(stderr, "%d, %d\n", bar, step);

  switch(debugMode) {
  case normalMode: // normal mode (synchronized with music)
    {
      while(reg->bar < bar) {
	if(gradate.count) {
	  scr->gradatePalette(gradate.count);
	  if(gradate.count >= gradate.maxdif)
	    gradate.count = 0;
	}
	scr->doExpose();
	scr->doKeyRelease();
	scr->doEnterWindow();
	scr->doLeaveWindow();
	usleep(400);		// 400us
      }
      while(reg->bar == bar && reg->step < step) {
	if(gradate.count) {
	  scr->gradatePalette(gradate.count);
	  if(gradate.count >= gradate.maxdif)
	    gradate.count = 0;
	}
	scr->doExpose();
	scr->doKeyRelease();
	scr->doEnterWindow();
	scr->doLeaveWindow();
	usleep(400);		// 400us
      }
    }
    break;
  case emulationMode: // emulation mode
    {
      static unsigned bar0 = 0, step0 = 0;
      unsigned long sleep_time = (60*1000*1000/tempo/timebase *
				  ((bar-bar0)*4*timebase+step-step0));
      if(verbose == verbose_debug)
	fprintf(stderr, "%u(%u), %u(%u), %lu\n",
		bar, bar0, step, step0, sleep_time);
      if(bar > bar0 || (bar == bar0 && step > step0)) {
	struct timeval tv1, tv;
	gettimeofday(&tv1, NULL);
	tv1.tv_usec += sleep_time;
	if(tv1.tv_usec > 1000*1000) {
	  tv1.tv_sec += tv1.tv_usec / (1000*1000);
	  tv1.tv_usec %= 1000*1000;
	}
	const unsigned long sleep_time_unit = 400;	// 400us
	while(gettimeofday(&tv, NULL),
	      tv.tv_sec < tv1.tv_sec ||
	      (tv.tv_sec == tv1.tv_sec && tv.tv_usec < tv1.tv_usec)) {
	  if(gradate.count) {
	    scr->gradatePalette(gradate.count);
	    if(gradate.count >= gradate.maxdif)
	      gradate.count = 0;
	  }
	  scr->doExpose();
	  scr->doKeyRelease();
	  scr->doEnterWindow();
	  scr->doLeaveWindow();
	  usleep(sleep_time_unit);
	}
      }
      bar0 = bar;
      step0 = step;
    }
    break;
  case stepMode: // step execution mode
    {
      unsigned oldstep;
      oldstep = debugStep;
      do {
	if(gradate.count) {
	  scr->gradatePalette(gradate.count);
	  if(gradate.count >= gradate.maxdif)
	    gradate.count = 0;
	}
	scr->doExpose();
	scr->doKeyRelease();
	scr->doEnterWindow();
	scr->doLeaveWindow();
	usleep(5*1000);		// 5ms
      } while(debugStep == oldstep);
    }
    break;
  case nonstopMode: // nonstop mode
    usleep(10*1000);	// 10ms
    break;
  }
}


/*
 * fopen() with path search ignoring case
 */
FILE* WrdFile::dos_fopen(char* path, char* mode)
{
  FILE* fp = NULL;
  char* newpath = NULL;
  int i;
  for(i = 0; files[i] != NULL; i++) {
    if(!strcasecmp(files[i], path)) {
      newpath = new char[strlen(this->path)+strlen(files[i])+1];
      sprintf(newpath, "%s%s", this->path, files[i]);
      break;
    }
  }
  if(newpath != NULL) {
    fp = fopen(newpath, mode);
    delete[] newpath;
  }
  return fp;
}


/*
 * get filename with path search ignoring case
 */
char* WrdFile::dos_filename(char* path)
{
  char* newpath = NULL;
  int i;
  for(i = 0; files[i] != NULL; i++) {
    if(!strcasecmp(files[i], path)) {
      newpath = new char[strlen(this->path)+strlen(files[i])+1];
      sprintf(newpath, "%s%s", this->path, files[i]);
      break;
    }
  }
  return newpath;
}


/*
 * execute external program
 */
void WrdFile::exec(const char* file_name)
{
  if(verbose != verbose_quiet)
    fprintf(stderr, "Executing `%s'.\n", file_name);

  char* fname = new char[strlen(file_name)+1];
  strcpy(fname, file_name);

  const int max_arg = 10;
  int argc = 0;
  const char* argv[max_arg];
  argv[argc++] = fname;
  int i;
  for(i = argc; i < max_arg; i++)
    argv[i] = NULL;
  for(i = 0; i < (int)strlen(fname); i++) {
    if(fname[i] == ' ') {
      fname[i] = '\0';
      while(fname[++i] == ' ');
      argv[argc++] = &fname[i];
    }
  }

  if(!strcmp(argv[0], "mag") ||
     !strcmp(argv[0], "MAG")) {	// MAG loader
    for(i = 1; i < argc; i++) {
      if(*argv[i] != '-') {
	if(verbose != verbose_quiet)
	  fprintf(stderr, "Loading MAG file `%s'.\n", argv[i]);
	const char* magfile = NULL;
//	if((magfile = dos_fopen(argv[i])) != NULL) {
	  scr->loadMag(magfile, -1, -1, 1, 1, 1);
//	  delete[] magfile;
//	}
      }
    }
  }

  // wrdcmd2.exe
  // kobanmag, magd
  // kmxp *.kdd
  // IPK.EXE
}


/*
 * set to access to graphic planes
 */
void GraphicScreen::gmode(unsigned char sw)
{
  (void) sw;
}


/*
 * wait until key is pressed or play reaches to specified measure
 */
void WrdScreen::inkey(unsigned bar)
{
  (void) bar;
}


/*
 * set to loop count
 */
void WrdScreen::loop(unsigned count)
{
  (void) count;
}


/*
 * draw a character
 */
void WrdScreen::drawFont(int num, int x, int y, int atr)
{
  int color;	// ???
  switch(atr>>4) {
  case 0:
    color = pixel[16+0];
    break;
  case 2:
    color = pixel[16+5];
    break;
  case 4:
    color = pixel[16+2];
    break;
  case 6:
    color = pixel[16+6];
    break;
  case 8:
    color = pixel[16+3];
    break;
  case 10:
    color = pixel[16+7];
    break;
  case 12:
    color = pixel[16+4];
    break;
  case 14:
    color = pixel[16+1];
    break;
  default:
    color = 0;		// ???
    break;
  }
  int t = atr&0x0f;
  XSetForeground(XDisplay, XGC, text_palet[color]);
  int i, j;
  for(j = 0; j < 16; j++) {
    for(i = 0; i < 16; i++) {
      bool dot;
      dot = fontdata[num].data[j] & (1<<(15-i));
      if((dot && (t == 1 || t == 2 || t == 8)) || (!dot && t == 4))
	XDrawPoint(XDisplay, XWindow, XGC, x+i, y+j);
    }
  }
  if(t == 8)	// underline
    XDrawLine(XDisplay, XWindow, XGC, x, y+16, x+16, y+16);
  XFlush(XDisplay);
}


/*
 * allocate virtual screens
 */
void GraphicScreen::getVS(unsigned n)
{
  n_vsc = n;
  vsc = new Pixmap[n_vsc];
  unsigned i;
  for(i = 0; i < n_vsc; i++) {
    vsc[i] =
      XCreatePixmap(XDisplay, XWindow, WinWidth, WinHeight, DDepth);
  }
}


/*
 * release virtual screens
 */
void GraphicScreen::releaseVS(void)
{
  if(n_vsc > 0 && vsc != NULL) {
    unsigned i;
    for(i = 0; i < n_vsc; i++)
      XFreePixmap(XDisplay, vsc[i]);
    delete[] vsc;
    vsc = NULL;
  }
}


/*
 * copy between virtual screen and real screen
 */
void GraphicScreen::vcopy(int sx1, int sy1, int sx2, int sy2, int tx, int ty,
			  int ss, int ts, int mode)
{
  unsigned width  = abs((int)sx2-(int)sx1)+1;
  unsigned height = abs((int)sy2-(int)sy1)+1;
  XCopyArea(XDisplay, (mode ? vsc[ss] : gvram[ss]),
	    (mode ? gvram[ts] : vsc[ts]),
	    XGC, sx1, sy1, width, height, tx, ty);
  if(graph && mode && ts == dsp)
    XCopyArea(XDisplay, gvram[dsp], XWindow, XGC, 
	      tx, ty, width, height, tx, ty);
  XFlush(XDisplay);
}


/*
 * redraw text screen
 */
void WrdScreen::redrawText(int x1, int y1, int x2, int y2, bool restore_bg)
{
  int tx1, ty1, tx2, ty2;
  tx1 = x1/8+1;
  ty1 = y1/16+1;
  tx2 = x2/8+1;
  ty2 = y2/16+1;

//  fprintf(stderr, "redrawText: (%2d,%2d,%2d,%2d) -> (%2d,%2d,%2d,%2d)\n",
//	  x1, y1, x2, y2, tx1, ty1, tx2, ty2);

  int y;
  for(y = ty1; y <= ty2; y++) {
    int x = tx1;
    if(tx1 > 1 && tram[y-1][(x-1)-1].charset == charset_kanji1) {
      x = tx1-1;
#ifdef KANJI
      putchar2b_(x, y, tram[y-1][x-1].col,
		 tram[y-1][x-1].chr, tram[y-1][x+1-1].chr, restore_bg);
#endif
      x += 2;
    }
    for(/**/; x <= tx2; x++) {
      switch(tram[y-1][x-1].charset) {
      case charset_kanji1:
#ifdef KANJI
	putchar2b_(x, y, tram[y-1][x-1].col,
		   tram[y-1][x-1].chr, tram[y-1][x+1-1].chr, restore_bg);
#endif
	x++;
	break;
      case charset_kanji2:
	// ???
	break;
      case charset_iso:
	putchar_(x, y, tram[y-1][x-1].col, tram[y-1][x-1].chr, restore_bg);
	break;
      case charset_kana:
#ifdef KANJI
	putchar2b_(x, y, tram[y-1][x-1].col, SSO, tram[y-1][x-1].chr,
		   restore_bg);
#endif
	break;
      }
    }
  }
}


/*
 * SIGCLD handler
 */
static void sigcld_handler(int sig)
{
  (void) sig;
  if(wait(NULL) == -1)
    perror("wait");
  signal(SIGCHLD, sigcld_handler);
}


