/*
 * server_impl.cc -- client/server interface (implementation)
 *
 * Copyright (C) 1997,1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: server_impl.cc,v 1.2 1998-03-25 08:31:55+09 satoshi Exp $

#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <cctype>
#include <unistd.h>
#include <netdb.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <cerrno>

#include "shmem.h"
#include "server_impl.h"
#include "playmidi.h"
#include "filelist.h"

#define DEBUG

#define SERVICE	"eplaymidi"
#define DEFAULT_PORT 10206	// ???

extern FileList* filelist;
extern bool addFile(const char* args);
extern int gus_dev, ext_dev, sb_dev;
extern int play_gus, play_sb, play_ext;
extern int playing;

extern void load_sysex(int, char *);
extern void seq_control(int dev, int chn, int p1, int p2);

static const int backlog = 80;

static const char* lockfile = "/tmp/eplaymidi.pid";
static void removelock(void);


/*
 * Server
 */
Server_Impl::Server_Impl(const char* argv0)
: Server(argv0)
{
  is_server = true;

  int lock_fd;
  if(argv0 != NULL)
    return;

  if((lock_fd = creat(lockfile, 0644)) < 0) {
    perror("creat(lockfile, 0644)");
    exit(EXIT_FAILURE);
  }
  char buf[16];
  sprintf(buf, "%d\n", getpid());
  write(lock_fd, buf, strlen(buf));
  close(lock_fd);
  atexit(removelock);

  // socket initalization
  int i;
  for(i = 0; i < MAXCLIENT; i++)
    fd[i] = -1;

  sockaddr_in myaddr_in;
  bzero((char*)&myaddr_in, sizeof(sockaddr_in));
  myaddr_in.sin_family = AF_INET;
  myaddr_in.sin_addr.s_addr = INADDR_ANY;
  
  servent* sp;
  if((sp = getservbyname(SERVICE, "tcp")) == NULL) {
#ifdef linux
    myaddr_in.sin_port = htons(DEFAULT_PORT);
#else
    myaddr_in.sin_port = DEFAULT_PORT;
#endif
  } else
    myaddr_in.sin_port = sp->s_port;
#ifdef linux
  port = ntohs(myaddr_in.sin_port);
#else
  port = myaddr_in.sin_port;
#endif

  if((ls = socket(AF_INET, SOCK_STREAM, 0)) == -1)
    perror("socket");
  const int one = 1;
  setsockopt(ls, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(int));
  if(bind(ls, (sockaddr*)&myaddr_in, sizeof(sockaddr_in)) == -1)
    perror("bind");
  if(listen(ls, backlog) == -1)
    perror("listen");

  // shared memory initialization
  key_t key = ftok((char*)lockfile, 1);
  creg = new CRegister();	// ???
}


Server_Impl::~Server_Impl()
{
  if(close(ls) == -1)
    perror("close");
  int i;
  for(i = 0; i < MAXCLIENT && fd[i] != -1; i++)
    if(close(fd[i]) == -1)
      perror("close");
  delete creg;
}


/*
 * remove lock file
 */
void removelock(void)
{
  unlink(lockfile);
}


/*
 * select
 */
void Server_Impl::select(bool wait)
{
  int width;
  fd_set readfds, writefds, exceptfds;
  int i;

  width = getdtablesize();
  FD_ZERO(&readfds);
  FD_ZERO(&writefds);
  FD_ZERO(&exceptfds);
  FD_SET(ls, &readfds);
  for(i = 0; i < MAXCLIENT; i++) {
    if(fd[i] != -1 )
      FD_SET(fd[i], &readfds);
  }

  timeval timeout = { 0, 0 };
  ::select(width, &readfds, &writefds, &exceptfds, wait ? NULL : &timeout);

  if(FD_ISSET(ls, &readfds)) {
    sockaddr_in peeraddr_in;
    int addrlen = sizeof(peeraddr_in);
    bzero((char*)&peeraddr_in, addrlen);
    int new_client;
    if((new_client = accept(ls, (sockaddr*)&peeraddr_in, &addrlen)) != -1) {
      for(i = 0; i < MAXCLIENT && fd[i] != -1; i++);
      if(i != MAXCLIENT) {
	fd[i] = new_client;
#ifdef DEBUG
	fprintf(stderr, "New connection\n");
#endif
      } else {
#ifdef DEBUG
	fprintf(stderr, "Too many clients\n");
#endif
      }
    } else
      perror("accept");
  }
  for(i = 0; i < MAXCLIENT; i++) {
    if(fd[i] != -1) {
      if(FD_ISSET(fd[i], &readfds))
	talk(i);
    }
  }
}


#define BUFSIZE	4096

/*
 * talk to a client
 */
void Server_Impl::talk(int n)
{
  bool isclosed = false;
#ifdef DEBUG
  fprintf(stderr, "client %d(%d): ", n, fd[n]);
#endif

  static char buf[BUFSIZE];
  int len;
  int s;

  s = fd[n];
  bzero(buf, BUFSIZE);
  if((len = read(s, buf, BUFSIZE)) >= 0) {
    buf[len] = '\0';
    char* p;
    if((p = strrchr(buf, '\n')) != NULL)
      *p = '\0';
    if((p = strrchr(buf, '\r')) != NULL)	// ???
      *p = '\0';
#ifdef DEBUG
    fprintf(stderr, "[%s]\n", buf);
#endif
  } else
    perror("read");
  if(len > 0) {
    volatile Register* reg = creg->reg;
    switch(buf[0]) {
    case '+': // new file
      if(addFile(&buf[1])) {
	reg->seqid++;
	sprintf(buf, "%lu\n", reg->seqid);
	write(s, buf, strlen(buf));
#ifdef DEBUG
	fprintf(stderr, "server: <%lu>\n", reg->seqid);
#endif
      } else {
	write(s, "0\n", 2);
#ifdef DEBUG
	fprintf(stderr, "server: <0>\n");
#endif
      }
      break;
    case 'P': // play back
      if(reg->status == STATUS_PAUSE)
	reg->status = STATUS_PLAY;
      if(reg->status == STATUS_IDLE)
	filelist->allundone();
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 's': // stop playing
      if(reg->status == STATUS_PLAY || reg->status == STATUS_PAUSE) {
	playing = 0;
	filelist->alldone();
      }
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 'p': // make a pause
      {
	if(reg->status == STATUS_PLAY) {
	  reg->status = STATUS_PAUSE;
	  int i;
	  for(i = 0; i < 32; i++) {	// All Note Off
	    if(play_ext)
	      seq_control(ext_dev, i, 0x7b, 0x00);
	    if(play_sb)
	      seq_control(sb_dev, i, 0x7b, 0x00);
	    if(play_gus)
	      seq_control(gus_dev, i, 0x7b, 0x00);
	  }
	}
	write(s, "1\n", 2);
#ifdef DEBUG
	fprintf(stderr, "server: <1>\n");
#endif
      }
      break;
    case 'c': // leave a pause
      if(reg->status == STATUS_PAUSE)
	reg->status = STATUS_PLAY;
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 'b': // play the previous song
      if(reg->status == STATUS_PLAY || reg->status == STATUS_PAUSE) {
	playing = 0;
	filelist->prev();
      }
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 'n': // play the next song
      if(reg->status == STATUS_PLAY || reg->status == STATUS_PAUSE)
	playing = 0;
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 'T': // skew tempo
      {
	double x = atof(&buf[1]);
	if(x < 25.0) {
	  write(s, "0\n", 2);
#ifdef DEBUG
	  fprintf(stderr, "server: <0>\n");
#endif
	} else {
	  reg->skew = x;
	  write(s, "1\n", 2);
#ifdef DEBUG
	  fprintf(stderr, "server: <1>\n");
#endif
	}
      }
      break;
    case 'f': // fade out
      if(reg->status == STATUS_PLAY)
	reg->status = STATUS_FADE;
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 'l': // clear the list
      playing = 0;
      filelist->clear();
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 'L': // get the play list
      {
	const char** list = filelist->getList();
	if(list != NULL) {
	  int i;
	  for(i = 0; list[i] != NULL; i++);
	  sprintf(buf, "%d\n", i);
	  errno = 0;
	  if(write(s, buf, strlen(buf)) < 0)
	    perror("write");
#ifdef DEBUG
	  fprintf(stderr, "server: <%d\n", i);
#endif
	  int j;
	  for(j = 0; j < i; j++) {
	    errno = 0;
	    if(write(s, list[j], strlen(list[j])) < 0)
	      perror("write");
	    errno = 0;
	    if(write(s, "\n", 1) < 0)
	      perror("write");
#ifdef DEBUG
	    fprintf(stderr, "%s\n", list[j]);
#endif
	  }
#ifdef DEBUG
	  fprintf(stderr, ">\n");
#endif
	} else {
	  errno = 0;
	  if(write(s, "0\n", 2) < 0)
	    perror("write");
#ifdef DEBUG
	  fprintf(stderr, "server: <0>\n");
#endif
	}
      }
      break;
    case 'q': // quit
      playing = 0;
      creg->initialize();
      delete this;
      exit(EXIT_SUCCESS);
    case 'M': // master volume
      {
	unsigned char x = atoi(&buf[1]);
	if(reg->status == STATUS_PLAY && x < 128) {
	  unsigned char sysex[] =
	    { 0xf0, 0x7f, 0x7f, 0x04, 0x01, 0x00/**/, x };
	  load_sysex(sizeof(sysex), sysex);
	}
	write(s, "1\n", 2);
#ifdef DEBUG
	fprintf(stderr, "server: <1>\n");
#endif
      }
      break;
    case 'I': // instrument type
      switch(buf[1]) {
      case 1:
	reg->sc55 = true;
	break;
      case 2:
	reg->sc88 = true;
	break;
      case 3:
	reg->sc88pro = true;
	break;
      case 4:
	reg->xg = true;
	break;
      default:
	break;
      }
      write(s, "1\n", 2);
#ifdef DEBUG
      fprintf(stderr, "server: <1>\n");
#endif
      break;
    case 'Q':
      isclosed = true;
      break;
    default:
      fprintf(stderr, "invalid control command: `%s'\n", buf);
      break;
    }
  } else
    isclosed = true;

  if(isclosed) {
    printf("Client %d connection closed\n",n);
    if(close(fd[n]) == -1)
      perror("close");
    fd[n] = -1;
  }
}
