/*
 * rcpfile.cc -- Recomposer RCP/R36/G18/G36 file format analyzer
 *
 * Copyright (C) 1995-1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: rcpfile.cc,v 1.11 1998-04-08 17:24:30+09 satoshi Exp $
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#if defined(linux)
#include <sys/soundcard.h>
#elif defined(__FreeBSD__)
#include <machine/soundcard.h>
#endif
#include <iostream.h>
#include "midifile.h"
#include "kconv.h"
#include "shmem.h"

#define MAX_PATH 80	// maximum length of path
#define MAX_FILENAME 20	// maximum length of file name

static const unsigned SYSEX_INTERVAL = (20*1000);
	// interval time between system exclusive messages
static const unsigned SYSEX_INTERVAL_LONG = (40*1000);
	// interval time between system exclusive messages

static const unsigned char EX = 0xf0;	// Exclusive Status

static const unsigned char EOX = 0xf7;	// End Of Exclusive
static const unsigned char RQ1 = 0x11;	// Data Request 1
static const unsigned char DT1 = 0x12;	// Data Set 1

static const unsigned char GT = 0x80;	// GT
static const unsigned char VE = 0x81;	// VEL
static const unsigned char CH = 0x82;	// CH
static const unsigned char CS = 0x83;	// Clear Sum
static const unsigned char SS = 0x84;	// Send Sum

// Manufacturer's ID number
#define ROLAND	0x41
#define YAMAHA	0x43
#define UNM	0x7e	// Universal Non-realtime Message
#define URM	0x7f	// Universal Realtime Message

#if 0
#define Seq_Circuits (0x01) // Sequential Circuits Inc.
#define Big_Briar    (0x02) // Big Briar Inc.          
#define Octave       (0x03) // Octave/Plateau          
#define Moog         (0x04) // Moog Music              
#define Passport     (0x05) // Passport Designs        
#define Lexicon      (0x06) // Lexicon
#define Tempi        (0x20) // Bon Tempi               
#define Siel         (0x21) // S.I.E.L.                
#define KAWAI        (0x41) // ???
#define Roland       (0x42) // ???
#define KORG         (0x42) // ???
#endif

// Device's ID number
//#define 		0x10
#define BROADCAST	0x7f	// Broadcast

// Model's ID number
#define CM64	0x16	// CM-64/CM-32x/MT-32/D110/D10/D20
#define U220	0x2b	// U220/U20
#define GS	0x42	// GS format (SC-55/SC-155/CM-300/CM-500/etc...)
#define LCD	0x45	// SC-55/SC-155 (DISPLAY)
#define XG	0x4c	// XG

static const unsigned MAX_LOOP_LEVEL = 16;  // maximum nesting level of loop???
static const unsigned COMMENT_LENGTH = 22/*20*/;
static const unsigned USER_EXCLUSIVE_LENGTH = 24;
static const unsigned MAX_EXCLUSIVE_LENGTH = 256;	// ???

extern char path[MAX_PATH+1];

extern const char* inst_sc88_pro[128][128];
extern const char* inst_sc88_88[128][128];
extern const char* inst_sc88_55[128][128];

extern void init_inst_name(void);

void rcpread(void);
void preprocess(void);

static void readheader(void);
static void readtrack(void);
static int mgetc(void);
static void gsdfile(const char* file);
static int preprocess_sysex(unsigned char* ex, unsigned char ch,
			    unsigned char gt, unsigned char vel);
static void send_sysex(const unsigned char* ex, int length);
void check_sysex(const unsigned char* ex, int length);

static unsigned n_track;	// number of tracks
static unsigned char user_exclusive[8][USER_EXCLUSIVE_LENGTH];
				// user define exclusive message
static int init_tempo;		// tempo initialize
static bool is_g36;		// G18/G36 format
bool is_gs;			// data for GS
bool is_gm;			// data for GM
bool is_sc88;			// data for SC-88
bool is_sc88pro;		// data for SC-88Pro
bool is_xg;			// data for XG
static char gsd_file[12+1];
static char gsd_file2[12+1];
static bool gsd_exist;
static char cm6_file[12+1];
static bool cm6_exist;
static int play_bias;

static long total_st;
static bool title_flag;

extern volatile Register* reg;


/*
 * RCP file analyzer
 */
void rcpread(void)
{
  is_g36 = false;
  is_gs = false;
  is_gm = false;
  is_sc88 = false;
  is_sc88pro = false;
  is_xg = false;
  gsd_exist = false;
  cm6_exist = false;
  total_st = 0;
  title_flag = true;

  static bool flag = true;
  if(flag) {
    init_inst_name();
    flag = false;
  }
  readheader();
  int i;
  for(i = 0; i < (int)n_track; i++)
    readtrack();

  {
    for(Mf_currtime = 99; (long)Mf_currtime <= total_st; 
	Mf_currtime += 4*reg->timebase/48)
      Mf_seqspecific(0, 0, NULL);	// step
  }

  fprintf(stderr, "Data for %s\n",
	  (is_xg ? "XG" :
	   is_sc88pro ? "SC-88Pro" :
	   is_sc88 ? "SC-88" : "SC-55"));
  reg->xg = is_xg;
  reg->sc88pro = is_sc88pro;
  reg->sc88 = is_sc88;
  reg->sc55 = !(is_sc88pro || is_sc88 || is_xg);

#if 0
  int etime = total_st*60/reg->timebase/init_tempo;
  reg->playtime = etime*100;
  fprintf(stderr, "Estimated play time: %2d\'%2d\"\n", etime/60, etime%60);
#endif
}


/*
 *
 */
void preprocess(void)
{
  extern load_sysex(int, char*);

  if(gsd_exist)
    gsdfile(gsd_file);

  int i;
  for(i = 0; i <= 0x0f; i++) {
    unsigned char ex1[12] = {
      0xf0, ROLAND, 0x10, GS, DT1,
      CS, 
      0x40, GT, 0x00,	// 40 4x 00: TONE MAP NUMBER 
      0x00,		// 00:SELECTED, 01:SC-55 MAP, 02:SC-88 MAP,
			// 03:SC-88Pro MAP
      SS, EOX};
    unsigned char ex2[12] = {
      0xf0, ROLAND, 0x10, GS, DT1,
      CS, 
      0x40, GT, 0x01,	// 40 4x 01: TONE MAP-0 NUMBER 
      VE,		// 01:SC-55 MAP, 02:SC-88 MAP, 03:SC-88Pro MAP
      SS, EOX};

    unsigned char ex[12];
    memcpy(ex, ex1, 12);
    int len = preprocess_sysex(ex, 0, 0x40+i, 0);
    load_sysex(len, ex);

    usleep(SYSEX_INTERVAL);

    memcpy(ex, ex2, 12);
    len = preprocess_sysex(ex, 0, 0x40+i,
			   is_sc88pro ? 0x03 : (is_sc88 ? 0x02 : 0x01));
    load_sysex(len, ex);

    if(i < 0x0e)
      usleep(SYSEX_INTERVAL);
  }
}


/*
 * read header
 */
static void readheader(void)
{
  int i, j;
  char header[32];
  for(i = 0; i < 32; i++)
    header[i] = mgetc();
  const char* header_string = "RCM-PC98V2.0(C)COME ON MUSIC\x0d\x0a\0\0";
  const char* header_strg36 = "COME ON MUSIC RECOMPOSER RCP3.0\0";
  if(strncmp(header, header_string, 32)) {
    if(strncmp(header, header_strg36, 32)) {
      mferror("not RCP file");
      return;
    } else
      is_g36 = true;
  }

  char title[64+1];
  char title_[64*2+1];
  for(i = 0; i < 64; i++)
    title[i] = mgetc();
  title[64] = '\0';
  kconv((unsigned char*)title, (unsigned char*)title_);
  memcpy((void*)reg->title, title_, 64+1);
  fprintf(stderr, "Title: %s\n", title_);

  char memo[360+1];
  char memo_[360*2+1];
  unsigned time_base;
  int meter_num;
  int meter_denom;
  int key;
  int time_base_u;

  if(!is_g36) {	// RCP/R36
    for(i = 0; i < 336; i++)
      memo[i] = mgetc();
    memo[336] = '\0';
    kconv((unsigned char*)memo, (unsigned char*)memo_);
    memcpy((void*)reg->memo, memo_, 336);

    for(i = 0; i < 16; i++)
      if(mgetc() != 0x40)
	mferror("unexpected data");

    if((time_base = mgetc()) != 48 && time_base != 60 && time_base != 80 &&
       time_base != 96 && time_base != 120 && time_base != 160 &&
       time_base != 224 && time_base != 240) {
      fprintf(stderr, "invalid time base: %d.\n", time_base);
      time_base = 48;
    }

    if((init_tempo = mgetc()) < 8 || 250 < init_tempo) {
      fprintf(stderr, "invalid tempo: %d.\n", init_tempo);
      init_tempo = 120;
    }

    meter_num = mgetc();
    meter_denom = mgetc();

    if((key = mgetc()) > 0x1f) {
      fprintf(stderr, "invalid key: %d.\n", key);
      key = 0;
    }
    Mf_keysig(key & 0x03, (key & 0x10) ? 1 : 0);

    if((play_bias = mgetc()) < -36 || 36 < play_bias) {
      fprintf(stderr, "invalid play bias: %d.\n", play_bias);
      play_bias = 0;
    }

    for(i = 0; i < 12; i++)
      cm6_file[i] = tolower(mgetc());
    cm6_file[12] = '\0';
    if(strlen(cm6_file) > 0 && strcmp(cm6_file, "            ")) {
      char *p;
      if((p = strrchr(cm6_file, ' ')) != NULL)
	*p = '\0';
      fprintf(stderr, "CM6 file: %s\n", cm6_file);
    fprintf(stderr, "CM6 file not supported yet.\n");
      cm6_exist = true;
    }

    for(i = 0; i < 4; i++)	// reserved
      if(mgetc() != 0)
	mferror("reserved data is not 0");

    for(i = 0; i < 12; i++)
      gsd_file[i] = tolower(mgetc());
    gsd_file[12] = '\0';
    if(strlen(gsd_file) > 0 && strcmp(gsd_file, "            ")) {
      char *p;
      if((p = strrchr(gsd_file, ' ')) != NULL)
	*p = '\0';
      fprintf(stderr, "GSD file: %s\n", gsd_file);
      gsd_exist = true;
    }

    for(i = 0; i < 4; i++)	// reserved
      if(mgetc() != 0)
	mferror("reserved data is not 0.");

    if((n_track = mgetc()) == 0)	// RCM Ver.2.5 or older
      n_track = 18;	// !!!debug!!!
    if(n_track != 18 && n_track != 36) {
      fprintf(stderr, "invalid number of tracks: %d\n", n_track);
      n_track = 18;	// !!!debug!!!
    }

    if((time_base_u = mgetc()) != 0 &&
       !(time_base_u == 1 && time_base == 224)) {
      mferror("invalid time base.");
      time_base_u = 0;
    }
    time_base += time_base_u << 8;
    reg->timebase = time_base;
    fprintf(stderr, "Time Base: %u\n", time_base);

    for(i = 0; i < 14; i++)	// reserved
      if(mgetc() != 0)
	mferror("reserved data is not 0.");
    for(i = 0; i < 16; i++)
      mgetc();

    // rhythm definition
    for(i = 0; i < 32; i++) {
      for(j = 0; j < 14; j++) {
	mgetc();		// instrument name
      }
      mgetc();		// key number
      mgetc();		// gate time
    }

    // user exclusive data
    for(i = 0; i < 8; i++) {
      for(j = 0; j < 24; j++)
	(void)mgetc();		// memo
      for(j = 0; j < (int)USER_EXCLUSIVE_LENGTH; j++)
	user_exclusive[i][j] = mgetc();	// exclusive data
    }
  } else {	// G18/G36
    for(i = 0; i < 64; i++)	// reserved
      if(mgetc() != 0x20)
	mferror("reserved data is not 0x20");
    for(i = 0; i < 360; i++)
      memo[i] = mgetc();
    memo[360+1] = '\0';
    kconv((unsigned char*)memo, (unsigned char*)memo_);
    memcpy((void*)reg->memo, memo_, 360);

    n_track = mgetc();
    (void)mgetc();
    if(n_track != 18 && n_track != 36) {
      fprintf(stderr, "invalid number of tracks: %d\n", n_track);
      n_track = 18;	// !!!debug!!!
    }

    time_base = mgetc();
    time_base_u = mgetc();
    time_base += time_base_u << 8;
    reg->timebase = time_base;
    fprintf(stderr, "Time Base: %u\n", time_base);

    if((init_tempo = mgetc()) < 8 || 250 < init_tempo) {
      fprintf(stderr, "invalid tempo: %d.\n", init_tempo);
      init_tempo = 120;
    }
    (void)mgetc();

    meter_num = mgetc();
    meter_denom = mgetc();

    if((key = mgetc()) > 0x1f) {
      fprintf(stderr, "invalid key: %d.\n", key);
      key = 0;
    }
    Mf_keysig(key & 0x03, (key & 0x10) ? 1 : 0);

    if((play_bias = mgetc()) < -36 || 36 < play_bias) {
      fprintf(stderr, "invalid play bias: %d.\n", play_bias);
      play_bias = 0;
    }

    for(i = 0; i < 6; i++)	// reserved
      if(mgetc() != 0)
	mferror("reserved data is not 0");
    for(i = 0; i < 16; i++)	// reserved
      if(mgetc() != 0x40)
	mferror("reserved data is not 0x40");
    for(i = 0; i < 112; i++)	// reserved
      if(mgetc() != 0)
	mferror("reserved data is not 0");

    for(i = 0; i < 12; i++)
      gsd_file[i] = tolower(mgetc());
    gsd_file[12] = '\0';
    for(i = 0; i < 4; i++)
      (void)mgetc();
    if(strlen(gsd_file) > 0 && strcmp(gsd_file, "            ")) {
      char *p;
      if((p = strrchr(gsd_file, ' ')) != NULL)
	*p = '\0';
      fprintf(stderr, "GSD file: %s\n", gsd_file);
      gsd_exist = true;
    }

    for(i = 0; i < 12; i++)
      gsd_file2[i] = tolower(mgetc());
    gsd_file2[12] = '\0';
    for(i = 0; i < 4; i++)
      (void)mgetc();
    if(strlen(gsd_file2) > 0 && strcmp(gsd_file2, "            ")) {
      char *p;
      if((p = strrchr(gsd_file2, ' ')) != NULL)
	*p = '\0';
      fprintf(stderr, "GSD file: %s\n", gsd_file2);
      gsd_exist = true;
    }

    for(i = 0; i < 12; i++)
      cm6_file[i] = tolower(mgetc());
    cm6_file[12] = '\0';
    for(i = 0; i < 4; i++)
      (void)mgetc();
    if(strlen(cm6_file) > 0 && strcmp(cm6_file, "            ")) {
      char *p;
      if((p = strrchr(cm6_file, ' ')) != NULL)
	*p = '\0';
      fprintf(stderr, "CM6 file: %s\n", cm6_file);
    fprintf(stderr, "CM6 file not supported yet.\n");
      cm6_exist = true;
    }

    for(i = 0; i < 80; i++)	// reserved
      if(mgetc() != 0)
	mferror("reserved data is not 0");

    // rhythm definition
    for(i = 0; i < 128; i++) {
      for(j = 0; j < 14; j++) {
	mgetc();		// instrument name
      }
      mgetc();		// key number
      mgetc();		// gate time
    }

    // user exclusive data
    for(i = 0; i < 8; i++) {
      for(j = 0; j < 24; j++)
	(void)mgetc();		// memo
      for(j = 0; j < (int)USER_EXCLUSIVE_LENGTH; j++)
	user_exclusive[i][j] = mgetc();	// exclusive data
    }
  }

  Mf_header(1, n_track, time_base);
}


/*
 * read track data
 */
static void readtrack(void)
{
  unsigned long size;		// track data size
  unsigned char* track_data;	// track data
  unsigned char ch;		// current MIDI channel
  int mode;			// track mode
  char track_comment[36];
  unsigned char note;		// note number
  unsigned char st;		// step time
  unsigned char st2;		// step time (2nd byte)
  unsigned short step;		// step time
  unsigned char gt;		// gate time
  unsigned char gt2;		// gate time (2nd byte)
  unsigned short gate;		// gate time
  unsigned char vel;		// velocity
  bool roland_base_init = false;
  unsigned char roland_base_addr0 = 0;
  unsigned char roland_base_addr1 = 0;
  unsigned char roland_device_id = 0x10;
  unsigned char roland_model_id = CM64;
  bool yamaha_base_init = false;
  bool yamaha_dev_init = false;
  unsigned char yamaha_base_addr0 = 0;
  unsigned char yamaha_base_addr1 = 0;
  unsigned char yamaha_device_id = 0x10;	// ???
  unsigned char yamaha_model_id = XG;		// ???
  int cur_tempo;		// current tempo
  char comment[COMMENT_LENGTH+1];
  int cmt_ptr = 0;
  unsigned measure = 1;		// measure no.
  unsigned same_next = 0;	// next address of same measure
  unsigned cnt;			// step counter
  int key_offset;		// key number offset
  int st_offset;		// step time offset
  int track_no;
  unsigned nesting_level = 0;	// nesting level of loop
  unsigned short loop_start_addr[MAX_LOOP_LEVEL];  // start address of loop
  unsigned short loop_count[MAX_LOOP_LEVEL];	// loop counter

  int i;
  for(i = 0; i < (int)MAX_LOOP_LEVEL; i++)
    loop_count[i] = 0;

  struct {	// for tie
    unsigned long time;
    unsigned char ch;
  } noteoff[128];
  for(i = 0; i < 128; i++) {
    noteoff[i].time = 0;
    noteoff[i].ch = ch;
  }

  Mf_currtime = 0;
  Mf_trackstart();

  size = mgetc() + (mgetc() << 8);
  if((track_data = new unsigned char[size]) == NULL) {
    Mf_error("Cannot allocate track data buffer");
    return;
  }

  if(is_g36) {
    (void)mgetc();
    (void)mgetc();
  }

  track_no = mgetc();	// track number
//if(track_no == 0);	// ???

  cur_tempo = init_tempo;

  Mf_tempo((unsigned long)(1000*1000*60/init_tempo));

  (void)mgetc();	// rhythm

  if(31 < (ch = mgetc()) && ch < 0xff)
    fprintf(stderr, "invalid midi channel: %2x\n", ch);
  // ch == 0xff: off

  key_offset = mgetc()-128;	// KEY#+
  if(key_offset == -128) {	// off
    key_offset = 0;
  } else {
    if(key_offset < -64 || 63 < key_offset) {
      fprintf(stderr, "invalid KEY#+: %d\n", key_offset);
      key_offset = 0;
    } else
      key_offset += play_bias;
  }

  st_offset = (int)(signed char) mgetc();	// ST+
  if(st_offset < -99 || 99 < st_offset)
    fprintf(stderr, "invalid ST+: %d\n", st_offset);
  Mf_currtime = 99+st_offset;

  switch(mode = mgetc()) {
  case 0: // play
//    fprintf(stderr, "Mode: play  ");
    break;
  case 1: // mute
//    fprintf(stderr, "Mode: mute  ");
#if 0
    return;
#endif
    break;
  case 4: // rec
//    fprintf(stderr, "Mode: record ");
    break;
  default:
    fprintf(stderr, "invalid mode: %d\n", mode);
  }

  for(i = 0; i < 36; i++)
    track_comment[i] = mgetc();
  Mf_text(0x03, 36, track_comment);	// Sequence/Track Name

  for(i = (is_g36 ? 46 : 44); i < size; i++)
    track_data[i] = mgetc();

  for(cnt = (is_g36 ? 46 : 44); mode != 1 && cnt < (unsigned)size; /**/) {
    if(Mf_currtime > reg->timebase*init_tempo*30) {	// debug (30 min)
      Mf_error("Too large track size, aborting.");
      break;
    }

    if(!is_g36) {
      note = track_data[cnt++];
      step = st = track_data[cnt++];
      gate = gt = track_data[cnt++];
      vel  = track_data[cnt++];
    } else {
      note = track_data[cnt++];
      vel  = track_data[cnt++];
      st   = track_data[cnt++];
      st2  = track_data[cnt++];
      step = st + (st2<<8);
      gt   = track_data[cnt++];
      gt2  = track_data[cnt++];
      gate = gt + (gt2<<8);
    }

    if(note < 0x80) {
      if(vel > 0 && gate > 0) {	// note
	if(vel > 127) {
	  Mf_error("invalid velocity");
	  vel = 127;		// debug!!!
	}
	// tie parse routine by Yutaka Oiwa
	unsigned long currtime = Mf_currtime;

	if(noteoff[note].time > Mf_currtime) { // overlapped
	  noteoff[note].time = 0;
	  // noteoff and noteon suppressed
	} else { // not overlapped
	  if (noteoff[note].time) { // pending noteoff exists
	    // save current time
	    Mf_currtime = noteoff[note].time;
	    Mf_noteoff(noteoff[note].ch, note+key_offset, 0);
	    Mf_currtime = currtime;
	    noteoff[note].time = 0;
	  }
	  if(ch != 0xff)
	    Mf_noteon(ch, note+key_offset, vel);
	}
	if (gate <= st) { 
	  // ST > GT : tone is completely inside step time
	  Mf_currtime += gate;
	  if(ch != 0xff)
	    Mf_noteoff(ch, note + key_offset, 0);
	  Mf_currtime = currtime + step;
	} else {
	  // GT > ST : tone may be overlapped
	  //             => note-off delayed.
	  noteoff[note].time = currtime + gate;
	  noteoff[note].ch = ch;
	  Mf_currtime += step;
	}
      } else {		// rest
	Mf_currtime += step;
      }
      continue;
    }
    // command
    switch(note) {
    case 0x90:	// User Exclusive 1
    case 0x91:	// User Exclusive 2
    case 0x92:	// User Exclusive 3
    case 0x93:	// User Exclusive 4
    case 0x94:	// User Exclusive 5
    case 0x95:	// User Exclusive 6
    case 0x96:	// User Exclusive 7
    case 0x97:	// User Exclusive 8
      {
	unsigned char ex[USER_EXCLUSIVE_LENGTH+2];
	ex[0] = 0xf0;
	memcpy(&ex[1], &user_exclusive[note-0x90][0], 
	       sizeof(unsigned char[USER_EXCLUSIVE_LENGTH]));
	ex[USER_EXCLUSIVE_LENGTH+1] = EOX;
	int len = preprocess_sysex(ex, ch, gt, vel);
	check_sysex(ex, len);
	if(ch != 0xff)
	  Mf_sysex(len, (char*)ex);
	Mf_currtime += step;
      }
      break;
    case 0x98:	// Channel Exclusive
      {
	bool flag = true;
	while(flag && track_data[cnt] == 0xf7) {
	  unsigned char ex[MAX_EXCLUSIVE_LENGTH+2];
	  ex[0] = 0xf0;
	  int i;
	  for(i = 1; (flag && cnt <= size-(is_g36 ? 6 : 4) &&
		      track_data[cnt] == 0xf7 &&
		      i <= MAX_EXCLUSIVE_LENGTH+2-2); /**/) {
	    cnt++;
	    if(!is_g36 && track_data[cnt++] != 0)
	      Mf_error("Step Time of Channel Exclusive 2nd Event is not 0.");
	    if((ex[i++] = track_data[cnt++]) == EOX)
	      flag = false;
	    if((ex[i++] = track_data[cnt++]) == EOX)
	      flag = false;
	    if(is_g36) {
	      if((ex[i++] = track_data[cnt++]) == EOX)
		flag = false;
	      if((ex[i++] = track_data[cnt++]) == EOX)
		flag = false;
	      if((ex[i++] = track_data[cnt++]) == EOX)
		flag = false;
	    }
	    // not `||' but `&&' ???

//	    fprintf(stderr, "%x: %02x %02x %02x %02x\n", cnt,
//		    track_data[cnt+0], track_data[cnt+1],
//		    track_data[cnt+2], track_data[cnt+3]);

//	    if(track_data[cnt+2] == EOX || track_data[cnt+3] == EOX) {
//	      fprintf(stderr, "----------------\n");
//	      break;
//	    }
	  }
	  int len = preprocess_sysex(ex, ch, gt, vel);
	  check_sysex(ex, len);
	  if(ch != 0xff)
	    Mf_sysex(len, (char*)ex);
	}
	Mf_currtime += step;
      }
      break;
    case 0xf7:	// 2nd Event
//      Mf_error("Something's wrong: 2nd Event is appeared.");
      fprintf(stderr, "Something's wrong: 2nd Event is appeared: %d\n",cnt);
      break;
    case 0xe1:	// BnkLPrg

      fprintf(stderr, "BnkLPrg is not supported yet.\n");

      Mf_currtime += step;
      break;
    case 0xe2:	// bank & program change (BnkMPrg)
      if(vel > 127)
	fprintf(stderr, "Invalid bank no. (MSB): %d\n", vel);
      if(gt > 127)
	fprintf(stderr, "Invalid program no.: %d\n", gt);
      if(ch != 0xff) {
	Mf_parameter(ch, 0x00, vel);	// Bank Number MSB
	Mf_program(ch, gt);
      }
      // ???
      if(!is_sc88pro) {
	if(inst_sc88_55[gt][vel] == NULL) {
	  if(inst_sc88_88[gt][vel] == NULL) {
	    if(inst_sc88_pro[gt][vel] == NULL) {
	      fprintf(stderr, "No Instrument: prg:%3d, bnk:%3d\n",
		      gt+1, vel);
	    } else {
	      is_sc88pro = true;
	      fprintf(stderr, "Matches: prg:%3d, bnk:%3d (SC-88Pro)\n",
		      gt+1, vel);
	    }
	  } else {
	    if(!is_sc88) {
	      is_sc88 = true;
	      fprintf(stderr, "Matches: prg:%3d, bnk:%3d (SC-88)\n",
		      gt+1, vel);
	    }
	  }
	}
      }
      Mf_currtime += step;
      break;
    case 0xe5:	// key scan
      switch(gt) {
      case 12: // suspend playing
      case 18: // increase play bias
      case 23: // stop playing
      case 32: // show main screen
      case 33: // show 11th track
      case 34: // show 12th track
      case 35: // show 13th track
      case 36: // show 14th track
      case 37: // show 15th track
      case 38: // show 16th track
      case 39: // show 17th track
      case 40: // show 18th track
      case 48: // show 10th track
      case 49: // show 1st track
      case 50: // show 2nd track
      case 51: // show 3rd track
      case 52: // show 4th track
      case 53: // show 5th track
      case 54: // show 6th track
      case 55: // show 7th track
      case 56: // show 8th track
      case 57: // show 9th track
      case 61: // mute 1st track
	break;
      }
      fprintf(stderr, "key scan is not supported yet.\n");
      Mf_currtime += step;
      break;
    case 0xe6:	// MIDI channel change
      if(gt == 0) {
	fprintf(stderr, "MIDI channel off\n");
	ch = 0xff;
      } else
	ch = gt-1;
      if(ch > 31)
	fprintf(stderr, "invalid MIDI ch: %d\n", ch);
      Mf_currtime += step;
      break;
    case 0xe7:	// tempo change
      if(vel != 0) {
	fprintf(stderr, "tempo graduation is not supported yet.\n");
      }
      if(gt == 0)
	fprintf(stderr, "Invalid tempo change: GT is 0.\n");
      cur_tempo = init_tempo*gt/64;
      if(ch != 0xff) {
	if(cur_tempo == 0)
	  fprintf(stderr, "tempo change sets cur_tempo to 0.\n");
	else
	  Mf_tempo((unsigned long)(1000*1000*60/cur_tempo));
      }
      Mf_currtime += step;
      break;
    case 0xea:	// channel after touch (channel pressure)
//      fprintf(stderr, "after touch: %2x %2x\n", note, gt);
      if(ch != 0xff)
	Mf_chanpressure(ch, gt);
      Mf_currtime += step;
      break;
    case 0xeb:	// control change
      if(ch != 0xff)
	Mf_parameter(ch, gt, vel);

      // PgBnkMs etc. ???

      if(gt == 0x20) {	// Bank Number LSB
	if(vel > 0x03)
	  is_xg = true;
	if(vel == 0x02)	// SC-88 MAP
	  if(!is_sc88pro)
	    is_sc88 = true;
	if(vel == 0x03)	// SC-88Pro MAP
	  is_sc88pro = true;
      }
      if(gt == 0x5e)	// Effect 4 (Delay Send Level)
	if(!is_sc88pro)
	  is_sc88 = true;
      Mf_currtime += step;
      break;
    case 0xec:	// program change
      if(ch != 0xff)
	Mf_program(ch, gt);
      Mf_currtime += step;
      break;
    case 0xed:	// after touch polyphonic (polyphonic key pressure)
//      fprintf(stderr, "after touch poly.: %2x %2x %2x\n", note, gt, vel);
      if(ch != 0xff)
	Mf_pressure(ch, gt, vel);
      Mf_currtime += step;
      break;
    case 0xee:	// pitch bend
      if(ch != 0xff)
	Mf_pitchbend(ch, gt, vel);
      Mf_currtime += step;
      break;
    case 0xdd:	// Roland base
      if(gt > 127 || vel > 127)
	fprintf(stderr, "Invalid Roland base address.\n");
      roland_base_addr0 = gt;
      roland_base_addr1 = vel;
      roland_base_init = true;
      Mf_currtime += step;
      break;
    case 0xdf:	// Roland device
      if(gt > 127 || vel > 127)
	fprintf(stderr, "Invalid Roland device or model ID.\n");
      roland_device_id = gt;
      roland_model_id = vel;
      Mf_currtime += step;
      break;
    case 0xde:	// Roland para
      {
	if(gt > 127 || vel > 127)
	  fprintf(stderr, "Invalid Roland parameter or offset address.\n");
	if(!roland_base_init)
	  fprintf(stderr, "RolPara used before initializing RolBase.\n");
	unsigned char ex[11];
	ex[0] = 0xf0;
	ex[1] = ROLAND;
	ex[2] = roland_device_id;
	ex[3] = roland_model_id;
	ex[4] = DT1;
	unsigned char cs = 0;
	cs += ex[5] = roland_base_addr0;
	cs += ex[6] = roland_base_addr1;
	cs += ex[7] = gt;
	cs += ex[8] = vel;
	ex[9] = 128 - (cs & 0x7f);
	ex[10] = EOX;
	check_sysex(ex, 11);
	if(ch != 0xff)
	  Mf_sysex(11, (char*)ex);
	Mf_currtime += step;
      }
      break;
    case 0xd0:	// Yamaha base
      if(gt > 127 || vel > 127)
	fprintf(stderr, "Invalid Yamaha base address.\n");
      yamaha_base_addr0 = gt;
      yamaha_base_addr1 = vel;
      yamaha_base_init = true;
      Mf_currtime += step;
      break;
    case 0xd2:	// Yamaha device
      if(gt > 127 || vel > 127)
	fprintf(stderr, "Invalid Yamaha device or model ID.\n");
      yamaha_device_id = gt;
      yamaha_model_id = vel;
      yamaha_dev_init = true;
      Mf_currtime += step;
      break;
    case 0xd1:	// Yamaha para
      {
	if(gt > 127 || vel > 127)
	  fprintf(stderr, "Invalid Yamaha parameter or offset address.\n");
	if(!yamaha_base_init)
	  fprintf(stderr, "YamPara used before initializing YamBase.\n");
	if(!yamaha_dev_init)
	  fprintf(stderr, "YamPara used before initializing YamDev#.\n");
	unsigned char ex[9];
	ex[0] = 0xf0;
	ex[1] = YAMAHA;
	ex[2] = yamaha_device_id;
	ex[3] = yamaha_model_id;
	ex[4] = yamaha_base_addr0;
	ex[5] = yamaha_base_addr1;
	ex[6] = gt;
	ex[7] = vel;
	ex[8] = EOX;
	check_sysex(ex, 9);
	if(ch != 0xff)
	  Mf_sysex(9, (char*)ex);
	Mf_currtime += step;
      }
      break;
    case 0xd3:	// XG para ???
      {
	if(gt > 127 || vel > 127)
	  fprintf(stderr, "Invalid XG parameter or offset address.\n");
//      if(!yamaha_base_init)
//	fprintf(stderr, "XG Para used before initializing YamBase.\n");
//      if(!yamaha_dev_init)
//	fprintf(stderr, "XG Para used before initializing YamDev#.\n");
	unsigned char ex[9];
	ex[0] = 0xf0;
	ex[1] = YAMAHA;
	ex[2] = yamaha_device_id;
	ex[3] = yamaha_model_id;
	ex[4] = yamaha_base_addr0;
	ex[5] = yamaha_base_addr1;
	ex[6] = gt;
	ex[7] = vel;
	ex[8] = EOX;
	check_sysex(ex, 9);
	if(ch != 0xff)
	  Mf_sysex(9, (char*)ex);

	fprintf(stderr, "XG Para is not supported yet.\n");

	Mf_currtime += step;
      }
      break;
    case 0xf5:	// key change
      if(ch != 0xff)
	Mf_keysig(st & 0x03, (st & 0x10) ? 1 : 0);
      break;
    case 0xf6:	// comment start
      cmt_ptr = 0;
      if(!is_g36) {
	comment[cmt_ptr++] = gt;
	comment[cmt_ptr++] = vel;
      } else {
	comment[cmt_ptr++] = vel;
	comment[cmt_ptr++] = st;
	comment[cmt_ptr++] = st2;
	comment[cmt_ptr++] = gt;
	comment[cmt_ptr++] = gt2;
      }
      for(/**/; cmt_ptr < (is_g36 ? 22 : 20); cnt += (is_g36 ? 6 : 4)) {
	if(track_data[cnt] != 0xf7)	// 2nd Event
	  break;
	if(!is_g36) {
	  if(track_data[cnt+1] != 0)
	    Mf_error("Comment's step is not 0");
	  comment[cmt_ptr++] = track_data[cnt+2];
	  comment[cmt_ptr++] = track_data[cnt+3];
	} else {
	  comment[cmt_ptr++] = track_data[cnt+1];
	  comment[cmt_ptr++] = track_data[cnt+2];
	  if(cmt_ptr < 22) {
	    comment[cmt_ptr++] = track_data[cnt+3];
	    comment[cmt_ptr++] = track_data[cnt+4];
	    comment[cmt_ptr++] = track_data[cnt+5];
	  } else {
	    if(track_data[cnt+3] != ' ' ||
	       track_data[cnt+4] != ' ' ||
	       track_data[cnt+5] != ' ')
	      fprintf(stderr,
		      "The rest of comment is not filled with 0x20.\n");
	    comment[cmt_ptr++] = track_data[cnt+3];
	    comment[cmt_ptr++] = track_data[cnt+4];
	    comment[cmt_ptr++] = track_data[cnt+5];
	  }
	}
      }
      comment[cmt_ptr] = '\0';
      char comment_[COMMENT_LENGTH*2+1];
      // kanji code conversion
      kconv((unsigned char*)comment, (unsigned char*)comment_);
//    fprintf(stderr, "Comment: [%s]\n", comment_);	// debug
      if(ch != 0xff)
	Mf_text(0x01, cmt_ptr, comment_);	// Text Event
      break;
    case 0xf8:	// loop end
//      fprintf(stderr,"]\n");
      if(nesting_level == 0) {
	Mf_error("Misplaced loop end.");
      } else {
	if(loop_count[nesting_level-1] == 0) {
	  // loop count: 2--256 (0 means 256?)
	  if(step == 0)
	    fprintf(stderr, "Repeat count is 0.\n");
	  if(step == 1)
	    fprintf(stderr, "Repeat count is 1.\n");
	  loop_count[nesting_level-1] = step;	// initialize counter
	}
	if(--loop_count[nesting_level-1] > 0) {
	  cnt = loop_start_addr[nesting_level-1];
//	  fprintf(stderr,"[\n");
	} else
	  nesting_level--;
      }
      break;
    case 0xf9:	// loop start
//      fprintf(stderr, "[\n");
      if(nesting_level > MAX_LOOP_LEVEL) {
	Mf_error("Too deeply nested.");
      } else {
	loop_start_addr[nesting_level] = cnt;
	loop_count[nesting_level++] = 0;
      }
      break;
    case 0xfc:	// same measure ???
      if(same_next != 0) {
	measure++;
//	fprintf(stderr, "same ed: %d %d\n", measure, cnt);
	cnt = same_next;
	same_next = 0;
      } else {
	same_next = cnt;
	cnt -= is_g36 ? 6 : 4;
	do {
	  if(is_g36) {
	    step = ((track_data[cnt+3]<<8) | track_data[cnt+2]+1);
	    // same measure no.
	    gate = ((track_data[cnt+5]<<8) | track_data[cnt+4]);  // offset
	    if(n_track == 36)
	      gate = gate * 6 - 242;
#if 0
	    fprintf(stderr,
		    "%5u: same st: %5d %5d (%02x %02x %02x %02x %02x)\n",
		    cnt, step, gate,
		    track_data[cnt+1], track_data[cnt+2], track_data[cnt+3],
		    track_data[cnt+4], track_data[cnt+5]);
#endif
	    cnt = gate;
	  } else {
	    st  = track_data[cnt+1];
	    gt  = track_data[cnt+2];
	    vel = track_data[cnt+3];
	    cnt = (vel<<8) | (gt & 0xfc);
//	    fprintf(stderr, "same st: %u %u\n", ((gt & 0x03)<<8)+st+1, cnt);
	  }
	  if(cnt < (is_g36 ? 46 : 44) || size <= cnt) {
	    fprintf(stderr, "invalid same measure offset: %u\n", cnt);
	    cnt = same_next;
	    same_next = 0;
	    break;
	  }
	} while(track_data[cnt] == 0xfc);
      }
      break;
    case 0xfd:	// measure end
      measure++;
      if(same_next != 0) {
//	fprintf(stderr, "same ed: %d %d\n", measure, cnt);
	cnt = same_next;
	same_next = 0;
      } else
//	fprintf(stderr, "measure: %d %d\n", measure, cnt);
      if(track_no == 1)
	Mf_seqspecific(1, 0, NULL);	// bar
      break;
    case 0xfe:	// end of track
/*    Mf_eot();
      cnt = size;*/
      break;
    case 0xc0:	// DX7 function
    case 0xc1:	// DX parameter
    case 0xc2:	// DX RERF
    case 0xc3:	// TX function
    case 0xc5:	// FB-01 P parameter
    case 0xc6:	// FB-01 S System
    case 0xc7:	// TX81Z V VCED
    case 0xc8:	// TX81Z A ACED
    case 0xc9:	// TX81Z P PCED
    case 0xca:	// TX81Z S System
    case 0xcb:	// TX81Z E EFFECT
    case 0xcc:	// DX7-2 R REMOTE SW
    case 0xcd:	// DX7-2 A ACED
    case 0xce:	// DX7-2 P PCED
    case 0xcf:	// TX802 P PCED
    case 0xdc:	// MKS-7
      fprintf(stderr, "Unsupported Command: %2x %2x %2x\n", note, gt, vel);
      Mf_currtime += step;
      break;
    default:
      fprintf(stderr, "Unknown Command: %2x %2x %2x\n", note, gt, vel);
      break;
    }
  }
  if(nesting_level > 0)
    Mf_error("Loop not closed");

  // final pending note off
  for(note = 0; note < 128; note++) {
    if(noteoff[note].time > 0) {
      unsigned long currtime = Mf_currtime;
      Mf_currtime = noteoff[note].time;
      Mf_noteoff(noteoff[note].ch, note+key_offset, 0);
      Mf_currtime = currtime;
    }
  }

  delete[] track_data;

  Mf_trackend();

//  fprintf(stderr, "Total ST: %ld\n", Mf_currtime-99);

  if(title_flag) {
    fprintf(stderr, "Track Size   CH  Key  ST+ Mode Total ST\n");
    title_flag = false;
  }
  fprintf(stderr, "%2d  %6lu  ", track_no, size);
  fprintf(stderr, ((ch!=255) ? "%3d" : "off"), ch+1);
  fprintf(stderr, "  %3d  %3d    %1d  %7ld\n",
	  key_offset, st_offset, mode, Mf_currtime-99);

  if(Mf_currtime-99 > total_st)
    total_st = Mf_currtime-99;
}


/*
 * read a character
 */
static int mgetc(void)
{
  int c;
  if((c = Mf_getc()) == EOF) {
    mferror("premature EOF");
    exit(EXIT_FAILURE);
  }
  return c;
}


inline void expand(unsigned char* dst, const unsigned char* src)
{
  dst[0] = /*(src[0] & 0xf0) |*/ ((src[0] & 0xf0) >> 4);
  dst[1] = (src[0] & 0x0f) /*| ((src[0] & 0x0f) << 4)*/;
}

inline void compress(unsigned char* dst, const unsigned char* src)
{
  dst[0] = (((src[0] & 1) << 3) |
	    ((src[1] & 1) << 2) | 
	    ((src[2] & 1) << 1) |
	     (src[3] & 1));
}

inline void compress122(unsigned char* dst, const unsigned char* src)
{
  dst[0] = (((src[0] & 1) << 4) |
	    ((src[1] & 3) << 2) |
	     (src[2] & 3));
}


/*
 * GSD file analyzer
 */
static void gsdfile(const char *file)
{
  const unsigned GSD_LENGTH = 0x0a71;

  char filename[MAX_PATH+MAX_FILENAME+1];
  if(strlen(path) > 0)
    sprintf(filename, "%s%s", path, file);
  else
    strcpy(filename, file);
  FILE* fp;
  if((fp = fopen(filename, "r")) == NULL) {
    char msg[sizeof("Can't open GSD file `'.")+12];
    sprintf(msg, "Can't open GSD file `%s'.", filename);
    Mf_error(msg);
    return;
  }
  const char* header_string = "COME ON MUSIC\0GS CONTROL 1.0\0\0\0\0";
  int i;
  for(i = 0; i < 32; i++) {
    if(fgetc(fp) != header_string[i]) {
      mferror("Not GSD file.");
      fclose(fp);
      return;
    }
  }
  unsigned char data[GSD_LENGTH];
  for(i = 32; i <= GSD_LENGTH; i++) {
    data[i] = fgetc(fp);
  }
  fclose(fp);

  cerr << "Sending exclusive messages..." << endl;

  unsigned char ex[6+3+128+2] = {	// exclusive message
    0xf0, ROLAND, 0x10, GS, DT1,	// ROLAND ???
    CS,
    // data (128 bytes),
    // SS, EOX
  };
  int len;

  // System Parameters
  ex[5] = CS;
  ex[6] = 0x40;
  ex[7] = 0x00;
  ex[8] = 0x00;
  ex[9+0] = data[0x0020+0];
  ex[9+1] = data[0x0020+1];
  ex[9+2] = data[0x0020+2];
  ex[9+3] = data[0x0020+3];
  ex[9+4] = SS;
  ex[9+5] = EOX;
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// MASTER TUNE

  usleep(SYSEX_INTERVAL);

#define sysex_individual(addr_h,addr_m,addr_l,value) \
  { \
      ex[5] = CS; \
      ex[6] = addr_h; \
      ex[7] = addr_m; \
      ex[8] = addr_l; \
      ex[9+0] = value; \
      ex[9+1] = SS; \
      ex[9+2] = EOX; \
  }

  sysex_individual(0x40, 0x00, 0x04, data[0x24]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// MASTER VOLUME
  usleep(SYSEX_INTERVAL);
  reg->mvol = data[0x24];

  sysex_individual(0x40, 0x00, 0x05, data[0x25]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// MASTER KEY-SHIFT
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x00, 0x06, data[0x26]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// MASTER PAN
  usleep(SYSEX_INTERVAL);
  reg->mpan = data[0x26];

  sysex_individual(0x40, 0x01, 0x30, data[0x27]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// REVERB MACRO
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x31, data[0x28]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// REVERB CHARACTER
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x32, data[0x29]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// REVERB PRE-LPF
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x33, data[0x2a]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// REVERB LEVEL
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x34, data[0x2b]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// REVERB TIME
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x35, data[0x2c]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// REVERB DELAY FEEDBACK
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x36, data[0x2d]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// REVERB SEND LEVEL TO CHORUS ???
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x38, data[0x2e]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS MACRO
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x39, data[0x2f]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS PRE-LPF
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x3a, data[0x30]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS LEVEL
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x3b, data[0x31]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS FEEDBACK
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x3c, data[0x32]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS DELAY
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x3d, data[0x33]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS RATE
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x3e, data[0x34]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS DEPTH
  usleep(SYSEX_INTERVAL);

  sysex_individual(0x40, 0x01, 0x3f, data[0x35]);
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// CHORUS SEND LEVEL TO REVERB
  usleep(SYSEX_INTERVAL);

  ex[5] = CS;
  ex[6] = 0x40;
  ex[7] = 0x01;
  ex[8] = 0x10;
  ex[9+0] = data[0xaf+0];
  ex[9+1] = data[0xaf+1];
  ex[9+2] = data[0xaf+2];
  ex[9+3] = data[0xaf+3];
  ex[9+4] = data[0xaf+4];
  ex[9+5] = data[0xaf+5];
  ex[9+6] = data[0xaf+6];
  ex[9+7] = data[0xaf+7];
  ex[9+8] = data[0xaf+8];
  ex[9+9] = data[0xaf+9];
  ex[9+10] = data[0xaf+10];
  ex[9+11] = data[0xaf+11];
  ex[9+12] = data[0xaf+12];
  ex[9+13] = data[0xaf+13];
  ex[9+14] = data[0xaf+14];
  ex[9+15] = data[0xaf+15];
  ex[9+16] = SS;
  ex[9+17] = EOX;
  len = preprocess_sysex(ex, 0, 0, 0);
  send_sysex(ex, len);		// VOICE RESERVE
  usleep(SYSEX_INTERVAL);

  // Part Data
  for(i = 0; i < 16; i++) {
    unsigned addr;

    reg->bnk[i] = data[0x0036+0x7a*i+0x00];	// BANK MSB ???
    reg->prg[i] = data[0x0036+0x7a*i+0x01];
    reg->rhythm[i] = data[0x0036+0x7a*i+0x15];
    reg->vol[i] = data[0x0036+0x7a*i+0x19];	// volume ???
    reg->pan[i] = data[0x0036+0x7a*i+0x1c];
    reg->cho[i] = data[0x0036+0x7a*i+0x21];
    reg->rev[i] = data[0x0036+0x7a*i+0x22];

    // Patch Part -- Patch Block 0-F
    // Packet 1
    ex[5] = CS;
    addr = 0x90 + ((i == 9) ? 0 : (i < 9) ? i+1 : i) * 0xe0;
    ex[6] = 0x48;
    ex[7] = (addr & 0x7f80)>>7;
    ex[8] = addr & 0x7f;
    expand(&ex[9+0], &data[0x0036+0x7a*i+0x00]);	// TONE NUMBER (CC#00)
    expand(&ex[9+2], &data[0x0036+0x7a*i+0x01]);	// TONE NUMBER
    compress(&ex[9+4], &data[0x0036+0x7a*i+0x03]);	// Rx.PITCH BEND
							// Rx.CH PRESSURE(CAf)
							// Rx.PROGRAM CHANGE
							// Rx.CONTROL CHANGE
    compress(&ex[9+5], &data[0x0036+0x7a*i+0x07]);	// Rx.POLY PRESSURE(PAf)
							// Rx.NOTE MESSAGE
							// Rx.RPN
							// Rx.NRPN
    compress(&ex[9+6], &data[0x0036+0x7a*i+0x0b]);	// Rx.MODURATION
							// Rx.VOLUME
							// Rx.PANPOT
							// Rx.EXPRESSION
    compress(&ex[9+7], &data[0x0036+0x7a*i+0x0f]);	// Rx.HOLD1
							// Rx.PORTAMENT
							// Rx.SOSTENUTO
							// Rx.SOFT
    expand(&ex[9+8], &data[0x0036+0x7a*i+0x02]);	// Rx.CHANNEL
    compress122(&ex[9+10], &data[0x0036+0x7a*i+0x13]);	// MONO/POLY MODE
							// ASSIGN MODE
							// USE FOR RHYTHM PART
    expand(&ex[9+12], &data[0x0036+0x7a*i+0x16]);	// PITCH KEY SHIFT
    ex[9+14] = data[0x0036+0x7a*i+0x17];		// PITCH OFFSET FINE
    ex[9+15] = data[0x0036+0x7a*i+0x18];		// PITCH OFFSET FINE
    expand(&ex[9+16], &data[0x0036+0x7a*i+0x19]);	// PART LEVEL
    expand(&ex[9+18], &data[0x0036+0x7a*i+0x1b]);	// VELOCITY SENSE OFFSET
    expand(&ex[9+20], &data[0x0036+0x7a*i+0x1c]);	// PART PANPOT
    expand(&ex[9+22], &data[0x0036+0x7a*i+0x1a]);	// VELOCITY SENSE DEPTH
    expand(&ex[9+24], &data[0x0036+0x7a*i+0x1d]);	// KEYBOARD RANGE LOW
    expand(&ex[9+26], &data[0x0036+0x7a*i+0x1e]);	// KEYBOARD RANGE HIGH
    expand(&ex[9+28], &data[0x0036+0x7a*i+0x21]);	// CHORUS SEND LEVEL
    expand(&ex[9+30], &data[0x0036+0x7a*i+0x22]);	// REVERB SEND LEVEL
    expand(&ex[9+32], &data[0x0036+0x7a*i+0x23]);	// TONE MODIFY1
    expand(&ex[9+34], &data[0x0036+0x7a*i+0x24]);	// TONE MODIFY2
    expand(&ex[9+36], &data[0x0036+0x7a*i+0x25]);	// TONE MODIFY3
    expand(&ex[9+38], &data[0x0036+0x7a*i+0x26]);	// TONE MODIFY4
    expand(&ex[9+40], &data[0x0036+0x7a*i+0x27]);	// TONE MODIFY5
    expand(&ex[9+42], &data[0x0036+0x7a*i+0x28]);	// TONE MODIFY6
    expand(&ex[9+44], &data[0x0036+0x7a*i+0x29]);	// TONE MODIFY7
    expand(&ex[9+46], &data[0x0036+0x7a*i+0x2a]);	// TONE MODIFY8
    ex[9+48] = 0;
    ex[9+49] = 0;
    ex[9+50] = 0;
    ex[9+51] = 0;
    expand(&ex[9+52], &data[0x0036+0x7a*i+0x2b]);	// SCALE TUNING C
    expand(&ex[9+54], &data[0x0036+0x7a*i+0x2c]);	// SCALE TUNING C#
    expand(&ex[9+56], &data[0x0036+0x7a*i+0x2d]);	// SCALE TUNING D
    expand(&ex[9+58], &data[0x0036+0x7a*i+0x2e]);	// SCALE TUNING D#
    expand(&ex[9+60], &data[0x0036+0x7a*i+0x2f]);	// SCALE TUNING E
    expand(&ex[9+62], &data[0x0036+0x7a*i+0x30]);	// SCALE TUNING F
    expand(&ex[9+64], &data[0x0036+0x7a*i+0x31]);	// SCALE TUNING F#
    expand(&ex[9+66], &data[0x0036+0x7a*i+0x32]);	// SCALE TUNING G
    expand(&ex[9+68], &data[0x0036+0x7a*i+0x33]);	// SCALE TUNING G#
    expand(&ex[9+70], &data[0x0036+0x7a*i+0x34]);	// SCALE TUNING A
    expand(&ex[9+72], &data[0x0036+0x7a*i+0x35]);	// SCALE TUNING A#
    expand(&ex[9+74], &data[0x0036+0x7a*i+0x36]);	// SCALE TUNING B
    expand(&ex[9+76], &data[0x0036+0x7a*i+0x1f]);    // CC1 CONTROLLER NUMBER
    expand(&ex[9+78], &data[0x0036+0x7a*i+0x20]);    // CC2 CONTROLLER NUMBER
    expand(&ex[9+80], &data[0x0036+0x7a*i+0x37]);    // MOD PITCH CONTROL
    expand(&ex[9+82], &data[0x0036+0x7a*i+0x38]);    // MOD TVF CUTOFF CONTROL
    expand(&ex[9+84], &data[0x0036+0x7a*i+0x39]);    // MOD AMPLITUDE CONTROL
    ex[9+86] = 0;
    ex[9+87] = 0;
    expand(&ex[9+88], &data[0x0036+0x7a*i+0x3a]);    // MOD LFO1 RATE CONTROL
    expand(&ex[9+90], &data[0x0036+0x7a*i+0x3b]);    // MOD LFO1 PITCH DEPTH
    expand(&ex[9+92], &data[0x0036+0x7a*i+0x3c]);    // MOD LFO1 TVF DEPTH
    expand(&ex[9+94], &data[0x0036+0x7a*i+0x3d]);    // MOD LFO1 TVA DEPTH
    expand(&ex[9+96], &data[0x0036+0x7a*i+0x3e]);    // MOD LFO2 RATE CONTROL
    expand(&ex[9+98], &data[0x0036+0x7a*i+0x3f]);    // MOD LFO2 PITCH DEPTH
    expand(&ex[9+100], &data[0x0036+0x7a*i+0x40]);   // MOD LFO2 TVF DEPTH
    expand(&ex[9+102], &data[0x0036+0x7a*i+0x41]);   // MOD LFO2 TVA DEPTH
    expand(&ex[9+104], &data[0x0036+0x7a*i+0x42]);   // BEND PITCH CONTROL
    expand(&ex[9+106], &data[0x0036+0x7a*i+0x43]);   // BEND TVF CUTOFF CONTROL
    expand(&ex[9+108], &data[0x0036+0x7a*i+0x44]);   // BEND AMPLITUDE CONTROL
    ex[9+110] = 0;
    ex[9+111] = 0;
    expand(&ex[9+112], &data[0x0036+0x7a*i+0x45]);   // BEND LFO1 RATE CONTROL
    expand(&ex[9+114], &data[0x0036+0x7a*i+0x46]);   // BEND LFO1 PITCH DEPTH
    expand(&ex[9+116], &data[0x0036+0x7a*i+0x47]);   // BEND LFO1 TVF DEPTH
    expand(&ex[9+118], &data[0x0036+0x7a*i+0x48]);   // BEND LFO1 TVA DEPTH
    expand(&ex[9+120], &data[0x0036+0x7a*i+0x49]);   // BEND LFO2 RATE CONTROL
    expand(&ex[9+122], &data[0x0036+0x7a*i+0x4a]);   // BEND LFO2 PITCH DEPTH
    expand(&ex[9+124], &data[0x0036+0x7a*i+0x4b]);   // BEND LFO2 TVF DEPTH
    expand(&ex[9+126], &data[0x0036+0x7a*i+0x4c]);   // BEND LFO2 TVA DEPTH
    ex[9+128] = SS;
    ex[9+129] = EOX;
    len = preprocess_sysex(ex, 0, 0, 0);
    send_sysex(ex, len);

    usleep(SYSEX_INTERVAL_LONG);

    // Packet 2
    ex[5] = CS;
    addr += 128;
    ex[6] = 0x48;
    ex[7] = (addr & 0x7f80)>>7;
    ex[8] = addr & 0x7f;
    expand(&ex[9+0], &data[0x0036+0x7a*i+0x4d]);    // CAf PITCH CONTROL
    expand(&ex[9+2], &data[0x0036+0x7a*i+0x4e]);    // CAf TVF CUTOFF CONTROL
    expand(&ex[9+4], &data[0x0036+0x7a*i+0x4f]);    // CAf AMPLITUDE CONTROL
    expand(&ex[9+6], &data[0x0036+0x7a*i+0x50]);    // CAf LFO1 RATE CONTROL
    ex[9+8] = 4;
    ex[9+9] = 0;
    expand(&ex[9+10], &data[0x0036+0x7a*i+0x51]);    // CAf LFO1 PITCH DEPTH
    expand(&ex[9+12], &data[0x0036+0x7a*i+0x52]);    // CAf LFO1 TVF DEPTH
    expand(&ex[9+14], &data[0x0036+0x7a*i+0x53]);    // CAf LFO1 TVA DEPTH
    expand(&ex[9+16], &data[0x0036+0x7a*i+0x54]);    // CAf LFO2 RATE CONTROL
    expand(&ex[9+18], &data[0x0036+0x7a*i+0x55]);    // CAf LFO2 PITCH DEPTH
    expand(&ex[9+20], &data[0x0036+0x7a*i+0x56]);    // CAf LFO2 TVF DEPTH
    expand(&ex[9+22], &data[0x0036+0x7a*i+0x57]);    // CAf LFO2 TVA DEPTH
    expand(&ex[9+24], &data[0x0036+0x7a*i+0x58]);    // PAf PITCH CONTROL
    expand(&ex[9+26], &data[0x0036+0x7a*i+0x59]);    // PAf TVF CUTOFF CONTROL
    expand(&ex[9+28], &data[0x0036+0x7a*i+0x5a]);    // PAf AMPLITUDE CONTROL
    expand(&ex[9+30], &data[0x0036+0x7a*i+0x5b]);    // PAf LFO1 RATE CONTROL
    ex[9+32] = 4;
    ex[9+33] = 0;
    expand(&ex[9+34], &data[0x0036+0x7a*i+0x5c]);    // PAf LFO1 PITCH DEPTH
    expand(&ex[9+36], &data[0x0036+0x7a*i+0x5d]);    // PAf LFO1 TVF DEPTH
    expand(&ex[9+38], &data[0x0036+0x7a*i+0x5e]);    // PAf LFO1 TVA DEPTH
    expand(&ex[9+40], &data[0x0036+0x7a*i+0x5f]);    // PAf LFO2 RATE CONTROL
    expand(&ex[9+42], &data[0x0036+0x7a*i+0x60]);    // PAf LFO2 PITCH DEPTH
    expand(&ex[9+44], &data[0x0036+0x7a*i+0x61]);    // PAf LFO2 TVF DEPTH
    expand(&ex[9+46], &data[0x0036+0x7a*i+0x62]);    // PAf LFO2 TVA DEPTH
    expand(&ex[9+48], &data[0x0036+0x7a*i+0x63]);    // CC1 PITCH CONTROL
    expand(&ex[9+50], &data[0x0036+0x7a*i+0x64]);    // CC1 TVF CUTOFF CONTROL
    expand(&ex[9+52], &data[0x0036+0x7a*i+0x65]);    // CC1 AMPLITUDE CONTROL
    ex[9+54] = 0;
    ex[9+55] = 0;
    expand(&ex[9+56], &data[0x0036+0x7a*i+0x66]);    // CC1 LFO1 RATE CONTROL
    expand(&ex[9+58], &data[0x0036+0x7a*i+0x67]);    // CC1 LFO1 PITCH DEPTH
    expand(&ex[9+60], &data[0x0036+0x7a*i+0x68]);    // CC1 LFO1 TVF DEPTH
    expand(&ex[9+62], &data[0x0036+0x7a*i+0x69]);    // CC1 LFO1 TVA DEPTH
    expand(&ex[9+64], &data[0x0036+0x7a*i+0x6a]);    // CC1 LFO2 RATE CONTROL
    expand(&ex[9+66], &data[0x0036+0x7a*i+0x6b]);    // CC1 LFO2 PITCH DEPTH
    expand(&ex[9+68], &data[0x0036+0x7a*i+0x6c]);    // CC1 LFO2 TVF DEPTH
    expand(&ex[9+70], &data[0x0036+0x7a*i+0x6d]);    // CC1 LFO2 TVA DEPTH
    expand(&ex[9+72], &data[0x0036+0x7a*i+0x6e]);    // CC2 PITCH CONTROL
    expand(&ex[9+74], &data[0x0036+0x7a*i+0x6f]);    // CC2 TVF CUTOFF CONTROL
    expand(&ex[9+76], &data[0x0036+0x7a*i+0x70]);    // CC2 AMPLITUDE CONTROL
    ex[9+78] = 0;
    ex[9+79] = 0;
    expand(&ex[9+80], &data[0x0036+0x7a*i+0x71]);    // CC2 LFO1 RATE CONTROL
    expand(&ex[9+82], &data[0x0036+0x7a*i+0x72]);    // CC2 LFO1 PITCH DEPTH
    expand(&ex[9+84], &data[0x0036+0x7a*i+0x73]);    // CC2 LFO1 TVF DEPTH
    expand(&ex[9+86], &data[0x0036+0x7a*i+0x74]);    // CC2 LFO1 TVA DEPTH
    expand(&ex[9+88], &data[0x0036+0x7a*i+0x75]);    // CC2 LFO2 RATE CONTROL
    expand(&ex[9+90], &data[0x0036+0x7a*i+0x76]);    // CC2 LFO2 PITCH DEPTH
    expand(&ex[9+92], &data[0x0036+0x7a*i+0x77]);    // CC2 LFO2 TVF DEPTH
    expand(&ex[9+94], &data[0x0036+0x7a*i+0x78]);    // CC2 LFO2 TVA DEPTH
    ex[9+96] = SS;
    ex[9+97] = EOX;
    len = preprocess_sysex(ex, 0, 0, 0);
    send_sysex(ex, len);

    if(i < 15)
      usleep(SYSEX_INTERVAL_LONG);
  }
  // Rhythm Map
#if 0
  for(i = 0; i < 2; i++) {
    int j;

    ex[5] = CS;
    ex[6] = 0x49;
    ex[7] = (i == 0) ? 0x02 : 0x12;
    ex[8] = 0x00;
    for(j = 0; j < 28; j++)
      ex[9+j] = 0;
    for(j = 27; j < 64; j++)
      ex[9+j] = data[0x7d6+0x148*i+j-27];
    ex[9+64] = SS;
    ex[9+65] = EOX;
    len = preprocess_sysex(ex, 0, 0, 0);
    send_sysex(ex, len);

    usleep(SYSEX_INTERVAL_LONG);

    for(j = 45; j < 65; j++)
      ex[9+j] = 0;

  }
#endif
}


/*
 * preprocess system exclusive message
 */
static int preprocess_sysex(unsigned char *ex, unsigned char ch,
			    unsigned char gt, unsigned char vel)
{
  unsigned char cs = 0;		// check sum
  int len = 0;
  int i;
  for(i = 0; i < MAX_EXCLUSIVE_LENGTH+2 && ex[i] != EOX; i++) {
    switch(ex[i]) {
    case GT: // gt
      cs += ex[len++] = gt;
      break;
    case VE: // ve
      cs += ex[len++] = vel;
      break;
    case CH: // ch
      cs += ex[len++] = ch;
      break;
    case CS: // cs (Clear Sum)
      cs = 0;
      break;
    case SS: // ss (Send Sum)
      ex[len++] = 128 - (cs & 0x7f);
      break;
    default:
      cs += ex[len++] = ex[i];
      break;
    }
  }
  ex[len++] = EOX;
  return len;
}


/*
 * send system exclusive message
 */
static void send_sysex(const unsigned char *ex, int length)
{
  extern void load_sysex(int length, char* data);
  extern void seqbuf_dump();

  load_sysex(length, ex);
  seqbuf_dump();
}


/*
 * check system exclusive message
 */
void check_sysex(const unsigned char* ex, int length)
{
  static unsigned char addr_sc88[99*3] = {
    0x00, 0x00, 0x7f,
    0x00, 0x01, 0x00,
    0x00, 0x01, 0x01,
    0x00, 0x01, 0x02,
    0x00, 0x01, 0x03,
    0x00, 0x01, 0x04,
    0x00, 0x01, 0x05,
    0x00, 0x01, 0x06,
    0x00, 0x01, 0x07,
    0x00, 0x01, 0x08,
    0x00, 0x01, 0x09,
    0x00, 0x01, 0x0a,
    0x00, 0x01, 0x0b,
    0x00, 0x01, 0x0c,
    0x00, 0x01, 0x0d,
    0x00, 0x01, 0x0e,
    0x00, 0x01, 0x0f,
    0x00, 0x01, 0x10,
    0x00, 0x01, 0x11,
    0x00, 0x01, 0x12,
    0x00, 0x01, 0x13,
    0x00, 0x01, 0x14,
    0x00, 0x01, 0x15,
    0x00, 0x01, 0x16,
    0x00, 0x01, 0x17,
    0x00, 0x01, 0x18,
    0x00, 0x01, 0x19,
    0x00, 0x01, 0x1a,
    0x00, 0x01, 0x1b,
    0x00, 0x01, 0x1c,
    0x00, 0x01, 0x1d,
    0x00, 0x01, 0x1e,
    0x00, 0x01, 0x1f,
    0x40, 0x01, 0x37,
    0x40, 0x01, 0x40,
    0x40, 0x01, 0x50,
    0x40, 0x01, 0x51,
    0x40, 0x01, 0x52,
    0x40, 0x01, 0x53,
    0x40, 0x01, 0x54,
    0x40, 0x01, 0x55,
    0x40, 0x01, 0x56,
    0x40, 0x01, 0x57,
    0x40, 0x01, 0x58,
    0x40, 0x01, 0x59,
    0x40, 0x01, 0x5a,
    0x40, 0x02, 0x00,
    0x40, 0x02, 0x01,
    0x40, 0x02, 0x02,
    0x40, 0x02, 0x03,

    0x40, 0x40, 0x00,
    0x40, 0x41, 0x00,
    0x40, 0x42, 0x00,
    0x40, 0x43, 0x00,
    0x40, 0x44, 0x00,
    0x40, 0x45, 0x00,
    0x40, 0x46, 0x00,
    0x40, 0x47, 0x00,
    0x40, 0x48, 0x00,
    0x40, 0x49, 0x00,
    0x40, 0x4a, 0x00,
    0x40, 0x4b, 0x00,
    0x40, 0x4c, 0x00,
    0x40, 0x4d, 0x00,
    0x40, 0x4e, 0x00,
    0x40, 0x4f, 0x00,
    0x40, 0x40, 0x01,
    0x40, 0x41, 0x01,
    0x40, 0x42, 0x01,
    0x40, 0x43, 0x01,
    0x40, 0x44, 0x01,
    0x40, 0x45, 0x01,
    0x40, 0x46, 0x01,
    0x40, 0x47, 0x01,
    0x40, 0x48, 0x01,
    0x40, 0x49, 0x01,
    0x40, 0x4a, 0x01,
    0x40, 0x4b, 0x01,
    0x40, 0x4c, 0x01,
    0x40, 0x4d, 0x01,
    0x40, 0x4e, 0x01,
    0x40, 0x4f, 0x01,
    0x40, 0x40, 0x20,
    0x40, 0x41, 0x20,
    0x40, 0x42, 0x20,
    0x40, 0x43, 0x20,
    0x40, 0x44, 0x20,
    0x40, 0x45, 0x20,
    0x40, 0x46, 0x20,
    0x40, 0x47, 0x20,
    0x40, 0x48, 0x20,
    0x40, 0x49, 0x20,
    0x40, 0x4a, 0x20,
    0x40, 0x4b, 0x20,
    0x40, 0x4c, 0x20,
    0x40, 0x4d, 0x20,
    0x40, 0x4e, 0x20,
    0x40, 0x4f, 0x20
  };
  static unsigned char addr_sc88pro[61*3] = {
    0x40, 0x03, 0x00,
    0x40, 0x03, 0x03,
    0x40, 0x03, 0x04,
    0x40, 0x03, 0x05,
    0x40, 0x03, 0x06,
    0x40, 0x03, 0x07,
    0x40, 0x03, 0x08,
    0x40, 0x03, 0x09,
    0x40, 0x03, 0x0a,
    0x40, 0x03, 0x0b,
    0x40, 0x03, 0x0c,
    0x40, 0x03, 0x0d,
    0x40, 0x03, 0x0e,
    0x40, 0x03, 0x0f,
    0x40, 0x03, 0x10,
    0x40, 0x03, 0x11,
    0x40, 0x03, 0x12,
    0x40, 0x03, 0x13,
    0x40, 0x03, 0x14,
    0x40, 0x03, 0x15,
    0x40, 0x03, 0x16,
    0x40, 0x03, 0x17,
    0x40, 0x03, 0x18,
    0x40, 0x03, 0x19,
    0x40, 0x03, 0x1b,
    0x40, 0x03, 0x1c,
    0x40, 0x03, 0x1d,
    0x40, 0x03, 0x1e,
    0x40, 0x03, 0x1f,
    0x40, 0x40, 0x21,
    0x40, 0x41, 0x21,
    0x40, 0x42, 0x21,
    0x40, 0x43, 0x21,
    0x40, 0x44, 0x21,
    0x40, 0x45, 0x21,
    0x40, 0x46, 0x21,
    0x40, 0x47, 0x21,
    0x40, 0x48, 0x21,
    0x40, 0x49, 0x21,
    0x40, 0x4a, 0x21,
    0x40, 0x4b, 0x21,
    0x40, 0x4c, 0x21,
    0x40, 0x4d, 0x21,
    0x40, 0x4e, 0x21,
    0x40, 0x4f, 0x21,
    0x40, 0x40, 0x22,
    0x40, 0x41, 0x22,
    0x40, 0x42, 0x22,
    0x40, 0x43, 0x22,
    0x40, 0x44, 0x22,
    0x40, 0x45, 0x22,
    0x40, 0x46, 0x22,
    0x40, 0x47, 0x22,
    0x40, 0x48, 0x22,
    0x40, 0x49, 0x22,
    0x40, 0x4a, 0x22,
    0x40, 0x4b, 0x22,
    0x40, 0x4c, 0x22,
    0x40, 0x4d, 0x22,
    0x40, 0x4e, 0x22,
    0x40, 0x4f, 0x22
  };
  if(ex[1] == ROLAND && (ex[3] == GS || ex[3] == LCD)) {
    is_gs = true;
    if(!is_sc88pro && length >= 8) {
      int i;
      if(!is_sc88) {
	for(i = 0; i < 99; i++) {
	  if(ex[5] == addr_sc88[i*3+0] &&
	     ex[6] == addr_sc88[i*3+1] &&
	     ex[7] == addr_sc88[i*3+2]) {
	    is_sc88 = true;
	    fprintf(stderr, "Matches: %02x %02x %02x (SC-88)\n",
		    ex[5], ex[6], ex[7]);
	    break;
	  }
	}
      }
      for(i = 0; i < 61; i++) {
	if(ex[5] == addr_sc88pro[i*3+0] &&
	   ex[6] == addr_sc88pro[i*3+1] &&
	   ex[7] == addr_sc88pro[i*3+2]) {
	  is_sc88pro = true;
	  fprintf(stderr, "Matches: %02x %02x %02x (SC-88Pro)\n",
		  ex[5], ex[6], ex[7]);
	  break;
	}
      }
    }
  } else {
    if(ex[1] == YAMAHA && ex[3] == XG)
      is_xg = true;
  }
}

