/*
 * mlist.cc -- multiple list
 *
 * Copyright (C) 1996-1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: mlist.cc,v 1.2 1998-03-25 08:49:25+09 satoshi Exp $

#include <stdlib.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xresource.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Grip.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/xpm.h>
#include <algo.h>
#include <list.h>
#include "xmidisel.h"
#include "mlist.h"

#ifndef NeedWidePrototypes
extern "C" void XawScrollbarSetThumb(Widget, float, float);
#endif

const char* MList::className = "MList";


MList::MList(Selector* sel_, Widget parent_, const char* name_,
	     unsigned columns_, const char** titles_)
: sel(sel_), parent(parent_), name(name_), need_scrollbar(false), 
  columns(columns_), rows(0), titles(NULL), list(NULL), longest(NULL),
  justify(NULL), index(-1), first(0), cur_file(0), shown_columns(columns),
  width(NULL), proportion(NULL)
{
  if(columns == 0)
    return;
  titles = new char*[columns];
  unsigned i;
  for(i = 0; i < columns; i++)
    titles[i] = strdup(titles_[i]);

  justify = new Justify[columns];
  for(i = 0; i < columns; i++) {
    char res_name[24];
    sprintf(res_name, "list%u.Justify", i);
    const char* justify_str = sel->getResource(className, name, res_name);
    if(!strcmp("", justify_str) || !strcmp("left", justify_str))
      justify[i] = left;
    else if(!strcmp("center", justify_str))
      justify[i] = center;
    else if(!strcmp("right", justify_str))
      justify[i] = right;
    else {
      fprintf(stderr, "invalid justify `%s'.\n", justify_str);
      justify[i] = left;
    }
  }

  width = new int[columns];
  for(i = 0; i < columns; i++)
    width[i] = -1;
  const char* res_str;
  proportion = new double[columns];
  res_str = sel->getResource(className, name, "Proportion");
  const char* p = res_str;
  for(i = 0; i < columns; i++)
    proportion[i] = strtod(p, &p);
  res_str = sel->getResource(className, name, "rowSpacing");
  if(strcmp("", res_str))
    rowSpacing = atoi(res_str);
  res_str = sel->getResource(className, name, "internalHeight");
  if(strcmp("", res_str))
    internalHeight = atoi(res_str);
  res_str = sel->getResource(className, name, "internalWidth");
  if(strcmp("", res_str))
    internalWidth = atoi(res_str);

  paned = XtVaCreateManagedWidget(name, panedWidgetClass, parent,
				  XtNborderWidth, 0,
				  XtNorientation, XtEhorizontal,
				  NULL);
  XtAddEventHandler(paned, StructureNotifyMask, False,
		    (XtEventHandler)ResizeCB, this);
  titlePaned = new Widget[columns+1];
  titleW = new Widget[columns];
  listW = new Widget[columns];
  for(i = 0; i < columns; i++)
    createTitleW(i);
  createSHVPaned();
}


/*
 * create a list paned
 */
void MList::createTitleW(unsigned i)
{
  char panedname[16];
  sprintf(panedname, "paned%u", i);
  titlePaned[i] = XtVaCreateWidget(panedname, panedWidgetClass, paned,
				   XtNborderWidth,       0,
				   XtNresizeToPreferred, True,
				   XtNallowResize,       True,
				   NULL);
  XtAddEventHandler(titlePaned[i], StructureNotifyMask, False,
		    (XtEventHandler)ResizeCB2, this);
  char titlename[16];
  sprintf(titlename, "title%u", i);
  titleW[i] = XtVaCreateWidget(titlename, commandWidgetClass, titlePaned[i],
			       XtNborderWidth, 0,
			       XtNlabel,       titles[i],
			       XtNshowGrip,    False,
			       XtNallowResize, True,
			       NULL);
  char listname[16];
  sprintf(listname, "list%u", i);
  int width = 1;//600.0*proportion[i];	// ???
  listW[i] = XtVaCreateWidget(listname, simpleWidgetClass, titlePaned[i],
			      XtNborderWidth, 0,
			      XtNwidth,       width,
			      XtNshowGrip,    False,
			      XtNallowResize, True,
			      NULL);
  XtAddEventHandler(listW[i], ExposureMask, False,
		    (XtEventHandler)ExposeCB, this);
  Widget widget_list[] = { titleW[i], listW[i] };
  XtManageChildren(widget_list, XtNumber(widget_list));
  XtManageChild(titlePaned[i]);
}


/*
 * create a paned with show button, hide button and vertical scroll bar
 */
void MList::createSHVPaned(void)
{
  char panedname[16];
  sprintf(panedname, "paned%u", columns);
  titlePaned[columns] = XtVaCreateWidget(panedname, panedWidgetClass, paned,
					 XtNborderWidth, 0,
					 XtNshowGrip,    False,
					 XtNallowResize, True,
					 XtNskipAdjust,  True,
					 XtNmin,         24,	// ???
					 XtNmax,         24,	// ???
					 NULL);
  buttonBox = XtVaCreateWidget("box", boxWidgetClass, titlePaned[columns],
			       XtNborderWidth, 0,
			       XtNorientation, XtEhorizontal,
			       XtNshowGrip,    False,
			       NULL);
  showButton = XtVaCreateWidget("show", commandWidgetClass, buttonBox,
				XtNborderWidth, 0,
				XtNlabel,       "<",	// ???
				NULL);
  XtAddCallback(showButton, XtNcallback, (XtCallbackProc)ShowCB, this);
  hideButton = XtVaCreateWidget("hide", commandWidgetClass, buttonBox,
				XtNborderWidth, 0,
				XtNlabel,       ">",	// ???
				NULL);
  XtAddCallback(hideButton, XtNcallback, (XtCallbackProc)HideCB, this);
  vscrollbar =
    XtVaCreateWidget("vscrollbar", scrollbarWidgetClass, titlePaned[columns],
//		     XtNwidth,             14,	// ???
//		     XtNthickness,         14,
		     XtNmappedWhenManaged, need_scrollbar,
		     NULL);
  XtAddCallback(vscrollbar, XtNscrollProc, (XtCallbackProc)ScrollCB, this);
  XtAddCallback(vscrollbar, XtNjumpProc, (XtCallbackProc)JumpCB, this);

  Widget widget_list[] = { showButton, hideButton };
  XtManageChildren(widget_list, XtNumber(widget_list));
  Widget widget_list2[] = { buttonBox, vscrollbar };
  XtManageChildren(widget_list2, XtNumber(widget_list2));
  XtManageChild(titlePaned[columns]);

/*  if(XtIsRealized(buttonBox)) {
    Dimension width;
    XtVaGetValues(buttonBox, XtNwidth, &width, NULL);
    XawPanedSetMinMax(titlePaned[columns], width, width);*/
//  XtVaSetValues(vscrollbar, XtNwidth, 14, NULL);	// ???
//  XtVaSetValues(vscrollbar, XtNthickness, 14, NULL);	// ???
/*  }*/
}


/*
 * initialize X resources
 */
void MList::initialize(void)
{
  d = XtDisplay(paned);
  Window w = XtWindow(paned);

  XGCValues gcv;
  gcv.graphics_exposures = False;
  gc = XCreateGC(d, w, GCGraphicsExposures, &gcv);
#ifdef KANJI
  char** missing_charset_list;
  int missing_charset_count;
  char* def_string;
  fontset = XCreateFontSet(d, sel->getResource(className, name, "FontSet"),
			   &missing_charset_list, &missing_charset_count,
			   &def_string);
#endif
  xfs = XLoadQueryFont(d, sel->getResource(className, name, "Font"));
  XSetFont(d, gc, xfs->fid);
  foreground =
    sel->allocColor(sel->getResource(className, name, "Foreground"));
  XSetForeground(d, gc, foreground);
  background =
    sel->allocColor(sel->getResource(className, name, "Background"));
  XSetBackground(d, gc, background);
  highlightBackground =
    sel->allocColor(sel->getResource(className, name, "highlightBackground"));

/*
  Dimension width;
  XtVaGetValues(buttonBox, XtNwidth, &width, NULL);
  XawPanedSetMinMax(titlePaned[columns], width, width);*/
//  XtVaSetValues(vscrollbar, XtNwidth, 14, NULL);	// ???
//  XtVaSetValues(vscrollbar, XtNthickness, 14, NULL);	// ???

//  readXpm(sel->getResource(className, name, "folderXpm"), &folder.pixmap,
//	  &folder.width, &folder.height);
}


MList::~MList()
{
}


/*
 * change list
 */
void MList::change(unsigned rows_, const char** list_,
		   const unsigned* longest_)
{
/*  if(list != NULL)
    delete[] list;*/
  if(longest != NULL)
    delete[] longest;

  list = list_;
  rows = rows_;

  unsigned i;
  longest = new unsigned[columns];
  if(longest_ != NULL) {
    for(i = 0; i < columns; i++)
      longest[i] = longest_[i];
  } else {
    // calculate longest width
    for(i = 0; i < columns; i++) {
      longest[i] = 0;
      unsigned j;
      for(j = 0; j < rows; j++) {
	Dimension width;
	const char* str;
	if((str = list[i*rows+j]) != NULL) {
#ifdef KANJI
	  XRectangle ink, logical;
	  XmbTextExtents(fontset, str, strlen(str), &ink, &logical);
	  width = ink.width;
#else
	  width = XTextWidth(xfs, str, strlen(str));
#endif
	} else
	  width = 0;
	if(width > longest[i])
	  longest[i] = width;
      }
    }
  }

  index = (rows > 0) ? 0 : -1;
  first = 0;
  cur_file = 0;
  for(i = 0; i < shown_columns; i++)
    redraw(listW[i]);

  Dimension h;
  XtVaGetValues(listW[0], XtNheight, &h, NULL);
  if(rows > 0 && h < rows * (xfs->ascent + xfs->descent+1)) {
    float shown = (float)h / (float)(xfs->ascent+xfs->descent+1) / (float)rows;
    XawScrollbarSetThumb(vscrollbar, 0.0, shown);
    XtMapWidget(vscrollbar);
    need_scrollbar = true;
  } else {
    XtUnmapWidget(vscrollbar);
    need_scrollbar = false;
  }
}


/*
 * redraw specified region
 */
void MList::redraw(Widget widget, int x, int y, int width, int height)
{
  if(rows == 0)
    return;
  unsigned col;
  for(col = 0; col < columns; col++)
    if(widget == listW[col])
      break;
  if(widget == NULL || col == columns)
    return;
  if(width == -1 || height == -1) {
#if 0
    Dimension width_, height_;
    XtVaGetValues(widget,
		  XtNwidth,  &width_,
		  XtNheight, &height_,
		  NULL);
    width = width_;
    height = height_;
#endif
    width = widget->core.width;
    height = widget->core.height;
  }
  Dimension wd;
  XtVaGetValues(widget, XtNwidth, &wd, NULL);
  XSetForeground(d, gc, background);
  Window w = XtWindow(widget);
  XFillRectangle(d, w, gc, (justify[col] == left) ? x : 0, y,
		 (justify[col] == left) ? width : wd, height);
  if(list != NULL) {
    int ht = xfs->ascent+xfs->descent+1;
    int yy = y/ht*ht+xfs->ascent;
    XSetForeground(d, gc, foreground);
    unsigned i;
    for(i = first+y/ht; i < rows && yy <= y+height+xfs->ascent;
	i++, yy += ht) {
      if((int)i == index) {
	XSetForeground(d, gc, highlightBackground);
	XFillRectangle(d, w, gc, 0, yy-xfs->ascent-1, wd, ht);
	XSetForeground(d, gc, foreground);
      }
      const char* str;
      if((str = list[col*rows+i]) != NULL) {
#ifdef KANJI
	XRectangle ink, logical;
	XmbTextExtents(fontset, str, strlen(str), &ink, &logical);
	int xx =
	  ((justify[col] == center) ?
	   (max((int)wd, (int)longest[col])-ink.width)/2+internalWidth :
	   ((justify[col] == right) ?
	    internalWidth+longest[col]-ink.width+
	    max((int)(wd-longest[col]-2*internalWidth), 0) :
	    internalWidth));
	XmbDrawString(d, w, fontset, gc, xx, yy, str, strlen(str));
#else
	XDrawString(d, w, gc, internalWidth, yy, str, strlen(str));
#endif
      }
    }
  }
}


/*
 * highlight the cursor
 */
void MList::highlight(int n)
{
  if(rows == 0)
    return;
  Dimension* width = new Dimension[shown_columns];
  unsigned i;
  for(i = 0; i < shown_columns; i++)
    XtVaGetValues(listW[i], XtNwidth, &width[i], NULL);
  int height = xfs->ascent+xfs->descent+1;
  const char* str;
  if(index != -1) {
    int y = (index-first)*height+xfs->ascent;
    XSetForeground(d, gc, background);
    for(i = 0; i < shown_columns; i++)
      XFillRectangle(d, XtWindow(listW[i]), gc, 0, y-xfs->ascent-1,
		     width[i], height);
    XSetForeground(d, gc, foreground);
    if(list != NULL) {
      for(i = 0; i < shown_columns; i++) {
	if((str = list[i*rows+index]) != NULL) {
#ifdef KANJI
	  XRectangle ink, logical;
	  XmbTextExtents(fontset, str, strlen(str), &ink, &logical);
	  int x =
	    ((justify[i] == center) ?
	     (max((int)width[i], (int)longest[i])-ink.width)/2+internalWidth :
	     ((justify[i] == right) ?
	      internalWidth+longest[i]-ink.width+
	      max((int)(width[i]-longest[i]-2*internalWidth), 0) :
	      internalWidth));
	  XmbDrawString(d, XtWindow(listW[i]), fontset, gc, x, y, str,
			strlen(str));
#else
	  XDrawString(d, XtWindow(listW[i]), gc, internalWidth, y, str,
		      strlen(str));
#endif
	}
      }
    }
  }
  if(list != NULL) {
    int y = (n-first)*height+xfs->ascent;
    XSetForeground(d, gc, highlightBackground);
    for(i = 0; i < shown_columns; i++)
      XFillRectangle(d, XtWindow(listW[i]), gc, 0, y-xfs->ascent-1,
		     width[i], height);
    XSetForeground(d, gc, foreground);
    for(i = 0; i < shown_columns; i++) {
      if((str = list[i*rows+n]) != NULL) {
#ifdef KANJI
	XRectangle ink, logical;
	XmbTextExtents(fontset, str, strlen(str), &ink, &logical);
	int x =
	  ((justify[i] == center) ?
	   (max((int)width[i], (int)longest[i])-ink.width)/2+internalWidth :
	   ((justify[i] == right) ?
	    internalWidth+longest[i]-ink.width+
	    max((int)(width[i]-longest[i]-2*internalWidth), 0) :
	    internalWidth));
	XmbDrawString(d, XtWindow(listW[i]), fontset, gc, x, y, str,
		      strlen(str));
#else
	  XDrawString(d, XtWindow(listW[i]), gc, internalWidth, y, str,
		      strlen(str));
#endif
      }
    }
  }
  index = n;
  delete[] width;
}


/*
 * 
 */
void MList::select(int /*x*/, int y, bool flag)
{
  int height = xfs->ascent+xfs->descent+1;
  unsigned n = y/height+first;
  if(n < rows)
    highlight(n);
  cur_file = n;
}


/*
 * 
 */
void MList::up(void)
{
  if(cur_file > 0) {
    cur_file--;
    if(cur_file < first) {
      first--;
      redraw();
      if(need_scrollbar) {
	Dimension h;
	XtVaGetValues(listW[0], XtNheight, &h, NULL);
	float top = (float)first / (float)rows;
	XawScrollbarSetThumb(vscrollbar, top, -1.0);
      }
    }
    highlight(cur_file);
  }
}


/*
 *
 */
void MList::down(void)
{
  if(cur_file < rows-1) {
    cur_file++;
    Dimension h;
    XtVaGetValues(listW[0], XtNheight, &h, NULL);
    h /= xfs->ascent + xfs->descent+1;
    if(cur_file > first+h-1) {
      first++;
      redraw();
      if(need_scrollbar) {
	XtVaGetValues(listW[0], XtNheight, &h, NULL);
	float top = (float)first / (float)rows;
	XawScrollbarSetThumb(vscrollbar, top, -1.0);
      }
    }
    highlight(cur_file);
  }
}


/*
 * scroll up
 */
void MList::scroll_up(void)
{
}


/*
 * scroll down
 */
void MList::scroll_down(void)
{
}


/*
 * compute each pane size
 */
void MList::resize(void)
{
  Dimension wd, wd2;
  XtVaGetValues(paned, XtNwidth, &wd, NULL);
  XtVaGetValues(titlePaned[columns], XtNwidth, &wd2, NULL);
  wd -= wd2;

  unsigned i;
  Dimension sum = 0;
  for(i = 0; i < shown_columns; i++)
    sum += longest[i];
  double sum_prop = 0.0;
  for(i = 0; i < shown_columns; i++)
    sum_prop += proportion[i];

  Dimension* width = new Dimension[shown_columns];
#if 0
  if(sum <= wd) {
printf("over\n");
    for(i = 0; i < columns; i++)
      width[i] = wd*proportion[i];
  } else {
printf("under\n");
    sum = 0;
    for(i = 0; i < columns; i++) {
      sum += width[i] =
	/*min((int)(wd*proportion[i]), */(int)longest[i]/*)*/;
    }
    for(i = 0; i < columns; i++)
      width[i] -= (sum-wd)*proportion[i];
  }
#endif
  for(i = 0; i < shown_columns; i++)	// ???
    width[i] = (Dimension)(wd*proportion[i]/sum_prop);	//???
  for(i = 0; i < shown_columns; i++) {
    XtVaSetValues(titlePaned[i], XtNwidth, width[i], NULL);
    XtVaSetValues(titlePaned[i], XtNpreferredPaneSize, width[i], NULL);
  }
#if 0
  printf("Longest: ");
  for(i = 0; i < columns; i++)
    printf("%d ", longest[i]);
  printf("\n");
  printf("Actual:  ");
  for(i = 0; i < columns; i++)
    printf("%d ", width[i]);
  printf("\n");
  printf("Percent: ");
  for(i = 0; i < columns; i++)
    printf("%g ", (double)width[i]/(double)wd*100.);
  printf("\n");
#endif
  delete[] width;
}


/*
 * scroll callback
 */
void ScrollCB(Widget w, MList* self, int position)
{
  if(position >= 0)
    self->scroll_down();
  else
    self->scroll_up();
}


/*
 * jump callback
 */
void JumpCB(Widget w, MList* self, float* percent_ptr)
{
  float percent = *percent_ptr;
  Dimension h;
  XtVaGetValues(self->listW[0], XtNheight, &h, NULL);
  h /= self->xfs->ascent + self->xfs->descent+1;
  if(h < self->rows) {
    unsigned first1 = (unsigned)((self->rows)*percent);
    if(first1 != self->first) {
      self->first = first1;
      self->redraw();
    }
  }
}


/*
 * Expose event handler for listW
 */
void ExposeCB(Widget w, MList* self, XEvent* event)
{
  if(event->type == Expose) {
    XExposeEvent& e = event->xexpose;
    self->redraw(w, e.x, e.y, e.width, e.height);
  }
}


/*
 * StructureNotify event handler for paned
 */
void ResizeCB(Widget /*w*/, MList* self, XConfigureEvent* event, Boolean* /**/)
{
  if(event->above == None)
    return;
//  printf("Resize(%d,%d,%d,%d)\n",
//	 event->x,event->y,event->width,event->height);
  self->resize();

  Dimension h;
  XtVaGetValues(self->listW[0], XtNheight, &h, NULL);
#ifdef Use_Xaw3d
  // change grip height
  char str[256];
  sprintf(str, "*%s.Grip.height: %u", self->name, h);
  self->sel->setResource(str);
  unsigned i;
  for(i = 0; i < self->shown_columns-1; i++) {
    XtVaSetValues(self->titlePaned[i], XtNshowGrip, False, NULL);
    XtVaSetValues(self->titlePaned[i], XtNshowGrip, True, NULL);
  }
#endif

#if 0
  Widget* pw;
  for(pw = ((PanedRec*)self->paned)->composite.children;
      pw < (((PanedRec*)self->paned)->composite.children
	    + ((PanedRec*)self->paned)->paned.num_panes);
      pw++) {
    Widget gw = ((Pane)(*pw)->core.constraints)->grip;
    if(gw != NULL) {
//    XtVaSetValues(((GripRec*)gw)->threeD, XtNheight, h, NULL);
//    XtVaSetValues(gw, XtNheight, h, NULL);
//      gw->core.height = h;
//      (*((GripClassRec*)XtClass(gw))->core_class.expose)(gw, NULL, NULL);
//      Dimension hh;
//      XtVaGetValues(gw, XtNheight, &hh, NULL);
    }
  }
#endif

  if(self->rows > 0 &&
     h < self->rows * (self->xfs->ascent + self->xfs->descent+1)) {
    float shown =
      (float)h / (float)(self->xfs->ascent+self->xfs->descent+1)
	/ (float)self->rows;
    XawScrollbarSetThumb(self->vscrollbar, 0.0, shown);
    XtMapWidget(self->vscrollbar);
    self->need_scrollbar = true;
  } else {
    XtUnmapWidget(self->vscrollbar);
    self->need_scrollbar = false;
  }
}


/*
 * StructureNotify event handler for titlePaned
 */
void ResizeCB2(Widget widget, MList* self, XConfigureEvent* event,
	       Boolean* /**/)
{
  if(event->above == None || self->rows == 0)
    return;
  unsigned col;
  for(col = 0; col < self->columns; col++)
    if(widget == self->titlePaned[col])
      break;
  if(widget == NULL || col == self->columns)
    return;
//  printf("Resize2: paned%u ",col);
//  printf("%d -> %d\n",self->width[col],event->width);
  if(self->width[col] == -1)
    self->width[col] = event->width;
  else {
    if(event->width < self->width[col]) {	// shrunk
      if(self->justify[col] != MList::left) {
	self->redraw(self->listW[col], 0, 0, event->width, event->height);
//	printf("redraw()\n");
      }
    }
    self->width[col] = event->width;
  }
}


/*
 * show column
 */
void ShowCB(Widget /*widget*/, MList* self, void* /**/)
{
  if(self->shown_columns < self->columns) {
    XtDestroyWidget(self->titlePaned[self->columns]);
    XawPanedSetRefigureMode(self->paned, False);
    self->createTitleW(self->shown_columns);
    self->shown_columns++;
    self->createSHVPaned();
    self->resize();
    XawPanedSetRefigureMode(self->paned, True);
  }
}


/*
 * hide column
 */
void HideCB(Widget /*widget*/, MList* self, void* /**/)
{
  if(self->shown_columns > 1) {
    XtDestroyWidget(self->titlePaned[self->shown_columns-1]);
    self->titlePaned[self->shown_columns-1] = NULL;
    self->shown_columns--;
    self->resize();
  }
}


#if 0
bool MList::readXpm(const char* xpmfile, Pixmap* pixmap,
		   int* width, int* height)
{
  Colormap cmap = DefaultColormap(XtDisplay(paned), 0);
  XpmAttributes attr;
  attr.valuemask = XpmColormap | XpmSize /*| XpmCloseness*/;
  attr.colormap  = cmap;
  attr.exactColors = False;
//attr.closeness = closeness;
  Pixmap mask;
  int result =
    XpmReadFileToPixmap(XtDisplay(paned), XtWindow(paned), (char*)xpmfile,
			pixmap, &mask, &attr);
  *width  = attr.width;
  *height = attr.height;
  if(result != XpmSuccess && result != XpmColorError) {
    fprintf(stderr, "XpmReadFileToPixmap: error reading %s\n", xpmfile);
    return false;
  }
  return true;
}
#endif


