/*
 * fileinfo.cc -- file information
 *
 * Copyright (C) 1996-1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: fileinfo.cc,v 1.2 1998/03/14 10:57:10 satoshi Exp $

#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include "fileinfo.h"
#include "kconv.h"


static bool compare_str(const char* n1, const char* n2);
static bool compare_date(time_t t1, time_t t2);
static bool compare_size(unsigned long s1, unsigned long s2);

FileInfo::FileInfo(const char* path_)
: path(strdup(path_)), title(NULL), date(0), size(0), type(Unknown)
{
  name = strrchr(path, '/');
  if(name == NULL)
    name = path;
  else
    name++;

  if((type = _isdirectory()) != Dir)
    if((type = _ismidifile()) != Unknown)
      _getTitle();

  struct stat buf;
  if(stat(path, &buf) == -1)
    perror("stat");
  else {
    size = (type != Dir) ? buf.st_size : 0;
    date = buf.st_mtime;
  }

}


FileInfo::~FileInfo()
{
//  delete[] path;
//  delete[] name;
//  delete[] title;
}


FileInfo::file_type FileInfo::_isdirectory(void)
{
  struct stat buf;
  if(stat(path, &buf) == -1) {
    perror("stat");
    return Unknown;
  }
  return S_ISDIR(buf.st_mode) ? Dir : Unknown;
}


FileInfo::file_type FileInfo::_ismidifile(void)
{
  FILE* fp;
  errno = 0;
  if((fp = fopen(path, "r")) == NULL) {
    if(errno != EACCES)
      fprintf(stderr, "Can't open file `%s'.\n", path);
    return Unknown;
  }
  // RCP file
  const char* rcp_header = "RCM-PC98V2.0(C)COME ON MUSIC\x0d\x0a\0\0";
  const char* g36_header = "COME ON MUSIC RECOMPOSER RCP3.0\0";
  char header[32];
  int i;
  for(i = 0; i < 32; i++)
    header[i] = fgetc(fp);
  if(!strncmp(header, rcp_header, 32)) {
    fclose(fp);
    return RCP;
  }
  if(!strncmp(header, g36_header, 32)) {
    fclose(fp);
    return RCP3;
  }
  rewind(fp);

  // Standard MIDI File
  const char* smf_header = "MThd";
  for(i = 0; i < 4; i++)
    header[i] = fgetc(fp);
  if(!strncmp(header, smf_header, 4)) {
    fclose(fp);
    return SMF;
  }

  fclose(fp);
  return Unknown;
}


void FileInfo::_getTitle(void)
{
  if(type == RCP || type == RCP3 || type == SMF) {
    FILE* fp;
    errno = 0;
    if((fp = fopen(path, "r")) == NULL) {
      if(errno != EACCES)
	fprintf(stderr, "Can't open file `%s'.\n", path);
      return;
    }
    int i;
    if(type == RCP || type == RCP3) {
      fseek(fp, 0x20, SEEK_SET);
      char* title_ = new char[64+1];
      char* title__ = new char[64*2+1];
      for(i = 0; i < 64; i++)
	title_[i] = fgetc(fp);
      title_[64] = '\0';
      fclose(fp);
      kconv((unsigned char*)title_, (unsigned char*)title__);
      delete[] title_;
      title = title__;
    }
    if(type == SMF) {
      fseek(fp, 0x04, SEEK_SET);
      char* title_ = new char[255+1];
      char* title__ = new char[255*2+1];
      int len = 0;
      int c;
      c = fgetc(fp);
      while(!feof(fp)) {
	if(c == 0xff) {			// meta event
	  if((c = fgetc(fp)) == 0x03)	// Sequence/Track Name text
	    break;
	} else
	  c = fgetc(fp);
      }
      len = fgetc(fp);
      for(i = 0; i < len; i++)
	title_[i] = fgetc(fp);
      title_[i] = '\0';
      fclose(fp);
      kconv((unsigned char*)title_, (unsigned char*)title__);
      delete[] title_;
      title = title__;
    }
  }
}


const char* FileInfo::getDateStr(void) const
{
  struct tm* tm;
  if((tm = localtime(&date)) == NULL)
    return "";
  time_t today = time(NULL);
  double diff = difftime(today, date);
  char str[12+1];
  if(diff > 6*60*60*24*30 || diff < 1*60*60)
    // more than 6 months old or more than 1 hour into the future
    strftime(str, sizeof(str), "%b %d  %Y", tm);
  else
    strftime(str, sizeof(str), "%b %d %H:%M", tm);
  return strdup(str);
}


const char* FileInfo::getSizeStr(void) const
{
  char* str = new char[16+1];
  char* p = &str[16];
  *p-- = '\0';
  if(size > 0) {
    unsigned long s;
    int i;
    for(s = size, i = 1; s > 0; s /= 10, i++) {
      *p-- = '0' + s % 10;
      if(i == 3 && s/10 > 0) {
	*p-- = ',';
	i = 0;
      }
    }
  } else
    *p-- = '0';
  return ++p;
}


bool compare_dir(const FileInfo& f1, const FileInfo& f2, bool* r)
{
  bool d1 = f1.isdirectory();
  bool d2 = f2.isdirectory();
  if(d1 || d2) {
    *r = !(d1 && d2) ? d1 : compare_str(f1.name, f2.name);
    return true;
  } else
    return false;
}


bool compare_name_a(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : compare_str(f1.name, f2.name);
}


bool compare_name_d(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : !compare_str(f1.name, f2.name);
}


bool compare_title_a(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : compare_str(f1.title, f2.title);
}


bool compare_title_d(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : !compare_str(f1.title, f2.title);
}


static bool compare_str(const char* n1, const char* n2)
{
  if(n1 == NULL)
    return false;
  if(n2 == NULL)
    return true;
  int l1 = strlen(n1);
  int l2 = strlen(n2);
  int i;
  for(i = 0; i < l1 && i < l2; i++) {
    if(n1[i] != n2[i])
      return n1[i] < n2[i];
  }
  return (l1 > l2);
}


bool compare_date_a(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : compare_date(f1.date, f2.date);
}


bool compare_date_d(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : !compare_date(f1.date, f2.date);
}


static bool compare_date(time_t t1, time_t t2)
{
  return t1 < t2;
}


bool compare_size_a(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : compare_size(f1.size, f2.size);
}


bool compare_size_d(const FileInfo& f1, const FileInfo& f2)
{
  bool r;
  return compare_dir(f1, f2, &r) ? r : !compare_size(f1.size, f2.size);
}


static bool compare_size(unsigned long s1, unsigned long s2)
{
  return s1 < s2;
}
