/*
 * GVector.java -- multi version vector
 *
 * (C)1993, 1994, 1995 Institute for New Generation Computer Technology
 *	(Read COPYRIGHT for detailed information.) 
 *
 * (C)1996, 1997 Japan Information Processing Development Center
 *      (Read COPYRIGHT-JIPDEC for detailed information.)
 *
 * Copyright (C) 1998,1999 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * $Id: GVector.java,v 1.2 1999-03-06 23:32:30+09 satoshi Exp $
 */

package kl1.lang;

/**
 * This class represents a vector object.
 *
 * @author Satoshi KURAMOCHI
 */
public final class GVector extends GDObj {
  static { name = "vector"; }

  private GVector next;		// when null, shallowed
  private int index;		// size or index
  private boolean iscnst;
  /*
    The following field "body" is actually used as a union:
    when shallowed, as actually the body of the vector;
    when deep, i.e., for difference records, the element that differs.
    */
  private KL1Object body[];

  private static final String functor_element_2 = "element_2".intern();
  private static final String functor_size_1 = "size_1".intern();
  private static final String functor_set__element_3 = "set__element_3".intern();
  private static final String functor_set__element_4 = "set__element_4".intern();
  private static final String functor_split_3 = "split_3".intern();
  private static final String functor_join_2 = "join_2".intern();
  private static final String functor_vector_1 = "vector_1".intern();


  // shallowing
  private void shallow() {
    if(next != null) {
      GVector v, last, next;

      // Go down to the shallowed version inverting pointers
      last = null;
      v = this;
      do {
	next = v.next;
	v.next = last;
	last = v;
	v = next;
      } while (v != null);
      v = last;

      // Update physical vector elements tracing the inverted pointers
      int size = v.index;
      KL1Object body[] = v.body;
      next = v.next;
      do {
	int index;
	index = next.index;
	v.index = index;
	v.body = new KL1Object[]{ body[index] };
	body[index] = next.body[0];
	v = next;
	next = v.next;
      } while (next != null);
      v.index = size;
      v.body = body;
    }
  }


  // basic method definitions
  public KL1Object gunify(GDObj that_) {
    if(!(that_ instanceof GVector))
      return GDObj.FAILURE;
    GVector that = (GVector)that_;
    shallow();
    int size = index;
    that.shallow();
    if(that.index != size)
      return GDObj.FAILURE;
    for(int k=0; k<size; k++) {
      shallow();
      KL1Object retval = body[k].gunify(that.body[k]);
      if(retval != GDObj.SUCCESS)
	return retval;
      that.shallow();
    }
    return GDObj.SUCCESS;
  }


  public void unify(KL1Machine mach, KL1Object that_) {
    if(mach.UNIFYDEBUG)
      mach.print("Unify with " + this.print() + "," + that_.print());
    while (that_ instanceof Var) {
      KL1Object temp = ((Var)that_).refers;
      if(temp == that_) { // that is undef cell
	((Var)that_).refers = this;
	return;
      } else {
	if(temp instanceof Var && ((Var)temp).refers == that_) {
	  mach.resume_goals(temp, this);
	  return;
	}
      }
      that_ = temp;
    }
    // that is bound
    int size;
    if(!(that_ instanceof GVector))
      unify_fail(mach);
    GVector that = (GVector)that_;
    shallow();
    size = index;
    that.shallow();
    if(that.index != size)
      unify_fail(mach);
    for(int k=0; k<size; k++) {
      KL1Object x = that.body[k];
      shallow();
      body[k].shallow_unify(mach, x);
      that.shallow();
    }
    return;
  }


  // Generic method
  private final void body_element_2(KL1Machine mach, String method,
				    KL1Object argv[]) {
    shallow();
    int position[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, position, 0, 0, index)
       != null)
      argv[1].shallow_unify(mach, body[position[0]]);
  }


  private final void body_size_1(KL1Machine mach, String method,
				 KL1Object argv[]) {
    shallow();
    unify_value(mach, argv[0], new IntAtom(index));
  }


  private final void body_set__element_3(KL1Machine mach, String method,
					 KL1Object argv[]) {
    shallow();
    int size = index;
    int position[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, position, 0, 0, size) == null)
      return;
    GVector newvect = new GVector();
    if(!iscnst) {
      KL1Object olddata = body[position[0]];
      index = position[0];
      body[position[0]] = argv[1];
      newvect.body = body;
      body = new KL1Object[]{ olddata };
      next = newvect;
    } else {
      KL1Object newbody[] = new KL1Object[size];
      System.arraycopy(body, 0, newbody, 0, size);
      newbody[position[0]] = argv[1];
      newvect.body = newbody;
    }
    newvect.next = null;
    newvect.index = size;
    newvect.iscnst = false;
    unify_value(mach, argv[2], newvect);
  }


  private final void body_set__element_4(KL1Machine mach, String method,
					 KL1Object argv[]) {
    shallow();
    int size = index;
    int position[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, position, 0, 0, size) == null)
      return;
    GVector newvect = new GVector();
    KL1Object olddata = body[position[0]];
    argv[1].shallow_unify(mach, olddata);
    if(!iscnst) {
      index = position[0];
      body[position[0]] = argv[2];
      newvect.body = body;
      body = new KL1Object[]{ olddata };
      next = newvect;
    } else {
      KL1Object newbody[] = new KL1Object[size];
      System.arraycopy(body, 0, newbody, 0, size);
      newbody[position[0]] = argv[2];
      newvect.body = newbody;
    }
    newvect.next = null;
    newvect.index = size;
    newvect.iscnst = false;
    unify_value(mach, argv[3], newvect);
  }


  private final void body_split_3(KL1Machine mach, String method,
				  KL1Object argv[]) {
    shallow();
    int size = index;
    int split_point[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, split_point, 0, 0, size+1)
       == null)
      return;
    int lower_size = size-split_point[0];
    KL1Object upper_body[] = new KL1Object[split_point[0]];
    KL1Object lower_body[] = new KL1Object[lower_size];
    GVector upper = new GVector();
    GVector lower = new GVector();
    System.arraycopy(body, 0, upper_body, 0, split_point[0]);
    System.arraycopy(body, split_point[0], lower_body, 0, lower_size);
    upper.next = lower.next = null;
    upper.index = split_point[0];
    lower.index = lower_size;
    upper.iscnst = lower.iscnst = false;
    upper.body = upper_body;
    lower.body = lower_body;
    unify_value(mach, argv[1], upper);
    unify_value(mach, argv[2], lower);
  }


  private final void body_join_2(KL1Machine mach, String method,
				 KL1Object argv[]) {
    shallow();
    int size1 = index;
    KL1Object anotherq;
    if((anotherq = argv[0].deref()) instanceof Var) {
      suspend_generic_goal(mach, (Var)anotherq, method, argv);
      return;
    }
    GVector another = (GVector)anotherq;
    if(!(another instanceof GVector)) {
      mach.debug_print("### " + another.print() + " ###\n");
      mach.fatal("Illegal argument to string join");
    }
    another.shallow();
    int size2 = another.index;
    int newsize = size1+size2;
    KL1Object newbody[] = new KL1Object[newsize];
    shallow();
    another.shallow();
    GVector newvect = new GVector();
    System.arraycopy(body, 0, newbody, 0, size1);
    System.arraycopy(another.body, 0, newbody, size1, size2);
    newvect.next = null;
    newvect.index = newsize;
    newvect.iscnst = false;
    newvect.body = newbody;
    unify_value(mach, argv[1], newvect);
  }


  // Generic Method Table
  public void generic(KL1Machine mach, String method, KL1Object argv[]) {
    if(method == functor_element_2)
      body_element_2(mach, method, argv);
    else if(method == functor_size_1)
      body_size_1(mach, method, argv);
    else if(method == functor_set__element_3)
      body_set__element_3(mach, method, argv);
    else if(method == functor_set__element_4)
      body_set__element_4(mach, method, argv);
    else if(method == functor_split_3)
      body_split_3(mach, method, argv);
    else if(method == functor_join_2)
      body_join_2(mach, method, argv);
    else
      mach.fatal("undefined method: " + method);
  }


  // guard generic methods
  private KL1Object guard_element_2(KL1Object argv[]) {
    shallow();
    int position[] = new int[1];
    KL1Object ret;
    if((ret = set_gintarg_within_range(position, argv[0], 0, index))
       != GDObj.SUCCESS)
      return ret;
    argv[1] = body[position[0]];
    return GDObj.SUCCESS;
  }


  private boolean guard_vector_1(KL1Object argv[]) {
    shallow();
    argv[0] = new IntAtom(index);
    return true;
  }


  public KL1Object ggeneric(String method, KL1Object argv[]) {
    boolean ret;
    if(method == functor_element_2)
      return guard_element_2(argv);
    else if(method == functor_vector_1)
      ret = guard_vector_1(argv);
    else
      ret = false;
    return ret ? GDObj.SUCCESS : GDObj.FAILURE;
  }


  public String print() {
    shallow();
    int size = index;
    int limit = /*(size > g_length) ? g_length :*/ size;	// ???
    String s = "{";
    if(size != 0)
      s += body[0].print();
    for(int k = 1; k < limit; k++) {
      s += ",";
      if(size > 0)
	s += body[k].print();
    }
    if(limit != size)
      s += ",..";
    s += "}";
    return s;
  }


  public IntAtom compare(GDObj that_) {
    GVector that = (GVector)that_;
    shallow();
    int size1 = index;
    that.shallow();
    int size2 = index;
    int limit = (size1 <= size2 ? size1 : size2);
    for(int k = 0; k < limit; k++) {
      shallow();
      KL1Object elem = body[k], tmp;
      that.shallow();
/*
      tmp = builtin_compare_terms(elem, that.body[k]);
      if(!(tmp instanceof IntAtom))
	return (IntAtom)tmp;	// ???
      if(((IntAtom)tmp).value != 0)
	return (IntAtom)tmp;*/
    }
    if(size1 != size2) {
      return new IntAtom(size1 >= size2 ? size1+1 : -(size2+1));
    } else {
      return new IntAtom(0);
    }
  }


  public IntAtom hash(int level) {
    shallow();
    int size = index;
    if(level == 0 || size == 0) {
      return new IntAtom(0 + 0xb5d*size);
    } else {
      KL1Object h1, h2, h3;
      level--;
/*    h1 = hash_kl1_term(body[0], level);
      if(!(h1 instanceof IntAtom))
	return h1;
      h2 = hash_kl1_term(body[size>>1], level);
      if(!(h1 instanceof IntAtom))
	return h2;
      h3 = hash_kl1_term(body[size-1], level);
      if(!(h1 instanceof IntAtom))
	return h3;*/
      return
	new IntAtom(/*0x813*((IntAtom)h1).value
		    + 0x425*((IntAtom)h2).value
		    + 0x3c9*((IntAtom)h3).value
		    +*/ 0xb5d*size);
    }
  }


  // new_vector function
  public static KL1Object _new(KL1Machine mach, KL1Object argv[]) {
    if(argv.length != 1)
      error_in_new(mach, "Too few or too many arguments");
    KL1Object init;
    if((init = argv[0].deref()) instanceof Var)
      return suspend_new(mach, (Var)init, GVector.class, argv);
    KL1Object body[] = null;
    int size = 0;
    if(init instanceof IntAtom) {
      size = ((IntAtom)init).value;
      if(size < 0)
	error_in_new(mach, "Negative size specified");
      body = new KL1Object[size];
      for(int k = 0; k < size; k++)
	body[k] = new IntAtom(0);
    } else if(init == SymAtom.nil || init instanceof Cons) {
      for(size = 0; ; size++) {
	if(init == SymAtom.nil)
	  break;
	init = ((Cons)init).cdr;
	if((init = init.deref()) instanceof Var)
	  return suspend_new(mach, (Var)init, GVector.class, argv);
	if(init != SymAtom.nil && !(init instanceof Cons))
	  error_in_new(mach, "Illegal parameter");
      }
      init = argv[0];
      body = new KL1Object[size];
      for(int k = 0; k < size; k++) {
	if((init = init.deref()) instanceof Var)
	  return suspend_new(mach, (Var)init, GVector.class, argv);
	body[k] = ((Cons)init).car;
	init = ((Cons)init).cdr;
      }
    } else
      error_in_new(mach, "Illegal parameter");
    return new GVector(body, false);
  }


  /**
   * Constructs a vector object.
   */
  public GVector() {}


  /**
   * Constructs a vector object.
   *
   * @param  body    the elements.
   * @param  iscnst  whether or not this is a constant.
   */
  public GVector(KL1Object body[], boolean iscnst) {
    next = null;
    this.iscnst = iscnst;
    if(body != null) {
      index = body.length;
      this.body = new KL1Object[index];
      System.arraycopy(body, 0, this.body, 0, index);
    } else {
      index = 0;
      this.body = null;
    }
  }

  // Interface with builtin predicates
  public IntAtom size_of_vector() {
    shallow();
    return new IntAtom(index);
  }


  public KL1Object element_of_vector(IntAtom k) {
    shallow();
    if(k.value < 0 || index <= k.value) {
      return null;
    } else {
      return body[k.value];
    }
  }
}
