/*
 * GString.java -- string
 *
 * (C)1993, 1994, 1995 Institute for New Generation Computer Technology
 *	(Read COPYRIGHT for detailed information.) 
 *
 * (C)1996, 1997 Japan Information Processing Development Center
 *      (Read COPYRIGHT-JIPDEC for detailed information.)
 *
 * Copyright (C) 1998,1999 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * $Id: GString.java,v 1.3 1999-03-06 23:32:08+09 satoshi Exp $
 */

package kl1.lang;

/**
 * This class represents a string object.
 *
 * @author Satoshi KURAMOCHI
 */
public final class GString extends GDObj {
  static { name = "string"; }

  private GString next;		// when null, shallowed
  private int index;		// size or index
  private boolean iscnst;
//private boolean ismbdiff;	// reserved for future use
  /*
    The following field "body" is actually used as a union:
    when shallowed, as actually the body of the string;
    when deep, i.e., for difference records, the element that differs.
    */
  private char body[];

  private static final int ELEMSIZE = 8;


  private static final String functor_string_2 = "string_2".intern();
  private static final String functor_element_2 = "element_2".intern();
  private static final String functor_size_1 = "size_1".intern();
  private static final String functor_element__size_1 = "element__size_1".intern();
  private static final String functor_set__element_3 = "set__element_3".intern();
  private static final String functor_split_3 = "split_3".intern();
  private static final String functor_join_2 = "join_2".intern();
  private static final String functor_search__character_4 = "search__character_4".intern();
  private static final String functor_less__than_1 = "less__than_1".intern();
  private static final String functor_not__less__than_1 = "not__less__than_1".intern();
  private static final String functor_estring_3 = "estring_3".intern();


  // shallowing
  private void shallow() {
    if(next != null) {
      GString s, last, next;

      // Go down to the shallowed version inverting pointers
      last = null;
      s = this;
      do {
	next = s.next;
	s.next = last;
	last = s;
	s = next;
      } while (s != null);
      s = last;

      // Update physical string elements tracing the inverted pointers
      int size = s.index;
      char body[] = s.body;
      next = s.next;
      do {
	int index;
	index = next.index;
	s.index = index;
	s.body = new char[]{ body[index] };
	body[index] = next.body[0];
	s = next;
	next = s.next;
      } while (next != null);
      s.index = size;
      s.body = body;
    }
  }


  // basic method definitions
  public KL1Object gunify(KL1Object that_) {
    if (!(that_ instanceof GString))
      return GDObj.FAILURE;
    GString that = (GString)that_;
    shallow();
    int size = index;
    that.shallow();
    if (that.index != size)
      return GDObj.FAILURE;
    if(next == null) {
      if(body.length != that.body.length)
	return GDObj.FAILURE;
      else
	for(int i = 0; i < body.length; i++)
	  if(body[i] != that.body[i])
	    return GDObj.FAILURE;
    } else {
      for (int k=0; k<size; k++) {
	char c = that.body[k];
	shallow();
	if (body[k] != c)
	  return GDObj.FAILURE;
	that.shallow();
      }
    }
    return GDObj.SUCCESS;
  }


  public void unify(KL1Machine mach, KL1Object that_) {
    if(mach.UNIFYDEBUG)
      mach.print("Unify with " + this.print() + "," + that_.print());
    while (that_ instanceof Var) {
      KL1Object temp = ((Var)that_).refers;
      if (temp == that_) { // that is undef cell
	((Var)that_).refers = this;
	return;
      } else {
	if(temp instanceof Var && ((Var)temp).refers == that_) {
	  mach.resume_goals(temp, this);
	  return;
	}
      }
      that_ = temp;
    }
    // that is bound
    int size;
    if (!(that_ instanceof GString))
      unify_fail(mach);
    GString that = (GString)that_;
    shallow();
    size = index;
    that.shallow();
    if(that.index != size)
      unify_fail(mach);
    if(next == null) {
      if(body.length != that.body.length)
	unify_fail(mach);
      else
	for(int i = 0; i < body.length; i++)
	  if(body[i] != that.body[i])
	    unify_fail(mach);
    } else {
      for (int k=0; k<size; k++) {
	char c = that.body[k];
	shallow();
	if (body[k] != c)
	  unify_fail(mach);
	that.shallow();
      }
    }
    return;
  }


  // Generic method
  private final void body_string_2(KL1Machine mach, String method,
				   KL1Object argv[]) {
    shallow();
    unify_value(mach, argv[0], new IntAtom(index));
    unify_value(mach, argv[1], new IntAtom(ELEMSIZE));
  }


  private final void body_element_2(KL1Machine mach, String method,
				    KL1Object argv[]) {
    shallow();
    int position[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, position, 0, 0, index)
       != null)
      unify_value(mach, argv[1], new IntAtom(body[position[0]]));
  }


  private final void body_size_1(KL1Machine mach, String method,
				 KL1Object argv[]) {
    shallow();
    unify_value(mach, argv[0], new IntAtom(index));
  }


  private final void body_element__size_1(KL1Machine mach, String method,
					  KL1Object argv[]) {
    shallow();
    unify_value(mach, argv[0], new IntAtom(ELEMSIZE));
  }


  private final void body_set__element_3(KL1Machine mach, String method,
					 KL1Object argv[]) {
    shallow();
    int size = index;
    int position[] = new int[1];
    int newelem[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, position, 0, 0, size) == null ||
       set_intarg_within_range(mach, method, argv, newelem, 1, 0, 256) == null)
      return;
    GString newstr = new GString();
    if (!iscnst) {
      char olddata = body[position[0]];
      index = position[0];
      body[position[0]] = (char)newelem[0];
      newstr.body = body;
      body = new char[]{ olddata };
      next = newstr;
    } else {
      char newbody[] = new char[size];
      System.arraycopy(body, 0, newbody, 0, size);
      newbody[position[0]] = (char)newelem[0];
      newstr.body = newbody;
    }
    newstr.next = null;
    newstr.index = size;
    newstr.iscnst = false;
    unify_value(mach, argv[2], newstr);
  }


  private final void body_split_3(KL1Machine mach, String method,
				  KL1Object argv[]) {
    shallow();
    int size = index;
    int split_point[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, split_point, 0, 0, size+1)
       == null)
      return;
    int lower_size = size-split_point[0];
    char upper_body[] = new char[split_point[0]];
    char lower_body[] = new char[lower_size];
    GString upper = new GString();
    GString lower = new GString();
    System.arraycopy(body, 0, upper_body, 0, split_point[0]);
    System.arraycopy(body, split_point[0], lower_body, 0, lower_size);
    upper.next = lower.next = null;
    upper.index = split_point[0];
    lower.index = lower_size;
    upper.iscnst = lower.iscnst = false;
//  upper.ismbdiff = lower.ismbdiff = false;
    upper.body = upper_body;
    lower.body = lower_body;
    unify_value(mach, argv[1], upper);
    unify_value(mach, argv[2], lower);
  }


  private final void body_join_2(KL1Machine mach, String method,
				 KL1Object argv[]) {
    shallow();
    int size1 = index;
    KL1Object anotherq;
    if((anotherq = argv[0].deref()) instanceof Var) {
      suspend_generic_goal(mach, (Var)anotherq, method, argv);
      return;
    }
    GString another = (GString)anotherq;
    if (!(another instanceof GString)) {
      mach.debug_print("### " + another.print() + " ###\n");
      mach.fatal("Illegal argument to string join");
    }
    another.shallow();
    int size2 = another.index;
    int newsize = size1+size2;
    char newbody[] = new char[newsize];
    shallow();
    another.shallow();
    GString newstr = new GString();
    System.arraycopy(body, 0, newbody, 0, size1);
    System.arraycopy(another.body, 0, newbody, size1, size2);
    newstr.next = null;
    newstr.index = newsize;
    newstr.iscnst = false;
    newstr.body = newbody;
    unify_value(mach, argv[1], newstr);
  }


  private final void body_search__character_4(KL1Machine mach, String method,
					      KL1Object argv[]) {
    shallow();
    int start[] = new int[1], end[] = new int[1], code[] = new int[1];
    if(set_intarg_within_range(mach, method, argv, start, 0, 0, index) == null ||
       set_intarg_within_range(mach, method, argv, end, 1, 0, index) == null ||
       set_intarg_within_range(mach, method, argv, code, 2, 0, 256) == null)
      return;
    if (start[0] <= end[0]) {
      for (int k=start[0]; k<=end[0]; k++) {
	if (body[k] == (char)code[0]) {
	  unify_value(mach, argv[3], new IntAtom(k));
	  return;
	}
      }
    } else {
      for (int k=start[0]; k>=end[0]; k--) {
	if (body[k] == (char)code[0]) {
	  unify_value(mach, argv[3], new IntAtom(k));
	  return;
	}
      }
    }
    unify_value(mach, argv[3], new IntAtom(-1));
  }


  // Generic Method Table
  public void generic(KL1Machine mach, String method, KL1Object argv[]) {
    if(method == functor_string_2)
      body_string_2(mach, method, argv);
    else if(method == functor_element_2)
      body_element_2(mach, method, argv);
    else if(method == functor_size_1)
      body_size_1(mach, method, argv);
    else if(method == functor_element__size_1)
      body_element__size_1(mach, method, argv);
    else if(method == functor_set__element_3)
      body_set__element_3(mach, method, argv);
    else if(method == functor_split_3)
      body_split_3(mach, method, argv);
    else if(method == functor_join_2)
      body_join_2(mach, method, argv);
    else if(method == functor_search__character_4)
      body_search__character_4(mach, method, argv);
    else
      mach.fatal("undefined method");
  }


  // guard generic methods
  private KL1Object guard_element_2(KL1Object argv[]) {
    shallow();
    int position[] = new int[1];
    KL1Object ret;
    if((ret = set_gintarg_within_range(position, argv[0], 0, index))
       != GDObj.SUCCESS)
      return ret;
    argv[1] = new IntAtom(body[position[0]]);
    return GDObj.SUCCESS;
  }


  private boolean guard_string_2(KL1Object argv[]) {
    shallow();
    argv[0] = new IntAtom(index);
    argv[1] = new IntAtom(ELEMSIZE);
    return true;
  }


  private static int compare_two_strings(GString s1, GString s2) {
    s1.shallow();
    int size1 = s1.index;
    s2.shallow();
    int size2 = s2.index;
    int minsize = ((size1 < size2) ? size1 : size2);
    if (s1 != null) {
      // s1 and s2 are different versions of the same string
      for (int k=0; k<minsize; k++) {
	s1.shallow();
	char c = s1.body[k];
	s2.shallow();
	if (c != s2.body[k]) {
	  return ((c < s2.body[k]) ? -(k+1) : k+1);
	}
      }
    } else {
      for (int k=0; k<minsize; k++) {
	if (s1.body[k] != s2.body[k]) {
	  return ((s1.body[k] < s2.body[k]) ? -(k+1) : k+1);
	}
      }
    }
    if (size1 != size2) {
      return ((size1<size2) ? -(size1+1) : size2+1);
    } else {
      return 0;
    }
  }


  private boolean guard_less__than_1(KL1Object argv[]) {
    if (!(argv[0] instanceof GString))
      return false;
    GString other = (GString)argv[0];
    int cmp = compare_two_strings(this, other);
    return (cmp < 0);
  }


  private boolean guard_not__less__than_1(KL1Object argv[]) {
    if (!(argv[0] instanceof GString))
      return false;
    GString other = (GString)argv[0];
    int cmp = compare_two_strings(this, other);
    return (cmp >= 0);
  }


  private boolean guard_estring_3(KL1Object argv[]) {
    shallow();
    int size = index;
    if(((IntAtom)argv[0]).value != size ||
       ((IntAtom)argv[1]).value != ELEMSIZE)
      return false;
    KL1Object tmp = argv[2];
    for(int k=0; k<size; k++, tmp = ((Cons)tmp).cdr) {
      if(((IntAtom)((Cons)tmp).car).value != body[k])
	return false;
    }
    return true;
  }


  public KL1Object ggeneric(String method, KL1Object argv[]) {
    boolean ret;
    if(method == functor_element_2)
      return guard_element_2(argv);
    else if(method == functor_string_2)
      ret = guard_string_2(argv);
    else if(method == functor_less__than_1)
      ret = guard_less__than_1(argv);
    else if(method == functor_not__less__than_1)
      ret = guard_not__less__than_1(argv);
    else if(method == functor_estring_3)
      ret = guard_estring_3(argv);
    else
      ret = false;
    return ret ? GDObj.SUCCESS : GDObj.FAILURE;
  }


  public String print() {
    String s = "\"";
    shallow();
/*
    int size = index;
    int limit = (size > g_length ? g_length : size);
    for (int k = 0; k < limit; k++) {
      int c = body[k];
      if (!isgraph(c) && c != ' ') {
	s += ("\\"
	      + '0'+((c>>6)&3)
	      + '0'+((c>>3)&7)
	      + '0'+(c&7));
      } else {
	if (c == '\\' || c == '"')
	  s += "\\";
	s += c;
      }
    }
    if (limit != size)
      s += "..";
*/
    s += body;	// debug ???
    s += "\"";
    return s;
  }


  public IntAtom compare(GDObj that_) {
    GString that = (GString)that_;
    shallow();
    int size1 = index;
    that.shallow();
    int size2 = index;
    int limit = (size1 <= size2 ? size1 : size2);
    for (int k = 0; k < limit; k++) {
      shallow();
      char elem = body[k];
      that.shallow();
      if (elem != that.body[k])
	return new IntAtom(elem > that.body[k] ? k+1 : -(k+1));
    }
    if (size1 != size2)
      return new IntAtom(size1 >= size2 ? size1+1 : -(size2+1));
    else
      return new IntAtom(0);
  }


  public IntAtom hash() {
    shallow();
    int size = index;
    if (size == 0) {
      return new IntAtom(0);
    } else {
      return
	new IntAtom(0x813 * body[0] +
		    0x425 * body[size>>1] +
		    0x3c9 * body[size-1]);
    }
  }


  // new_string function
  // ???
  public static KL1Object byte_string_new(KL1Machine mach, KL1Object argv[]) {
//  if (argv.length != 1)
//    error_in_new(mach, "Too few or too many arguments");
    KL1Object init;
    if((init = argv[0].deref()) instanceof Var)
      return suspend_new(mach, (Var)init, GString.class, argv);
    char body[] = null;
    KL1Object elem;
    int size = 0;
    if (init instanceof IntAtom) {
      size = ((IntAtom)init).value;
      if (size < 0)
	error_in_new(mach, "Negative size specified");
      body = new char[size];
    } else if (init == SymAtom.nil || init instanceof Cons) {
      for (size = 0; ; size++) {
	if (init == SymAtom.nil)
	  break;
	elem = ((Cons)init).car;
	if((elem = elem.deref()) instanceof Var)
	  return suspend_new(mach, (Var)elem, GString.class, argv);
	if (!(elem instanceof IntAtom) || ((IntAtom)elem).value < 0
	    || 256 <= ((IntAtom)elem).value) {
	  error_in_new(mach, "Illegal parameter");
	}
	init = ((Cons)init).cdr;
	if((init = init.deref()) instanceof Var)
	  return suspend_new(mach, (Var)init, GString.class, argv);
	if (init != SymAtom.nil && !(init instanceof Cons))
	  error_in_new(mach, "Illegal parameter");
      }
      init = argv[0];
      body = new char[size];
      for (int k=0; k<size; k++) {
	if((init = init.deref()) instanceof Var)
	  return suspend_new(mach, (Var)init, GString.class, argv);
	elem = ((Cons)init).car;
	if((elem = elem.deref()) instanceof Var)
	  return suspend_new(mach, (Var)elem, GString.class, argv);
	body[k] = (char)((IntAtom)elem).value;
	init = ((Cons)init).cdr;
      }
    } else {
      error_in_new(mach, "Illegal parameter");
    }
    return new GString(body, false);
  }


/*
q gd_new_string(size,g_allocp)
     long size;
     q *g_allocp;
{
  q argv[1];
  argv[0] = makeint(size);
  return byte__string_g_new(1,argv,g_allocp);
}


q gd_list_to_string(list,g_allocp)
     q list;
     q *g_allocp;
{
  q argv[2];
  argv[0] = list;
  return byte__string_g_new(1,argv,g_allocp);
}


q convert_c_string_to_klic_string(cstr,g_allocp)
     char *cstr;
     q *g_allocp;
{
  q argv[1];
  q str;
  long len = strlen(cstr);
  argv[0] = makeint(len);
  str = byte__string_g_new(1,argv,g_allocp);
  if (!G_ISREF(str)) {
    BCOPY(cstr, ((struct byte_string_object *)functorp(str)).body, len);
  }
  return str;
}


q convert_binary_c_string_to_klic_string(cstr,len,g_allocp)
     char *cstr;
     long len;
     q *g_allocp;
{
  q argv[1];
  q str;
  argv[0] = makeint(len);
  str = byte__string_g_new(1,argv,g_allocp);
  if (!G_ISREF(str)) {
    BCOPY(cstr, ((struct byte_string_object *)functorp(str)).body, len);
  }
  return str;
}


char *convert_klic_string_to_c_string(s)
     q s;
{
  extern char *malloc_check();
  struct byte_string_object *str =
    (struct byte_string_object *)functorp(s);
  char *cstr;
  Shallow(str);
  cstr = (char *)malloc_check(str.index+1);
  BCOPY((char *)str.body, cstr, str.index);
  cstr[str.index] = '\0';
  return cstr;
}
*/

  public static KL1Object _new(KL1Machine mach, KL1Object argv[]) {
    if(argv.length != 2)
      error_in_new(mach, "Too few or too many arguments");
    KL1Object elemsize;
    if((elemsize = argv[1].deref()) instanceof Var)
      return suspend_new(mach, (Var)elemsize, GString.class, argv);
    if(!(elemsize instanceof IntAtom))
      error_in_new(mach, "Non-integer parameter");
    if(((IntAtom)elemsize).value != ELEMSIZE)
      error_in_new(mach, "Only byte strings are supported now");
//  return byte_string_new(mach, new KL1Object[]{ argv[0] });
    return byte_string_new(mach, argv);	// ???
  }


  /**
   * Constructs a string object.
   */
  public GString() {
  }


  /**
   * Constructs a string object.
   *
   * @param  body    a string.
   */
  public GString(String body) {
    this(body, false);
  }


  /**
   * Constructs a string object.
   *
   * @param  body    a string.
   * @param  iscnst  whether or not this is a constant.
   */
  public GString(String body, boolean iscnst) {
    next = null;
    this.iscnst = iscnst;
    if(body != null) {
      index = body.length();
      this.body = body.toCharArray();
    } else {
      index = 0;
      this.body = null;
    }
  }


  /**
   * Constructs a string object.
   *
   * @param  body    a string.
   */
  public GString(char body[]) {
    this(body, false);
  }


  /**
   * Constructs a string object.
   *
   * @param  body    a string.
   * @param  iscnst  whether or not this is a constant.
   */
  public GString(char body[], boolean iscnst) {
    next = null;
    this.iscnst = iscnst;
    if(body != null) {
      index = body.length;
      this.body = new char[index];
      System.arraycopy(body, 0, this.body, 0, index);
    } else {
      index = 0;
      this.body = null;
    }
  }


  public String toString() {
    shallow();
    return String.valueOf(body);
  }


  public int string_size() {
    shallow();
    return index;
  }


  // Interface with builtin
  public IntAtom size_of_string() {
    shallow();
    return new IntAtom(index);
  }


  public KL1Object element_of_string(IntAtom k) {
    shallow();
    if(k.value < 0 || index <= k.value) {
      return null;
    } else {
      return new IntAtom(body[k.value]);
    }
  }
}
