/*
 * GIO.java -- file I/O
 *
 * (C)1993, 1994, 1995 Institute for New Generation Computer Technology
 *	(Read COPYRIGHT for detailed information.) 
 *
 * (C)1996, 1997 Japan Information Processing Development Center
 *      (Read COPYRIGHT-JIPDEC for detailed information.)
 *
 * Copyright (C) 1998,1999 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * $Id: GIO.java,v 1.3 1999-03-06 23:38:12+09 satoshi Exp $
 */

package kl1.lang;

import java.io.*;

/**
 * This class represents a file I/O object.
 *
 * @author Satoshi KURAMOCHI
 */
public final class GIO extends GCObj {
  static { name = "file_io"; }

  private int linecount;
  private KL1Object inname, outname;
  private KL1Object stream;
	  // saved input stream for suspension by some other reasons
  private InputStream infile;
  private PrintStream outfile;

  private static final String functor_getc_1 = "getc_1".intern();
  private static final String functor_ungetc_1 = "ungetc_1".intern();
  private static final String functor_fread_2 = "fread_2".intern();
  private static final String functor_linecount_1 = "linecount_1".intern();
  private static final String functor_putc_1 = "putc_1".intern();
  private static final String functor_fwrite_2 = "fwrite_2".intern();
  private static final String functor_fwrite_1 = "fwrite_1".intern();
  private static final String functor_feof_1 = "feof_1".intern();
  private static final String functor_fseek_3 = "fseek_3".intern();
  private static final String functor_ftell_1 = "ftell_1".intern();
  private static final String functor_fflush_1 = "fflush_1".intern();
  private static final String functor_fclose_1 = "fclose_1".intern();
  private static final String functor_sync_1 = "sync_1".intern();


  private KL1Object message = null;


  private int do_flush() {
    outfile.flush();
    return outfile.checkError() ? -1 : 0;
  }


  private void CheckInput(KL1Machine mach) {
    if (infile == null) message_error(mach);
    if (outfile != null) do_flush();
  }


  private void CheckOutput(KL1Machine mach) {
    if (outfile == null) message_error(mach);
  }


  // basic method definitions
  public KL1Object active_unify(KL1Machine mach, KL1Object that) {
    if(stream != null) {
      that = stream;
      stream = null;
    }
    while(true) {
      if(that instanceof Cons) {
	message = ((Cons)that).car.deref();
	if(message instanceof Var) {
	  mach.rest_of_stream = message;
	  stream = that;
	  return GDObj.SUCCESS;
	}
	if(message instanceof IntAtom) {
	  int c = ((IntAtom)message).value;
	  mach.klic_putc(c, outfile);
//	    mach.fatal("putc failed");
	} else if(message instanceof Functor) {
	  /*** Input Messages ***/
	  if(((Functor)message).functor == functor_getc_1) {
	    CheckInput(mach);
	    int c = mach.klic_getc(infile);
	    if(c=='\n') { linecount++; }
	    unify_value(mach, ((Functor)message).args[0], new IntAtom(c));
	  } /*else if(((Functor)message).functor == functor_ungetc_1) {
	    CheckInput(mach);
	    KL1Object temp = ((Functor)message).args[0].deref();
	    if(temp instanceof Var) {
	      mach.rest_of_stream = temp;
	      stream = that;
	      return GDObj.SUCCESS;
	    }
	    if(!(temp instanceof IntAtom))
	      message_error(mach);
	    int c = ((IntAtom)temp).value;
	    if(c=='\n') { linecount--; }
	    ungetc(c, infile);
	  }*/ else if(((Functor)message).functor == functor_fread_2) {
	    CheckInput(mach);
	    KL1Object temp = ((Functor)message).args[0].deref();
	    if(temp instanceof Var) {
	      mach.rest_of_stream = temp;
	      stream = that;
	      return GDObj.SUCCESS;
	    }
	    if(!(temp instanceof IntAtom))
	      message_error(mach);
	    int n = ((IntAtom)temp).value;
	    if(n<0) message_error(mach);
	    byte buf[] = new byte[n];
	    n = mach.klic_fread(buf, infile);
	    
	    for (int k = 0; k < n; k++) {
	      if(buf[k] == '\n') { linecount++; }
	    }
	    GString string = new GString(new String(buf));
	    unify_value(mach, ((Functor)message).args[1], string);
	  } else if(((Functor)message).functor == functor_linecount_1) {
	    CheckInput(mach);
	    unify_value(mach, ((Functor)message).args[0],
			new IntAtom(linecount));
	  }
	  /*** Output Messages ***/
	  else if(((Functor)message).functor == functor_putc_1) {
	    CheckOutput(mach);
	    KL1Object temp = ((Functor)message).args[0].deref();
	    if(temp instanceof Var) {
	      mach.rest_of_stream = temp;
	      stream = that;
	      return GDObj.SUCCESS;
	    }
	    if(!(temp instanceof IntAtom))
	      message_error(mach);
	    int c = ((IntAtom)temp).value;
	    mach.klic_putc(c, outfile);
//	      mach.fatal("putc failed");
	  } else if(((Functor)message).functor ==  functor_fwrite_2) {
	    CheckInput(mach);
	    KL1Object temp = ((Functor)message).args[0].deref();
	    if(temp instanceof Var) {
	      mach.rest_of_stream = temp;
	      stream = that;
	      return GDObj.SUCCESS;
	    }
	    if(!(temp instanceof GString))
	      message_error(mach);
	    GString str = (GString)temp;
	    int written = mach.klic_fwrite(str.toString(), outfile);
	    unify_value(mach, ((Functor)message).args[1], new IntAtom(written));
	  } else if(((Functor)message).functor == functor_fwrite_1) {
	    CheckOutput(mach);
	    KL1Object temp = ((Functor)message).args[0].deref();
	    if(temp instanceof Var) {
	      mach.rest_of_stream = temp;
	      stream = that;
	      return GDObj.SUCCESS;
	    }
	    if(!(temp instanceof GString))
	      message_error(mach);
	    GString str = (GString)temp;
	    mach.klic_fwrite(str.toString(), outfile);	// ???
	  }
	  /*** Common Messages ***/
	  else if(((Functor)message).functor == functor_feof_1) {
	    boolean iseof;
	    if(infile != null)
	      iseof = mach.feof(infile);
	    else
	      iseof = mach.feof(outfile);
	    unify_value(mach, ((Functor)message).args[0],
			new IntAtom(iseof ? 1 : 0));
	  } /*else if(((Functor)message).functor == functor_fseek_3) {
	    KL1Object temp = ((Functor)message).args[0].deref();
	    if(temp instanceof Var) {
	      mach.rest_of_stream = temp;
	      stream = that;
	      return GDObj.SUCCESS;
	    }
	    if(!(temp instanceof IntAtom))
	      message_error(mach);
	    int offset = ((IntAtom)temp).value;
	    temp = ((Functor)message).args[1].deref();
	    if(temp instanceof Var) {
	      mach.rest_of_stream = temp;
	      stream = that;
	      return GDObj.SUCCESS;
	    }
	    if(!(temp instanceof IntAtom))
	      message_error(mach);
	    int whence = ((IntAtom)temp).value;
	    Object file = (infile == null ? outfile : infile);
	    int result = fseek(file, offset, whence);
	    unify_value(mach, ((Functor)message).args[2], new IntAtom(result));
	  }*/ /*else if(((Functor)message).functor == functor_ftell_1) {
	    int result;
	    Object file = (infile == null ? outfile : infile);
	    result = ftell(file);
	    unify_value(mach, ((Functor)message).args[0], new IntAtom(result));
	  }*/ else if(((Functor)message).functor == functor_fflush_1) {
	    int result = do_flush();
	    unify_value(mach, ((Functor)message).args[0], new IntAtom(result));
	  } else if(((Functor)message).functor == functor_fclose_1) {
	    boolean result;
	    if(outfile != null && outfile != System.out &&
	       outfile != System.err) {
	      outfile.close();
	      result = outfile.checkError();
	      outfile = null;
	    } else
	      result = false;
	    if(result == false) {
	      if(infile != null && infile != System.in) {
		try {
		  infile.close();
		} catch(IOException e) {
		  result = true;
		}
		infile = null;
	      }
	    }
	    unify_value(mach, ((Functor)message).args[0],
			new IntAtom(result ? -1 : 0));
	  } else if(((Functor)message).functor == functor_sync_1) {
	    unify_value(mach, ((Functor)message).args[0], new IntAtom(0));
	  } else {
	    message_error(mach);
	  }
	} else {
	  message_error(mach);
	}
	that = ((Cons)that).cdr;
	continue;
      } else if(that == SymAtom.nil) {
	if(infile != null && infile != System.in) {
	  try {
	    infile.close();
	  } catch(IOException e) {}
	}
	if(outfile != null && outfile != System.out && outfile != System.err)
	  outfile.close();
	mach.rest_of_stream = null;
	return GDObj.SUCCESS;
      } else if(that instanceof Var) {
	KL1Object temp = ((Var)that).refers;
	if(temp instanceof Var && ((Var)temp).refers == that) {
	  stream = null;
	  mach.rest_of_stream = that;
	  return GDObj.SUCCESS;
	} else {
	  that = temp;
	  continue;
	}
      }
      break;
    }
    return null;	// ???
  }


  private void message_error(KL1Machine mach) {
    String inname_ = inname.toString();
    String outname_ = outname.toString();
    mach.debug_print("Illegal message " + message.print()
		     + " to I/O stream for " + inname_
		     + ((inname_.equals("") && outname_.equals("") &&
			 inname_.compareTo(outname_) != 0) ? "/" : "")
		     + outname_ + "\n");
    mach.fatal("message error");
  }


  public String print() {
    return "$$FILE I/O$";
  }


  public static KL1Object _new(KL1Machine mach, KL1Object argv[]) {
    /*
       Arguments are:
       0: Input file object
       1: Its name
       2: Output file object
       3: Its name
       */
    if(argv.length != 4)
      error_in_new(mach, "Arity mismatch");

    KL1Object infile = argv[0].deref();
    if(infile instanceof Var)
      return suspend_new(mach, (Var)infile, GIO.class, argv);
    if(infile != SymAtom.nil && !(infile instanceof GObject))
      mach.fatal("First argument for file creation is not a pointer object");
    KL1Object inpath = argv[1].deref();
    if(inpath instanceof Var)
      return suspend_new(mach, (Var)inpath, GIO.class, argv);

    KL1Object outfile = argv[2].deref();
    if(outfile instanceof Var)
      return suspend_new(mach, (Var)outfile, GIO.class, argv);
    if(outfile != SymAtom.nil && !(outfile instanceof GObject))
      mach.fatal("Third argument for file creation is not a pointer object");
    KL1Object outpath = argv[3].deref();
    if(outpath instanceof Var)
      return suspend_new(mach, (Var)outpath, GIO.class, argv);

    GIO newobj = new GIO();
    newobj.infile =
      (infile == SymAtom.nil) ? null : (InputStream)((GObject)infile).object;
    newobj.outfile =
      (outfile == SymAtom.nil) ? null : (PrintStream)((GObject)outfile).object;
    newobj.linecount = 0;
    newobj.inname = inpath;
    newobj.outname = outpath;
    newobj.stream = null;
    Var var = make_hook_var(newobj);
    return var;
  }


  /**
   * Constructs a file I/O object.
   */
  public GIO() {
  }
}
