/*
 * GCode.java -- predicate
 *
 * (C)1993, 1994, 1995 Institute for New Generation Computer Technology
 *	(Read COPYRIGHT for detailed information.) 
 *
 * (C)1996, 1997 Japan Information Processing Development Center
 *      (Read COPYRIGHT-JIPDEC for detailed information.)
 *
 * Copyright (C) 1998,1999 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * $Id: GCode.java,v 1.2 1999-03-06 04:39:29+09 satoshi Exp $
 */

package kl1.lang;

/**
 * This class represents a predicate object.
 *
 * @author Satoshi KURAMOCHI
 */
public final class GCode extends GDObj {
  static { name = "predicate"; }

  private Predicate pdesc;	// atom tag
  private GModule module_obj;
  private String predicate_name;

  private static final String functor_name_1 = "name".intern();
  private static final String functor_module_1 = "module".intern();
  private static final String functor_arity_1 = "arity".intern();
  private static final String functor_apply_1 = "apply".intern();
  private static final String functor_vector_1 = "vector".intern();
  private static final String functor_element_2 = "element".intern();
  private static final String functor_predicate_0 = "predicate".intern();


  // basic method definitions
  public KL1Object gunify(GDObj that) {
    if (!(that instanceof GCode) || !pdesc.equals(((GCode)that).pdesc))
      return GDObj.FAILURE;
    else
      return GDObj.SUCCESS;
  }


  public void unify(KL1Machine mach, KL1Object that) {
    if(mach.UNIFYDEBUG)
      mach.print("Unify with " + this.print() + "," + that.print());
    while (that instanceof Var) {
      KL1Object temp = ((Var)that).refers;
      if (temp == that) { // that is undef cell
	((Var)that).refers = this;
	return;
      } else {
	if(temp instanceof Var && ((Var)temp).refers == that) {
	  mach.resume_goals(temp, this);
	  return;
	}
      }
      that = temp;
    }
    // that is bound
    if (!(that instanceof GCode) || !pdesc.equals(((GCode)that).pdesc))
      unify_fail(mach);
    else
      return;
  }



  // Generic method
  private final void body_module_1(KL1Machine mach, String method,
				   KL1Object argv[]) {
    argv[0].shallow_unify(mach, module_obj);
  }


  private final void body_name_1(KL1Machine mach, String method,
				 KL1Object argv[]) {
    argv[0].shallow_unify(mach, new SymAtom(predicate_name));
  }


  private final void body_arity_1(KL1Machine mach, String method,
				  KL1Object argv[]) {
    argv[0].shallow_unify(mach, new IntAtom(pdesc.arity));
  }


  private final void body_apply_1(KL1Machine mach, String method,
				  KL1Object argv[]) {
    KL1Object argv0 = argv[0].deref();
    if(argv0 instanceof Var) {
      suspend_generic_goal(mach, (Var)argv0, method, argv);
      return;
    }
    if (!(argv0 instanceof GVector))
      error_in_method(mach, "Illegal argument specification", "apply");
    GVector func = (GVector)argv0;

    KL1Object args[] = new KL1Object[2];
    func.ggeneric(functor_vector_1, args);
    int size = ((IntAtom)args[0]).value;
    if (pdesc.arity != size)
      error_in_method(mach, "Arity mismatch", "apply");
    Goalrec goal = new Goalrec(null, pdesc, null);
    goal.args = new KL1Object[size];
    for(int i = 0; i < size; ++i){
      args[0] = new IntAtom(i);
      func.ggeneric(functor_element_2, args);
      goal.args[i] = args[1];
    }
    mach.resume_same_prio(goal);
  }


  private final void body_call(KL1Machine mach, String method,
			       KL1Object argv[]) {
    int size = argv.length;
    Goalrec goal = new Goalrec(null, pdesc, null);
    goal.args = new KL1Object[size];
    for(int i = 0; i < size; ++i)
      goal.args[i] = argv[i];
    mach.resume_same_prio(goal);
  }


  public void generic(KL1Machine mach, String method, KL1Object argv[]) {
    if(method == functor_module_1)
      body_module_1(mach, method, argv);
    else if(method == functor_name_1)
      body_name_1(mach, method, argv);
    else if(method == functor_arity_1)
      body_arity_1(mach, method, argv);
    else if(method == functor_apply_1)
      body_apply_1(mach, method, argv);
    else if(method.startsWith("call_"))		// ???
      body_call(mach, method, argv);
    else
      mach.fatal("undefined method: " + method);
  }


  public String print() {
    return ("predicate#(" + module_obj.name_ + ":" + predicate_name + "/"
	    + pdesc.arity + ")");
  }


  private KL1Object guard_arity_1(KL1Object argv[]) {
    argv[0] =  new IntAtom(pdesc.arity);
    return GDObj.SUCCESS;
  }


  private KL1Object guard_predicate_0(KL1Object argv[]) {
    return (pdesc != null) ? GDObj.SUCCESS : GDObj.FAILURE;
  }


  public KL1Object ggeneric(String method, KL1Object argv[]) {
    if(method == functor_predicate_0)
      return guard_predicate_0(argv);
    else if(method == functor_arity_1)
      return guard_arity_1(argv);
    else
      return GDObj.FAILURE;
  }


  // new predicate
  public static KL1Object _new(KL1Machine mach, KL1Object argv[]) {
    if (argv.length != 3)
      error_in_new(mach, "Too few or too many arguments");

    KL1Object argv0 = argv[0].deref();
    if(argv0 instanceof Var)
      return suspend_new(mach, (Var)argv0, GCode.class, argv);
    if(!(argv0 instanceof GModule))
      error_in_new(mach, "First parameter is not a module");
    GModule module_obj = (GModule)argv0;

    KL1Object argv1 = argv[1].deref();
    if(argv1 instanceof Var)
      return suspend_new(mach, (Var)argv1, GCode.class, argv);
    if(!(argv1 instanceof SymAtom))
      error_in_new(mach, "Non-atom parameter");
    String predname = ((SymAtom)argv1).name;

    KL1Object argv2 = argv[2].deref();
    if(argv2 instanceof Var)
      return suspend_new(mach, (Var)argv2, GCode.class, argv);
    if(!(argv2 instanceof IntAtom))
      error_in_new(mach, "Non-Integer parameter");
    int arity = ((IntAtom)argv2).value;

    Predicate predaddr =
      mach.locate_predicate_in_module(module_obj.name_, predname, arity);
    if (predaddr == null)
      error_in_new(mach, ("Nonexistent predicate " + module_obj.name_ + ":"
			  + predname + "/" + arity));

    return new GCode(predaddr, module_obj, predname);
  }


  /**
   * Constructs a predicate object.
   *
   * @param  pdesc  the predicate descriptor
   * @param  module_obj  the module where the predicate is defined.
   * @param  predicate_name  the name of the predicate.
   */
  public GCode(Predicate pdesc, GModule module_obj, String predicate_name) {
    this.pdesc = pdesc;
    this.module_obj = module_obj;
    this.predicate_name = predicate_name;
  }
}
